/*
 * Macintosh interface routines for Macraigor Systems PCI card
 *  2000 Apple Computer, Inc.
 *
 * 3 Mar 2000 - release to Moto as ver 1.0 (aek)
 */
 
#include 	"jtagdefs.h"
#define		CURRENT_LIB_VERSION "1.0"

/*
 * Mac specific includes
 */
#include <NameRegistry.h>
#include <Pci.h>
#include <Gestalt.h>
#include <Timer.h>

/*
 * Internal function prototypes
 */
static pciJtagRegs *findPCICard(void);
static long byteCount(unsigned long length);
static unsigned char readLastBit(void);
static int stateMove( JTAG_STATE_TYPE newState, unsigned char lastData);
static int doTMS(unsigned char value, unsigned char count);
static int timedOut(long long timeoutTime);
static long long primeTimeoutTime(int timeout);
static long long currentTime(void);
static void csrTimeout(char *rtn);

/*
 * Internal static data 
 */
static volatile pciJtagRegs *regPtr = 0;

static unsigned char shadow_gpOut;
static unsigned char shadow_cmdReg;
static JTAG_STATE_TYPE currentState;
static int jtagError = 0;
static char *failedRoutineName;

static char *libraryVersion = CURRENT_LIB_VERSION;

static unsigned long		aaplAdrs[4];
static RegPropertyValueSize	adrSize = sizeof(aaplAdrs);
static char					slotName[32];
static RegPropertyValueSize	nameSize = sizeof(slotName);
static unsigned long		assignedAdrs[4][5];
static RegPropertyValueSize	assignedAdrsSize = sizeof(assignedAdrs);

/*
 * External functions
 */

void Jtag_InitializeController(unsigned short port, 
							   unsigned long speed)
{
	if(port){
	};		/* since I don't have any idea what port should be, ignore it */
	
	jtagError = 0;
	if((regPtr = findPCICard()) != 0){

	/*
	 * initialize command register
	 */
	 shadow_cmdReg = NORMAL_MODE | CHIP_IN_RESET | ( 7 << NSCNT_SHIFT) | ((speed - 1) & 0x7);
	 shadow_gpOut  = TRST_INACTIVE;

	 regPtr->cmdStat = shadow_cmdReg;		/* reset lattice chip */
	 asm (eieio);

	 shadow_cmdReg = NORMAL_MODE | CHIP_NOT_IN_RESET | ( 7 << NSCNT_SHIFT) | ((speed - 1) & 0x7);
	 regPtr->cmdStat = shadow_cmdReg;		/* take chip out of reset */
	 regPtr->gpOut   = shadow_gpOut;
	 asm(eieio);

	}
	else
	 jtagError = CARD_NOT_FOUND;
}

void Jtag_ResetTap(bool assert_trst, 
				   JTAG_STATE_TYPE state_e)
{

	int i, ok;

	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	currentState = TEST_LOGIC_RESET;

	if(assert_trst){			/* Perform state move with trst active, then make it inactive */
		shadow_gpOut |= TRST_ACTIVE;
		regPtr->gpOut = shadow_gpOut;
		asm(eieio);
	}

	for(i = 0; i < 5; i++){
		ok = stateMove(TEST_LOGIC_RESET, 0);
	}
	
	if(assert_trst){
		shadow_gpOut  &= ~TRST_ACTIVE;
		regPtr->gpOut = shadow_gpOut;
		asm(eieio);
	}
	
	stateMove(state_e, 0);
}

bool Jtag_InError(void)
{
	return(jtagError);
}

void Jtag_ScanOut(SHIFT_REGISTER_TYPE jtag_register_e,
				  int length,
				  SCAN_DATA_TYPE out_data[],
				  JTAG_STATE_TYPE state_e
				 )
{
    int 			ok;
    unsigned int 	bits_remaining;
    unsigned char 	last_bit;
    long 			first_byte;
    long 			last_byte;
    long 			index;
	long long		timeoutTime;  
	
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
    if (jtag_register_e == DATA_REGISTER)
        ok = stateMove(SHIFT_DR, 0);
    else
        ok = stateMove(SHIFT_IR, 0);

    /* TRAVERSE ARRAY : FIRST TO LAST */
    first_byte = 0;
    last_byte = byteCount(length) - 1;

	timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
     if(timeoutTime <= currentTime()) goto timeoutFail;
	}

    for (index = first_byte; index != last_byte; index++)
    {
        /* Wait for data register empty */
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
    	while (regPtr->cmdStat & (DATA_OUT_FULL)){
    	 if(timeoutTime <= currentTime()) goto timeoutFail;
		}
        /* Output next data byte to scan chain */    
        regPtr->IOLow = out_data[index];

    }
     
    /* Get bit count to send in last byte */
    bits_remaining = length & 0x0007;
    bits_remaining = (bits_remaining == 0) ? 8 : bits_remaining;

    /* Get last bit and save it for state move */
    last_bit = (out_data[index] >> (bits_remaining - 1)) & 0x01;

    /* Send all but the last bit */
    if (bits_remaining > 1)
    {
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

      	shadow_cmdReg &= ~NSCNT_MASK;
      	shadow_cmdReg |= ((bits_remaining - 2) << NSCNT_SHIFT) & NSCNT_MASK;
		regPtr->cmdStat = shadow_cmdReg;
		regPtr->IOLow	= out_data[index];
    }

    ok = stateMove(state_e, last_bit);
    
    return;
    
timeoutFail:
	csrTimeout("Jtag_ScanOut()");
}


void Jtag_ScanIn(SHIFT_REGISTER_TYPE jtag_register_e, int length, unsigned char in_data[], JTAG_STATE_TYPE state_e)
{
    int ok = 1;
    unsigned int bits_remaining;
    long first_byte;
    long last_byte;
    long long timeoutTime;
   	int i;

	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;

    /* Traverse array, first to last */
    first_byte = 0;
    last_byte = byteCount(length) - 1;

    if (jtag_register_e == DATA_REGISTER)
        ok = stateMove(SHIFT_DR, 0);
    else
        ok = stateMove(SHIFT_IR, 0);

    for (i = first_byte; i < last_byte; i++)
    {
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        /* Output fill value to scan chain */    
		regPtr->IOLow =  0;

		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}
               
        /* Read in next byte from scan chain */
        in_data[i] = regPtr->IOLow;
    }
  
    /* Get bit count of last byte */
    bits_remaining = length & 0x0007;
    bits_remaining = (bits_remaining == 0) ? 8 : bits_remaining;


    /* Send everything but the last bit */
    if (bits_remaining > 1)
    {
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        shadow_cmdReg &= ~NSCNT_MASK;
        shadow_cmdReg |= ((bits_remaining - 2) << NSCNT_SHIFT) & NSCNT_MASK;

		regPtr->cmdStat = shadow_cmdReg;

        /* Send stuff bit */
		regPtr->IOLow = 0;

		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        /* Read in last byte of data */
        in_data[i] = regPtr->IOLow;
    }

    ok = stateMove(state_e, 0);

    /* Or in the last input bit from input msb */
    in_data[i] = (in_data[i] >> 1) | readLastBit();

    /* Fix for last byte to be lsb shifted */
    in_data[i] = in_data[i] >> (8 - bits_remaining);
     
    return;
    
timeoutFail:
	csrTimeout("Jtag_ScanIn()");
}

/*
 * combined scanOut and scanIn
 */
void Jtag_ScanIO(SHIFT_REGISTER_TYPE jtag_register_e,
				 int length,
				 SCAN_DATA_TYPE out_data[],
				 SCAN_DATA_TYPE in_data[],
				 JTAG_STATE_TYPE state_e
				 )
{
    int ok = 1;
	unsigned int i;
    unsigned int bits_remaining;
    unsigned char last_bit;
    long first_byte;
    long last_byte;
    unsigned long long timeoutTime;

	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
    if (jtag_register_e == DATA_REGISTER)
        ok = stateMove(SHIFT_DR, 0);
    else
        ok = stateMove(SHIFT_IR, 0);

    /* Traverse array, first to last */
    first_byte = 0;
    last_byte  = byteCount(length) - 1;


    for (i = first_byte; i != last_byte; i++)
    {
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & DATA_OUT_FULL){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        /* Output next data byte to scan chain */    
        regPtr->IOLow = out_data[i];
		asm(eieio);
        
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        /* Read in next byte from scan chain */
        in_data[i] = regPtr->IOLow;
		asm(eieio);
    }
        
    /* Get bit count in last byte */
    bits_remaining = length & 0x0007;
    bits_remaining = (bits_remaining == 0) ? 8 : bits_remaining;

    /* Get last bit and save it for state move */
    last_bit = (out_data[last_byte] >> (bits_remaining - 1)) & 0x01;

    /* Send out all but last bit */
    if (bits_remaining > 1)
    {
		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & DATA_OUT_FULL){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        shadow_cmdReg &= ~NSCNT_MASK;
        shadow_cmdReg |= ((bits_remaining - 2) << NSCNT_SHIFT) & NSCNT_MASK;
		regPtr->cmdStat = shadow_cmdReg;
		asm(eieio);
		regPtr->IOLow = out_data[last_byte];
		asm(eieio);

		timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	    while (regPtr->cmdStat & DATA_IN_EMPTY){
	     if(timeoutTime <= currentTime()) goto timeoutFail;
		}

        in_data[last_byte] = regPtr->IOLow;
		asm(eieio);
    }

    ok = stateMove(state_e, last_bit);


    /* Or in the last bit from input msb */
    in_data[last_byte] = (in_data[last_byte] >> 1) | readLastBit();

    /* Fix for last byte to be lsb shifted */
    in_data[last_byte] = in_data[last_byte] >> (8 - bits_remaining);

    return;
    
timeoutFail:
	csrTimeout("Jtag_ScanIO()");
}


void Jtag_WritePin(bool assert,
				   OUTPUT_PIN_TYPE pin_e
				 )
{
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	if(assert){
		switch(pin_e){
		 case PIN_0PIN:
		  shadow_gpOut |= 1;
		  break;
		 case PIN_1PIN:
		  shadow_gpOut |= 2;
		  break;
		 case PIN_2PIN:
		  shadow_gpOut |= 4;
		  break;
		 case PIN_3PIN:
		  shadow_gpOut |= 8;
		  break;
		 case RESET_PIN:
		  shadow_gpOut |= 0x80;
		  break;
		}
		shadow_gpOut |= (0x01 << pin_e);
	}
	else{
		switch(pin_e){
		 case PIN_0PIN:
		  shadow_gpOut &= ~1;
		  break;
		 case PIN_1PIN:
		  shadow_gpOut &= ~2;
		  break;
		 case PIN_2PIN:
		  shadow_gpOut &= ~4;
		  break;
		 case PIN_3PIN:
		  shadow_gpOut &= ~8;
		  break;
		 case RESET_PIN:
		  shadow_gpOut &= ~0x80;
		  break;
		}
	}
	regPtr->gpOut = shadow_gpOut;
	asm(eieio);
}

/*
 * Set jtag clock divisor (0-7) in the cmd register
 * Assuming 33.33MHz PCI clock:
 * 1 = 8.3  MHz
 * 2 = 8.3  MHz
 * 3 = 5.2  MHz
 * 4 = 3.3  MHz
 * 5 = 1.8  MHz
 * 6 =   970 KHz
 * 7 =   500 KHz
 * 8 =     4 KHz
 */
void Jtag_SetSpeed( unsigned long speed )
{
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	shadow_cmdReg &= ~0x7;
	shadow_cmdReg |= (speed - 1 ) & 0x7;
	regPtr->cmdStat = shadow_cmdReg;
	asm(eieio);
	
}

void Jtag_EnterLoopBackMode()
{
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	shadow_cmdReg |= LOOPBACK_MODE;
	regPtr->cmdStat = shadow_cmdReg;
	asm(eieio);
}

void Jtag_ExitLoopBackMode()
{
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	shadow_cmdReg &= ~LOOPBACK_MODE;
	regPtr->cmdStat = shadow_cmdReg;
	asm(eieio);
}

void Jtag_SetTRST(bool assert)
{
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	if(assert){
		shadow_gpOut |= TRST_ACTIVE;
		regPtr->gpOut = shadow_gpOut;
	}
	else{
		shadow_gpOut  &= ~TRST_ACTIVE;
		regPtr->gpOut = shadow_gpOut;
	}
	asm(eieio);

}

void Jtag_WriteOutputs( unsigned char value)
{
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return;
	}
		
	jtagError = 0;
	value &= GP_OUTS_MASK;
	shadow_gpOut = (shadow_gpOut & ~GP_OUTS_MASK) | value;
	regPtr->gpOut = shadow_gpOut;
	asm(eieio);
}

static unsigned char
readLastBit(void)
{
	unsigned char	bitValue = 0;
	long long		timeoutTime;
		
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return 0;
	}
		
	timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	 if(timeoutTime <= currentTime()) goto timeoutFail;
	}
	
	bitValue = regPtr->cmdStat & LAST_DATA_IN_BIT;
	return bitValue;

timeoutFail:
	csrTimeout("readLastBit()");
	return 0;
}

static long
byteCount(unsigned long length)
{
	long numBytes;
	numBytes = length / 8;
	numBytes = ((length % 8) == 0) ? numBytes : numBytes + 1;
	return numBytes;
}

static int
doTMS(unsigned char value, unsigned char count)
{
    unsigned short val;
    long long timeoutTime;
    
    int ok = 1;
    
	if(regPtr == 0){
		 jtagError = CARD_NOT_FOUND;
		 return 0;
	}
		
    /* wait for data out to empty */
	timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	 if(timeoutTime <= currentTime()) goto timeoutFail;
	}
	
	/* Set the TMS register. This is endian dependent */
	val = (value << 8) | (count - 1);
	regPtr->TMSReg = val;
	asm(eieio);

	/* Read status register and wait for data in to be full */
	timeoutTime = primeTimeoutTime(COMMAND_TIMEOUT);
	while (regPtr->cmdStat & (DATA_OUT_FULL | DATA_IN_EMPTY)){
	 if(timeoutTime <= currentTime()) goto timeoutFail;
	}
	asm(eieio);

    return ok;

timeoutFail:
	csrTimeout("doTMS()");
	ok = 0;
	return ok;
}


static int 
stateMove( JTAG_STATE_TYPE newState, unsigned char lastData)
{	
	int ok = 1;
	
	/*
	 * TMS shift value to get from currentState to newState
	 */
    static const unsigned char tmsValue[6][6] = {

         /* TO           TLR   RTI   PDR   PIR   SDR   SIR  
         /* FROM TLR*/  {0x01, 0x00, 0x0a, 0x16, 0x02, 0x06},		/* TLR - Test Logic Reset */
         /*      RTI*/  {0x07, 0x00, 0x05, 0x0b, 0x01, 0x03},		/* RTI - Run Test Idle */
         /*      PDR*/  {0x1f, 0x03, 0x17, 0x2f, 0x01, 0x0f},		/* PDR - Pause DR */
         /*      PIR*/  {0x1f, 0x03, 0x17, 0x2f, 0x07, 0x01},		/* PIR - Pause IR */
         /*      SDR*/  {0x1f, 0x03, 0x01, 0x2f, 0x00, 0x00},		/* SDR - Select DR Scan */
         /*      SIR*/  {0x1f, 0x03, 0x17, 0x01, 0x00, 0x00}		/* SIR - Select IR Scan */
	};

	/*
	 * Number of TCKs needed to get from currentState to newState
	 */
    static const unsigned char tmsClocks[6][6] = {

         /* TO           TLR   RTI   PDR   PIR   SDR   SIR  
         /* FROM TLR*/  {1,    1,    5,    6,    4,    5},
         /*      RTI*/  {3,    1,    4,    5,    3,    4},
         /*      PDR*/  {5,    3,    6,    7,    2,    6},
         /*      PIR*/  {5,    3,    6,    7,    5,    2},
         /*      SDR*/  {5,    3,    2,    7,    0,    0},
         /*      SIR*/  {5,    4,    6,    2,    0,    0}
    };

    if (currentState >= MAX_JTAG_STATE)
    {
        /* Not in a known state, force Lattice chip to TEST_LOGIC_RESET */
        ok = doTMS(0xFF, 7);
        currentState = TEST_LOGIC_RESET;
    }

	/* go to new state */
	
    if (ok)
    {    
        ok = doTMS(tmsValue[currentState][newState] | (lastData << 7),
                   tmsClocks[currentState][newState]);

        currentState = newState;
    }

    return ok;
}

/*
 * try to find the PCI JTAG card, and return the register base
 * adr
 */
 
pciJtagRegs *findPCICard(void)
{

#define CARDNAME "pci152c,1"

	OSErr					err;
	long 					result;
	RegEntryID				entry;
	RegEntryIter			iter;
	Boolean 				done = false;
	RegEntryIterationOp		iterOp = kRegIterDescendants;
	int						i;

	/*
	 * Check that this is a PCI Mac
	 */
	if(Gestalt( gestaltNameRegistryVersion, &result) != noErr) goto fail;

	RegistryEntryIDInit(&entry);
	if(RegistryEntryIterateCreate( &iter ) != noErr) goto fail;

	/*
	 * search the name registry for a PCI card with the correct vendor ID
	 * we only support one card for now
	 */
	if((err = RegistryEntrySearch(&iter, iterOp, &entry, &done, "name", CARDNAME, sizeof(CARDNAME))) != noErr) goto fail;
	RegistryEntryIterateDispose(&iter);
	err = RegistryPropertyGet(&entry, "assigned-addresses", (void *)&assignedAdrs, &assignedAdrsSize);
	if(err != noErr) goto fail;
	err = RegistryPropertyGet(&entry, "AAPL,address", (void *)&aaplAdrs, &adrSize);
	if(err != noErr) goto fail;
	err = RegistryPropertyGet(&entry, "AAPL,slot-name", (void *)&slotName, &nameSize);
	if(err != noErr) goto fail;
	/*
	 * card was found, enable I/O and memory space access in PCI config reg at offset 4
	 */
	err = ExpMgrConfigWriteByte(&entry, (LogicalAddress)4, 3);
	if(err != noErr) goto fail;
	/*
	 * return the correct address space by checking he assigned-addresses property
	 */
	for(i=0; i < 4; i++){
		if(assignedAdrs[i][4] == 0x00000020){
		 return((pciJtagRegs *)aaplAdrs[i]);
		}
	}
fail:
	return(0);
}

static long long
primeTimeoutTime(int timeout)
{
	UnsignedWide currentTime;
	long long theTime;

	Microseconds(&currentTime);
	theTime = ((currentTime.hi  << 32) | currentTime.lo) + timeout;
	return theTime;
}

static long long
currentTime(void)
{
	UnsignedWide currentTime;
	Microseconds(&currentTime);
	return((currentTime.hi << 32) | currentTime.lo);
}

static void
csrTimeout(char *rtn)
{
	failedRoutineName = rtn;
	jtagError = 1;
}