package sorcererII;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

public class SorcererAudioInputStream extends InputStream {

  @Override
  public int read() throws IOException {
    if(seekStart() == -1) return -1;
    return readByte();
  }
  
  private AudioInputStream _audioInputStream;
  private double _sampleFrequency;
  private int _bytesPerFrame;
  private int _channels;


  private int _numBytes;
  private byte[] _audioBytes;
  private int _numBytesRead = 0;
  private int _currentSample;
  private double _samplesPerBit;

  private boolean _currentState = false;
  private int _upperThreshold = 40;
  private int _lowerThreshold = -40;
  private boolean _trace = false;
  private Logger _logger;
  
  public SorcererAudioInputStream (final File fileIn, final double bitFrequency, final Logger logger) 
  throws IOException
  {
    _logger = logger;
    try {
      _audioInputStream = AudioSystem.getAudioInputStream( fileIn );
    }
    catch(final UnsupportedAudioFileException e) {
      if(_logger != null) _logger.log(Level.SEVERE, "Audio file format is not supported", e);
      throw new IOException(e);
    }

    //
    // Get the format and check we can decode it
    //
    AudioFormat audioFormat = _audioInputStream.getFormat();

    //...
    if(_trace) System.out.println( audioFormat );
    if(_logger != null) _logger.log(Level.INFO, "Reading audio format:" + audioFormat);

    _sampleFrequency = (double)audioFormat.getFrameRate();

    //
    // Get the frame size and check we can use it
    //
    _bytesPerFrame = audioFormat.getFrameSize();
    _channels = audioFormat.getChannels();

    AudioFormat.Encoding encoding = audioFormat.getEncoding();

    // TODO Handle dc offset...
    int dcOffset = 0;

    if( encoding == AudioFormat.Encoding.PCM_UNSIGNED )
    {
      dcOffset = 1 << ((_bytesPerFrame / _channels) * 8 - 1);
    }

    final int f = (1 ^ (8 * ((_bytesPerFrame / _channels) -1))) * _channels;
    _upperThreshold *= f;
    _lowerThreshold *= f;
    _upperThreshold += dcOffset;
    _lowerThreshold += dcOffset;
   
    if( _bytesPerFrame > 2 )
    {
      if(_channels != 2) {
        if(_trace) System.out.println( "Sorry cannot support " + _bytesPerFrame + " bytes per frame " );
        throw new IOException ("Sorry cannot support " + _bytesPerFrame + " bytes per frame ");
      }
    }

    _numBytes = 1024 * _bytesPerFrame;
    _audioBytes = new byte[ _numBytes ];
    _currentSample = _numBytes;

    _samplesPerBit = (_sampleFrequency / bitFrequency  );

    if(_trace) System.out.println( "Samples Per Bit " + _samplesPerBit );
  }

  private int readSample() throws IOException {
    int sample;
    if(_bytesPerFrame == 1) {
      sample =  _audioBytes[ _currentSample++ ];
    }
    else if(_bytesPerFrame == 2){
      sample = (_audioBytes[ _currentSample ] & 0xff) + 
      ((_audioBytes[ _currentSample + 1 ] & 0xff) <<8);
      // sign extend...
      if((sample & 0x8000) != 0) {
        sample |= 0xffff0000;
      }
      _currentSample +=2;
    }
    else if(_bytesPerFrame == 4 && _channels == 2) {
      int s1 = (_audioBytes[ _currentSample ] & 0xff) + 
      ((_audioBytes[ _currentSample + 1 ] & 0xff) <<8);
      // sign extend...
      if((s1 & 0x8000) != 0) {
        s1 |= 0xffff0000;
      }
      _currentSample +=2; 
      int s2 = (_audioBytes[ _currentSample ] & 0xff) + 
      ((_audioBytes[ _currentSample + 1 ] & 0xff) <<8);
      // sign extend...
      if((s2 & 0x8000) != 0) {
        s2 |= 0xffff0000;
      }
      _currentSample +=2;
      sample = s1+s2;
    }
    else {
      if(_logger != null) _logger.log(Level.SEVERE, "Cannot support "+_bytesPerFrame + " bytes per frame.");
      throw new IOException("Cannot support "+_bytesPerFrame + " bytes per frame.");
    }

    //    System.out.println(sample);

    return sample;
  }
  
  private int nextSample() throws IOException
  {
    int sample;

    if( _numBytesRead == -1 )
    {
      return 0;
    }
    else if( _currentSample < _numBytesRead )
    {
      sample = readSample();
    }
    else
    {
      _numBytesRead = _audioInputStream.read( _audioBytes );
      _currentSample = 0;

      if( _numBytesRead > 0 )
      {
        sample = readSample();
      }
      else
      {
        return 0;
      }
    }
    return sample;
  }

  private boolean next() throws IOException
  {
    int sample = nextSample();
    
    if( _numBytesRead == -1 ) return false;
    
    if( _currentState )
    {
      if( sample < _lowerThreshold )
      {
        _currentState = false;
      }
    }
    else
    {
      if( sample > _upperThreshold )
      {
        _currentState = true;
      }
    }

    return true;
  }

  private int readPeriod() throws IOException
  {
    boolean thisState = _currentState;

    int periodLength = 0;
    boolean more = false;

    while( ( thisState == _currentState ) && ( more = next() ) )
    {
      periodLength++;
    }

    if( more )
    {
      return periodLength;
    }
    else
    {
      return -1;
    }
  }

  private int readByte() throws IOException
  {
    int b = 0;
    for( int i = 0; i < 8; ++i )
    {
      b = b >> 1;
      final int bit = readBit();  
      if(bit == -1) return -1;
      if( bit == 1 )
      {
        b |= 0x80;
      }
    }    
    if(_trace) System.out.println( "Data: " + Integer.toHexString(b) );
    return b;
  }


  private int readBit() throws IOException
  {
    final int p0 = readPeriod();
    
    if(p0 == -1) return -1;

    double d1 = p0 - _samplesPerBit;
    if( d1 < 0 ) d1 = -d1;

    double d2 = p0 - (_samplesPerBit / 2.0);
    if( d2 < 0 ) d2 = -d2;

    double diff = d1 - d2;
    if( diff < 0 ) diff = -diff;
    if( diff < 5 )
    {
      if(_logger != null) _logger.log(Level.FINE, "Cannot determine if bit is 1 or 0");
    }

    if( d1 > d2 )
    {
      final int p1 = readPeriod();
      if(p1 == -1) return -1;
      return 1;
    }
    else
    {
      return 0;
    }
  }

  private int seekStart() throws IOException
  {
    boolean seeking = true;

    int p1 = readPeriod();if(p1 == -1) return -1;
    int p2 = readPeriod();if(p2 == -1) return -1;
    int p3 = readPeriod();if(p3 == -1) return -1;
    int syncCount = 0;
    int syncLength = 0;
    
    while( seeking )
    {
      double d1 = p1 - ( _samplesPerBit / 2.0 );
      double d2 = p2 - ( _samplesPerBit / 2.0 );
      double d3 = p3 - _samplesPerBit;
      double d4 = p3 - ( _samplesPerBit / 2.0 );
      if( d1 < 0 ) d1 = -d1;
      if( d2 < 0 ) d2 = -d2;
      if( d3 < 0 ) d3 = -d3;
      if( d4 < 0 ) d4 = -d4;
      boolean a1 = d1 <= _samplesPerBit / 4.0;
      boolean a2 = d2 <= _samplesPerBit / 4.0;
      boolean a3 = d3 <= _samplesPerBit / 4.0;
      boolean a4 = d4 <= _samplesPerBit / 4.0;

      if( a1 && a2 && a3 )
      {
        if(_trace) System.out.println( "good " +  p1 + " " + p2 + " " + p3 + " Samples Per bit " + ( ( p1+p2+p3 ) / 2 ) );
        if(_trace) System.out.println( "Found Start" );
        seeking = false;
      }

      else  
      {
        if (a1 && a2 && a4) {
          if(_trace) System.out.println( "sync (" + syncCount + ")" +  p1 + " " + p2 + " " +p3 );
          syncCount++;
          syncLength += p1 + p2 + p3;
        }
        else {
         
          if(_trace) System.out.println( "bad " +  p1 + " " + p2 + " " +p3 );
          syncCount = 0;syncLength = 0;
        }
        p1 = p2;
        p2 = p3;
        p3 = readPeriod();if(p3 == -1) return -1;
      }
    }
    
    if(syncCount > 30) {
     
      _samplesPerBit = (syncLength * 2.0)/ (syncCount * 3.0);
      if(_trace) System.out.println( "Re-calibrating samples-per-bit at " + _samplesPerBit);
    }
    return 0;
  }

}
