/*
 *	Routines in this file implement a message interface via
 *	UNIX/VMS pipes. Messages can consequently flow only
 *	between parent and child processes.
 *
 *	Copyright (C) 1983, 1984, 1985 by Chromatics, Inc.
 *
 */
#include <gks/gks_types.h>
#include <gks/message.h>
#include <gks/gksmsg.h>


extern PID mypid;


#define MAXPROC 5

static struct 
	{
	PID map_pid;
	int map_fdin;
	int map_fdout;
	} fdmap[MAXPROC];

static int nprocmap;


#ifdef LIB1500

static MSGHDR *Replymsg[MAXPROC];

#else

static MSGHDR stdhead;

#endif



/*
 *	allocate message
 */
MSGHDR *_allocmsg( len )
	int len;
	{

	return((MSGHDR *)malloc(len));

	}


/*
 *	free message
 */
VOID _freemsg( msg )
	MSGHDR *msg;
	{

#ifdef DEBUNK
	debunk("_freemsg");
#endif

	free(msg);

	}



#ifdef LIB1500
/*
 *	"send" reply back to GKS
 */
VOID _replymsg( msg )
	MSGHDR *msg;
	{
	register int i;

	for (i = 0; i < nprocmap; ++i)
		if (msg->msg_from == fdmap[i].map_pid)
			Replymsg[i] = msg;

	}

#endif


/*
 *	receive message
 */
MSGHDR *_recvmsg( pid )
	PID pid;
	{
	int n;

#ifdef LIB1500

	for (n = 0; n < nprocmap; ++n)
		if (fdmap[n].map_pid == pid)
			return (Replymsg[n]);

	return (NULL);

#else

	int msgin;
	MSGHDR *msg;
	int *dump;

#ifdef DEBUNK
	debunk("_recvmsg");
#endif

	if((msgin = pid2fdin(pid)) < 0)
		{
		Gksout("_recvmsg: unable to receive message from process %d\n",
			pid);

		return(NULL);
		}

#ifdef DEBUG
	Gksout("_recvmsg (head): %d, %d\n", msgin, sizeof(stdhead));
#endif

	if((n = readp(msgin, &stdhead, sizeof(stdhead))) != sizeof(stdhead))
		{
		Gksout("_recvmsg: can't read msg head from pipe %d, process %d\n",
			msgin, pid);
		dump = (int *)&stdhead;
		Gksout("_recvmsg: %d bytes returned, 0x%x 0x%x 0x%x\n",
			n, dump[0], dump[1], dump[2]);

		return(NULL);
		}

	if((msg = (MSGHDR *)malloc(stdhead.msg_length + sizeof(stdhead)))
		== NULL)
		{
		Gksout( "_recvmsg: unable to allocate %d bytes.\n",
			stdhead.msg_length);

		return(NULL);
		}

	cpybuf(msg, &stdhead, sizeof(stdhead));

#ifdef DEBUG
	Gksout("_recvmsg: fd = %d, body len = %d\n", msgin, stdhead.msg_length);
#endif

	if(stdhead.msg_length > 0 && 
		readp(msgin, ((char *)msg)+sizeof(stdhead), stdhead.msg_length)
			!= stdhead.msg_length)
		{
		Gksout("_recvmsg: can't read msg body from pipe %d, process %d\n",
			msgin, pid);

		return(NULL);
		}

	return(msg);

#endif	/* LIB1500 */

	}


/*
 *	read from pipe
 */
static int readp( fd, buf, n )
	int fd;
	char *buf;
	int n;
	{
	register int ret;
	int req;

#ifdef DEBUNK
	debunk("readp");
#endif

	for (req = n; n > 0; )
		{
		ret = read(fd, buf, n);
		if( ret < 0)
			return (0);
		if( ret == 0)
			return(req - n);
		n -= ret;
		buf += ret;
		}

	return(req - n);

	}


#ifndef LIB1500

/*
 *	get input file descriptor associated w/ pid
 */
static VOID pid2fdin( pid )
	PID pid;
	{
	int i;

#ifdef DEBUNK
	debunk("pid2fdin");
#endif

	for(i = 0; i < nprocmap; ++i)
		if(fdmap[i].map_pid == pid)
			return(fdmap[i].map_fdin);
	
	return(-1);

	}

#endif	/* not LIB1500 */


#ifndef LIB1500

/*
 *	get output file descriptor associated w/ pid
 */
static VOID pid2fdout( pid )
	PID pid;
	{
	int i;

#ifdef DEBUNK
	debunk("pid2fdout");
#endif

	for(i = 0; i < nprocmap; ++i)
		if(fdmap[i].map_pid == pid)
			return(fdmap[i].map_fdout);
	
	return(-1);

	}

#endif	/* not LIB1500 */


/*
 *	associate input/output file descriptors
 *	w/ pid
 */
BOOL _addmap( pid, fdin, fdout )
	PID pid;
	int fdin, fdout;
	{

#ifdef DEBUNK
	debunk("_addmap");
#endif

	if(nprocmap >= MAXPROC)
		return(FALSE);

	fdmap[nprocmap].map_pid  = pid;
	fdmap[nprocmap].map_fdin = fdin;
	fdmap[nprocmap].map_fdout = fdout;
	nprocmap++;

	return(TRUE);

	}


/*
 *	free up resources after child's death
 */
VOID _delmap( pid )
	PID pid;
	{
	int i;
	int fdi, fdo;

#ifndef LIB1500

#ifdef DEBUNK
	debunk("_delmap");
#endif

	for(i = 0; i < nprocmap; ++i)
		if(fdmap[i].map_pid == pid)
			break;
	
	if (i == nprocmap)
		return;							/* pid not found !! */
	
	--nprocmap;
	fdi = fdmap[i].map_fdin;
	fdo = fdmap[i].map_fdout;
	fdmap[i].map_pid = fdmap[nprocmap].map_pid;
	fdmap[i].map_fdin = fdmap[nprocmap].map_fdin;
	fdmap[i].map_fdout = fdmap[nprocmap].map_fdout;

	for (i = 0; i < nprocmap; ++i)
		if (fdmap[i].map_fdin == fdi || fdmap[i].map_fdout == fdo)
			return;

	close(fdi);
	close(fdo);

#endif

	}


/*
 *	duplicate map descriptor
 *
 *	returns:
 *
 *		0	error
 *	  < 0	dummy process id to use for communication w/workstation
 */
int _dupmap( pid )
	PID pid;
	{
	int i;

#ifdef DEBUNK
	debunk("_dupmap");
#endif

	if(nprocmap >= MAXPROC)
		return(0);

	for (i = 0; i < nprocmap; ++i)
		if (fdmap[i].map_pid == pid)
			{
			fdmap[nprocmap].map_pid  = pid = -nprocmap - 1;
			fdmap[nprocmap].map_fdin = fdmap[i].map_fdin;
			fdmap[nprocmap].map_fdout = fdmap[i].map_fdout;
			++nprocmap;
			return (pid);
			}
	
	return (0);

	}


#ifndef LIB1500
/*
 *	send message to process
 */
BOOL _sendmsg( msg )
	MSGHDR *msg;
	{
	int msgout;

#ifdef DEBUNK
	debunk("_sendmsg");
#endif

	msg -> msg_from = mypid;
	if((msgout = pid2fdout(msg -> msg_to)) < 0)
		{
		Gksout("_sendmsg: unable to send message to process %d\n",
			msg -> msg_to);

		return(FALSE);
		}

#ifdef DEBUG
	Gksout("_sendmsg: fd = %d, message length = %d\n",
		msgout, sizeof(stdhead) + msg -> msg_length);
#endif

	if(writep(msgout, msg, sizeof(stdhead) + msg->msg_length) !=
						   sizeof(stdhead) + msg->msg_length)
		{
		Gksout("_sendmsg: unable to write to pipe %d to process %d\n",
			msgout, msg->msg_to);
		return(FALSE);
		}

	_freemsg(msg);

	return(TRUE);

	}

#endif	/* LIB1500 */


/*
 *	write to pipe
 */
static int writep( fd, buf, n )
	int fd;
	char *buf;
	int n;
	{
	register int ret;
	int req;

#ifdef DEBUNK
	debunk("writep");
#endif

	for (req = n; n > 0; )
		{

#ifdef vms
		ret = write(fd, buf, ((n > 512) ? 512 : n));
#else
		ret = write(fd, buf, n);
#endif

		if( ret < 0)
			return (0);
		if( ret == 0)
			return(req - n);

		n -= ret;
		buf += ret;
		}

	return(req - n);

	}
