;
; These routines allocate RAM and download code 
; from PROM to RAM.
; Downld - load permanently resident code in high memory.
; Malloc - acquire permanent storage from high memory.
;
; Downld and Malloc allocate RAM at the top of
; memory, never to be freed.
;
; Kernel global variables :
;	HeapTop - highest address allocated to heap (these
;		routines assume HeapTop = 0).
;	StackBase - address of highest free byte of RAM.
;	Stackmin - least legal diff between HeapTop, StackBase.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Allocate RAM in high memory				;
; Return with carry set if not enough RAM.		;
; Low RAM addr of the code is returned on the stack.	;
; This routine is responsible for maintaining the	;
; global variable StackBase - address of highest free	;
; byte in RAM.  After all proms have been 'awakened'	;
; and their storage requirements satisfied, the powerup	;
; routine sets the stack to StackBase.			;
; Calling sequence :					;
;
;	pea	##count		; push # bytes of RAM needed.
;	jsl	>0,Malloc	; get pointer to RAM (on stack).
;	bcs	error		; branch if count too large.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

malloc:
			; count  11,s
		; ret addr, bank  8,s
	clc			; assume count is reasonable.
	php			; 7,s
	rep	#0x30
	pha			; 5,s
	phx			; 3,s
	phy			; 1,s
;
; Check count for reasonableness - error if
; StackBase - count < StackMin
;
	sec
	lda	>0,StackBase
	sbc	11,s
	bcs	$1
;
; Count too large, return to caller
; with carry set. 
;
	lda	7,s		; get callers psw.
	ora	##1		; set carry bit to flag error.
	sta	7,s
	brl	$done		; clean up stack and return.
$1:
;
; Allocation successful, adjust StackBase
; and return pointer to caller.
;
	sta	>0,StackBase	; save new addr of high RAM.
	inc	a		; point to low byte of allocation.
	sta	11,s		; save it for caller (replaces count).
$done:
	ply
	plx
	pla
	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Copy code from prom to ram.				;
; Return with carry set if not enough RAM.		;
; Low RAM addr of the code is returned on the stack.	;
; Calls Malloc to reserve the memory to be used.	;
; Calling sequence :					;
;
;	per	addr		; push prom address of code.
;	pea	##length	; push # bytes of code to copy.
;	jsl	>0,Downld	; copy the code, return with
;	bcs	dlerror		; start addr on stack.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

downld:
		; code start	; 14,s
		; code length	; 12,s
		; ret addr,bank ; 9,s
	clc			; assume no error.
	php			; 8,s
	rep	#0x30
	pha			; 6,s
	phx			; 4,s
	phy			; 2,s
	phb			; 1,s

	lda	12,s		; get byte count
	pha			; stack for Malloc.
	jsl	>0,Malloc	; allocate RAM.
	bcc	$1		; br if allocation successful.
	
	lda	8,s		; not enough RAM, set callers
	ora	##1		; carry to indicate error
	brl	$done		; and return.
$1:	
	lda	1,s		; get target address for mvn,
	tay			;   leave on stack for later.
	lda	14+2,s		; get source addr for mvn.
	tax
	lda	12+2,s		; get byte count.
	dec	a		; minus 1 because mvn is that way.
	mvn	>0xff,>0	; move prom to RAM.
	
	pla			; get target addr.
	sta	14,s		; store for return to caller.
$done:

;
; That's it, clean up stack and return
;
	plb
	ply
	plx

; addr	9,s
; count	7,s
; ret,bnk 4,s
; p	3,s
; a	1,s	
	
	lda	5,s
	sta	7,s
	lda	3,s	
	sta	5,s
	lda	1,s
	sta	3,s
	pla

	pla
	plp
	rtl

	end

