/*----------------------------------------------------------------------*/
/*  File        colors.c                                                */
/*                                                                      */
/*  Description This file contains functions and data needed to support */
/*              color independence.                                     */
/*----------------------------------------------------------------------*/
#include "tigademo.h"
#include "colors.h"

extern PATTERN patn;

/*----------------------------------------------------------------------*/
/*  Define global variables needed for color support                    */
/*----------------------------------------------------------------------*/
unsigned long   phys_color[NUM_LOG_COLORS];
short           max_colors, iscolor;
PTR             dither_patns;
static short    mn;
/*----------------------------------------------------------------------*/
/*  Define call to make_patn TMS340 extended function                   */
/*----------------------------------------------------------------------*/
#define make_patn(a,b)      \
            cp_cmd(USER_CP(mn),2,_WORD(a),_DWORD(b))
/*----------------------------------------------------------------------*/
/*  Logical palette for 1-bpp (2 colors) programmable palette           */
/*----------------------------------------------------------------------*/
PALET log_palet_1bpp[] =
{
    0x00,0x00,0x00,0x00,    /* BLACK                */
    0xFF,0xFF,0xFF,0xFF     /* WHITE                */
};
/*----------------------------------------------------------------------*/
/*  Logical palette for 2-bpp (4 colors) programmable palette           */
/*----------------------------------------------------------------------*/
PALET log_palet_2bpp[] =
{
    0x00,0x00,0x00,0x00,    /* BLACK                */
    0xFF,0xFF,0xFF,0xFF,    /* WHITE                */
    0x00,0x00,0x00,0x00,    /* BLACK                */
    0xFF,0xFF,0xFF,0xFF     /* WHITE                */
};
/*----------------------------------------------------------------------*/
/*  Logical palette for 4-bpp (16 colors) programmable palette          */
/*----------------------------------------------------------------------*/
PALET log_palet_4bpp[] =
{
    0x00,0x00,0x00,0x00,    /* BLACK                */
    0x20,0x20,0xB0,0x11,    /* BLUE                 */
    0x20,0xA0,0x20,0x22,    /* GREEN                */
    0x30,0xC0,0xC0,0x33,    /* CYAN                 */
    0xB0,0x30,0x30,0x44,    /* RED                  */
    0xB0,0x30,0xB0,0x55,    /* MAGENTA              */
    0x80,0x40,0x40,0x66,    /* BROWN                */
    0xC0,0xC0,0xC0,0x77,    /* LIGHT GRAY           */
    0x50,0x50,0x50,0x88,    /* DARK  GRAY           */
    0x20,0x20,0xFF,0x99,    /* LIGHT BLUE           */
    0x20,0xE0,0x20,0xAA,    /* LIGHT GREEN          */
    0x20,0xE0,0xE0,0xBB,    /* LIGHT CYAN           */
    0xE0,0x20,0x20,0xCC,    /* LIGHT RED            */
    0xE0,0x10,0xE0,0xDD,    /* LIGHT MAGENTA        */
    0xFF,0xFF,0x10,0xEE,    /* YELLOW               */
    0xFF,0xFF,0xFF,0xFF     /* WHITE                */
};
/*----------------------------------------------------------------------*/
/*  Logical palette for 8-bpp (256 colors) programmable palette         */
/*----------------------------------------------------------------------*/
PALET log_palet_8bpp[] =
{                                                   
    0x00,0x00,0x00,0x00,    /* BLACK                */
    0x20,0x20,0xB0,0x11,    /* BLUE                 */
    0x20,0xA0,0x20,0x22,    /* GREEN                */
    0x30,0xC0,0xC0,0x33,    /* CYAN                 */
    0xB0,0x30,0x30,0x44,    /* RED                  */
    0xB0,0x30,0xB0,0x55,    /* MAGENTA              */
    0x80,0x40,0x40,0x66,    /* BROWN                */
    0xC0,0xC0,0xC0,0x77,    /* LIGHT GRAY           */
    0x50,0x50,0x50,0x88,    /* DARK  GRAY           */
    0x20,0x20,0xFF,0x99,    /* LIGHT BLUE           */
    0x20,0xE0,0x20,0xAA,    /* LIGHT GREEN          */
    0x20,0xE0,0xE0,0xBB,    /* LIGHT CYAN           */
    0xE0,0x20,0x20,0xCC,    /* LIGHT RED            */
    0xE0,0x10,0xE0,0xDD,    /* LIGHT MAGENTA        */
    0xFF,0xFF,0x10,0xEE,    /* YELLOW               */
    0xFF,0xFF,0xFF,0xFF,    /* WHITE                */
    0x70,0xdb,0x93,0x9f,    /* AQUAMARINE           */
    0x9f,0x5f,0x9f,0x89,    /* BLUE_VIOLET          */
    0x5f,0x9f,0x9f,0x89,    /* CADET_BLUE           */
    0xff,0x7f,0x00,0x7f,    /* CORAL                */
    0x42,0x42,0x6f,0x51,    /* CORNFLOWER_BLUE      */
    0x2f,0x4f,0x2f,0x39,    /* DARK_GREEN           */
    0x4f,0x4f,0x2f,0x44,    /* DARK_OLIVE_GREEN     */
    0x99,0x32,0xcc,0x87,    /* DARK_ORCHID          */
    0x6b,0x23,0x8e,0x5e,    /* DARK_SLATE_BLUE      */
    0x2f,0x4f,0x4f,0x44,    /* DARK_SLATE_GRAY      */
    0x70,0x93,0xdb,0x9f,    /* DARK_TURQUOISE       */
    0x54,0x54,0x54,0x54,    /* DIM_GRAY             */
    0x8e,0x23,0x23,0x46,    /* FIREBRICK            */
    0x23,0x8e,0x23,0x46,    /* FOREST_GREEN         */
    0xcc,0x7f,0x32,0x7f,    /* GOLD                 */
    0xdb,0xdb,0x70,0xb7,    /* GOLDENROD            */
    0xc0,0xc0,0xc0,0xc0,    /* GRAY                 */
    0x93,0xdb,0x70,0x9f,    /* GREEN_YELLOW         */
    0x4f,0x2f,0x2f,0x39,    /* INDIAN_RED           */
    0x9f,0x9f,0x5f,0x89,    /* KHAKI                */
    0x8f,0x8f,0xbc,0x9e,    /* LIGHT_STEEL_BLUE     */
    0x32,0xcc,0x32,0x65,    /* LIME_GREEN           */
    0x8e,0x23,0x6b,0x5e,    /* MAROON               */
    0x32,0xcc,0x99,0x87,    /* MEDIUM_AQUAMARINE    */
    0x32,0x32,0xcc,0x65,    /* MEDIUM_BLUE          */
    0x6b,0x8e,0x23,0x5e,    /* MEDIUM_FOREST_GREEN  */
    0xea,0xea,0xad,0xd5,    /* MEDIUM_GOLDENROD     */
    0x8e,0x70,0xdb,0x9d,    /* MEDIUM_ORCHID        */
    0x42,0x6f,0x42,0x51,    /* MEDIUM_SEA_GREEN     */
    0x7f,0x00,0xff,0x7f,    /* MEDIUM_SLATE_BLUE    */
    0x7f,0xff,0x00,0x7f,    /* MEDIUM_SPRING_GREEN  */
    0x70,0xdb,0xdb,0xb7,    /* MEDIUM_TURQUOISE     */
    0xdb,0x70,0x93,0x9f,    /* MEDIUM_VIOLET_RED    */
    0x2f,0x2f,0x4f,0x39,    /* MIDNIGHT_BLUE        */
    0x23,0x23,0x8e,0x46,    /* NAVY                 */
    0x23,0x23,0x8e,0x46,    /* NAVY_BLUE            */
    0xcc,0x32,0x32,0x65,    /* ORANGE               */
    0xff,0x00,0x7f,0x7f,    /* ORANGE_RED           */
    0xdb,0x70,0xdb,0xb7,    /* ORCHID               */
    0x8f,0xbc,0x8f,0x9e,    /* PALE_GREEN           */
    0xbc,0x8f,0x8f,0x9e,    /* PINK                 */
    0xea,0xad,0xea,0xd5,    /* PLUM                 */
    0x6f,0x42,0x42,0x51,    /* SALMON               */
    0x23,0x8e,0x6b,0x5e,    /* SEA_GREEN            */
    0x8e,0x6b,0x23,0x5e,    /* SIENNA               */
    0x32,0x99,0xcc,0x87,    /* SKY_BLUE             */
    0x00,0x7f,0xff,0x7f,    /* SLATE_BLUE           */
    0x00,0xff,0x7f,0x7f,    /* SPRING_GREEN         */
    0x23,0x6b,0x8e,0x5e,    /* STEEL_BLUE           */
    0xdb,0x93,0x70,0x9f,    /* TAN                  */
    0xd8,0xbf,0xd8,0xcf,    /* THISTLE              */
    0xad,0xea,0xea,0xd5,    /* TURQUOISE            */
    0x4f,0x2f,0x4f,0x44,    /* VIOLET               */
    0xcc,0x32,0x99,0x87,    /* VIOLET_RED           */
    0xd8,0xd8,0xbf,0xcf,    /* WHEAT                */
    0x99,0xcc,0x32,0x87     /* YELLOW_GREEN         */
};
/*----------------------------------------------------------------------*/
/*  Compute number of palette entries in each logical palette           */
/*----------------------------------------------------------------------*/
#define LOG_PALET_1BPP          (sizeof(log_palet_1bpp)/sizeof(PALET))
#define LOG_PALET_2BPP          (sizeof(log_palet_2bpp)/sizeof(PALET))
#define LOG_PALET_4BPP          (sizeof(log_palet_4bpp)/sizeof(PALET))
#define LOG_PALET_8BPP          (sizeof(log_palet_8bpp)/sizeof(PALET))

/*----------------------------------------------------------------------*/
/*  name        create_dithered_patterns                                */
/*                                                                      */
/*  synopsis    s = create_dithered_patterns( void )                    */
/*              short   s;      status: 0=Error, 1=OK                   */
/*                                                                      */
/*  description This function is called by the init_colors() function   */
/*              to create 64 dithered patterns used to simulate 64      */
/*              "colors" when the psize is 1 or 2.                      */
/*----------------------------------------------------------------------*/
static short create_dithered_patterns()
{
    PTR             gpPatn;
    register short  intensity;

    if( (dither_patns = gpPatn = gsp_malloc(64 * ((16*16)/8))) == 0L )
    {
        printf( "Not enough memory to generate dithered patterns\n" );
        return(0);
    }
    if( (mn = install_rlm( "makepatn %f" )) < 0 )
    {
        printf( "Error creating dithered patterns\n" );
        return(0);
    }
    for( intensity=0; intensity < 64; intensity++ )
    {
        make_patn( intensity, gpPatn );
        gpPatn += 16*16;
    }
    flush_module(mn);   
    return(1);
}

/*----------------------------------------------------------------------*/
/*  name        set_dither                                              */
/*                                                                      */
/*  synopsis    void set_dither(intensity)                              */
/*              short intensity;    intensity, range 0-63               */
/*                                                                      */
/*  description This functions sets the current TIGA pattern to the     */
/*              dither pattern corresponding to the argument            */
/*              "intensity".  These patterns must have been previously  */
/*              created by the create_dither_patterns() function.       */
/*----------------------------------------------------------------------*/
void set_dither( intensity )
unsigned long intensity;
{
    patn.data = dither_patns + (intensity * 16L * 16L);
    set_patn( &patn );
}

/*----------------------------------------------------------------------*/
/*  name        init_colors                                             */
/*                                                                      */
/*  synopsis    void init_colors(class,psize)                           */
/*              short class;    color class                             */
/*              short psize;    pixel size                              */
/*                                                                      */
/*  description This function is called at initialization time to       */
/*              initialize the physical colors based on the current     */
/*              color class and pixel size.                             */
/*                                                                      */
/*              For the tigademo program, pixel sizes of 1 and 2 are    */
/*              treated as monochrome, and dithered patterns are        */
/*              used to simulate various colors.  The get_nearest_color */
/*              function is used to initialize physical colors when the */
/*              pixel size is greater than 2.                           */
/*----------------------------------------------------------------------*/
void init_colors(color_class, psize)
short color_class, psize;
{
    PALET   *palet;
    short   i, palet_entries;

    /*------------------------------------------------------------------*/
    /*  Select palette and maximum color based on current pixel size.   */
    /*------------------------------------------------------------------*/
    switch(psize)
    {
        case 1:     palet_entries   = LOG_PALET_1BPP;
                    max_colors      = NUM_LOG_COLORS;
                    palet           = log_palet_1bpp;
                    iscolor         = 0;
                    break;
        case 2:     palet_entries   = LOG_PALET_2BPP;
                    max_colors      = NUM_LOG_COLORS;
                    palet           = log_palet_2bpp;
                    iscolor         = 0;
                    break;
        case 4:     palet_entries   = LOG_PALET_4BPP;
                    max_colors      = LOG_PALET_4BPP;
                    palet           = log_palet_4bpp;
                    iscolor         = 1;
                    break;
        default:    palet_entries   = LOG_PALET_8BPP;
                    max_colors      = NUM_LOG_COLORS;
                    palet           = log_palet_8bpp;
                    iscolor         = 1;
                    break;
    };
    /*------------------------------------------------------------------*/
    /*  Setup palette if color class indicates it's programmable.       */
    /*------------------------------------------------------------------*/
    if( !(color_class & 1) )
        set_palet( palet_entries, 0, palet );
    /*------------------------------------------------------------------*/
    /*  Convert all logical colors to closest physical values.          */
    /*------------------------------------------------------------------*/
    if( iscolor )
        /*--------------------------------------------------------------*/
        /*  For color (>2bpp), use get_nearest_color                    */
        /*--------------------------------------------------------------*/
        for( i=0; i<NUM_LOG_COLORS; i++ )
            phys_color[i] = get_nearest_color( log_palet_8bpp[i].r,
                log_palet_8bpp[i].g, log_palet_8bpp[i].b, log_palet_8bpp[i].i );
    else
    {
        /*--------------------------------------------------------------*/
        /*  For monochrome (1,2bpp), use intensity scale (0-63)         */
        /*--------------------------------------------------------------*/
        create_dithered_patterns();
        for( i=0; i<NUM_LOG_COLORS; i++ )
            phys_color[i] = (unsigned long)(log_palet_8bpp[i].i / 4);
        set_colors( 0xFFFFFFFF, 0x0 );
    }
}


