/*-----------------------------------------------------------------------*/
/*                                 TIGA                                  */
/*       Copyright (c) 1988-1990 Texas Instruments Incorporated.         */
/*                         All Rights Reserved                           */
/*-----------------------------------------------------------------------*/
/*  TiGA - Graphics Manager                                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*  void set_palet(palet_size, offset, palet_ary)                        */
/*                                                                       */
/*  This function loads the global "palet" array with the palette values */
/*  in the parameters and then goes on to load the palette itself.       */
/*  The offset/size combination allows a palette to be loaded piecemeal  */
/*  Rather than all at once. This function is designed to work with      */
/*  the Brooktree 453 palette.                                           */
/*                                                                       */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*    int  set_palet_entry(index, r, g, b, i)                            */
/*                                                                       */
/*    This function loads one entry of the global "palet" array,         */
/*    and then goes on to load the palette itself. If the index is       */
/*    not valid it returns FALSE (0).                                    */
/*-----------------------------------------------------------------------*/
/*  07/20/88  Original Version Written                      Graham Short */
/*  09/27/88  Added set_palet_entry                         Graham Short */
/*  01/26/89  NEC Version                                   Graham Short */
/*  03/16/90  Added palette mask initialization             W.S.Egr      */
/*-----------------------------------------------------------------------*/
#include <gsptypes.h>
#include <gspglobs.h>
#include "oem.h"

#define FALSE  0
#define TRUE   1

#define COLOR_MONOCHROME_FLAG 1

/* NEC memory mapped hardware */
#define ADDR_REG_SEL    B478_WRITE
#define ADDR_GUN_VAL    B478_VALUE
#define PALET_MASK      0xFF

set_palet(palet_size, offset, palet_ary)
short                   palet_size;
short                   offset;
register unsigned char *palet_ary;
{
   register unsigned char  *pal_ptr;
   register unsigned char  *pal_ary;
   register short          count;
   register unsigned short *gun_val = (unsigned short *) ADDR_GUN_VAL;
   register unsigned short *reg_sel;

   /* check for zero palette size */
   if (!palet_size)
      return;

   /* check that the offset/size combination does not get larger than
      the entire palet */
   if   ((offset + palet_size) > config.mode.palet_size)
      palet_size = (config.mode.palet_size - offset);
    
   pal_ary = (unsigned char *) palet_ary;
   pal_ptr = (unsigned char *) &(palet[offset]);

   /* initialize palette write mask */
   reg_sel = (unsigned short *) B478_MASK;
   *reg_sel = (1 << (config.mode.disp_psize)) - 1;

   /* select palette register */
   reg_sel = (unsigned short *) ADDR_REG_SEL;
   *reg_sel = offset;
      
   /* test for monochrome */
   if (((MONITORINFO10 *)monitorinfo)->flags10 & COLOR_MONOCHROME_FLAG)
      /* monochrome */
      while (palet_size--)
      {
         pal_ary    += 3;                          /* skip to intensity */
         *gun_val   = *pal_ary;                    /* load red       */
         *gun_val   = *pal_ary;                    /* load green     */
         *gun_val   = *pal_ary;                    /* load blue      */
         *pal_ptr++ = 0;                           /* save red       */
         *pal_ptr++ = 0;                           /* save green     */
         *pal_ptr++ = 0;                           /* save blue      */
         *pal_ptr++ = (*pal_ary++) & PALET_MASK;   /* save intensity */
      }
   else
      /* color */
      while (palet_size--)
      {
         *gun_val   = *pal_ary;                   /* load red       */
         *pal_ptr++ = (*pal_ary++) & PALET_MASK;  /* save red       */
         *gun_val   = *pal_ary;                   /* load green     */
         *pal_ptr++ = (*pal_ary++) & PALET_MASK;  /* save green     */
         *gun_val   = *pal_ary;                   /* load blue      */
         *pal_ptr++ = (*pal_ary++) & PALET_MASK;  /* save blue      */
         *pal_ptr++ = (*pal_ary++) & PALET_MASK;  /* save intensity */
      }
}



set_palet_entry(index, r, g, b, i)
register long index;     
short         r, g, b, i;
{
   register unsigned char  *pal_ary;
   register unsigned short *gun_val = (unsigned short *) ADDR_GUN_VAL;
   register unsigned short *reg_sel;

   /* check that the index is not larger than the entire palet */
   if   ((index > config.mode.palet_size) || (index < 0))
      return(FALSE);

   /* initialize palette write mask */
   reg_sel = (unsigned short *) B478_MASK;
   *reg_sel = (2 << (config.mode.disp_psize)) - 1;

   /* select palette register */
   reg_sel = (unsigned short *) ADDR_REG_SEL;
   *reg_sel = index;
      
   /* test for monochrome */
   if (((MONITORINFO10 *)monitorinfo)->flags10 & COLOR_MONOCHROME_FLAG)
   {
      /* monochrome */
      *gun_val       = i;                /* load red       */
      *gun_val       = i;                /* load green     */
      *gun_val       = i;                /* load blue      */
      palet[index].r = 0;                /* save red       */
      palet[index].g = 0;                /* save green     */
      palet[index].b = 0;                /* save blue      */
      palet[index].i = i & PALET_MASK;   /* save intensity */
   }
   else
   {
      /* color */
      *gun_val       = r;               /* load red       */
      palet[index].r = r & PALET_MASK;  /* save red       */
      *gun_val       = g;               /* load green     */
      palet[index].g = g & PALET_MASK;  /* save green     */
      *gun_val       = b;               /* load blue      */
      palet[index].b = b & PALET_MASK;  /* save blue      */
      palet[index].i = i & PALET_MASK;  /* save intensity */
   }

   return(TRUE);
}



dm_set_palet_entry(data_ptr)
short *data_ptr;
{
   register          long  index;     
   register unsigned char  *pal_ary;
   register unsigned short *gun_val = (unsigned short *) ADDR_GUN_VAL;
   register unsigned short *reg_sel = (unsigned short *) ADDR_REG_SEL;
   short                   r, g, b, i;

   /* check that the index is not larger than the entire palet */
   index = *(long *)data_ptr++;
   if   ((index > config.mode.palet_size) || (index < 0))
      return(FALSE);
   
   /* initialize the local "palet" array */
   r = *data_ptr++ & PALET_MASK;
   g = *data_ptr++ & PALET_MASK;
   b = *data_ptr++ & PALET_MASK;
   i = *data_ptr   & PALET_MASK;

   *reg_sel = index;
      
   /* test for monochrome */
   if (((MONITORINFO10 *)monitorinfo)->flags10 & COLOR_MONOCHROME_FLAG)
   {
      /* monochrome */
      *gun_val       = i;                /* load red       */
      *gun_val       = i;                /* load green     */
      *gun_val       = i;                /* load blue      */
      palet[index].r = 0;                /* save red       */
      palet[index].g = 0;                /* save green     */
      palet[index].b = 0;                /* save blue      */
      palet[index].i = i & PALET_MASK;   /* save intensity */
   }
   else
   {
      /* color */
      *gun_val       = r;               /* load red       */
      palet[index].r = r & PALET_MASK;  /* save red       */
      *gun_val       = g;               /* load green     */
      palet[index].g = g & PALET_MASK;  /* save green     */
      *gun_val       = b;               /* load blue      */
      palet[index].b = b & PALET_MASK;  /* save blue      */
      palet[index].i = i & PALET_MASK;  /* save intensity */
   }

   return(TRUE);
}
