/* MODULE NAME: SIM3D.C                                              */
/*                                                                   */
/* MODULE DESCRIPTION:                                               */
/*   This module contains the main entry point for an example of a   */
/* 3 dsp device simulation.  It expands the standard SIM56000        */
/* command set by trapping the "new" instructions before they are    */
/* sent to dsp_docmd().  The new instructions allow selection of     */
/* one of the three simulated devices for display and provide the    */
/* capability to load or save all device states with a single        */
/* command.  The user may expand the trapcodes array and create his  */
/* own special functions.  unctions.  e SIM56000 (display version)   */
/* simulator by loading dspt1.sim with the simulator LOAD S command. */
/*                                                                   */
/* FUNCTIONS:                                                        */
/* main(argc,argv) - single dsp, no window                           */
/*                                                                   */
#include "config.h"
#include "brackets.h"
#include "dspexec.h"
#include "dsp_sdef.h"
#include "dsp_sdcl.h"

#define NUM_DSPS 3 /* Number of simultaneously simulated devices */

/* main simulator entry point */
main (argc,argv)
int argc;
char **argv;
{
   static char *trapcodes[]={"d0","d1","d2","svall","ldall"}; /* instructions to trap */

   char cmdstr[81]; /* Will hold command strings input from terminal */
   int i,j,k;
   struct dsp_var *dspn[NUM_DSPS]; /* dsp device pointers */
   dsp_const.num_dsps=NUM_DSPS;    /* number of dsp devices */
   dsp_const.dsp_list= dspn;       /* The simulator needs this. */
   dsp_const.viewdev=dsp_const.breakdev=0; /* Start with device 0 */

   for (i=0;i<NUM_DSPS;i++) dsp_new(i,DSP56001); /* Create new devices. */
   dsplwinit();    /* Initialize window and keyboard parameters */
   dsp_revision(0);  /* Display the simulator revision number */

   /* Begin with execution of macros specified in command line */

   if (argc == 2){ /* macro name in command line? */
      if (!strcmp(argv[1],"resume")){ /* Special macro name. */
         /* reload all simulator state files from DSP_SIMn.mem */
         dsp_ldall("DSP_SIM");
         }
      else dsp_docmd(0,argv[1]); /* Other macros named <name>.cmd */
      }

   /* This is the main execution loop.  If the device is at a breakpoint, */
   /* the next command is requested from a macro file or from the terminal.*/
   /* Commands that cause the device to begin executing will set the */
   /* "executing" flag in the dsp structure. */

   for(;;){ 
      for (i=0;i<NUM_DSPS;i++){
         if (dspn[i]->executing) dsp_exec(i); /* execute cycles until breakpoint is reached */
         else { /* Execution halted for one device */
            if (i!=dsp_const.viewdev){ /* Is device displayed? */
               dsp_const.breakdev=i; /* Change device that will respond to CTRLC */
               dsp_const.viewdev=i; /* Change viewed device index to currently halted device */
               dsp_wredo(i); /* redraw the whole screen of halted device */
               }
            do { /* Get commands for the halted device until not halted */
               if (dspn[i]->in_macro) dsp_gmcmd(i,cmdstr); /* get macro command */
               if (!dspn[i]->in_macro) dsp_gtcmd(i,cmdstr); /* get terminal command*/

               /* Trap any special user defined instructions.  Simulation */
               /* command input strings can be added to the trapcodes list */
               /* defined above and their actions defined here. The expansions */
               /* shown here allows the user to view a particular device in */
               /* a multiple device system, to save all device states or load all */
               /* device states in a multiple device system */

               for (j=0,k=0;k< (sizeof(trapcodes)/sizeof (char *)) ;k++){ /* trap some instructions to stop specific device */
                  if (!strcmp(cmdstr,trapcodes[k])){
                     switch (k){
                        case 0:
                        case 1:
                        case 2:
                           dspn[k]->CTRLBR=1; /* halt at device k after "go" */
                           dsp_const.breakdev=k; /* Will respond to CTRLBR */
                           dsp_const.viewdev=k; /* Change viewed device index to currently halted device */
                           dsp_wredo(k); /* redraw the whole screen of halted device */
                           dsp_docmd(i,"go"); /* substitute go command */
                           j=1; /* means skip the next dsp_docmd */
                           break;
                        case 3: dsp_svall("DSP_SAV"); /* save all dsp states to files */
                           j=1;
                           break;
                        case 4: dsp_ldall("DSP_SAV"); /* load all dsp states from files */
                           j=1;
                           break;
                        }
                     break; /* break out of for loop after command found */
                     }
                  }
               if (!j) dsp_docmd(i,cmdstr);
               } while (!dspn[i]->executing);
            }
         }
      }   
}   /* end of main */


