
/*
 *	This file contains the NVRAM load and update functions.
 *
 *	Written by 	Ramakrishna Rao
 *
 *	$Log:   /b/gregs/i960/util/nvram.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:38:28   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:27:06   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:24:58   gregs
 * Initial revision.
 * 
 *    Rev 1.0   30 Mar 1992 17:09:00   pvcs
 * Initial revision.
 *
 *	Copyright (c) 1992 by Hughes LAN Systems
 */

#include <types.h>
#include <nvram.h>

NVRAM_HDR	Nvram_header;

static	void	*NvramAdd;
static	int	NvramSize;

/*
 * name		InitNvram
 * 
 * synopsis	InitNvram(nvram, size)
 *		void	*nvram; <<	nvram starting address
 *		int	size; <<	of nvram in bytes)
 *
 * description	This function should be called before the nvram primitives
 *		are called. It sets up the nvram header if this is the first
 *		time the eeprom is being used. 
 *
 * returns	NVRAM_AOK	nvram is ready 
 *		NVRAM_NOSUCH	The NVRAM block is not defined
 *		NVRAM_ALNERR	Alignment error. offset or the size is odd
 *		NVRAM_CSERR	Checksum error.
 */
InitNvram(void *nvram, int size)

	{

	NvramAdd = nvram;
	NvramSize = size;

	if (Nvram_Load(nvram, &Nvram_header, NVRAM_HDR_SIZE) != NVRAM_AOK)
		{
		memset(&Nvram_header, 0, NVRAM_HDR_SIZE);
		Nvram_header.nvr_recs[0].start_addr = (byte *)nvram;
		Nvram_header.nvr_recs[0].size = NVRAM_HDR_SIZE;
		return Nvram_Updt(nvram, &Nvram_header, NVRAM_HDR_SIZE); 
		}
	return NVRAM_AOK;
	}

/*
 *	Nvram_Load(offset, buffer addr, Size)
 *
 *	Returns  NVRAM_AOK	If successful
 *		 NVRAM_NOSUCH	The NVRAM block is not defined
 *		 NVRAM_ALNERR	Alignment error. offset or the size is odd
 *		 NVRAM_CSERR	Checksum error.
 *
 *	Assumption	The NVRAM header is already read and is available
 *			in the Nvram_header.
 */
Nvram_Load(offset,bufadr,size)
byte	*offset;
byte	*bufadr;
int	size;	
{
	int	found;
	u_short *cksump;
	byte	*cp;
	int	i;

	if(((int)offset & 1) || (size & 1))
		return NVRAM_ALNERR;

	/* Verify that the block exists in the header */
	found = FALSE;
	for(i = 0; i < MAX_NVRAM_RECS && Nvram_header.nvr_recs[i].size; i++)
	{
		if(offset == Nvram_header.nvr_recs[i].start_addr && 
			size == Nvram_header.nvr_recs[i].size)
		{
			found = TRUE;
			break;
		}
	}
	if(found == FALSE)
	{
		if(!(offset == (byte *)NvramAdd && size == NVRAM_HDR_SIZE))
			return NVRAM_NOSUCH;
	}
	cksump = (u_short *)(offset + size - 2);
	if(cksum((u_short *)offset, size >> 1) != 0xffff)
		return NVRAM_CSERR;
	memcpy(bufadr,offset,size - 2);
	return NVRAM_AOK;
}

/*
 *	Nvram_Update(offset,bufadr,size)
 *
 *	Returns  NVRAM_AOK	If successful
 *		 NVRAM_ALNERR	Alignment error. offset or the size is odd
 *
 *	Assumption	The NVRAM header is already read and is available
 *			in the Nvram_header.
 *
 *	If the block being updated is new an entry is added into the
 *	header structure. The header structure contains all the blocks
 *	defined, their start address and the size.
 *
 */
Nvram_Updt(offset,bufadr,size)
byte	*offset;
byte	*bufadr;
int	size;	
{
	int	found;
	u_short *cksump;
	byte	*cp;
	int	i;

	if(((int)offset & 1) || (size & 1))
		return NVRAM_ALNERR;

	/* Verify that the block exists in the header */
	found = FALSE;
	for(i = 0; i < MAX_NVRAM_RECS && Nvram_header.nvr_recs[i].size; i++)
	{
		if(offset == Nvram_header.nvr_recs[i].start_addr)
		{
			if(size != Nvram_header.nvr_recs[i].size) {
				return NVRAM_BADSIZE;
			}
			/**
			if(size != Nvram_header.nvr_recs[i].size)
				{
				Nvram_header.nvr_recs[i].size = size;
				Nvram_Updt(NVRAM_HDR_ADDR,&Nvram_header,
					NVRAM_HDR_SIZE);
				}
			**/
			found = TRUE;
			break;
		}
	}
	if(found == FALSE)
	{
		if (i >= MAX_NVRAM_RECS)
			return NVRAM_NOSPACE;
		/* A new block is being created */
		Nvram_header.nvr_recs[i].size = size;
		Nvram_header.nvr_recs[i].start_addr = offset;
		Nvram_Updt(NvramAdd, &Nvram_header, NVRAM_HDR_SIZE);
	}
	cksump = (u_short *)(bufadr + size - 2);
	*cksump = 0;
	*cksump = ~cksum((u_short *)bufadr, size >> 1);
	EnableNvram();
	memcpy(offset,bufadr,size);
	DisableNvram();
	return NVRAM_AOK;
}

