/* admfunc.c - Defines the Administration Bus lower level primitives for the function card */
/*----------------------------------------------------------------

   Hughes LAN System.
   Mountain View, CA.

   CopyRight(c) 1991.
   All Rights Reserved

   Author:  Pedro E. Rangel

   $Log:   /b/gregs/i960/adminbus/admfunc.c_v  $
 * 
 *    Rev 1.4   12 Oct 1993 09:36:08   franks
 * No change.
 * 
 *    Rev 1.3   29 Sep 1993 10:19:40   franks
 * No change.
 * 
 *    Rev 1.2   10 Sep 1993 10:13:20   gregs
 * No change.
 * 
 *    Rev 1.1   30 Jul 1993 13:32:46   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:27:26   gregs
 * Initial revision.
 * 
 *    Rev 1.2   14 Apr 1992 16:31:26   kwok
 * Put the driver in the ACTIVE state even thought there may not
 * be an init request from the SYSCARD.
 * 
 *    Rev 1.1   13 Apr 1992 18:31:00   kwok
 * Rename adm_isr() to adm_poll(). Pass the offset to the response buffer
 * to the immediate command handler. Improve the adm_init() response time.
 * 
 *    Rev 1.0   30 Mar 1992 16:48:48   pvcs
 * Initial revision.

   Rev    Date    Who   Comments
  -----  -------- ---   ------------------------------------------------
  00.00  08/17/91 PER   Initial software version.
  ----------------------------------------------------------------
  00.01	11/08/91  KK	Test, Debug on the i960 
  ----------------------------------------------------------------
  ----------------------------------------------------------------
  00.02	02/19/92  KK	define ADM_ENTRY for the admin bus MIBs 
  ----------------------------------------------------------------
  00.03 04/09/92  KK	Change the polling function name from
			adm_isr to adm_poll to make it consistent
			with the SYSCARD and we pass the slot
			number as the parameter. 
			In adm_poll, we pass the offset to the 
			Response buffer to the immediate command
			handler.
			adm_cmd returns ok if the command has been
			received by the SYSCARD. It should be up to
			the caller to check for an "ACK" or "NAK"
			response.
  ----------------------------------------------------------------
*/
#ifdef BASE10T
#include <confsys.h>
#include <vectors.h>
#endif

#include <krnl.h>
#include <types.h>
#if !defined(KLHR) && !defined(STTG)
#include <dbd.h>
#endif
#include <netbuf.h>
#include <tcpip.h>
#include <admdrv.h>


extern	IF_ENTRY	FAR *AdmIfentry;

extern	int MyDeviceName(char *, int);
extern	int MyDeviceDescription(char *, int);
extern	int MyBootFilename(char *, int);
extern	NID *	NyNid();
extern	in_name	MyBootpServer();

ADMIN	AdmPtr;
ADMIN_ENTRY	AdminCounters;
char	FAR 	*AdminBusAddress;
int	AdmMaxProtocols;	/* we are going to support	*/
int	AdmMaxCommands;		/* immedate command we are going to support */

int AdmConState[1] = {ADM_CON_CLOSED};
int (FAR **ProtocolTable)();
int (FAR **ImmRspTable)();

extern int FAR DummyHandler();
int FAR iRspCmdHandler();

#ifdef BASE10T
void (F_TYPE INTERRUPT FAR *oldptr)() = NULL;
void INTERRUPT FAR adm_irss();
#endif

static SEM	CmdSem;		/* Immediate Command Semaphore */
static volatile int AdmState = ADM_INACTIVE;

/*----------------------------------------------------------------
	Initialializes the Administration Bus.  This function 
	clears the ADMINISTRATION BUS shared RAM structure.  It
	detects if the System Card is present. If present then it
	will establish and initialization procedure.
  ----------------------------------------------------------------*/
adm_init(StartAdd, max_prot, max_cmd)
char FAR	*StartAdd;	/* starting address for the admin bus memory */
int	max_prot;	/* max number of protocols to be supported */
int	max_cmd;	/* max number of immediate commands to be supported */

	{
	ulong TimeOut;
	int i;

	AdminBusAddress = StartAdd;
	AdmMaxProtocols = max_prot; 
	AdmMaxCommands = max_cmd;
	CreatSemaphore(&CmdSem, 1);	/* only one command is active */
	ADM_WRITE(0, AdmPtr.shmInitialized, AdmPtr, 0);
	AdmState = ADM_INACTIVE;
	ProtocolTable = (int(FAR **)())lmalloc((max_prot + 1) * sizeof(int(FAR *)()));
	ImmRspTable = (int(FAR **)())lmalloc((max_cmd + 1) * sizeof(int(FAR *)()));
	if (ProtocolTable == NULL || ImmRspTable == NULL)
		{
#ifdef BASE10T
		AdmDebugMessage("Admin bus error: Not enough memory\n");
		blinking();
#endif
		return E_FAILURE;
		}
	for (i = 0; i <= max_prot; i++)
		*(ProtocolTable + i) = DummyHandler;
	for (i = 0; i <= max_cmd; i++)
		*(ImmRspTable + i) = iRspCmdHandler;
	if ( IsValidSlot(0) != 0 )
		{
		TimeOut = RealTimeTicks() + TIMEOUT;
		while ( RealTimeTicks() < TimeOut)
			{
			adm_poll(0);
			if (AdmState == ADM_ACTIVE)
				break;
#ifdef i960
			ReSchedule();
#else
			ReSchedule(IDLE_PRIORITY);
#endif
			}
		/*
		 *	Put myself in a ready state.
		 */
		if (AdmState == ADM_INACTIVE)
			{
			Initialize();
			AdmState = ADM_ACTIVE;
			}
		}
#ifdef BASE10T
	oldptr = GetVector(IDLE_TASK);
    	SetVector(IDLE_TASK, adm_irss);
#endif
	return 0;
	}
/*----------------------------------------------------------------
	Stop the device driver. Tell the relayer card to stop
	receiving frames.
--------------------------------------------------------------------*/
adm_stop()
	{
#ifdef BASE10T
	if (oldptr != NULL)
		SetVector(IDLE_TASK, oldptr);
#endif
	}
/*----------------------------------------------------------------
	This function clears the ADMIN BUS shared RAM, and initializes
	it to the default parameters.
  ----------------------------------------------------------------*/
Initialize()
	{
	int	i;
	int	TxBuffers;
	int	RxBuffers;
	ADMIN	*admp = &AdmPtr;
	
#ifdef ADM_DEBUG
	AdmDebugMessage("initialize\n");
#endif
	AdminCounters.AdmInits++;
	memset((char FAR *)admp, '\0', ADMSIZE);
	adm_memset(0, '\0', SHARED_SIZE);	/* clear the whole shared memory */
	TxBuffers = TXBUFFOFF;
	RxBuffers = RXBUFFOFF;
	admp->shmMaxBuffers = ADMBUFF;
	for ( i = 0; i < ADMBUFF; i++ )
		{
		admp->shmFromBuffers[i]= TxBuffers;
		TxBuffers += BUFFSIZE;
		}
	for ( i = 0; i < ADMBUFF; i++ )
		{
		admp->shmToBuffers[i]= RxBuffers;
		RxBuffers += BUFFSIZE;
		}
	/*
	 *	Immediate command response buffers.
	 */
	admp->shmToImmedCmdOff = CMDTOBUFOFF;
	admp->shmToImmedRspOff = RSPTOBUFOFF;
	admp->shmFromImmedCmdOff = CMDFROMBUFOFF;
	admp->shmFromImmedRspOff = RSPFROMBUFOFF;
	admp->shmImmedSize = CMD_RSP_BUF_SIZE;
	/*
	 *	Virtual console buffers.
	 */
	admp->shmConToBufOffset = ADMCONB_TO_OFFSET;
	admp->shmConFromBufOffset = ADMCONB_FROM_OFFSET;
	admp->shmConBufSize = ADMCONB_SIZE;
	/*
	 *	My personaliies ...
	 */
	admp->shmState = ADM_STATE_OPERATIONAL;		/* Card state is ok*/
#ifdef i960
	ncopy((NID FAR *)admp->shmPhysicalAddress, (NID FAR *)MyNid(0));
#else
	ncopy((NID FAR *)admp->shmPhysicalAddress, (NID FAR *)MyNid());
#endif
	admp->shmIpAddress = _initp->in_me;
	admp->shmRouterAddress = _initp->net_gway;
	admp->shmNetMask = _initp->net_mask;
	admp->shmBootServer = MyBootpServer();
	MyBootFilename(admp->shmBootFile, ADM_BOOTFILE_SIZE);
	MyDeviceName(admp->shmDeviceName, ADM_DEVICENAME_SIZE);
	MyDeviceDescription(admp->shmDeviceDescription, ADM_DEVICEDESCR_SIZE);
	InitAdmConMap(admp->shmConMap);
	admp->shmVersion = ADM_VERSION;
	admp->shmPower = PowerRequired();
	admp->shmSnmpAuth = snmp_access();
	MySnmpCommunity(admp->shmSnmpCommunity, ADM_COMMUNITY_SIZE);
	adm_write(0, 0, ADMSIZE, (char FAR*)admp);
	_initp->relayer = FALSE;
	AdmConState[0] = ADM_CON_CLOSED;
	AdmCloseUpcall(0);
	}

/*
 *	Set the state of the FUNCARD.
 */
AdmSetState(int state)
	{
	AdmPtr.shmState = state;
	ADM_WRITE(0, AdmPtr.shmState, AdmPtr, state);
	}
AdmSetNid(NID FAR *nid)
	{
#ifdef i960
	ncopy((NID FAR *)AdmPtr.shmPhysicalAddress, (NID FAR *)MyNid(0));
#else
	ncopy((NID FAR *)AdmPtr.shmPhysicalAddress, (NID FAR *)MyNid());
#endif
	adm_write(0, (char *)&AdmPtr.shmPhysicalAddress[0] - (char *)&AdmPtr,
		sizeof(AdmPtr.shmPhysicalAddress), (char FAR *)nid);
	}
AdmSetIP(in_name ip)
	{
	AdmPtr.shmIpAddress = ip;
	ADM_WRITE(0, AdmPtr.shmIpAddress, AdmPtr, AdmPtr.shmIpAddress);
	/*
	 *	If I am the relayer and I have a new ip,
	 *	then I'll send an immediate command to the syscard.
	 */	
	if (_initp->relayer != FALSE && ip != AdmPtr.shmIpAddress)
		AdmChangeIP(ip);
	}
AdmSetRouter(in_name ip)
	{
	AdmPtr.shmRouterAddress = ip;
	ADM_WRITE(0, AdmPtr.shmRouterAddress, AdmPtr, AdmPtr.shmRouterAddress);
	}

AdmSetNetMask(in_name netmask)
	{
	AdmPtr.shmNetMask = netmask;
	ADM_WRITE(0, AdmPtr.shmNetMask, AdmPtr, AdmPtr.shmNetMask);
	/*
	 *	If I am the relayer and I have a new netmask,
	 *	then I'll send an immediate command to the syscard.
	 */	
	if (_initp->relayer != FALSE && netmask != AdmPtr.shmNetMask)
		AdmChangeNetMask(netmask);
	}

AdmSetBootServer(in_name ip)
	{
	AdmPtr.shmBootServer = ip;
	ADM_WRITE(0, AdmPtr.shmBootServer, AdmPtr, AdmPtr.shmBootServer);
	}
AdmSetConMap(int index, int value)
	{
	AdmPtr.shmConMap[index] = value;
	ADM_WRITE(0, AdmPtr.shmConMap[index], AdmPtr, AdmPtr.shmConMap[index]);
	}
AdmSetBootFile(char FAR * filename)
	{
	strcpy((char FAR *)AdmPtr.shmBootFile, (char FAR *)filename);
	adm_write(0, (char *)&AdmPtr.shmBootFile[0] - (char *)&AdmPtr, 
		strlen((char FAR *)filename) + 1, filename);
	}

AdmUpdateAliveCounter()
	{
	AdmPtr.shmAliveCounter++;
	ADM_WRITE(0, AdmPtr.shmAliveCounter, AdmPtr, AdmPtr.shmAliveCounter);
	}

AdmSetSnmpAuth(int auth)
	{
	AdmPtr.shmSnmpAuth = auth;
	ADM_WRITE(0, AdmPtr.shmSnmpAuth, AdmPtr, auth);
	}

AdmSetCommunity(char FAR *community)
	{
	strcpy((char FAR *)AdmPtr.shmSnmpCommunity, (char FAR *)community);
	adm_write(0, (char *)&AdmPtr.shmSnmpCommunity[0] - (char *)&AdmPtr, 
		strlen((char FAR *)community) + 1, community);
	}
/*----------------------------------------------------------------
	Send a buffer through the ADMIN BUS interface.  This function
	is a asynchronous function ( it will return once the buffer has
	been successfully written int the shared ram). 
  ----------------------------------------------------------------*/
adm_send(int slot, char FAR*buff, uint len)
	{
	short AckBuffList, TxBuffList, BufFillTmp;
	int	err = E_FAILURE;
	int	i;
	int	FillBit;
	ulong	TimeOut;

	if ( AdmState != ADM_ACTIVE || IsValidSlot(0) == 0 || 
		len > BUFFSIZE)
		{
		AdmIfentry->outerrpkts++;
		return(err);		/* return operation error */
		}
	ADM_READ(slot, AdmPtr.shmFromBufFill, AdmPtr);
	TxBuffList = AdmPtr.shmFromBufFill;
	ADM_READ(slot, AdmPtr.shmToBufAck, AdmPtr);
	AckBuffList = AdmPtr.shmToBufAck;
	if ((i = GetFreeBuffer(TxBuffList, AckBuffList)) == -1)
		{
		AdminCounters.AdmTimeouts++;
		AdmIfentry->outdiscardpkts++;
		return -2; /*E_NORESOURCES*/;
		}
	/* We have found a buffer */
	/*
	 *	find out where is the buffer
	 */
	ADM_READ(slot, AdmPtr.shmFromBuffers[i], AdmPtr);
	/*
	 *	Copy the packet to the buffer
	 */
	adm_write(slot, AdmPtr.shmFromBuffers[i], len, (char FAR *)buff);
	/*
	 *	Set the bit to tell the FUNCARD that the
	 *	buffer is full.
	 */
	FillBit = 1 << i;
	TxBuffList |= FillBit;
	ADM_WRITE(slot, AdmPtr.shmFromBufFill, AdmPtr, TxBuffList);
	AdmIfentry->outbytecnt += len;
	AdmIfentry->outucastpkts++;
	err = E_SUCCESS;
	return err;
	}
#ifdef BASE10T
/*----------------------------------------------------------------
	This interrupt service routine gets call by the USER TIMER
	interrupt service routine.
  ----------------------------------------------------------------*/
void INTERRUPT FAR adm_irss()
	{
	int	i;

	if (i++ > 10)
		{
		AdmUpdateAliveCounter();
		i = 0;
		}
	adm_poll(0);			/* service interrupt */
	}

#endif
/*----------------------------------------------------------------
	Administration Bus polling/interrupt routine 
  ----------------------------------------------------------------*/
adm_poll(slot)
int	slot;	/* slot number, 0 for the FUNCARD	*/

	{
	int	Cmd;

#ifdef BASE10T
	Ei();
#endif

	ADM_READ(slot, AdmPtr.shmInitRequest, AdmPtr);
	if ( AdmPtr.shmInitRequest == 1 )
		{
		/*
		 *	init request from the SYSCARD.
		 */
		ADM_WRITE(slot, AdmPtr.shmInitialized, AdmPtr, 0);
 		ADM_WRITE(slot, AdmPtr.shmInitRequest, AdmPtr, 0);
		Initialize();
		ADM_WRITE(slot, AdmPtr.shmInitialized, AdmPtr, 1);
		AdmState = ADM_ACTIVE;
		AdmConState[slot] = ADM_CON_CLOSED;
		}
	else if (AdmState == ADM_ACTIVE)
		{	/* process command */
		ADM_READ(slot, AdmPtr.shmToImmedCmdAct, AdmPtr);
		ADM_READ(slot, AdmPtr.shmToImmedCmdOff, AdmPtr);
		if ((Cmd = AdmPtr.shmToImmedCmdAct) != 0) 
			{
			if (Cmd < ICMDMIN  || ICMDMAX < Cmd)
				{
				/*
				 * If command is out of range, return
				 * a Nak. (*ImmRspTable[0])() should
				 * handle this.
				 */
				Cmd = 0;
				}
			/* perform command */
			AdminCounters.AdmRcvdICmds++;
			(**(ImmRspTable + Cmd))(slot, Cmd, 
				AdmPtr.shmToImmedCmdOff,
				AdmPtr.shmFromImmedRspOff);
			/* tell the SYSCARD we are done */
			ADM_WRITE(slot, AdmPtr.shmToImmedCmdAct, AdmPtr, 0);
			}
		CheckRxBufs(slot, 
			(char *)&AdmPtr.shmToBufFill - (char*)&AdmPtr,
			(char *)&AdmPtr.shmFromBufAck - (char*)&AdmPtr,
			(char *)&AdmPtr.shmToBuffers[0] - (char*)&AdmPtr);
		CheckTxCompletion(slot,
			(char *)&AdmPtr.shmFromBufFill - (char*)&AdmPtr,
			(char *)&AdmPtr.shmToBufAck - (char*)&AdmPtr);
		}
	}

/*----------------------------------------------------------------
	This is the handler for the Immediate Command.  Once the
	application has initialized itself. It must add the appropriate
	handlers to service the Immediate commands from the System Card.
  ----------------------------------------------------------------*/
int FAR iRspCmdHandler(int slot, int Cmd, int CmdOffset, int RspOffset)
	{
	ADM_IRSP Rsp;

	AdminCounters.AdmUnknownICmds++;
	memset((char FAR *)&Rsp, '\0', sizeof(Rsp));
	Rsp.iRspType = IRSPNACK;
	Rsp.iRspLength = sizeof(Rsp);
	Rsp.iRspData.iRspReason = IRSPNAK_NOT_SUPPORTED;
	adm_write(slot, RspOffset, sizeof(Rsp), (char FAR *)&Rsp);
	}

/*----------------------------------------------------------------
	Send an immediate command to the SYSCARD Card.
  ----------------------------------------------------------------*/
adm_cmd(int slot, ADM_ICMD *CmdPtr, int length, ADM_IRSP *RspBuffer)
/*
 *	slot:	1 to (MAXSLOT - 1)
 *	CmdPtr:	the comamnd buffer
 *	length:	length of *CmdPtr
 *	RspBuffer:	Response to be returned to here
 */
	{
	int err = E_FAILURE;
	ulong TimeOut;

	AdminCounters.AdmIssuedICmds++;
	/* has the system been initialized */
	if ( !IsValidSlot(0) || AdmState != ADM_ACTIVE )
		return err;
	RcvSignal(&CmdSem);	/* block if somebody is using command */
	/*
	 *	Get the offset to the command buffer,
	 *	write the command to the command buffer in the shared ram,
	 *	set the ImmedCmdAct to be non-zero,
	 *	send an interrup signal to the FUNCARD and
	 *	wait for the response.
	 */
	ADM_READ(slot, AdmPtr.shmFromImmedCmdOff, AdmPtr);
	adm_write(slot, AdmPtr.shmFromImmedCmdOff, length, (char FAR *)CmdPtr);
	ADM_WRITE(slot, AdmPtr.shmFromImmedCmdAct, AdmPtr, 
		CmdPtr->iCmdType);
	ADM_READ(slot, AdmPtr.shmFromImmedCmdAct, AdmPtr);
	TimeOut = RealTimeTicks() + TIMEOUT;
	while (  AdmPtr.shmFromImmedCmdAct != 0 && RealTimeTicks() < TimeOut )
		{
#ifdef i960
		ReSchedule();
#else
		ReSchedule(IDLE_PRIORITY);
#endif
		ADM_READ(slot, AdmPtr.shmFromImmedCmdAct, AdmPtr);
		}
	if ( AdmPtr.shmFromImmedCmdAct == 0 )/* command completed */
		{
		ADM_READ(slot, AdmPtr.shmToImmedRspOff, AdmPtr);
		adm_read(slot, AdmPtr.shmToImmedRspOff, 
			sizeof(ADM_IRSP), (char FAR *)RspBuffer);
		err = E_SUCCESS;
		}
	else
		AdminCounters.AdmTimeouts++;
	SendSignal(&CmdSem);	/* let other commands come through */
	return(err);		/* return operation error */
	}

/*
 * name		AdmFuncardReset	
 *
 * synopsis	AdmFuncardReset()
 *
 * description	Send the "I am going to re-boot" immediate command 
 *		from the FUNCARD to the SYSCARD.
 *
 * returns	0		SYSCARD has accepted the command	
 *		non-zero	error code
 */

AdmFuncardReset()

	{
	int	dummy;
	/*
	 *	send a "I am going to re-boot"  immediate command message to the
	 *	SYSCARD.
	 */
	return AdmImmedCommand(0, ICMD_FUNCARD_RESET, (void FAR *)&dummy, 0);
	}

/*
 * name		AdmChangeIP
 *
 * synopsis	AdmChangeIP(ip)
 *		in_name	ip;	<< the new ip address
 *
 * description	Send the "my ip has been changed" immediate command 
 *		from the FUNCARD to the SYSCARD.
 *
 * returns	0		SYSCARD has accepted the command	
 *		non-zero	error code
 */

AdmChangeIP(in_name ip)

	{
	return AdmImmedCommand(0, ICMDCHANGEIP, (void FAR *)&ip, sizeof(ip));
	}

/*
 * name		AdmChangeNetMask
 *
 * synopsis	AdmChangeNetMask(NetMask)
 *		in_name	NetMask;	<< the new net mask
 *
 * description	Send the "my netmask has been changed" immediate command 
 *		from the FUNCARD to the SYSCARD.
 *
 * returns	0		SYSCARD has accepted the command	
 *		non-zero	error code
 */

AdmChangeNetMask(in_name NetMask)

	{
	return AdmImmedCommand(0, ICMD_CHANGE_NETMASK, 
		(void FAR *)&NetMask, sizeof(NetMask));
	}

AmISysCard()
	{
	return 0; /* No, I am not the system card	*/
	}


