/**			       
*
*	Program Name:	Admin Bus virtual console driver
*
*	Filename:	admcon.c
*
*	$Log:   /b/gregs/i960/adminbus/admcon.c_v  $
 * 
 *    Rev 1.4   12 Oct 1993 09:36:06   franks
 * No change.
 * 
 *    Rev 1.3   29 Sep 1993 10:19:38   franks
 * No change.
 * 
 *    Rev 1.2   10 Sep 1993 10:13:18   gregs
 * No change.
 * 
 *    Rev 1.1   30 Jul 1993 13:32:44   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:27:26   gregs
 * Initial revision.
 * 
 *    Rev 1.1   13 Apr 1992 18:30:14   kwok
 * Check if the immediate command response is an ACK or a NACK after
 * calling adm_cmd().
 * 
 *    Rev 1.0   30 Mar 1992 16:48:48   pvcs
 * Initial revision.
*
*	Creation Date:	10.30.91
*
*	Date:		2.19.92	
*
*	Version:	1.0
*
*	Programmers:	K Kong
*
*	Modifications:	2.19.92	K Kong
*			Adding support for the Admin bus MIBs.
*			The console cannot be opened more than once.
*
*	Comments:	The file contains the functions to 
*			"open", "close", "read" and "write" 
*			from/to the admin bus virtual console.
*			The admin bus virtual console supports
*			"raw" mode operation only. It is up to the 
*			individual FUNCARD to interpret the character(s).
*
*	Two Console i/o buffers are defined in the shared
*	ram area - one for input and one for output. the format of the
*	console i/o buffers are:-
*
*	struct
*		{
*		short	length;		size of the circular buffer.
*		short	NextGetc;	index to the circular 
*					buffer to get the next character.  
*					The recipient updates it.
*		short	NextPutc;	index to the circular buffer to
*				 	put the next character. 
*					The sender updates it. 
*		char	CirBuffer[];	the circular buffer.
*		}
*		ADMCONB;
*	
*	Two immediate commands are defined, OPEN and CLOSE. Only the 
*	SYSCARD can issue the OPEN command and CLOSE can be issued by
*	either the SYSCARD or the FUNCARD.
*	
*	To open a virtual console connection from the SYSCARD to the FUNCARD
*	will be:=
*	
*	1	SYSCARD issues the OPEN command.
*
*	2	FUNCARD responses with IRSPACK to accept the OPEN request.
*
*	3	SYSCARD sends console input to the FUNCARD by writing to the
*		"CirBuffer" in the "input" ADMCONB buffer. "NextPutc"
*		in the "input" ADMCONB buffer is updated. 
*
*	4	FUNCARD polls the "NextGetc" in the "input" ADMCONB 
*		buffer to check for any console input. If "NextPutc" is 
*		not the same as "NextGetc", then there is character in 
*		the "CirBuffer".  The FUNCARD updates "NextGetc" 
*		after reading data from the "CirBuffer".
*
*	5	FUNCARD sends console output to the FUNCARD by writing to the
*		"CirBuffer" in the "output" ADMCONB buffer. "NextPutc"
*		in the "output" ADMCONB buffer is updated. 
*
*	6	SYSCARD polls the "NextGetc" in the "output" ADMCONB 
*		buffer to check for any console output. If "NextPutc" is 
*		not the same as "NextGetc", then there is character in 
*		the "CirBuffer".  The SYSCARD updates "NextGetc" 
*		after reading data from the "CirBuffer".
*
*	7	Procedures 3, 4, 5, or 6 are repeated util either the SYSCARD 
*		or the FUNCARD issues the CLOSE command.
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/
#if defined(SYSCARD) || defined(BASE10T)
#include <portdefs.h>
#endif

#include <types.h>
#if !defined(KLHR) && !defined(STTG)
#include <krnl.h>
#include <dbd.h>
#endif
#include <netbuf.h>
#include <admdrv.h>

#define	min(a,b)	((a) > (b) ? (b) : (a))

#ifndef SYSCARD
extern	ADMIN	AdmPtr;
#else
extern int AdmState[];
#endif

/*
 * name		AdmConOpen	open an adm virtual console to a FUNCARD
 *
 * synopsis	AdmConOpen(slot)
 *		int	slot; <<	to the FUNCARD in this slot
 *					1 <= slot <= 13.
 *
 * description	It tries to open an adm virtual console connection to 
 *		a FUNCARD which is in the slot "slot". If the connection 
 *		has been established already, an error will be returned.
 *		The FUNCARD will be responsible for the initialization
 *		of the console input/output command area, i.e. "ADMCONB".
 *		The FUNCARD cannot OPEN the virtual console.
 *
 * returns	E_SUCCESS	connection has been established
 *		non-zero	error code
 *				- IRSPNAK_BUSY
 */

AdmConOpen(int slot)

	{
	ADM_ICMD	Cmd;	/* the command buffer	*/
	ADM_IRSP	Rsp;	/* the response buffer	*/
	int	ret;	/* return code		*/

	/*
	 *	The FUNCARD cannot "open" the virtual console.
	 */
	if (!AmISysCard())
		return E_FAILURE;
	/*
	 *	Return an error if it is opened
	 *	already.
	 */
	if (AdmConState[slot] == ADM_CON_OPENED)
		{
		return E_FAILURE;
		/***	K Kong 2.19.92, We do not allow more than 1 
		 *	open.
		AdmConClose(slot);
		*****/
		}
	/*
	 *	send a "open" immediate command message to the
	 *	FUNCARD on slot "slot".
	 */
	memset((char FAR *)&Rsp, '\0', sizeof(Rsp));
	Cmd.iCmdType = ICMD_CONS_OPEN;
	Cmd.iCmdLength = sizeof(Cmd.iCmdType) + sizeof(Cmd.iCmdLength);
	ret = adm_cmd(slot, &Cmd, Cmd.iCmdLength, &Rsp);
	if (ret == E_SUCCESS)
		{
		if (Rsp.iRspType == IRSPACK)
			{
			/*
			 *	ACK response, connection opened.
			 */
			AdmConState[slot] = ADM_CON_OPENED;
#ifdef SYSCARD
			AdmSlotState[slot].LastConnect = RealTimeTicks();
#endif
			}
		else
			{
			/*
			 *	If we get a NACK, 
			 *	we return the reason for the NACK to
			 *	the caller.
			 */
			ret = Rsp.iRspData.iRspReason;
			}
		}
	return ret;
	}

/*
 * name		AdmConClose	close an adm virtual console to a FUNCARD
 *
 * synopsis	AdmConClose(slot)
 *		int	slot; <<	In the SYSCARD:-
 *					to the FUNCARD in this slot
 *					1 <= slot <= 13.
 *					In the FUNCARD:-
 *					0
 *
 * description	For the SYSCARD, it closes the adm virtual console 
 *		connection to a FUNCARD which is in the slot "slot".  
 *		For the FUNCARD, it closes the adm virtual console
 *		to the SYSCARD.
 *		It is an error to close a "closed" console.
 *		All the data in the console input/output command area will
 *		be ignored.
 *
 * returns	0		connection has been closed
 *		non-zero	error code
 *				- IRSPNAK_NOTOPEN
 */

AdmConClose(int slot)

	{
	int	dummy;
	int	ret;	/* return code		*/

	if (AdmConState[slot] != ADM_CON_OPENED)
		{
		return IRSPNAK_NOTOPEN;
		}
	/*
	 *	Check if the card has been unplugged from the HUB.
	 */
#ifdef SYSCARD
	AdmSlotState[slot].LastDisconnect = RealTimeTicks();
	if (!IsValidSlot(slot) || AdmState[slot] != ADM_ACTIVE)
#else
	if (!IsValidSlot(slot))
#endif
		{
		AdmConState[slot] = ADM_CON_CLOSED;
		return 0;
		}
	/*
	 *	send a "close" immediate command message to the
	 *	FUNCARD on slot "slot".
	 */
	ret = AdmImmedCommand(slot, ICMD_CONS_CLOSE, (void FAR *)&dummy, 0);
	AdmConState[slot] = ADM_CON_CLOSED;
	return ret;
	}

/*
 * name		AdmGetch	get a character from the adm virtual console
 *
 * synopsis	int AdmGetch(slot)
 *		int	slot; <<	In the SYSCARD:-
 *					to the FUNCARD in this slot
 *					1 <= slot <= 13.
 *					In the FUNCARD:-
 *					0
 *
 * description	It reads without echoing a single character from the
 *		adm virtual console connection. 
 *
 * returns	the character read
 *		-1	there is no character to be read.
 */

AdmGetch(int slot)

	{
	ADMCONB	admcon;	/* the control data in the ADMCONB	*/
	byte	c;		/* put the character in here		*/
	uint	AdmConBOffset;	/* offset to the ADMCONB		*/
	uint	AdmCirBufOffset;	/* offset to the circular buffer */
	int	CircularOffset;	/* from the beginning of the ADMCONB	*/

	if (AdmConState[slot] != ADM_CON_OPENED)
		return -1;
	/*
	 *	Check if the card has been unplugged from the HUB.
	 */
#ifdef SYSCARD
	if (!IsValidSlot(slot) || AdmState[slot] != ADM_ACTIVE)
#else
	if (!IsValidSlot(slot))
#endif
		{
		AdmConState[slot] = ADM_CON_CLOSED;
#ifdef SYSCARD
		AdmSlotState[slot].LastDisconnect = RealTimeTicks();
#endif
		return -1;
		}
	/*
	 *	We use the "FROM" buffer for the SYSCARD and
	 *	the "TO" buffer for the FUNCARD.
	 */
	AdmConBOffset = AmISysCard() ? ADMCONB_FROM_OFFSET : ADMCONB_TO_OFFSET;
	CircularOffset = ((char *)&admcon.CirBuffer[0] - (char *)&admcon);

	AdmCirBufOffset = AdmConBOffset + CircularOffset; 
	adm_read(slot, AdmConBOffset, CircularOffset, (char FAR*)&admcon);
	if (admcon.NextGetc != admcon.NextPutc)
		{
		/*
		 *	There are characters to be read.
		 */
		adm_read(slot, AdmCirBufOffset + admcon.NextGetc,
			1, (char FAR*)&c);
		/*
		 *	increment NextGetc and wrap around if reach the
		 *	end of the circular buffer.
		 */
		admcon.NextGetc++;
		admcon.length--;
		admcon.NextGetc &= admcon.length;
		/*
		 *	update "NextGetc" in the ADMCONB
		 */
		adm_write(slot, 
			AdmConBOffset + ((char*)&admcon.NextGetc - (char*)&admcon),
			sizeof(admcon.NextGetc), 
			(char FAR*)&admcon.NextGetc);
		AdminCounters.AdmConsoleRcvdOctets++;
		return (int)c;
		}
	return -1;
	}
/*
 * name		AdmPutch	put a character to the adm virtual console
 *
 * synopsis	int AdmPutch(slot, c)
 *		int	slot; <<	In the SYSCARD:-
 *					to the FUNCARD in this slot
 *					1 <= slot <= 13.
 *					In the FUNCARD:-
 *					0
 *
 *		int	c; <<		character to be output.
 *					The upper 8 bits will be ignored.
 *
 * description	It writes a single character to the
 *		adm virtual console connection. 
 *
 * returns	c	successful.
 *		-1	there is no character to be read.
 */

AdmPutch(int slot, int c)
	{
	ADMCONB	admcon;	/* the control data in the ADMCONB	*/
	int	AdmConBOffset;	/* offset to the ADMCONB		*/
	int	AdmCirBufOffset;	/* offset to the circular buffer */
	int	CircularOffset;	/* from the beginning of the ADMCONB	*/
	/*
	 *	we do this because (*((char *)&c) may not equal to c
	 */
	byte	ch = (byte)c;

	if (AdmConState[slot] != ADM_CON_OPENED)
		return -1;
	/*
	 *	Check if the card has been unplugged from the HUB.
	 */
#ifdef SYSCARD
	if (!IsValidSlot(slot) || AdmState[slot] != ADM_ACTIVE)
#else
	if (!IsValidSlot(slot))
#endif
		{
		AdmConState[slot] = ADM_CON_CLOSED;
#ifdef SYSCARD
		AdmSlotState[slot].LastDisconnect = RealTimeTicks();
#endif
		return -1;
		}
	AdmConBOffset = AmISysCard() ? ADMCONB_TO_OFFSET : ADMCONB_FROM_OFFSET;
	CircularOffset = ((char *)&admcon.CirBuffer[0] - (char *)&admcon);
	AdmCirBufOffset = AdmConBOffset + CircularOffset; 
	/*
	 *	get the control data in the ADMCONB.
	 */
	adm_read(slot, AdmConBOffset, CircularOffset, (char FAR*)&admcon);
	/*
	 *	The circular buffer is full if
	 *	1	NextGetc - NextPutc == 1 or
	 *	2	NextGetc == 0 && NextPutc == length
	 */
	if ((admcon.NextGetc - admcon.NextPutc - 1) & --admcon.length)
		{
		/*
		 *	The circular buffer is not full
		 *	write the character into the circular buffer.
		 */
		adm_write(slot, AdmCirBufOffset + admcon.NextPutc,
			1, (char FAR *)&ch);
		/*
		 *	increment NextPutc and wrap around if reach the
		 *	end of the circular buffer.
		 */
		admcon.NextPutc++;
		admcon.NextPutc &= admcon.length;
		/*
		 *	update "NextPutc" in the ADMCONB
		 */
		adm_write(slot, 
			AdmConBOffset + ((char *)&admcon.NextPutc - (char *)&admcon),
			sizeof(admcon.NextPutc), 
			(char FAR*)&admcon.NextPutc);
		AdminCounters.AdmConsoleSentOctets++;
		return c;
		}
	return -1;
	}


/*
 * name		AdmPuts
 *
 * synopsis	AdmPuts(slot, s)
 *		int	slot; <<	In the SYSCARD:-
 *					to the FUNCARD in this slot
 *					1 <= slot <= 13.
 *					In the FUNCARD:-
 *					slot = 0.
 *		char	*s; <<		string to be output
 *
 * description	It writes a string to the adm virtual console connection.
 *
 * returns	number of bytes have been output.
 */

AdmPuts(int slot, char *s)

	{
	int	ret;
	int	length = 0;

		
	while (*s != '\0')
		{
		if ((ret = AdmPutch(slot, *s)) == -1)
			{
			if (!IsAdmConOpened(slot))
				break;
			ReSchedule(IDLE_PRIORITY);
			}
		else
			{
			s++;
			length++;
			}
		}
	return length;
	}

/*
 * name		AdmGets
 *
 * synopsis	int AdmGets(slot, buffer)
 *		int	slot; <<	In the SYSCARD:-
 *					to the FUNCARD in this slot
 *					1 <= slot <= 13.
 *					In the FUNCARD:-
 *					slot = 0.
 *		char	buffer[]; >>	String will be copied into here.
 *					buffer should of size
 *					CIRBUFFER_SIZE.
 *
 * description	It reads all the available characters from the input buffer
 *		and returns them in "buffer".
 *
 * returns	length of data copied into "buffer"
 */

AdmGets(int slot, char buffer[])

	{
	ADMCONB	admcon;	/* the control data in the ADMCONB	*/
	uint	AdmConBOffset;	/* offset to the ADMCONB		*/
	uint	AdmCirBufOffset;	/* offset to the circular buffer */
	int	CircularOffset;	/* from the beginning of the ADMCONB	*/
	int	length = 0;

	if (AdmConState[slot] != ADM_CON_OPENED)
		return 0;
	/*
	 *	Check if the card has been unplugged from the HUB.
	 */
#ifdef SYSCARD
	if (!IsValidSlot(slot) || AdmState[slot] != ADM_ACTIVE)
#else
	if (!IsValidSlot(slot))
#endif
		{
		AdmConState[slot] = ADM_CON_CLOSED;
#ifdef SYSCARD
		AdmSlotState[slot].LastDisconnect = RealTimeTicks();
#endif
		return 0;
		}
	/*
	 *	We use the "FROM" buffer for the SYSCARD and
	 *	the "TO" buffer for the FUNCARD.
	 */
	AdmConBOffset = AmISysCard() ? ADMCONB_FROM_OFFSET : ADMCONB_TO_OFFSET;
	CircularOffset = ((char *)&admcon.CirBuffer[0] - (char *)&admcon);
	AdmCirBufOffset = AdmConBOffset + CircularOffset; 
	adm_read(slot, AdmConBOffset, CircularOffset, (char FAR*)&admcon);
	if (admcon.NextGetc == admcon.NextPutc)
		{
		/*
		 *	there are no character to be read.
		 */
		return 0;
		}
	/*
	 *	There are data in the buffer to be read.
	 *	If "putc" > "getc"
	 *		then there are "putc - getc" bytes of data in the
	 *		circular buffer.
	 *	else	("getc" > "putc")
	 *		there are "buffersize - getc" + "putc" bytes of data
	 *		in the circular buffer.
	 */
	if (admcon.NextPutc > admcon.NextGetc)
		{
		length = admcon.NextPutc - admcon.NextGetc;
		adm_read(slot, AdmCirBufOffset + admcon.NextGetc,
			length, (char FAR *)buffer);
		/*
		 *	update "NextGetc" in the ADMCONB
		 */
		admcon.NextGetc += length;
		}
	else	/* admcon.NextGetc > admcon.NextPutc */
		{
		length = admcon.length - admcon.NextGetc;
		adm_read(slot, AdmCirBufOffset + admcon.NextGetc,
			length, (char FAR *)buffer);
		adm_read(slot, AdmCirBufOffset,
			admcon.NextPutc, (char FAR *)&buffer[length]);
		length += admcon.NextPutc;
		/*
		 *	update "NextGetc" in the ADMCONB
		 */
		admcon.NextGetc = admcon.NextPutc;
		}
	/*
	 *	Update "NextGetc".
	 */
	adm_write(slot, 
		AdmConBOffset + ((char*)&admcon.NextGetc - (char*)&admcon),
		sizeof(admcon.NextGetc), 
		(char FAR *)&admcon.NextGetc);
	buffer[length] = '\0';
	AdminCounters.AdmConsoleRcvdOctets += length;
	return length;
	}

IsAdmConOpened(int slot)
	{
#ifdef SYSCARD
	return (AdmConState[slot] == ADM_CON_OPENED && AdmState[slot] == ADM_ACTIVE);
#else
	return (AdmConState[slot] == ADM_CON_OPENED);
#endif
	}

#ifndef SYSCARD
/*
 * name		AdmOpenRequestInit
 *
 * synopsis	AdmOpenRequestInit(slot)
 *		int	slot; <<	open requested from this slot.
 *					for SYSCARD:-
 *					1 <= slot <= 13.
 *					for FUNCARD:-
 *					slot = 0.
 *
 * description	It initializes the Adm console data structure in a ready 
 *		state. This function should be called to put the 
 *		console driver in a ready state when an "open"
 *		request has been received and the application has 
 *		accepted the "open".
 *
 * returns	nothing
 */

void AdmOpenRequestInit(int slot)

	{
	ADMCONB	ConBuf;		/* the console buffer	*/

	/*
	 *	The application has accepted the open request
	 *	We have to initialize the "ADMCONB" for both
	 *	input and output. 
	 */
	/*
	 *	Get the offset of the to-FUNCARD buffer, from-FUNCARD
	 *	buffer and the length of the buffers.
	 */
	ADM_READ(slot, AdmPtr.shmConToBufOffset, AdmPtr);
	ADM_READ(slot, AdmPtr.shmConFromBufOffset, AdmPtr);
	ADM_READ(slot, AdmPtr.shmConBufSize, AdmPtr);
	/*
	 *	Initailise the to-FUNCARD and the from-FUNCARD
	 *	console i/o buffer
	 */
	ConBuf.length = min(CIRBUFFER_SIZE, AdmPtr.shmConBufSize - ADMCONB_UNPADDED_CTRL_SIZE);
	ConBuf.NextGetc = 0;
	ConBuf.NextPutc = 0;
	ConBuf.reserved = 0;
	adm_write(slot, AdmPtr.shmConToBufOffset, sizeof(ConBuf), 
		(char FAR *)&ConBuf);
	adm_write(slot, AdmPtr.shmConFromBufOffset, sizeof(ConBuf), 
		(char FAR *)&ConBuf);
	AdmConState[slot] = ADM_CON_OPENED;
	}

/*
 * name		AdmCloseRequestCleanUp
 *
 * synopsis	AdmCloseRequestCleanUp(slot)
 *		int	slot; <<	close requested from this slot.
 *					for SYSCARD:-
 *					1 <= slot <= 13.
 *					for FUNCARD:-
 *					slot = 0.
 *
 * description	It cleans up the adm console data structure. When this 
 *		function returns, the console will be in a CLOSE state.
 *
 * returns	nothing
 */

void AdmCloseRequestCleanUp(int slot)

	{
	/*
	 *	Tell the application that we are closed. 
	 */
	AdmConState[slot] = ADM_CON_CLOSED;
	}
#endif
