#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) mibget.c V6.2.3:cs.911111:7:7 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/**********************************************************************
	Management Information Base Module

	MIB maintenance module.

	File:		mibget.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911111	11/11/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	This file provides the operational routines for retrieving
	attribute data from the MIB. The operational routines are
	divided into object groups: SMT, MAC, PATH, PORT, and
	ATTACHMENT. The primary operational routine, GetMIBAttr()
	(see file mibproc.c), selects the appropriate group operation.
	The parameter interface to the object group operation routines
	is the same as for the primary operation.

	The attributes in the MIB fall into two categories: configuration
	and status. The configuration values are initialized at start-up
	to match the hardware's configuration or station's policies. These
	values either do not change at all, or can only be changed through
	a change operation, which would set the new value in the MIB
	data base. Therefore these values get retrived directly from the
	MIB data base. Status values consist of SMT-maintained values and
	current station status. The SMT status is kept up to date in the
	MIB data base by the XLNT Manager software and these values are
	reported directly from the MIB data base. The station status values,
	like counters, must be retrieved from their implementation-dependent 
	locations. Thus, the operation routines use interface functions to
	access these locations (see mibsysif.c). All of this means that
	only a small portion of the MIB attributes must be retrieved through
	interface routines. Most of the attributes are copied directly
	from the MIB data base.

	*** Updated to SMT 6.2 ***
	910122-005	LJP
		Removed XDI attributes xdiSMTOutIndex1 and xdiSMTOutIndex2.
	910301-001	LJP
		Put in range checking on entity value in routines
		GetMIBPATHAttr(), GetMIBPORTAttr(), and 
		GetMIBATTACHMENTAttr().
	910315-001	LJP
		Removed "&" from in front of variables that are actually
		arrays. Some compilers complain about this.
	911022-001	LJP
		Changed declaration of BuildTopology() from uInt16 to
		uChar. It is defined as uChar in mibsup.c.
	911111-001	LJP
		Changed code for fddiMACLongAliases, fddiMACShortAliases,
		fddiMACLongGrpAddrs, and fddiMACShortGrpAddrs to call
		GetMACData() instead of returning RC_NO_PARAM.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"fddihdr.h"
#include	"smtmacro.h"
#include	"smtmsg.h"
#include	"mibdefs.h"
#include	"mibtypes.h"
#include	"mibglbl.h"


/*********************************************************************
	Global Data
*********************************************************************/

static	uInt16
	SMTStationIdGrpList[] = {
		fddiSMTStationId, fddiSMTOpVersionId,
		fddiSMTHiVersionId, fddiSMTLoVersionId,
		fddiSMTManufacturerData, fddiSMTUserData,
		0 },

	SMTStationConfigGrpList[] = {
		fddiSMTMAC_Ct, fddiSMTNonMaster_Ct,
		fddiSMTMaster_Ct, fddiSMTPathsAvailable,
		fddiSMTConfigCapabilities, fddiSMTConfigPolicy,
		fddiSMTConnectionPolicy, fddiSMTReportLimit,
		fddiSMTT_Notify, fddiSMTStatusReporting,
		0 },

	SMTStatusGrpList[] = {
		fddiSMTECMState, fddiSMTCF_State,
		fddiSMTHoldState, fddiSMTRemoteDisconnectFlag,
		0 },

	SMTMIBOperationGrpList[] = {
		fddiSMTMsgTimeStamp, fddiSMTTransitionTimeStamp,
		fddiSMTSetCount, fddiSMTLastSetStationId,
		0 },

	MACCapabilitiesGrpList[] = {
		fddiMACFrameStatusCapabilities, fddiMACBridgeFunction,
		fddiMACT_MaxGreatestLowerBound,
		fddiMACTVXGreatestLowerBound,
		0 },

	MACConfigGrpList[] = {
		fddiMACPathsAvailable, fddiMACCurrentPath,
		fddiMACUpstreamNbr, fddiMACDownstreamNbr,
		fddiMACOldUpstreamNbr, fddiMACOldDownstreamNbr,
		fddiMACRootConcentratorMAC, fddiMACDup_Addr_Test,
		fddiMACPathsRequested, fddiMACDownstreamPORTType,
		0 },

	MACAddressGrpList[] = {
		fddiMACSMTAddress, fddiMACLongAliases,
		fddiMACShortAliases, fddiMACLongGrpAddrs,
		fddiMACShortGrpAddrs,
		0 },

	MACOperationGrpList[] = {
		fddiMACT_Req, fddiMACT_Neg,
		fddiMACT_Max, fddiMACTvxValue,
		fddiMACT_Min, fddiMACT_Pri0,
		fddiMACT_Pri1, fddiMACT_Pri2,
		fddiMACT_Pri3, fddiMACT_Pri4,
		fddiMACT_Pri5, fddiMACT_Pri6,
		fddiMACFrameStatus,
		0 },

	MACCountersGrpList[] = {
		fddiMACFrame_Ct, fddiMACCopied_Ct,
		fddiMACTransmit_Ct, fddiMACToken_Ct,
		0 },

	MACErrorCtrsGrpList[] = {
		fddiMACError_Ct, fddiMACLost_Ct,
		fddiMACTvxExpired_Ct, fddiMACNotCopied_Ct,
		fddiMACLate_Ct, fddiMACRingOp_Ct,
		0 },

	MACFrameErrorConditionGrpList[] = {
		fddiMACBaseFrame_Ct, fddiMACBaseError_Ct,
		fddiMACBaseLost_Ct, fddiMACBaseTimeFrameError,
		fddiMACFrameErrorThreshold, fddiMACFrameErrorRatio,
		0 },

	MACNotCopiedConditionGrpList[] = {
		fddiMACBaseNotCopied_Ct, fddiMACBaseTimeNotCopied,
		fddiMACNotCopiedThreshold, fddiMACBaseCopied_Ct,
		fddiMACNotCopiedRatio,
		0 },

	MACStatusGrpList[] = {
		fddiMACRMTState, fddiMACDa_Flag,
		fddiMACUnaDa_Flag, fddiMACFrameErrorCondition,
		fddiMACNotCopiedCondition, fddiMACLLCServiceAvailable,
		0 },

	MACRootMACStatusGrpList[] = {
		fddiMACMasterSlaveLoopStatus,
		fddiMACRootMACDownstreamPORTType,
		fddiMACRootMACCurrentPath,
		0 },

	PATHClassConfigGrpList[] = {
		fddiPATHClassTrace_MaxExpiration,
		fddiPATHClassTVXLowerBound,
		fddiPATHClassT_MaxLowerBound,
		0 },

	PATHConfigGrpList[] = {
		fddiPATHClassPATHClassType, fddiPATHClassPATHPORTOrder,
		fddiPATHClassPATHRingLatency, fddiPATHClassPATHTraceStatus,
		fddiPATHClassPATHSba, fddiPATHClassPATHSbaOverhead,
		fddiPATHClassPATHStatus, fddiPATHClassPATHConfiguration,
		fddiPATHClassPATHT_Rmode,
		0 },

	PORTConfigGrpList[] = {
		fddiPORTPC_Type, fddiPORTPC_Neighbor,
		fddiPORTConnectionPolicies, fddiPORTRemoteMACIndicated,
		fddiPORTCE_State, fddiPORTPathsRequested,
		fddiPORTMACPlacement, fddiPORTAvailablePaths,
		fddiPORTMACLoop_Time, fddiPORTFotxClass,
		0 },

	PORTOperationGrpList[] = {
		fddiPORTMaintLineState, fddiPORTTB_Max,
		fddiPORTBS_Flag,
		0 },

	PORTErrorCtrsGrpList[] = {
		fddiPORTEBError_Ct, fddiPORTLCTFail_Ct,
		0 },

	PORTLerGrpList[] = {
		fddiPORTLer_Estimate, fddiPORTLem_Reject_Ct,
		fddiPORTLem_Ct, fddiPORTBaseLer_Estimate,
		fddiPORTBaseLem_Reject_Ct, fddiPORTBaseLem_Ct,
		fddiPORTBaseLer_TimeStamp, fddiPORTLer_Cutoff,
		fddiPORTLer_Alarm,
		0 },

	PORTStatusGrpList[] = {
		fddiPORTConnectState, fddiPORTPCMState,
		fddiPORTPC_Withhold, fddiPORTLerCondition,
		0 },

	ATTACHMENTConfigGrpList[] = {
		fddiATTACHMENTClass, fddiATTACHMENTOpticalBypassPresent,
		fddiATTACHMENTI_MaxExpiration, fddiATTACHMENTInsertedStatus,
		fddiATTACHMENTInsertPolicy,
		0 };




/*********************************************************************
	External function definitions.
*********************************************************************/

extern	uInt32	GetSMTData();
extern	uInt32	GetMACData();
extern	uInt32	GetMACAddresses();
extern	uInt32	GetPATHData();
extern	uInt32	GetPORTData();
extern	uInt32	GetATTACHMENTData();  /* jlin 07/28/93 */
extern	uInt16	BuildConnectedResources ();
extern	uChar	BuildTopology ();	/* 911022-001 LJP */
extern	void	GetTimeStamp ();


/*********************************************************************
	MIB Get Operations.
*********************************************************************/

uInt32
GetMIBGroup (bufSize, bufData, groupList)
	uInt16		bufSize;
	uChar		*bufData;
	uInt16		groupList[];
/*********************************************************************
Function:	Build a MIB attribute group.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute group
					returned.
		groupList	= array of attribute IDs in the group.
Input:		groupList	= list of uInt16s of each MIB attribute
					in the group in the order they are
					to appear. The list is terminated
					by an attribute ID of 0.
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAM
			attribute not supported
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
*********************************************************************/
{
Int16		i;			/* loop counter */
TLVParamType	*p;			/* request attribute pointer */
uInt16		attrLen;		/* length of attribute */
uInt16		index;			/* entity index */
uInt32		result = RC_NO_PARAM;	/* return code */
uInt16		attrCount;		/* # of attributes in group */

/* Forward declarations */
uInt32	GetMIBSMTAttr ();
uInt32	GetMIBMACAttr ();
uInt32	GetMIBPATHAttr ();
uInt32	GetMIBPORTAttr ();
uInt32	GetMIBATTACHMENTAttr ();

	/* set initial attribute pointer */
	p = (TLVParamType *) bufData;
	index = p->MACINDEX;

	/*
	*	Loop through attribute list
	*/
	for (attrCount = 0, i = 0; groupList[i] != 0; i++)
	{
		/* get attribute */
		p->paramType = groupList[i];
		p->paramLen = 0;
		switch (p->paramType & MIB_OBJECT_MASK)
		{
		case fddiSMT:
			result = GetMIBSMTAttr (bufSize, bufData);
			break;

		case fddiMAC:
			p->MACINDEX = index;
			result = GetMIBMACAttr (bufSize, bufData);
			break;

		case fddiPATHClass:
			p->PATHINDEX = index;
			result = GetMIBPATHAttr (bufSize, bufData);
			break;

		case fddiPORT:
			p->PORTINDEX = index;
			result = GetMIBPORTAttr (bufSize, bufData);
			break;

		case fddiATTACHMENT:
			p->ATTACHINDEX = index;
			result = GetMIBATTACHMENTAttr (bufSize, bufData);
			break;
		}

		/* if buffer is full or invalid attribute */
		if ((result == RC_BUFFER_ERROR)
				|| (result == RC_ILLEGAL_PARAMETER))
			/* then exit */
			break;

		/* else if get was successful */
		else if (result == RC_SUCCESS)
		{
			/* move to the next position in the buffer */
			attrCount++;
			attrLen = p->paramLen + 4;
			bufData += attrLen;
			bufSize -= attrLen;
			p = (TLVParamType *) bufData;
		}
		/* else return value is OK, just skip this parameter */
		else
		{
			p->paramType = 0;
			p->paramLen = 0;
			result = RC_NO_PARAM;
		}
	}

	/*
	*	If result is RC_NO_PARAM, check if group actually has no
	*	attributes or just the last attribute was not available.
	*/
	if ((result == RC_NO_PARAM) && (attrCount > 0))
		/* group has some attributes in it */
		result = RC_SUCCESS;

	return (result);
}

uInt32
GetMIBSMTAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to get an fddiSMT attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAM
			attribute not supported
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
*********************************************************************/
{
TLVParamType	*p;			/* request attribute pointer */
uInt32		result = RC_SUCCESS;	/* return code */

	/*
	*	Check if buffer is minimum size.
	*	Header + data
	*	Length is the same for 8, 16, and 32 bit data types.
	*/
	if (bufSize < (sizeof (TLVHdrType) + sizeof (TLV8BitType)))
		return (RC_BUFFER_ERROR);

	/*
	*	Get format for buffer.
	*/
	p = (TLVParamType *) bufData;

	/*
	*	Clear data portion of buffer.
	*/
	MEMZERO (&p->paramValue, bufSize - sizeof (TLVHdrType));

	/*
	*	Length is the same for 8, 16, and 32 bit data types
	*	which is the most common size.
	*/
	p->paramLen = sizeof (TLV8BitType);

	/*
	*	Get value from MIB.
	*/
	switch (p->paramType)
	{
	/*
	*	fddiSMTStationIdGrp
	*/
	case fddiSMTStationIdGrp:
		result = GetMIBGroup (bufSize, bufData,
			SMTStationIdGrpList);
		break;

	case fddiSMTStationId:
		p->paramLen = sizeof (SMTStationIdType);
		/* 910315-001 LJP */
		MEMCOPY (p->SMTOTHER,
			&mib->SMTStationIdGrp.StationId,
			sizeof (SMTStationIdType));
		break;

	case fddiSMTOpVersionId:
		p->SMTPARAM16 = mib->SMTStationIdGrp.OpVersionId;
		break;

	case fddiSMTHiVersionId:
		p->SMTPARAM16 = mib->SMTStationIdGrp.HiVersionId;
		break;

	case fddiSMTLoVersionId:
		p->SMTPARAM16 = mib->SMTStationIdGrp.LoVersionId;
		break;

	case fddiSMTManufacturerData:
		p->paramLen = sizeof (SMTManufacturerDataType);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->SMTOTHER,
				&mib->SMTStationIdGrp.ManufacturerData,
				sizeof (SMTManufacturerDataType));
		break;

	case fddiSMTUserData:
		p->paramLen = sizeof (SMTUserDataType);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->SMTOTHER,
				mib->SMTStationIdGrp.UserData,
				sizeof (SMTUserDataType));
		break;

	/*
	*	fddiSMTStationConfigGrp
	*/
	case fddiSMTStationConfigGrp:
		result = GetMIBGroup (bufSize, bufData,
			SMTStationConfigGrpList);
		break;

	case fddiSMTMAC_Ct:
		p->SMTPARAM8 = mib->SMTStationConfigGrp.MAC_Ct;
		break;

	case fddiSMTNonMaster_Ct:
		p->SMTPARAM8 = mib->SMTStationConfigGrp.NonMaster_Ct;
		break;

	case fddiSMTMaster_Ct:
		p->SMTPARAM8 = mib->SMTStationConfigGrp.Master_Ct;
		break;

	case fddiSMTPathsAvailable:
		p->SMTPARAM8 = mib->SMTStationConfigGrp.PathsAvailable;
		break;

	case fddiSMTConfigCapabilities:
		p->SMTPARAM16 = mib->SMTStationConfigGrp.ConfigCapabilities;
		break;

	case fddiSMTConfigPolicy:
		p->SMTPARAM16 = mib->SMTStationConfigGrp.ConfigPolicy;
		break;

	case fddiSMTConnectionPolicy:
		p->SMTPARAM16 = mib->SMTStationConfigGrp.ConnectionPolicy;
		break;

	case fddiSMTReportLimit:
		p->SMTPARAM32 = mib->SMTStationConfigGrp.ReportLimit;
		break;

	case fddiSMTT_Notify:
		p->SMTPARAM16 = mib->SMTStationConfigGrp.T_Notify;
		break;

	case fddiSMTStatusReporting:
		p->SMTPARAM8 = mib->SMTStationConfigGrp.StatusReporting;
		break;

	/*
	*	fddiSMTStatusGrp
	*/
	case fddiSMTStatusGrp:
		result = GetMIBGroup (bufSize, bufData,
			SMTStatusGrpList);
		break;

	case fddiSMTECMState:
		p->SMTPARAM8 = mib->SMTStatusGrp.ECMState;
		break;

	case fddiSMTCF_State:
		p->SMTPARAM8 = mib->SMTStatusGrp.CF_State;
		break;

	case fddiSMTHoldState:
		p->SMTPARAM16 = mib->SMTStatusGrp.HoldState;
		break;

	case fddiSMTRemoteDisconnectFlag:
		p->SMTPARAM8 = mib->SMTStatusGrp.RemoteDisconnectFlag;
		break;

	/*
	*	fddiSMTMIBOperationGrp
	*/
	case fddiSMTMIBOperationGrp:
		result = GetMIBGroup (bufSize, bufData,
			SMTMIBOperationGrpList);
		break;

	case fddiSMTMsgTimeStamp:
		GetTimeStamp (&mib->SMTMIBOperationGrp.MsgTimeStamp);
		p->paramLen =
			sizeof (SMTTimeStamp);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->SMTOTHER,
				&mib->SMTMIBOperationGrp.MsgTimeStamp,
				sizeof (SMTTimeStamp));
		break;

	case fddiSMTTransitionTimeStamp:
		p->paramLen =
			sizeof (SMTTimeStamp);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->SMTOTHER,
				&mib->SMTMIBOperationGrp.TransitionTimeStamp,
				sizeof (SMTTimeStamp));
		break;

	case fddiSMTSetCount:
		p->paramLen =
			sizeof (SetCountType);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->SMTOTHER,
				&mib->SMTMIBOperationGrp.SetCount,
				sizeof (SetCountType));
		break;

	case fddiSMTLastSetStationId:
		p->paramLen =
			sizeof (SMTStationIdType);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->SMTOTHER,
				&mib->SMTMIBOperationGrp.LastSetStationId,
				sizeof (SMTStationIdType));
		break;


	/*
	*	XDI added attributes
	*/
	case xdiSMTBothWrapCapability:
		p->SMTPARAM8 = mib->XDISMTGrp.BothWrapCapability;
		break;

	case xdiSMTBothWrapPolicy:
		p->SMTPARAM8 = mib->XDISMTGrp.BothWrapPolicy;
		break;

	case xdiSMTTopology:
		mib->XDISMTGrp.Topology = BuildTopology ();
		p->SMTPARAM8 =
			mib->XDISMTGrp.Topology;
		break;

	/*
	*	910122-005	LJP
	*	Removed xdiSMTOutIndex1 and xdiSMTOutIndex2
	*/

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	return (result);
}

uInt32
GetMIBMACAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to get an fddiMAC attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAM
			attribute not supported
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
Notes:		For the attributes fddiMACLongAliases, fddiMACShortAliases,
		fddiMACLongGrpAddrs, and fddiMACShortGrpAddrs, this routine
		returns as many items in the group as will fit in the buffer.
*********************************************************************/
{
TLVParamType	*p;				/* attribute pointer */
uInt16		entity;				/* MAC index */
uInt32		result = RC_SUCCESS;		/* return code */
Flag		doGroup = CLEAR;		/* SET for groups */

	/*
	*	Check if buffer is minimum size.
	*	Header + index + data
	*	Length is the same for 8, 16, and 32 bit data types.
	*/
	if (bufSize < (sizeof (TLVHdrType) + sizeof (TLV16BitType)
			+ sizeof (TLV8BitType)))
		return (RC_BUFFER_ERROR);

	/*
	*	Get format for buffer.
	*/
	p = (TLVParamType *) bufData;

	/*
	*	Get index
	*/
	entity = p->MACINDEX;

	/*
	*	Clear data portion of buffer.
	*/
	MEMZERO (&p->paramValue, bufSize - sizeof (TLVHdrType));

	/*
	*	Reset parameter index and adjust for MIB array element.
	*/
	p->MACINDEX = entity;
	if (entity < 1 || entity > mib->SMTStationConfigGrp.MAC_Ct)
		return (RC_ILLEGAL_PARAMETER);
	entity--;

	/*
	*	Set length to most common size (index added later).
	*/
	p->paramLen = sizeof (TLV16BitType);

	/*
	*	Get value from MIB.
	*/
	switch (p->paramType)
	{
	/*
	*	fddiMACCapabilitiesGrp
	*/
	case fddiMACCapabilitiesGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACCapabilitiesGrpList);
		break;

	case fddiMACFrameStatusCapabilities:
		p->MACPARAM16 =
			mib->MACCapabilitiesGrp[entity].FrameStatusCapabilities;
		break;

	case fddiMACBridgeFunction:
		p->MACPARAM16 = mib->MACCapabilitiesGrp[entity].BridgeFunction;
		break;

	case fddiMACT_MaxGreatestLowerBound:
		p->MACPARAM32 =
			mib->MACCapabilitiesGrp[entity].T_MaxGreatestLowerBound;
		break;

	case fddiMACTVXGreatestLowerBound:
		p->MACPARAM32 =
			mib->MACCapabilitiesGrp[entity].TVXGreatestLowerBound;
		break;


	/*
	*	fddiMACConfigGrp
	*/
	case fddiMACConfigGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACConfigGrpList);
		break;

	case fddiMACPathsAvailable:
		p->MACPARAM8 = mib->MACConfigGrp[entity].PathsAvailable;
		break;

	case fddiMACCurrentPath:
		p->MACPARAM16 = mib->MACConfigGrp[entity].CurrentPath;
		break;

	case fddiMACUpstreamNbr:
		p->paramLen = sizeof (TLVIdType);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MCopyAddress (p->MACADDR,
				mib->MACConfigGrp[entity].UpstreamNbr);
		break;

	case fddiMACDownstreamNbr:
		p->paramLen = sizeof (TLVIdType);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MCopyAddress (p->MACADDR,
				mib->MACConfigGrp[entity].DownstreamNbr);
		break;

	case fddiMACOldUpstreamNbr:
		p->paramLen = sizeof (TLVIdType);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MCopyAddress (p->MACADDR,
				mib->MACConfigGrp[entity].OldUpstreamNbr);
		break;

	case fddiMACOldDownstreamNbr:
		p->paramLen = sizeof (TLVIdType);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MCopyAddress (p->MACADDR,
				mib->MACConfigGrp[entity].OldDownstreamNbr);
		break;

	case fddiMACRootConcentratorMAC:
		p->MACPARAM8 = mib->MACConfigGrp[entity].RootConcentratorMAC;
		break;

	case fddiMACDup_Addr_Test:
		p->MACPARAM16 = mib->MACConfigGrp[entity].Dup_Addr_Test;
		break;

	case fddiMACPathsRequested:
		p->MACPARAM8 = mib->MACConfigGrp[entity].PathsRequested;
		break;

	case fddiMACDownstreamPORTType:
		p->MACPARAM16 = mib->MACConfigGrp[entity].DownstreamPORTType;
		break;

	/*
	*	fddiMACAddressGrp
	*
	*	The number and storage format of the group and alias
	*	addresses is implementation dependent. An interface
	*	routine is needed to fill the parameter buffer with
	*	the current values.
	*/
	case fddiMACAddressGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACAddressGrpList);
		break;

	case fddiMACSMTAddress:
		p->paramLen = sizeof (TLVIdType);
		if (bufSize < (p->paramLen + sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MCopyAddress (p->MACADDR,
				mib->MACAddressGrp[entity].SMTAddress);
		break;

	case fddiMACLongAliases:
	case fddiMACShortAliases:
	case fddiMACLongGrpAddrs:
	case fddiMACShortGrpAddrs:
		result = GetMACAddresses (bufSize, p);
		break;

	/*
	*	fddiMACOperationGrp
	*
	*	Although these values are contained in the MAC or
	*	associated hardware, the MIB gets initialized with the
	*	start-up values. Any changes to these values also get
	*	recorded in the MIB.
	*/
	case fddiMACOperationGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACOperationGrpList);
		break;

	case fddiMACT_Req:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Req;
		break;

	case fddiMACT_Neg:
		result = GetMACData (p->paramType, entity, &p->MACPARAM32);
		break;

	case fddiMACT_Max:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Max;
		break;

	case fddiMACTvxValue:
		p->MACPARAM32 = mib->MACOperationGrp[entity].TvxValue;
		break;

	case fddiMACT_Min:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Min;
		break;

	case fddiMACT_Pri0:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri0;
		break;

	case fddiMACT_Pri1:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri1;
		break;

	case fddiMACT_Pri2:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri2;
		break;

	case fddiMACT_Pri3:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri3;
		break;

	case fddiMACT_Pri4:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri4;
		break;

	case fddiMACT_Pri5:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri5;
		break;

	case fddiMACT_Pri6:
		p->MACPARAM32 = mib->MACOperationGrp[entity].T_Pri6;
		break;

	case fddiMACFrameStatus:
		p->MACPARAM16 = mib->MACOperationGrp[entity].FrameStatus;
		break;


	/*
	*	fddiMACCounterGrp
	*
	*	This group retrieves the current status of MAC counters.
	*	The interface routine retrieves the counter value from
	*	the implementation-dependent location.
	*/
	case fddiMACCountersGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACCountersGrpList);
		break;

	case fddiMACFrame_Ct:
	case fddiMACCopied_Ct:
	case fddiMACTransmit_Ct:
	case fddiMACToken_Ct:
		result = GetMACData (p->paramType, entity, &p->MACPARAM32);
		break;

	/*
	*	fddiMACErrorCtrsGrp
	*
	*	This group retrieves the current status of MAC error counters.
	*	The interface routine retrieves the error counter value from
	*	the implementation-dependent location.
	*/
	case fddiMACErrorCtrsGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACErrorCtrsGrpList);
		break;

	case fddiMACError_Ct:
	case fddiMACLost_Ct:
	case fddiMACTvxExpired_Ct:
	case fddiMACNotCopied_Ct:
	case fddiMACRingOp_Ct:
		result = GetMACData (p->paramType, entity, &p->MACPARAM32);
		break;

	case fddiMACLate_Ct:
		result = GetMACData (p->paramType, entity, &p->MACPARAM16);
		break;

	/*
	*	fddiMACFrameErrorConditionGrp
	*/
	case fddiMACFrameErrorConditionGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACFrameErrorConditionGrpList);
		break;

	case fddiMACBaseFrame_Ct:
		p->MACPARAM32 =
			mib->MACFrameErrorConditionGrp[entity].BaseFrame_Ct;
		break;

	case fddiMACBaseError_Ct:
		p->MACPARAM32 =
			mib->MACFrameErrorConditionGrp[entity].BaseError_Ct;
		break;

	case fddiMACBaseLost_Ct:
		p->MACPARAM32 =
			mib->MACFrameErrorConditionGrp[entity].BaseLost_Ct;
		break;

	case fddiMACBaseTimeFrameError:
		p->paramLen = sizeof (SMTTimeStamp);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->MACOTHER,
			    &mib->MACFrameErrorConditionGrp[entity].BaseTimeFrameError,
				p->paramLen);
		break;

	case fddiMACFrameErrorThreshold:
		p->MACPARAM16 =
		    mib->MACFrameErrorConditionGrp[entity].FrameErrorThreshold;
		break;

	case fddiMACFrameErrorRatio:
		p->MACPARAM16 =
			mib->MACFrameErrorConditionGrp[entity].FrameErrorRatio;
		break;

	/*
	*	fddiMACNotCopiedConditionGrp
	*/
	case fddiMACNotCopiedConditionGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACNotCopiedConditionGrpList);
		break;

	case fddiMACBaseNotCopied_Ct:
		p->MACPARAM32 =
			mib->MACNotCopiedConditionGrp[entity].BaseNotCopied_Ct;
		break;

	case fddiMACBaseTimeNotCopied:
		p->paramLen = sizeof (SMTTimeStamp);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->MACOTHER,
			    &mib->MACNotCopiedConditionGrp[entity].BaseTimeNotCopied,
				p->paramLen);
		break;

	case fddiMACNotCopiedThreshold:
		p->MACPARAM16 =
		    mib->MACNotCopiedConditionGrp[entity].NotCopiedThreshold;
		break;

	case fddiMACBaseCopied_Ct:
		p->MACPARAM32 =
			mib->MACNotCopiedConditionGrp[entity].BaseCopied_Ct;
		break;

	case fddiMACNotCopiedRatio:
		p->MACPARAM16 =
			mib->MACNotCopiedConditionGrp[entity].NotCopiedRatio;
		break;

	/*
	*	fddiMACStatusGrp
	*/
	case fddiMACStatusGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACStatusGrpList);
		break;

	case fddiMACRMTState:
		p->MACPARAM8 = mib->MACStatusGrp[entity].RMTState;
		break;

	case fddiMACDa_Flag:
		p->MACPARAM8 = mib->MACStatusGrp[entity].Da_Flag;
		break;

	case fddiMACUnaDa_Flag:
		p->MACPARAM8 = mib->MACStatusGrp[entity].UnaDa_Flag;
		break;

	case fddiMACFrameErrorCondition:
		p->MACPARAM8 = mib->MACStatusGrp[entity].FrameErrorCondition;
		break;

	case fddiMACNotCopiedCondition:
		p->MACPARAM8 = mib->MACStatusGrp[entity].NotCopiedCondition;
		break;

	case fddiMACLLCServiceAvailable:
		p->MACPARAM8 = mib->MACStatusGrp[entity].LLCServiceAvailable;
		break;

	/*
	*	fddiMACRootMACStatusGrp
	*/
	case fddiMACRootMACStatusGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			MACRootMACStatusGrpList);
		break;

	case fddiMACMasterSlaveLoopStatus:
	case fddiMACRootMACDownstreamPORTType:
	case fddiMACRootMACCurrentPath:
		result = RC_NO_PARAM;
		break;

	/*
	*	XDI added attributes
	*/
	case xdiMACRMTDupPolicy1:
		p->MACPARAM8 = mib->XDIMACGrp[entity].RMTDupPolicy1;
		break;

	case xdiMACRMTDupPolicy2:
		p->MACPARAM8 = mib->XDIMACGrp[entity].RMTDupPolicy2;
		break;

	case xdiMACResourceIndex:
		p->MACPARAM16 = mib->XDISMTGrp.Port_Ct + entity + 1;
		break;

	case xdiMACConnectedIndex:
		p->MACPARAM16 = BuildConnectedResources
			(mib->XDISMTGrp.Port_Ct + entity + 1);
		break;

	case xdiMACOperational:
		p->MACPARAM8 =
			mib->XDIMACGrp[entity].Operational;
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	/*
	*	Adjust parameter length to include entity index.
	*	Skip adjustment on groups.
	*/
	if ((result == RC_SUCCESS) && !doGroup)
		p->paramLen += sizeof (p->paramValue.otherAttr.paramIndex);

	return (result);
}

uInt32
GetMIBPATHAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to get an fddiPATH attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAM
			attribute not supported
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
*********************************************************************/
{
TLVParamType	*p;				/* attribute pointer */
uInt16		entity;				/* PATH index */
uInt32		result = RC_SUCCESS;		/* return value */
Flag		doGroup = CLEAR;		/* SET for groups */

	/*
	*	Check if buffer is minimum size.
	*	Header + index + data
	*	Length is the same for 8, 16, and 32 bit data types.
	*/
	if (bufSize < (sizeof (TLVHdrType) + sizeof (TLV16BitType)
			+ sizeof (TLV8BitType)))
		return (RC_BUFFER_ERROR);

	/*
	*	Get format for buffer.
	*/
	p = (TLVParamType *) bufData;

	/*
	*	Get index
	*/
	entity = p->PATHINDEX;

	/*
	*	Clear data portion of buffer.
	*/
	MEMZERO (&p->paramValue, bufSize - sizeof (TLVHdrType));

	/*
	*	Reset parameter index and adjust for MIB array element.
	*/
	/*
	*	910301-001	LJP
	*	Put in range checking for entity value.
	*/
	p->PATHINDEX = entity;
        if (entity < 1 || entity > MAX_PATH_COUNT)
                return (RC_ILLEGAL_PARAMETER);
	entity--;

	/*
	*	Set length to most common size (index added later).
	*/
	p->paramLen = sizeof (TLV16BitType);

	/*
	*	Get value from MIB.
	*/
	switch (p->paramType)
	{
	/*
	*	Path Class Attributes
	*	fddiPATHConfigGrp
	*/
	case fddiPATHConfigGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PATHClassConfigGrpList);
		break;

	case fddiPATHClassTrace_MaxExpiration:
		p->PATHPARAM32 =
			mib->ClassConfigGrp.Trace_MaxExpiration;
		break;

	case fddiPATHClassTVXLowerBound:
		p->PATHPARAM32 =
			mib->ClassConfigGrp.TVXLowerBound;
		break;

	case fddiPATHClassT_MaxLowerBound:
		p->PATHPARAM32 =
			mib->ClassConfigGrp.T_MaxLowerBound;
		break;

	/*
	*	Path Class Path Attributes
	*	fddiPATHClassPATHConfigGrp
	*/
	case fddiPATHClassPATHConfigGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PATHConfigGrpList);
		break;

	case fddiPATHClassPATHClassType:
		p->PATHPARAM16 =
			mib->ClassPATHConfigGrp[entity].Type;
		break;

	case fddiPATHClassPATHPORTOrder:
		p->PATHPARAM16 =
			mib->ClassPATHConfigGrp[entity].PORTOrder;
		break;

	case fddiPATHClassPATHRingLatency:
		p->PATHPARAM32 =
			mib->ClassPATHConfigGrp[entity].RingLatency;
		break;

	case fddiPATHClassPATHTraceStatus:
		p->PATHPARAM16 =
			mib->ClassPATHConfigGrp[entity].TraceStatus;
		break;

	case fddiPATHClassPATHSba:
		p->PATHPARAM32 =
			mib->ClassPATHConfigGrp[entity].Sba;
		break;

	case fddiPATHClassPATHSbaOverhead:
		p->PATHPARAM16 =
			mib->ClassPATHConfigGrp[entity].SbaOverhead;
		break;

	case fddiPATHClassPATHStatus:
#if 1    
		/* jlin: bug fix. 06/01/93 */
		p->PATHPARAM16 =
		         ((mib->SMTStatusGrp.CF_State == CF_THRU) ||
			  (mib->SMTStatusGrp.CF_State == CF_ISOLATED))
			   ? Path_Status_Thru : Path_Status_Wrapped;
#else
		p->PATHPARAM16 =
			   mib->ClassPATHConfigGrp[entity].Status;
#endif
		break;

	case fddiPATHClassPATHConfiguration:
		result = RC_NO_PARAM;
		break;

	case fddiPATHClassPATHT_Rmode:
		p->PATHPARAM32 =
			mib->ClassPATHConfigGrp[entity].T_Rmode;
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	/*
	*	Adjust parameter length to include entity index.
	*	Skip adjustment on groups.
	*/
	if ((result == RC_SUCCESS) && !doGroup)
		p->paramLen += sizeof (p->paramValue.otherAttr.paramIndex);

	return (result);
}

uInt32
GetMIBPORTAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to get an fddiPORT attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAM
			attribute not supported
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
*********************************************************************/
{
TLVParamType	*p;				/* attribute pointer */
uInt16		entity;				/* PORT index */
uInt32		result = RC_SUCCESS;		/* return code */
Flag		doGroup = CLEAR;		/* SET for groups */

	/*
	*	Check if buffer is minimum size.
	*	Header + index + data
	*	Length is the same for 8, 16, and 32 bit data types.
	*/
	if (bufSize < (sizeof (TLVHdrType) + sizeof (TLV16BitType)
			+ sizeof (TLV8BitType)))
		return (RC_BUFFER_ERROR);

	/*
	*	Get format for buffer.
	*/
	p = (TLVParamType *) bufData;

	/*
	*	Get index
	*/
	entity = p->PORTINDEX;

	/*
	*	Clear data portion of buffer.
	*/
	MEMZERO (&p->paramValue, bufSize - sizeof (TLVHdrType));

	/*
	*	Reset parameter index and adjust for MIB array element.
	*/
	/*
	*	910301-001	LJP
	*	Put in range checking for entity value.
	*/
        p->PORTINDEX = entity;
        if (entity < 1 || entity > (mib->SMTStationConfigGrp.NonMaster_Ct 
			+ mib->SMTStationConfigGrp.Master_Ct))
                return (RC_ILLEGAL_PARAMETER);
	entity--;

	/*
	*	Set length to most common size (index added later).
	*/
	p->paramLen = sizeof (TLV16BitType);

	/*
	*	Get value from MIB.
	*/
	switch (p->paramType)
	{
	/*
	*	fddiPORTConfigGrp
	*/
	case fddiPORTConfigGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PORTConfigGrpList);
		break;

	case fddiPORTPC_Type:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].PC_Type;
		break;

	case fddiPORTPC_Neighbor:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].PC_Neighbor;
		break;

	case fddiPORTConnectionPolicies:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].ConnectionPolicies;
		break;

	case fddiPORTRemoteMACIndicated:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].RemoteMACIndicated;
		break;

	case fddiPORTCE_State:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].CE_State;
		break;

	case fddiPORTPathsRequested:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].PathsRequested;
		break;

	case fddiPORTMACPlacement:
		p->PORTPARAM16 =
			mib->PORTConfigGrp[entity].MACPlacement;
		break;

	case fddiPORTAvailablePaths:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].AvailablePaths;
		break;

	case fddiPORTMACLoop_Time:
		p->PORTPARAM32 =
			mib->PORTConfigGrp[entity].MACLoop_Time;
		break;

	case fddiPORTFotxClass:
		p->PORTPARAM8 =
			mib->PORTConfigGrp[entity].FotxClass;
		break;

	/*
	*	fddiPORTOperationGrp
	*/
	case fddiPORTOperationGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PORTOperationGrpList);
		break;

	case fddiPORTMaintLineState:
		p->PORTPARAM8 =
			mib->PORTOperationGrp[entity].MaintLineState;
		break;

	case fddiPORTTB_Max:
		p->PORTPARAM32 =
			mib->PORTOperationGrp[entity].TB_Max;
		break;

	case fddiPORTBS_Flag:
		p->PORTPARAM8 =
			mib->PORTOperationGrp[entity].BS_Flag;
		break;

	/*
	*	fddiPORTErrorCtrsGrp
	*/
	case fddiPORTErrorCtrsGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PORTErrorCtrsGrpList);
		break;

	case fddiPORTEBError_Ct:
		result = GetPORTData (p->paramType, entity, &p->PORTPARAM32);
		break;

	case fddiPORTLCTFail_Ct:
		p->PORTPARAM32 = mib->PORTErrorCtrsGrp[entity].LCTFail_Ct;
		break;

	/*
	*	fddiPORTLerGrp
	*/
	case fddiPORTLerGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PORTLerGrpList);
		break;

	case fddiPORTLer_Estimate:
		p->PORTPARAM8 =
			mib->PORTLerGrp[entity].Ler_Estimate;
		break;

	case fddiPORTLem_Reject_Ct:
		p->PORTPARAM32 =
			mib->PORTLerGrp[entity].Lem_Reject_Ct;
		break;

	case fddiPORTLem_Ct:
		p->PORTPARAM32 =
			mib->PORTLerGrp[entity].Lem_Ct;
		break;

	case fddiPORTBaseLer_Estimate:
		p->PORTPARAM8 =
			mib->PORTLerGrp[entity].BaseLer_Estimate;
		break;

	case fddiPORTBaseLem_Reject_Ct:
		p->PORTPARAM32 =
			mib->PORTLerGrp[entity].BaseLem_Reject_Ct;
		break;

	case fddiPORTBaseLem_Ct:
		p->PORTPARAM32 =
			mib->PORTLerGrp[entity].BaseLem_Ct;
		break;

	case fddiPORTBaseLer_TimeStamp:
		p->paramLen = sizeof (SMTTimeStamp);
		if (bufSize < (p->paramLen + sizeof (TLV16BitType)
				+ sizeof (TLVHdrType)))
			result = RC_BUFFER_ERROR;
		else
			/* 910315-001 LJP */
			MEMCOPY (p->PORTOTHER,
				&mib->PORTLerGrp[entity].BaseLer_TimeStamp,
				sizeof (mib->PORTLerGrp[entity].BaseLer_TimeStamp));
		break;

	case fddiPORTLer_Cutoff:
		p->PORTPARAM8 =
			mib->PORTLerGrp[entity].Ler_Cutoff;
		break;

	case fddiPORTLer_Alarm:
		p->PORTPARAM8 =
			mib->PORTLerGrp[entity].Ler_Alarm;
		break;

	/*
	*	fddiPORTStatusGrp
	*/
	case fddiPORTStatusGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			PORTStatusGrpList);
		break;

	case fddiPORTConnectState:
		p->PORTPARAM16 =
			mib->PORTStatusGrp[entity].ConnectState;
		break;

	case fddiPORTPCMState:
		p->PORTPARAM8 =
			mib->PORTStatusGrp[entity].PCMState;
		break;

	case fddiPORTPC_Withhold:
		p->PORTPARAM8 =
			mib->PORTStatusGrp[entity].PC_Withhold;
		break;

	case fddiPORTLerCondition:
		p->PORTPARAM8 =
			mib->PORTStatusGrp[entity].LerCondition;
		break;

	/* XDI added attributes */
	case xdiPORTResourceIndex:
		p->PORTPARAM16 = entity + 1;
		break;

	case xdiPORTConnectedIndex:
		p->PORTPARAM16 = BuildConnectedResources (entity + 1);
		break;

	case xdiPORTOperational:
		p->PORTPARAM8 =
			mib->XDIPORTGrp[entity].Operational;
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	/*
	*	Adjust parameter length to include entity index.
	*	Skip adjustment on groups.
	*/
	if ((result == RC_SUCCESS) && !doGroup)
		p->paramLen += sizeof (p->paramValue.otherAttr.paramIndex);

	return (result);
}

uInt32
GetMIBATTACHMENTAttr (bufSize, bufData)
	uInt16		bufSize;
	uChar		*bufData;
/*********************************************************************
Function:	MIB access routine to get an fddiATTACHMENT attribute.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
Return:		The value returned corresponds to the results codes used
		by the Parameter Manager Protocol:
		RC_SUCCESS
			attribute retrieved successfully
		RC_NO_PARAM
			attribute not supported
		RC_ILLEGAL_PARAMETER
			illegal parameter requested
		RC_BUFFER_ERROR
			if the buffer is not large enough to hold the
			requested attribute or attribute group.
*********************************************************************/
{
TLVParamType	*p;				/* attribute pointer */
uInt16		entity;				/* ATTACHMENT index */
uInt32		result = RC_SUCCESS;		/* return code */
Flag		doGroup = CLEAR;		/* SET for groups */

	/*
	*	Check if buffer is minimum size.
	*	Header + index + data
	*	Length is the same for 8, 16, and 32 bit data types.
	*/
	if (bufSize < (sizeof (TLVHdrType) + sizeof (TLV16BitType)
			+ sizeof (TLV8BitType)))
		return (RC_BUFFER_ERROR);

	/*
	*	Get format for buffer.
	*/
	p = (TLVParamType *) bufData;

	/*
	*	Get index
	*/
	entity = p->ATTACHINDEX;

	/*
	*	Clear data portion of buffer.
	*/
	MEMZERO (&p->paramValue, bufSize - sizeof (TLVHdrType));

	/*
	*	Reset parameter index and adjust for MIB array element.
	*/
	/*
	*	910301-001	LJP
	*	Put in range checking for entity value.
	*/
        p->ATTACHINDEX = entity;
        if (entity < 1 || entity > (mib->SMTStationConfigGrp.NonMaster_Ct 
			+ mib->SMTStationConfigGrp.Master_Ct))
                return (RC_ILLEGAL_PARAMETER);
	entity--;

	/*
	*	Set length to most common size (index added later).
	*/
	p->paramLen = sizeof (TLV16BitType);

	/*
	*	Get value from MIB.
	*/
	switch (p->paramType)
	{
	/*
	*	fddiATTACHMENTConfigGrp
	*/
	case fddiATTACHMENTConfigGrp:
		doGroup = SET;
		result = GetMIBGroup (bufSize, bufData,
			ATTACHMENTConfigGrpList);
		break;

	case fddiATTACHMENTClass:
		p->ATTACHPARAM16 =
			mib->ATTACHMENTConfigGrp[entity].Class;
		break;

	case fddiATTACHMENTOpticalBypassPresent:
		p->ATTACHPARAM8 =
			mib->ATTACHMENTConfigGrp[entity].OpticalBypassPresent;
		break;

	case fddiATTACHMENTI_MaxExpiration:
		p->ATTACHPARAM32 =
			mib->ATTACHMENTConfigGrp[entity].I_MaxExpiration;
		break;

	case fddiATTACHMENTInsertedStatus:
#if 1 /* jlin: Check the bypass swicth thru the interface 07/28/93 */
		result = GetATTACHMENTData (p->paramType, entity, 
			&mib->ATTACHMENTConfigGrp[entity].InsertedStatus);
#endif
		p->ATTACHPARAM8 =
			mib->ATTACHMENTConfigGrp[entity].InsertedStatus;
		break;

	case fddiATTACHMENTInsertPolicy:
		p->ATTACHPARAM8 =
			mib->ATTACHMENTConfigGrp[entity].InsertPolicy;
		break;

	default:
		result = RC_ILLEGAL_PARAMETER;
		break;
	}

	/*
	*	Adjust parameter length to include entity index.
	*	Do not adjust length on groups.
	*/
	if ((result == RC_SUCCESS) && !doGroup)
		p->paramLen += sizeof (p->paramValue.otherAttr.paramIndex);

	return (result);
}

