#ifndef __SMTMSG_H
#define __SMTMSG_H

/*
@(#) smtmsg.h V6.2.3:cs.910920.2:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.
*/
/*********************************************************************
	Station Management Header File
	
	SMT Messages Header File

	File:		smtmsg.h
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.910920.2	09/20/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This header file is used by all SMT modules that must communicate
	between processes.

	There are four types of messages: events, actions, requests,
	and indications. Events are generated asynchronously to notify
	one process of a change in another process. An example of this
	is when CMT changes the station configuration, it will notify
	the MIB of the new configuration. Actions are commands from
	one process to another; a Connect or Disconnect is an example
	of an action. A request is issued when one process requires
	information from another. An indication is the message sent in
	response to a request.

	A message consists of the following fields:

	destination	This is the ID number of the process who will
			receive this message.
	source		This is the ID number of the process who sent
			this message.
	type		This is the message type: event, action, or
			response. This value is coded to include the 
			message type, the associated entity, and the 
			message ID.
	typeInfo	This member is for additional type information.
			It is used generally to hold the MIB attribute
			ID when requesting or responding with information
			in the MIB.
	localID		Value reserved for MAP to specify. This field
			is set to 0 by frames generated within the
			Manager. For responses to the MAP, the value is
			set to the localID field of the request message.
	entity		This is the PHY or MAC index if the data contained
			in the message pertains to a PHY or MAC.
	data		A union of data sizes to hold a single value.
			This field is convenient for CSP to report new
			MIB values without having to allocate memory for
			the data.
	len1		This is the length of the buffer at p1.
	p1		A union of pointers to additional message buffers.
	len2		This is the length of the buffer at p2.
	p2		A pointer to the data buffer for the message.

	The process ID number is a manifest constant identifying either
	CSP, MIB, FBM, or MAP.

	The message type is a 16-bit value encoded as follows:
		0 x T P I I
		    | | | |
		    | | +-+- This is the message identifier.
		    | |
		    | +----- This is the process or object associated
		    |         with the message.
		    |
		    +------- This is the message type.

	The implementation of the communication mechanism between processes
	is left to the implementor. Different processes may have different
	communication requirements. Therefore, it is important to refer
	to the interface documentation for each process before selecting
	a communication mechanism.

	Modification History:

	*** UPDATED TO SMT 6.2 ***

	910404-005	LJP
		The setCount member for PMF frames in SendFrameBuf
		changed from TLVParamType to TLVSetCountType. This is
		more efficient.
*********************************************************************/

/*********************************************************************
	Data Structures
*********************************************************************/
#pragma align 1
#pragma pack  1

/*
*	Structure of info member for FBM_ACTION_SEND_FRAME.
*/
typedef struct SendFrameStruct {
	uChar	frameClass;		/* frame class to send */
	uChar	frameType;		/* frame type to send */
	uInt32	sentTransID;		/* transaction ID sent in frame */
	/*
	*	The following union contains a different data structure
	*	for each frame class.
	*/
	union {
		/*
		* NIF - Request or Announcement
		*
		*	FC		FC field to use must be SMT_INFO_FC
		*			(0x41) or SMT_NSA_FC (0x4F). Only
		*			used for NIF requests.
		*	destAddr	Destination address for frame. Only
		*			used for NIF requests.
		*	sourceMAC	Index of MAC to transmit frame.
		*	transID		Transaction ID to use. 0 if one
		*			should be generated.
		*/
		struct {
			uChar		FC;
			MACAddr48	destAddr;
			uInt16		sourceMAC;
			uInt32		transID;
		} nif;

		/*
		* SIF Configuration or Operation - Request
		*
		*	FC		FC field to use must be SMT_INFO_FC
		*			(0x41) or SMT_NSA_FC (0x4F).
		*	destAddr	Destination address for frame.
		*	sourceMAC	Index of MAC to transmit frame.
		*/
		struct {
			uChar		FC;
			MACAddr48	destAddr;
			uInt16		sourceMAC;
		} sif;

		/*
		* ECF - Request
		*
		*	FC		FC field to use must be SMT_INFO_FC
		*			(0x41) or SMT_NSA_FC (0x4F).
		*	destAddr	Destination address for frame.
		*	sourceMAC	Index of MAC to transmit frame.
		*/
		struct {
			uChar		FC;
			MACAddr48	destAddr;
			uInt16		sourceMAC;
		} ecf;

		/*
		* RDF - Response
		*
		*	sourceMAC	Index of MAC to transmit frame.
		*	reasonCode	Reason for rejecting the frame.
		*	requestFrame	Contents of request frame including
		*			the padding before the FC field.
		*/
		struct {
			uInt16		sourceMAC;
			uInt32		reasonCode;
		} rdf;

		/*
		* RAF - Announcement or Request
		*
		*	FC		FC field to use must be SMT_INFO_FC
		*			(0x41) or SMT_NSA_FC (0x4F).
		*	destAddr	Destination address for frame.
		*	sourceMAC	Index of MAC to transmit frame.
		*
		*	CURRENTLY NOT SUPPORTED
		*
		*/
		struct {
			uChar		FC;
			MACAddr48	destAddr;
			uInt16		sourceMAC;
		} raf;

		/*
		* ESF - Announcement or Request
		*
		*	FC		FC field to use must be SMT_INFO_FC
		*			(0x41) or SMT_NSA_FC (0x4F).
		*	destAddr	Destination address for frame.
		*	sourceMAC	Index of MAC to transmit frame.
		*	esfID		ESF identification code.
		*	infoLen		Length of additional ESF data.
		*	info		ESF data.
		*/
		struct {
			uChar		FC;
			MACAddr48	destAddr;
			uInt16		sourceMAC;
			MACAddr48	esfID;
		} esf;

		/*
		* PMF Get, Change, Add, or Remove - Request
		*
		*	FC		FC field to use must be SMT_INFO_FC
		*			(0x41) or SMT_NSA_FC (0x4F).
		*	destAddr	Destination address for frame.
		*	sourceMAC	Index of MAC to transmit frame.
		*	setCount	Value of set count to use in frame.
		*	attribute	Value of attribute to send in frame.
		*/
		struct {
			uChar		FC;
			MACAddr48	destAddr;
			uInt16		sourceMAC;
			TLVSetCountType	setCount;	/* 910404-005 LJP */
			TLVParamType	attribute;
		} pmf;
	} frameData;
} SendFrameBuf;

#define	SMT_SEND_BUF_SIZE	(sizeof (SendFrameBuf))

/*
*	Structure of info member of FBM_EVENT_FRAME_RECEIVED event.
*	Index of MAC that received the frame is given by member entity in
*	SMT message structure above.
*/
typedef	struct RecFrameStruct {
	uChar	EACbits;		/* E, A, and C bits */
	uChar	pad;			/* padding */
	uInt16	frameLen;		/* total length of frame */
} RecFrameBuf;

#define	SMT_REC_BUF_SIZE	(sizeof (RecFrameBuf))

/*
*	Structure used for all MIB action messages.
*/
typedef struct MIBMessageStruct {
	uInt32		result;		/* value returned by MIB routine */
	MACAddr48	SA;		/* MAC addr received in PMF (if PMF) */
	uInt16		macIndex;	/* index of MAC received PMF */
	SMTStationIdType setStationID;	/* address of station setting value */
	TLVSetCountType	setCount;	/* buffer for fddiSMTSetCount */
} MIBMsgType;

#define	MIB_MSG_SIZE		(sizeof (MIBMsgType))

/*
*	This is the message data structure.
*/
typedef	struct	SMTMessageStruct {
	uInt16	destination;		/* receiving process ID */
	uInt16	source;			/* sending process ID */
	uInt16	type;			/* message type */
	uInt16	typeInfo;		/* message content */
	uInt32	localID;		/* available for implementor */
	uInt16	entity;			/* MAC or PHY index (if needed) */
	uChar	pad[2];
	union {
		uChar		b8;		/* 8-bit data item */
		uInt16		b16;		/* 16-bit data item */
		uInt32		b32;		/* 32-bit data item */
		MACAddr48	addr;		/* 48-bit data item */
		SMTTimeStamp	timeStamp;	/* 64-bit data item */
	} data;
	uInt16	len1;			/* length of data at p1 */
	uInt16	len2;			/* length of data at p2 */
	union {
		uChar		*ptr;		/* generic pointer */
		SendFrameBuf	*sendFrameBuf;
		RecFrameBuf	*recFrameBuf;
		MIBMsgType	*mibMsgBuf;
	} p1;
	uChar	*p2;			/* additional data buffer */
} SMTMessage;


#pragma align 16
#pragma pack  16

/*********************************************************************
	Defined Message Values
*********************************************************************/

/*
*	Message process IDs.
*/
#define	MAP_MSG_ID	1
#define	FBM_MSG_ID	2
#define	MIB_MSG_ID	3
#define	CSP_MSG_ID	4

/*
*	Message types
*/
#define	EVENT_MSG	0x1000
#define	ACTION_MSG	0x2000
#define	RESPONSE_MSG	0x3000
#define	ERROR_MSG	0x4000

/*
*	Macro to extract message types.
*/
#define	MESSAGE_TYPE(i)		(i & 0xF000)

/*
*	Message processes and objects.
*/
#define	MAP_MSG		0x0100
#define	FBM_MSG		0x0200
#define	MIB_MSG		0x0300
#define	CSP_MSG		0x0400


/*********************************************************************
	Message Types
*********************************************************************/

/**********
*	FBM event messages.
**********/

/*
* The FBM_EVENT_NOTIFY_MIB message notifies the MIB of a change
* to an SMT managed attribute. The typeInfo member contains
* the MIB attribute ID for the attribute to change. The data field
* contains the new value.
*/
#define	FBM_EVENT_NOTIFY_MIB		(EVENT_MSG | FBM_MSG | 0x11)

/*
* The FBM_EVENT_FRAME_RECEIVED message notifies the MAP that FBM has
* received a message that does not belong to any of its protocols.
* Generally, this signifies receipt of a response frame in reply to
* a request sent by the MAP, or a status announcement frame.
*/
#define	FBM_EVENT_FRAME_RECEIVED	(EVENT_MSG | FBM_MSG | 0x12)


/**********
*	MIB event messages.
**********/

/*
* The MIB_EVENT_NOTIFY_CHANGE message reports changes in MIB attribute
* value to other tasks in SMT. This allows the other tasks to maintain
* current information without having to poll the MIB.
*/
#define	MIB_EVENT_NOTIFY_CHANGE		(EVENT_MSG | MIB_MSG | 0x11)
#define	MIB_EVENT_NOTIFY_SMT		(EVENT_MSG | MIB_MSG | 0x12)


/**********
*	CSP event messages.
**********/

/*
* The CSP_EVENT_NOTIFY_MIB message is used by CSP like the
* FBM_EVENT_NOTIFY_MIB message is used by the MIB.
*/
#define	CSP_EVENT_NOTIFY_MIB		(EVENT_MSG | CSP_MSG | 0x11)


/*
*	Action messages.
*/

/* FBM actions */
#define	FBM_ACTION_SEND_FRAME	(ACTION_MSG | FBM_MSG | 0x01)

/* MIB actions */
#define	MIB_ACTION_GET_ATTR	(ACTION_MSG | MIB_MSG | 0x01)
#define	MIB_ACTION_CHANGE_ATTR	(ACTION_MSG | MIB_MSG | 0x02)
#define	MIB_ACTION_ADD_ATTR	(ACTION_MSG | MIB_MSG | 0x03)
#define	MIB_ACTION_REMOVE_ATTR	(ACTION_MSG | MIB_MSG | 0x04)
#define	MIB_ACTION_ACTION_ATTR	(ACTION_MSG | MIB_MSG | 0x05)
#define	MIB_ACTION_DETECT_CONDITIONS\
				(ACTION_MSG | MIB_MSG | 0x06)

/* CSP actions */
#define	CSP_ACTION_FDDI_SMT	(ACTION_MSG | CSP_MSG | 0x11)
#define	CSP_ACTION_FDDI_MAC	(ACTION_MSG | CSP_MSG | 0x12)
#define	CSP_ACTION_FDDI_PORT	(ACTION_MSG | CSP_MSG | 0x13)

/*
*	Response messages.
*/

/* Station responses */

/* MAC responses */

/* PHY responses */

/* MAP responses */

/* FBM responses */
#define	FBM_RESPONSE_FRAME_SENT		(RESPONSE_MSG | FBM_MSG | 0x01)

/* MIB responses */
#define	MIB_RESPONSE_GET_ATTR		(RESPONSE_MSG | MIB_MSG | 0x01)
#define	MIB_RESPONSE_CHANGE_ATTR	(RESPONSE_MSG | MIB_MSG | 0x02)
#define	MIB_RESPONSE_ADD_ATTR		(RESPONSE_MSG | MIB_MSG | 0x03)
#define	MIB_RESPONSE_REMOVE_ATTR	(RESPONSE_MSG | MIB_MSG | 0x04)

/* CSP responses */


/*********
*	Error Message Types
*********/

#define	MIB_ERROR_MSG			(ERROR_MSG | MIB_MSG | MIB_MSG_ID)
#define CSP_ERROR_MSG			(ERROR_MSG | CSP_MSG | CSP_MSG_ID)
#define	FBM_ERROR_MSG			(ERROR_MSG | FBM_MSG | FBM_MSG_ID)


/*********************************************************************
	Implementor Defined Message Types
*********************************************************************/

#include	"smtmsgim.h"

#endif /* __SMTMSG_H */
