#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmpmfp.c V6.2.3:cs.911111:7:7 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Management Module

	PMF Protocol Processing Module

	File:		fbmpmfp.c
	Created:	09/06/90

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911111	11/11/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	Frame Based Management for SMT.  This module contains routines
	to build the Parameter Management Frames.

	Modification History:

	*** Updated to SMT 6.2 ***

	910404-004	LJP
		Changed ProcessPMFChange() so that it parses and skips
		through the parameters correctly. Also corrected the
		pointer value used for the authorization parameter.
	910605-001	LJP
		Changed ProcessPMFChange():
		(1) When copying the attribute to change, the copy
			length was calculated using outData->paramLen.
			It should be using the source length
			inData->paramLen.
		(2) The last set station ID attribute is built in
			HOST_ORDER so AttrNetOrder() does not need to
			be called.
	911111-001	LJP
		Copied and modified code from ProcessPMFChange()
		to ProcessPMFAdd() and ProcessPMFRemove() to implement
		the latter two functions.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	External Declarations
*********************************************************************/

extern	void	ClearFrameBuffer ();
extern	uInt32	BuildPMFGetResponse ();
extern	uInt32	BuildPMFChangeResponse ();
extern	uInt32	BuildPMFAddResponse ();
extern	uInt32	BuildPMFRemoveResponse ();
extern	uInt32	FBMChangeMIBAttr ();
extern	uInt32	FBMAddMIBAttr ();
extern	uInt32	FBMRemoveMIBAttr ();
extern	uInt16	ntohs ();
extern	uInt32	ntohl ();
extern	void	SendSMTFrame ();
extern	uInt16	ReasonCode_Param ();
extern	void	AttrNetOrder ();
extern	Flag	VerifyAuthorization ();
extern  uChar	*FindFrameParam ();

/*********************************************************************
	Parameter Management Frames
*********************************************************************/

uInt32
ValidateAuthorization (memPtr)
	uChar		*memPtr;
/*********************************************************************
Function:	This function finds the authorization parameter in
		the frame and verifies its value.
Parameters:	memPtr		= buffer of received frame.
Input:		memPtr		= contains received frame.
Output:		None.
Return:		Reason code
		RC_SUCCESS = authorization valid or no authorization
			necessary
		RC_NOT_AUTHORIZED = authorization invalid or
			authorization required and not found in frame
Modification History:
*********************************************************************/
{
AuthorizationParamType	*authorize;

	if (!fbmStationData.useAuthorization)
		return (RC_SUCCESS);

	authorize = (AuthorizationParamType *)
		FindFrameParam (AUTHORIZATION_PARAM_TYPE, memPtr);

	if (!authorize)
		return (RC_NOT_AUTHORIZED);

	return ((VerifyAuthorization (authorize->Authorization,
		ntohs (authorize->hdr.length)))
		? RC_SUCCESS : RC_NOT_AUTHORIZED);
}

uInt32
ProcessPMFGet (memPtr, frameLen, MACNum, EACbits)
	uChar		*memPtr;
	uInt16		frameLen;
	uInt16		MACNum;
	uChar		EACbits;
/*********************************************************************
Function:	This function handles the protocol processing for 
		a received PMF Get frame.
Parameters:	memPtr		= buffer of received frame.
		frameLen	= length of frame.
		MACNum		= index of MAC that received the frame.
		EACbits		= E, A, and C indicators.
Input:		memPtr		= contains received frame.
Output:		Generates appropriate response.
Return:		Value returned by BuildPMFGetResponse() which is the
		transaction ID used in the frame or 0 if the frame
		could not be built.
Modification History:
*********************************************************************/
{
uInt32		result;			/* build frame result */
uInt16		len;			/* total frame length */

	/*
	*	Clear frame buffer.
	*/
	ClearFrameBuffer (fbmFrameBuffer);

	/* build response frame */
	result = BuildPMFGetResponse (memPtr, fbmFrameBuffer, MACNum);

	/* send frame */
	len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
			+ SMT_FRAME_HDR_SIZE;
	SendSMTFrame (fbmFrameBuffer, len, MACNum);

	return (result);
}

uInt32
ProcessPMFChange (memPtr, frameLen, MACNum, EACbits)
	uChar		*memPtr;
	uInt16		frameLen;
	uInt16		MACNum;
	uChar		EACbits;
/*********************************************************************
Function:	This function handles the protocol processing for
		a received PMF Change frame.
Parameters:	memPtr		= buffer of received frame.
		frameLen	= length of frame.
		MACNum		= index of MAC that received the frame.
		EACbits		= E, A, and C indicators.
Input:		memPtr		= contains received frame.
Output:		Generates appropriate response.
Return:		Value returned by BuildPMFChangeResponse() which is the
		transaction ID used in the frame or 0 if the frame
		could not be built.
Modification History:
910404-004	LJP
*********************************************************************/
{
TLVSetCountType	*inSetCount,		/* input set count buffer */
		*outSetCount;		/* output set count */
TLVParamType	*lastID;		/* last set station ID buffer */
TLVParamType	*inData,		/* request MIB attribute */
		*outData;		/* response MIB attribute */

uChar		*ptr;			/* general pointer into frame buf */
uInt16		plen, len, infoLen;	/* length calculator */
SMTFrameHdrType	*reqHdr;		/* request frame header */
ParamHdrType	*pHdr;			/* frame parameter header */
uInt32		reasonCode;		/* result of change op */

	/*
	*	Clear response frame buffer.
	*/
	ClearFrameBuffer (fbmFrameBuffer);

	/*
	*	Get request parameters.
	*/
	inData = NULL;
	inSetCount = NULL;

        outSetCount = NULL;        /* jlin */
        outData = lastID = NULL;   /* jlin */

	reqHdr = (SMTFrameHdrType *) memPtr;
	ptr = memPtr + sizeof (SMTFrameHdrType);
	len = 0;
	infoLen = ntohs (reqHdr->smtHdr.InfoField_Length);

	while (len < infoLen)
	{
		pHdr = (ParamHdrType *) ptr;
		switch (ntohs (pHdr->type))
		{

		/*
		*	910404-004	LJP
		*	No need to set authorization pointer since
		*	ValidateAuthorization() searched for it.
		*/

		case fddiSMTSetCount:
			inSetCount = (TLVSetCountType *) ptr;
			break;

		default:
			/* verify MIB attribute */
			if (ntohs (pHdr->type) > fddiSMT)
				inData = (TLVParamType *) ptr;
			break;
		}

		/*
		*	910404-004	LJP
		*	Parameter length is length plus header, not
		*	just length.
		*/
		plen = ntohs (pHdr->length) + sizeof (ParamHdrType);
		ptr += plen;
		len += plen;
	}

	/*
	*	If no set count or parameter,
	*/
	/*
	*	910404-004	LJP
	*	If no set count or parameter, set reason code.
	*	Else check authorization.
	*/
	if (inSetCount == NULL) 
		reasonCode = RC_NO_PARAM; /* jlin: can not be RC_BAD_SET_COUNT */
	else if (inData == NULL)
		reasonCode = RC_ILLEGAL_PARAMETER;
	else
		reasonCode = ValidateAuthorization (memPtr);

	if (reasonCode == RC_SUCCESS)
	{       
		/*
		*	Get addresses in response frame for location of
		*	parameters.
		*/
		infoLen = 0;

		ptr = (uChar *) (fbmFrameHeader + 1); /* start of info field */
		len = ReasonCode_Param (ptr, 0);	/* skip reason code */
		infoLen += len;
		ptr += len;

		outSetCount = (TLVSetCountType *) ptr;	/* response set count */
		len = sizeof (TLVSetCountType) + sizeof (TLVHdrType);
		infoLen += len;
		ptr += len;

		lastID = (TLVParamType *) ptr;		/* response last ID */
		len = sizeof (TLVIdType) + sizeof (TLVHdrType);
		infoLen += len;
		ptr += len;

		outData = (TLVParamType *) ptr;		/* MIB data buffer */
		len = MAX_SMT_INFO_LEN - infoLen;	/* buffer size */

		/*
		*	Fill in MIB Change info using response frame as buffer.
		*/
		/* copy set count and put into host byte order */
		MEMCOPY (outSetCount, inSetCount, sizeof (TLVSetCountType));
		AttrNetOrder (outSetCount, HOST_ORDER);

		/* copy last set station id and put into host byte order */
		lastID->paramType = fddiSMTLastSetStationId;
		lastID->paramLen = sizeof (TLVIdType);
		MEMCOPY (&(lastID->SMTOTHER), &reqHdr->smtHdr.Station_ID,
			sizeof (SMTStationIdType));
		/*
		*	910605-001	LJP
		*	Last set station ID is already in host order.
		*	Do not call AttrNetOrder().
		*/
		/* AttrNetOrder (lastID, HOST_ORDER); */

		/* copy attribute and put into host byte order */
		/*
		*	910605-001	LJP
		*	Length of copy should use inData->paramLen,
		*	not outData->paramLen.
		*/
		MEMCOPY (outData, inData,
			ntohs (inData->paramLen) + sizeof (TLVHdrType));
		AttrNetOrder (outData, HOST_ORDER);

		/*
		*	Request change.
		*/
		reasonCode = FBMChangeMIBAttr (len, (uChar *) outData, 
					  outSetCount, &(lastID->SMTOTHER));
	}

	/*
	*	Complete frame building process.
	*/
	BuildPMFChangeResponse (memPtr, fbmFrameBuffer, MACNum,
		reasonCode, outSetCount, lastID, outData);

	/*
	*	Send frame.
	*/
	len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
		+ SMT_FRAME_HDR_SIZE;
	SendSMTFrame (fbmFrameBuffer, len, MACNum);

	return (ntohl (fbmFrameHeader->smtHdr.Transaction_ID));
}

uInt32
ProcessPMFAdd (memPtr, frameLen, MACNum, EACbits)
	uChar		*memPtr;
	uInt16		frameLen;
	uInt16		MACNum;
	uChar		EACbits;
/*********************************************************************
Function:	This function handles the protocol processing for
		a received PMF Add frame.
Parameters:	memPtr		= buffer of received frame.
		frameLen	= length of frame.
		MACNum		= index of MAC that received the frame.
		EACbits		= E, A, and C indicators.
Input:		memPtr		= contains received frame.
Output:		Generates appropriate response.
Return:		Value returned by BuildPMFAddResponse() which is the
		transaction ID used in the frame or 0 if the frame
		could not be built.
Modification History:
*********************************************************************/
{
TLVSetCountType	*inSetCount,		/* input set count buffer */
		*outSetCount;		/* output set count */
TLVParamType	*lastID;		/* last set station ID buffer */
TLVParamType	*inData,		/* request MIB attribute */
		*outData;		/* response MIB attribute */

uChar		*ptr;			/* general pointer into frame buf */
uInt16		plen, len, infoLen;	/* length calculator */
SMTFrameHdrType	*reqHdr;		/* request frame header */
ParamHdrType	*pHdr;			/* frame parameter header */
uInt32		reasonCode;		/* result of change op */

	/*
	*	Clear response frame buffer.
	*/
	ClearFrameBuffer (fbmFrameBuffer);

	/*
	*	Get request parameters.
	*/
	inData = NULL;
	inSetCount = NULL;

        outSetCount = NULL;        /* jlin */
        outData = lastID = NULL;   /* jlin */

	reqHdr = (SMTFrameHdrType *) memPtr;
	ptr = memPtr + sizeof (SMTFrameHdrType);
	len = 0;
	infoLen = ntohs (reqHdr->smtHdr.InfoField_Length);

	/*
	*	Loop through parameters in frame.
	*/
	while (len < infoLen)
	{
		pHdr = (ParamHdrType *) ptr;
		switch (ntohs (pHdr->type))
		{
		case fddiSMTSetCount:
			inSetCount = (TLVSetCountType *) ptr;
			break;

		default:
			/*
			*	Verify MIB attribute. If not valid attr,
			*	inData is not set. Error is handled below.
			*/
			if (ntohs (pHdr->type) > fddiSMT)
				inData = (TLVParamType *) ptr;
			break;
		}

		/*
		*	Parameter length is length plus header
		*/
		plen = ntohs (pHdr->length) + sizeof (ParamHdrType);
		ptr += plen;
		len += plen;
	}

	/*
	*	If no set count or parameter,
	*/
	if (inSetCount == NULL) 
		reasonCode = RC_NO_PARAM; /* jlin:  can not be RC_BAD_SET_COUNT */
	else if (inData == NULL)
		reasonCode = RC_ILLEGAL_PARAMETER;
	else
		/*
		*	Validate authorization parameter.
		*/
		reasonCode = ValidateAuthorization (memPtr);

	/*
	*	If frame is OK so far, continue processing.
	*/
	if (reasonCode == RC_SUCCESS)
	{       
		/*
		*	Get locations in response frame for parameters.
		*/
		infoLen = 0;

		ptr = (uChar *) (fbmFrameHeader + 1); /* start of info field */
		len = ReasonCode_Param (ptr, 0);	/* skip reason code */
		infoLen += len;
		ptr += len;

		outSetCount = (TLVSetCountType *) ptr;	/* response set count */
		len = sizeof (TLVSetCountType) + sizeof (TLVHdrType);
		infoLen += len;
		ptr += len;

		lastID = (TLVParamType *) ptr;		/* response last ID */
		len = sizeof (TLVIdType) + sizeof (TLVHdrType);
		infoLen += len;
		ptr += len;

		outData = (TLVParamType *) ptr;		/* MIB data buffer */
		len = MAX_SMT_INFO_LEN - infoLen;	/* buffer size */

		/*
		*	Fill in MIB Change info using response frame as buffer.
		*/
		/* copy set count and put into host byte order */
		MEMCOPY (outSetCount, inSetCount, sizeof (TLVSetCountType));
		AttrNetOrder (outSetCount, HOST_ORDER);

		/* copy last set station id and put into host byte order */
		lastID->paramType = fddiSMTLastSetStationId;
		lastID->paramLen = sizeof (TLVIdType);
		MEMCOPY (&(lastID->SMTOTHER), &reqHdr->smtHdr.Station_ID,
			sizeof (SMTStationIdType));

		/* copy attribute and put into host byte order */
		MEMCOPY (outData, inData,
			ntohs (inData->paramLen) + sizeof (TLVHdrType));
		AttrNetOrder (outData, HOST_ORDER);

		/*
		*	Request add.
		*/
		reasonCode = FBMAddMIBAttr (len, (uChar *) outData, 
			                outSetCount, &(lastID->SMTOTHER));
	}

	/*
	*	Complete frame building process.
	*/
	BuildPMFAddResponse (memPtr, fbmFrameBuffer, MACNum,
		reasonCode, outSetCount, lastID, outData);

	/*
	*	Send frame.
	*/
	len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
		+ SMT_FRAME_HDR_SIZE;
	SendSMTFrame (fbmFrameBuffer, len, MACNum);

	return (ntohl (fbmFrameHeader->smtHdr.Transaction_ID));
}

uInt32
ProcessPMFRemove (memPtr, frameLen, MACNum, EACbits)
	uChar		*memPtr;
	uInt16		frameLen;
	uInt16		MACNum;
	uChar		EACbits;
/*********************************************************************
Function:	This function handles the protocol processing for
		a received PMF Remove frame.
Parameters:	memPtr		= buffer of received frame.
		frameLen	= length of frame.
		MACNum		= index of MAC that received the frame.
		EACbits		= E, A, and C indicators.
Input:		memPtr		= contains received frame.
Output:		Generates appropriate response.
Return:		Value returned by BuildPMFRemoveResponse() which is the
		transaction ID used in the frame or 0 if the frame
		could not be built.
Modification History:
*********************************************************************/
{
TLVSetCountType	*inSetCount,		/* input set count buffer */
		*outSetCount;		/* output set count */
TLVParamType	*lastID;		/* last set station ID buffer */
TLVParamType	*inData,		/* request MIB attribute */
		*outData;		/* response MIB attribute */

uChar		*ptr;			/* general pointer into frame buf */
uInt16		plen, len, infoLen;	/* length calculator */
SMTFrameHdrType	*reqHdr;		/* request frame header */
ParamHdrType	*pHdr;			/* frame parameter header */
uInt32		reasonCode;		/* result of change op */

	/*
	*	Clear response frame buffer.
	*/
	ClearFrameBuffer (fbmFrameBuffer);

	/*
	*	Get request parameters.
	*/
	inData = NULL;
	inSetCount = NULL;

        outSetCount = NULL;        /* jlin */
        outData = lastID = NULL;   /* jlin */

	reqHdr = (SMTFrameHdrType *) memPtr;
	ptr = memPtr + sizeof (SMTFrameHdrType);
	len = 0;
	infoLen = ntohs (reqHdr->smtHdr.InfoField_Length);

	/*
	*	Loop through parameters in frame.
	*/
	while (len < infoLen)
	{
		pHdr = (ParamHdrType *) ptr;
		switch (ntohs (pHdr->type))
		{
		case fddiSMTSetCount:
			inSetCount = (TLVSetCountType *) ptr;
			break;

		default:
			/*
			*	Verify MIB attribute. If not valid attr,
			*	inData is not set. Error is handled below.
			*/
			if (ntohs (pHdr->type) > fddiSMT)
				inData = (TLVParamType *) ptr;
			break;
		}

		/*
		*	Parameter length is length plus header
		*/
		plen = ntohs (pHdr->length) + sizeof (ParamHdrType);
		ptr += plen;
		len += plen;
	}

	/*
	*	If no set count or parameter,
	*/
	if (inSetCount == NULL) 
		reasonCode = RC_NO_PARAM; /* jlin: can not be RC_BAD_SET_COUNT */
	else if (inData == NULL)
		reasonCode = RC_ILLEGAL_PARAMETER;
	else
		/*
		*	Validate authorization parameter.
		*/
		reasonCode = ValidateAuthorization (memPtr);

	/*
	*	If frame is OK so far, continue processing.
	*/
	if (reasonCode == RC_SUCCESS)
	{
		/*
		*	Get locations in response frame for parameters.
		*/
		infoLen = 0;

		ptr = (uChar *) (fbmFrameHeader + 1); /* start of info field */
		len = ReasonCode_Param (ptr, 0);	/* skip reason code */
		infoLen += len;
		ptr += len;

		outSetCount = (TLVSetCountType *) ptr;	/* response set count */
		len = sizeof (TLVSetCountType) + sizeof (TLVHdrType);
		infoLen += len;
		ptr += len;

		lastID = (TLVParamType *) ptr;		/* response last ID */
		len = sizeof (TLVIdType) + sizeof (TLVHdrType);
		infoLen += len;
		ptr += len;

		outData = (TLVParamType *) ptr;		/* MIB data buffer */
		len = MAX_SMT_INFO_LEN - infoLen;	/* buffer size */

		/*
		*	Fill in MIB Change info using response frame as buffer.
		*/
		/* copy set count and put into host byte order */
		MEMCOPY (outSetCount, inSetCount, sizeof (TLVSetCountType));
		AttrNetOrder (outSetCount, HOST_ORDER);

		/* copy last set station id and put into host byte order */
		lastID->paramType = fddiSMTLastSetStationId;
		lastID->paramLen = sizeof (TLVIdType);
		MEMCOPY (&(lastID->SMTOTHER), &reqHdr->smtHdr.Station_ID,
			sizeof (SMTStationIdType));

		/* copy attribute and put into host byte order */
		MEMCOPY (outData, inData,
			ntohs (inData->paramLen) + sizeof (TLVHdrType));
		AttrNetOrder (outData, HOST_ORDER);

		/*
		*	Request add.
		*/
		reasonCode = FBMRemoveMIBAttr (len, (uChar *) outData, 
			outSetCount, &(lastID->SMTOTHER));
	}

	/*
	*	Complete frame building process.
	*/
	BuildPMFRemoveResponse (memPtr, fbmFrameBuffer, MACNum,
		reasonCode, outSetCount, lastID, outData);

	/*
	*	Send frame.
	*/
	len = ntohs (fbmFrameHeader->smtHdr.InfoField_Length)
		+ SMT_FRAME_HDR_SIZE;
	SendSMTFrame (fbmFrameBuffer, len, MACNum);

	return (ntohl (fbmFrameHeader->smtHdr.Transaction_ID));
}

