#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmparam.c V6.2.3:cs.910807:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Services Process Module
	
	Frame Parameter Processing Module
	
	File:		fbmparam.c
	Created:	08/27/90
			Added for SMT 6.2

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.910807	08/19/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module contains routines that format the SMT frame 
	parameters.

	Modification History:
	910416-001	LJP
		Convert time stamp parameter to network byte order.
	910807-005	LJP
		In RejectedFrame_Param(), the value of maxLen is now
		reduced by the size of the parameter header and pad
		field.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	External Declarations
*********************************************************************/

extern	uInt16	htons ();
extern	uInt32	htonl ();
extern	uInt16	ntohs ();
extern	uInt32	ntohl ();
extern	uInt32	FBMGetMIBAttr ();
extern	void	AttrNetOrder ();
extern	uInt16	BuildAuthorization ();


/*********************************************************************
	Frame Parameter Support Routines
*********************************************************************/

/*
*	Array of TLV MIB attributes for MIB groups.
*/
TLVParamType		mibGrp[15];

static TLVParamType *
GetGroupAttr (base, attrID)
	TLVParamType	*base;
	uInt16		attrID;
/*********************************************************************
Function:	Get a specific attribute from within a MIB attribute
		group.
Parameters:	base	= pointer to start of MIB attribute group.
		index	= MIB attribute ID to get
Input:		None.
Output:		None.
Return:		Pointer to parameter type of requested parameter.
		NULL if not found.
Notes:		List of attributes must end with a 0 parameter type.
Modification History:
*********************************************************************/
{
	while (base->paramType != 0)
	{
		if (base->paramType == attrID)
			return (base);
		base = (TLVParamType *) (((uChar *) base) +
			base->paramLen + 4);
	}

	return (NULL);
}


/*********************************************************************
	Frame Parameter Routines
*********************************************************************/

uInt16
UNA_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the Upstream Neighbor Address parameter.
Parameters:	pBuffer	= pointer to location where UNA parameter
			should start.
		macID	= FBM index of MAC sending frame.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the UNA SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
UNAParamType	*param;
TLVParamType	mibData;

	/* cast pointer */
	param = (UNAParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (UNA_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (UNAParamType) - sizeof (ParamHdrType));

	/* fill in parameter */
	mibData.paramType = fddiMACUpstreamNbr;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	MCopyAddress (param->UNA, mibData.MACADDR);

	return (sizeof (UNAParamType));
}

uInt16
StationDescriptor_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Station Descriprot parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
StationDescParamType	*param;

	/* cast pointer */
	param = (StationDescParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (STATION_DESC_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (StationDescParamType) - sizeof (ParamHdrType));

	/* fill in parameter */
	param->NodeClass = (fbmStationData.masterCount == 0) 
		? SMT_Type_Station : SMT_Type_Concentrator;
	param->Mac_Ct = fbmStationData.macCount;
	param->NonMaster_Ct = fbmStationData.attachCount;
	param->Master_Ct = fbmStationData.masterCount;

	return (sizeof (StationDescParamType));
}

uInt16
StationState_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the Station State parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		macID	= FBM MAC index sending this frame.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
StationStateParamType	*param;
TLVParamType		mibData;
uChar			CF_State;

	/* cast pointer */
	param = (StationStateParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (STATION_STATE_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (StationStateParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->Topology = 0;
	param->DuplAddr = 0;

	/*
	*	Set Topology field
	*/

	/* set wrapped/thru */
	mibData.paramType = fddiSMTCF_State;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	CF_State = mibData.SMTPARAM8;
	if ((CF_State != CF_ISOLATED) && (CF_State != CF_THRU))
		param->Topology |= Topology_Wrapped;

	/* set unrooted concentrator */
	if ((fbmStationData.masterCount > 0)
			&& (CF_State == CF_ISOLATED))
		param->Topology |= Topology_Unrooted;

	/* set rooted station */
	else if (fbmStationData.attachCount == 1)
	{
		/* get S PC_Neighbor */
		mibData.paramType = fddiPORTPC_Neighbor;
		mibData.paramLen = 4;
		mibData.PORTINDEX = MIB_PHY_S;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		if (mibData.PORTPARAM8 != PC_Type_M)
			param->Topology |= Topology_Rooted_Station;
	}

	else
	{
	uChar	nbrA, nbrB;

		/* get A,B PC_Neighbor */
		mibData.paramType = fddiPORTPC_Neighbor;
		mibData.paramLen = 4;
		mibData.PORTINDEX = MIB_PHY_A;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		nbrA = mibData.PORTPARAM8;

		mibData.paramType = fddiPORTPC_Neighbor;
		mibData.paramLen = 4;
		mibData.PORTINDEX = MIB_PHY_B;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		nbrB = mibData.PORTPARAM8;

		if (nbrA == PC_Type_A)
			param->Topology |= Topology_Twisted_AA;

		if (nbrB == PC_Type_B)
			param->Topology |= Topology_Twisted_BB;

		if ((nbrA != PC_Type_M)
				&& (nbrB != PC_Type_M))
			param->Topology |= Topology_Rooted_Station;
	}

	if (fbmStationData.useSRP)
		param->Topology |= Topology_SRF;

	/*
	*	Set DuplAddr field
	*/
	if (fbmMACData[macID].myDuplicate)
		param->DuplAddr |= DuplAddr_My_Duplicate;

	if (fbmMACData[macID].myUNADuplicate)
		param->DuplAddr |= DuplAddr_My_UNA_Duplicate;

	return (sizeof (StationStateParamType));
}

uInt16
MsgTimeStamp_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Message Time Stamp parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
MsgTimeStampParamType	*param;
TLVParamType		mibData;

	/* cast pointer */
	param = (MsgTimeStampParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (MSG_TIME_STAMP_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (MsgTimeStampParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	mibData.paramType = fddiSMTMsgTimeStamp;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	MEMCOPY (&param->MsgTimeStamp, &mibData.SMTOTHER, mibData.paramLen);

	/*
	*	910416-001	LJP
	*	Convert to network order.
	*/
	param->MsgTimeStamp.hiword = htonl (param->MsgTimeStamp.hiword);
	param->MsgTimeStamp.loword = htonl (param->MsgTimeStamp.loword);

	return (sizeof (MsgTimeStampParamType));
}

uInt16
StationPolicies_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Station Policies parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
StationPoliciesParamType	*param;
TLVParamType			mibData;

	/* cast pointer */
	param = (StationPoliciesParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (STATION_POLICIES_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (StationPoliciesParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->ConfigPolicy = 0;
	param->ConnectionPolicy = 0;

	/*
	*	Get hold configuration policy
	*/
	mibData.paramType = fddiSMTConfigPolicy;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	param->ConfigPolicy = (mibData.SMTPARAM16 & Config_Hold_Available)
		? 1 : 0;

	/*
	*	Get connection policy and convert to frame format
	*/
	mibData.paramType = fddiSMTConnectionPolicy;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);

	if (mibData.SMTPARAM16 & Policy_rejectA_A)
		param->ConnectionPolicy |= FBM_rejectA_A;
	if (mibData.SMTPARAM16 & Policy_rejectB_B)
		param->ConnectionPolicy |= FBM_rejectB_B;
	if (mibData.SMTPARAM16 & Policy_rejectS_S)
		param->ConnectionPolicy |= FBM_rejectS_S;
	if ((mibData.SMTPARAM16 & Policy_rejectA_M)
			|| (mibData.SMTPARAM16 & Policy_rejectM_A))
		param->ConnectionPolicy |= FBM_rejectA_M;
	if ((mibData.SMTPARAM16 & Policy_rejectB_M)
			|| (mibData.SMTPARAM16 & Policy_rejectM_B))
		param->ConnectionPolicy |= FBM_rejectB_M;
	if ((mibData.SMTPARAM16 & Policy_rejectA_S)
			|| (mibData.SMTPARAM16 & Policy_rejectS_A))
		param->ConnectionPolicy |= FBM_rejectA_S;
	if ((mibData.SMTPARAM16 & Policy_rejectB_S)
			|| (mibData.SMTPARAM16 & Policy_rejectS_B))
		param->ConnectionPolicy |= FBM_rejectB_S;

	/* convert to network order */
	param->ConfigPolicy = htons (param->ConfigPolicy);
	param->ConnectionPolicy = htons (param->ConnectionPolicy);

	return (sizeof (StationPoliciesParamType));
}

uInt16
PathLatency_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Path Latency Contribution parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
PathLatencyParamType	*param;
TLVParamType		mibData;

	/* if path latency value is not set */
	if (fbmStationData.latencyRing1 == 0)
		/* then this parameter is not supported */
		return (0);

	/* cast pointer */
	param = (PathLatencyParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (PATH_LATENCY_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (PathLatencyParamType) - sizeof (ParamHdrType));

	/* set latency values */
	param->PathLatencyRing1 = fbmStationData.latencyRing1;
	param->PathLatencyRing2 = fbmStationData.latencyRing2;

	/*
	*	Set output port indices
	*	The output port index is the connected resource index
	*	of the last MAC on the path because the MAC is connected
	*	to the output port.
	*/

	/* get connected resource for primary MAC */
	mibData.paramType = xdiMACConnectedIndex;
	mibData.paramLen = 4;
	mibData.MACINDEX = PRIMARY_MAC + 1;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	param->PORTOutIndex1 = htons (mibData.MACPARAM16);

	/* get connected resource for secondary MAC (if available) */
	if (fbmStationData.macCount > 1)
	{
		mibData.paramType = xdiMACConnectedIndex;
		mibData.paramLen = 4;
		mibData.MACINDEX = SECONDARY_MAC + 1;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		param->PORTOutIndex2 = htons (mibData.MACPARAM16);
	}
	else
		param->PORTOutIndex2 = 0;


	return (sizeof (PathLatencyParamType));
}

uInt16
MACNeighbors_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the MAC Neighbors parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		macID	= FBM index of MAC sending the frame.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
MACNeighborsParamType	*param;
TLVParamType		mibData;

	/* cast pointer */
	param = (MACNeighborsParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (MAC_NEIGHBORS_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (MACNeighborsParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->MAC_Index = htons (fbmStationData.phyCount + macID + 1);

	mibData.paramType = fddiMACUpstreamNbr;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	MCopyAddress (param->UNA, mibData.MACADDR);

	mibData.paramType = fddiMACDownstreamNbr;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	MCopyAddress (param->DNA, mibData.MACADDR);

	return (sizeof (MACNeighborsParamType));
}

uInt16
PathDescriptor_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Path Descriptor parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= containes the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
PathDescParamType	*param;
TLVParamType		mibData;
PathPORTRecordType	*portRecord;
PathMACRecordType	*macRecord;
uInt16			i;

	/* cast pointer */
	param = (PathDescParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (PATH_DESC_PARAM_TYPE);
	param->hdr.length = 0;

	/* fill in parameter */
	portRecord = (PathPORTRecordType *) &param->data;
	for (i = 1; i <= fbmStationData.phyCount; i++, portRecord++)
	{
		/* get port type */
		mibData.paramType = fddiPORTPC_Type;
		mibData.paramLen = 4;
		mibData.PORTINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		portRecord->PORTType = mibData.PORTPARAM8;

		/* get connect state */
		mibData.paramType = fddiPORTConnectState;
		mibData.paramLen = 4;
		mibData.PORTINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		/* connect state is 8 bits and does not need converting */
		portRecord->ConnectState = mibData.PORTPARAM16;

		/* get remote port type */
		mibData.paramType = fddiPORTPC_Neighbor;
		mibData.paramLen = 4;
		mibData.PORTINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		portRecord->RemotePORTType = mibData.PORTPARAM8;

		/* get remote MAC */
		mibData.paramType = fddiPORTRemoteMACIndicated;
		mibData.paramLen = 4;
		mibData.PORTINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		portRecord->RemoteMAC = mibData.PORTPARAM8;

		/* get connected resource */
		mibData.paramType = xdiPORTConnectedIndex;
		mibData.paramLen = 4;
		mibData.PORTINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		portRecord->ConResourceIndex = htons (mibData.PORTPARAM16);

		/* accumulate length */
		param->hdr.length += sizeof (PathPORTRecordType);
	}

	macRecord = (PathMACRecordType *) portRecord;
	for (i = 1; i <= fbmStationData.macCount; i++, macRecord++)
	{
		/* get MAC address */
		mibData.paramType = fddiMACSMTAddress;
		mibData.paramLen = 4;
		mibData.MACINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		MCopyAddress (macRecord->MAC_Addr, mibData.MACADDR);

		/* get connected resource */
		mibData.paramType = xdiMACConnectedIndex;
		mibData.paramLen = 4;
		mibData.MACINDEX = i;
		FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL);
		macRecord->ConResourceIndex = htons (mibData.MACPARAM16);

		/* accumulate length */
		param->hdr.length += sizeof (PathMACRecordType);
	}

	/* save unconverted length */
	i = param->hdr.length;

	/* convert length */
	param->hdr.length = htons (param->hdr.length);

	return (i + sizeof (ParamHdrType));
}

uInt16
MACStatus_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the MAC Status parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		macID	= FBM index of MAC to get status.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
MACStatusParamType	*param;
TLVParamType		*mibptr;	/* pointer into group */

	/* cast pointer */
	param = (MACStatusParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (MAC_STATUS_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (MACStatusParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->MAC_Index = htons (fbmStationData.phyCount + macID + 1);

	/* get MAC operation group */
	mibGrp[0].paramType = fddiMACOperationGrp;
	mibGrp[0].paramLen = 4;
	mibGrp[0].MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibGrp), (char *) mibGrp,
		(SetCountType *) NULL);

	/* T_Req */
	mibptr = GetGroupAttr (mibGrp, fddiMACT_Req);
	if (mibptr)
		param->T_Req = htonl (mibptr->MACPARAM32);

	/* T_Neg */
	mibptr = GetGroupAttr (mibGrp, fddiMACT_Neg);
	if (mibptr)
		param->T_Neg = htonl (mibptr->MACPARAM32);

	/* T_Max */
	mibptr = GetGroupAttr (mibGrp, fddiMACT_Max);
	if (mibptr)
		param->T_Max = htonl (mibptr->MACPARAM32);

	/* TVXValue */
	mibptr = GetGroupAttr (mibGrp, fddiMACTvxValue);
	if (mibptr)
		param->TVXValue = htonl (mibptr->MACPARAM32);

	/* T_Min */
	mibptr = GetGroupAttr (mibGrp, fddiMACT_Min);
	if (mibptr)
		param->T_Min = htonl (mibptr->MACPARAM32);

	/* set SBA to 0 */
	param->SBA = 0;

	/* get frame count */
	mibGrp[0].paramType = fddiMACFrame_Ct;
	mibGrp[0].paramLen = 4;
	mibGrp[0].MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibGrp), (char *) mibGrp,
		(SetCountType *) NULL);
	param->Frame_Ct = htonl (mibGrp[0].MACPARAM32);

	/* get error count */
	mibGrp[0].paramType = fddiMACError_Ct;
	mibGrp[0].paramLen = 4;
	mibGrp[0].MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibGrp), (char *) mibGrp,
		(SetCountType *) NULL);
	param->Error_Ct = htonl (mibGrp[0].MACPARAM32);

	/* get lost count */
	mibGrp[0].paramType = fddiMACLost_Ct;
	mibGrp[0].paramLen = 4;
	mibGrp[0].MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibGrp), (char *) mibGrp,
		(SetCountType *) NULL);
	param->Lost_Ct = htonl (mibGrp[0].MACPARAM32);

	return (sizeof (MACStatusParamType));
}

uInt16
LERStatus_Param (pBuffer, phyID)
	uChar	*pBuffer;
	uInt16	phyID;
/*********************************************************************
Function:	Create the PORT LER Status parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		phyID	= FBM index of PHY to get LER status.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
LERStatusParamType	*param;
TLVParamType		*mibptr;	/* pointer into group */

	/* cast pointer */
	param = (LERStatusParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (LER_STATUS_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (LERStatusParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->PORT_Index = htons (phyID + 1);

	/* get LER group */
	mibGrp[0].paramType = fddiPORTLerGrp;
	mibGrp[0].paramLen = 4;
	mibGrp[0].PORTINDEX = phyID + 1;
	FBMGetMIBAttr (sizeof (mibGrp), (char *) mibGrp,
		(SetCountType *) NULL);

	/* get LER cutoff */
	mibptr = GetGroupAttr (mibGrp, fddiPORTLer_Cutoff);
	if (mibptr)
		param->Ler_Cutoff = mibptr->PORTPARAM8;

	/* get LER alarm */
	mibptr = GetGroupAttr (mibGrp, fddiPORTLer_Alarm);
	if (mibptr)
		param->Ler_Alarm = mibptr->PORTPARAM8;

	/* get LER estimate */
	mibptr = GetGroupAttr (mibGrp, fddiPORTLer_Estimate);
	if (mibptr)
		param->Ler_Estimate = mibptr->PORTPARAM8;

	/* get LEM reject count */
	mibptr = GetGroupAttr (mibGrp, fddiPORTLem_Reject_Ct);
	if (mibptr)
		param->Lem_Reject_Ct = htonl (mibptr->PORTPARAM32);

	/* get LEM count */
	mibptr = GetGroupAttr (mibGrp, fddiPORTLem_Ct);
	if (mibptr)
		param->Lem_Ct = htonl (mibptr->PORTPARAM32);

	return (sizeof (LERStatusParamType));
}

uInt16
FrameCounters_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the MAC Frame Counters parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		macID	= FBM index of MAC to get counts.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
FrameCountersParamType	*param;
TLVParamType		*mibptr;	/* pointer into group */

	/* cast pointer */
	param = (FrameCountersParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (FRAME_COUNTERS_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (FrameCountersParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->MAC_Index = htons (fbmStationData.phyCount + macID + 1);

	/* get MAC counters group */
	mibGrp[0].paramType = fddiMACCountersGrp;
	mibGrp[0].paramLen = 4;
	mibGrp[0].MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibGrp), (char *) mibGrp,
		(SetCountType *) NULL);

	/* receive count */
	mibptr = GetGroupAttr (mibGrp, fddiMACCopied_Ct);
	if (mibptr)
		param->Receive_Ct = htonl (mibptr->MACPARAM32);

	/* transmit count */
	mibptr = GetGroupAttr (mibGrp, fddiMACTransmit_Ct);
	if (mibptr)
		param->Transmit_Ct = htonl (mibptr->MACPARAM32);

	return (sizeof (FrameCountersParamType));
}

uInt16
FrameNotCopied_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the MAC Frame Not Copied Count parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		macID	= FBM index of MAC to get not copied count.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
FrameNotCopiedParamType	*param;
TLVParamType		mibData;

	/* cast pointer */
	param = (FrameNotCopiedParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (FRAME_NOT_COPIED_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (FrameNotCopiedParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->MAC_Index = htons (fbmStationData.phyCount + macID + 1);

	mibData.paramType = fddiMACNotCopied_Ct;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
		(SetCountType *) NULL);
	param->NotCopied_Ct = htonl (mibData.MACPARAM32);

	return (sizeof (FrameNotCopiedParamType));
}

uInt16
PriorityValues_Param (pBuffer, macID)
	uChar	*pBuffer;
	uInt16	macID;
/*********************************************************************
Function:	Create the MAC Priority Values parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		macID	= FBM index of MAC to get priority values.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
PriorityValuesParamType	*param;
TLVParamType		mibData;
Int16			count;		/* number of priorities supported */

	/* cast pointer */
	param = (PriorityValuesParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (PRIORITY_VALUES_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (PriorityValuesParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->MAC_Index = htons (fbmStationData.phyCount + macID + 1);

	/*
	*	Since the priority values are optional in the MIB,
	*	each one is retrieved separately. The value of
	*	count is incremented for each supported value. If
	*	no values are supported, then this parameter is
	*	not used.
	*/
	count = 0;

	/* get T_Pri0 */
	mibData.paramType = fddiMACT_Pri0;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri0 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri0 = 0;

	/* get T_Pri1 */
	mibData.paramType = fddiMACT_Pri1;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri1 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri1 = 0;

	/* get T_Pri2 */
	mibData.paramType = fddiMACT_Pri2;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri2 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri2 = 0;

	/* get T_Pri3 */
	mibData.paramType = fddiMACT_Pri3;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri3 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri3 = 0;

	/* get T_Pri4 */
	mibData.paramType = fddiMACT_Pri4;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri4 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri4 = 0;

	/* get T_Pri5 */
	mibData.paramType = fddiMACT_Pri5;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri5 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri5 = 0;

	/* get T_Pri6 */
	mibData.paramType = fddiMACT_Pri6;
	mibData.paramLen = 4;
	mibData.MACINDEX = macID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		/* fill in value */
		param->T_Pri6 = htonl (mibData.MACPARAM32);
		count++;
	}
	else
		param->T_Pri6 = 0;

	/* skip this parameter if no priority values are supported */
	if (count == 0)
	{
		param->hdr.type = 0;
		param->hdr.length = 0;
		return (0);
	}

	return (sizeof (PriorityValuesParamType));
}

uInt16
EBStatus_Param (pBuffer, phyID)
	uChar	*pBuffer;
	uInt16	phyID;
/*********************************************************************
Function:	Create the PORT EB Status parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		phyID	= FBM port index to get EB status.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
EBStatusParamType	*param;
TLVParamType		mibData;

	/* cast pointer */
	param = (EBStatusParamType *) pBuffer;

	/*
	*	Get MIB value. If not supported, then skip
	*	this parameter.
	*/
	mibData.paramType = fddiPORTEBError_Ct;
	mibData.paramLen = 4;
	mibData.PORTINDEX = phyID + 1;
	if (FBMGetMIBAttr (sizeof (mibData), (char *) &mibData,
			(SetCountType *) NULL) != RC_SUCCESS)
		return (0);

	/* fill in header */
	param->hdr.type = htons (EB_STATUS_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (EBStatusParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->PORT_Index = htons (phyID + 1);
	param->EbErrorCt = htonl (mibData.PORTPARAM32);

	return (sizeof (EBStatusParamType));
}

uInt16
Manufacturer_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Manufacturer Field parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
ManufacturerParamType	*param;
uChar			mibData[36];	/* 32-byte attribute */
TLVParamType		*mibptr;

	/* cast pointer */
	param = (ManufacturerParamType *) pBuffer;

	/*
	*	If attribute is not supported by the MIB,
	*	then skip this parameter.
	*/
	mibptr = (TLVParamType *) mibData;
	mibptr->paramType = fddiSMTManufacturerData;
	mibptr->paramLen = 0;
	if (FBMGetMIBAttr (sizeof (mibData), (char *) mibData,
			(SetCountType *) NULL) != RC_SUCCESS)
		return (0);

	/* fill in header */
	param->hdr.type = htons (MANUFACTURER_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (ManufacturerParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	MEMCOPY (&param->manuf, &mibptr->SMTOTHER,
		sizeof (SMTManufacturerDataType));

	return (sizeof (ManufacturerParamType));
}

uInt16
User_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the User Field parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
UserParamType	*param;
uChar		mibData[36];	/* 32-byte attribute */
TLVParamType	*mibptr;

	/* cast pointer */
	param = (UserParamType *) pBuffer;

	/*
	*	If attribute is not supported by the MIB,
	*	then skip this parameter.
	*/
	mibptr = (TLVParamType *) mibData;
	mibptr->paramType = fddiSMTUserData;
	mibptr->paramLen = 0;
	if (FBMGetMIBAttr (sizeof (mibData), (char *) mibData,
			(SetCountType *) NULL) != RC_SUCCESS)
		return (0);

	/* fill in header */
	param->hdr.type = htons (USER_PARAM_TYPE);
	param->hdr.length
		= htons (sizeof (UserParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	MEMCOPY (param->user_data, &mibptr->SMTOTHER,
		sizeof (SMTUserDataType));

	return (sizeof (UserParamType));
}

uInt16
Echo_Param (pBuffer, eBuffer, eSize)
	uChar	*pBuffer;
	uChar	*eBuffer;
	uInt16	eSize;
/*********************************************************************
Function:	Create the Echo Data parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		eBuffer	= pointer to echo data.
		eSize	= length of echo data.
Input:		Uses fbmStationData.
		eBuffer	= contains echo data to use. This data must be
			in network byte order.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
EchoParamType	*param;

	/* cast pointer */
	param = (EchoParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (ECHO_PARAM_TYPE);
	param->hdr.length 
		= htons (eSize);

	/* fill in parameter */
	MEMCOPY (param->echo_data, eBuffer, eSize);

	return (eSize + sizeof (ParamHdrType));
}

uInt16
ReasonCode_Param (pBuffer, code)
	uChar	*pBuffer;
	uInt32	code;
/*********************************************************************
Function:	Create the Reason Code parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		code	= reason code to use.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
ReasonCodeParamType	*param;

	/* cast pointer */
	param = (ReasonCodeParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (REASON_CODE_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (ReasonCodeParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	param->reason = htonl (code);

	return (sizeof (ReasonCodeParamType));
}

uInt16
RejectedFrame_Param (pBuffer, frame, maxLen, reqFrameLen)
	uChar		*pBuffer;
	SMTFrameHdrType	*frame;
	uInt16		maxLen;
        uInt16          reqFrameLen;
/*********************************************************************
Function:	Create the Rejected Frame Beginning parameter.
Parameters:	pBuffer		= pointer to location where parameter
				should start.
		frame		= pointer to rejected frame.
		maxLen		= maximum size of this parameter (i.e., # of
		bytes left in frame currently being built).
		reqFrameLen     = number of bytes actually received for this frame.

Input:		Uses fbmStationData.
		frame	= contains rejected frame starting with the FC
			field through the complete SMT info field.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
RejectedFrameParamType	*param;
uInt16			frameLen;

	/* cast pointer */
	param = (RejectedFrameParamType *) pBuffer;

	/* get length of rejected frame */
	frameLen = ntohs (frame->smtHdr.InfoField_Length)
		+ sizeof (SMTFrameHdrType) - FC_PAD_SIZE;

        /*
	 * Validate enough data was received for the specified frame
	 * length.
	 */
        if (frameLen > reqFrameLen)
	     frameLen = reqFrameLen;

	/*
	*	910807-005	LJP
	*	Reduce amount of space for rejected frame by the size of
	*	the parameter header and pad field.
	*/
	maxLen -= (sizeof (param->hdr) + sizeof (param->pad));

	/* set data size */
	if (frameLen > maxLen)
		frameLen = maxLen;

	/* fill in parameter */
	MEMCOPY (param->rejected_frame_beginning, &frame->macHdr.FC,
		frameLen);

	/* add padding to parameter length */
	/* 910807-005 LJP Use sizeof() for padding */
	frameLen += sizeof (param->pad);

	/* fill in header */
	param->hdr.type = htons (REJECTED_FRAME_PARAM_TYPE);
	param->hdr.length = htons (frameLen);

	return (frameLen + sizeof (ParamHdrType));
}

uInt16
SupportedVersions_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the SMT Supported Versions parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
SupportedVersionsParamType	*param;
TLVParamType			mibData;
uInt16				loVersion, hiVersion,
				count,
				*versionPtr;

	/* cast pointer */
	param = (SupportedVersionsParamType *) pBuffer;

	/* get low version */
	mibData.paramType = fddiSMTLoVersionId;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL);
	loVersion = mibData.SMTPARAM16;

	/* get high version */
	mibData.paramType = fddiSMTHiVersionId;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL);
	hiVersion = mibData.SMTPARAM16;

	/* get number of versions */
	count = hiVersion - loVersion + 1;
	param->numVersions = count;

	/* fill in header */
	param->hdr.type = htons (SUPPORTED_VERSIONS_PARAM_TYPE);

	/* length must end on 4-byte boundary */
	if (count & 1)
		count++;

	/* length is 2 bytes per version plus pad, numVersions and opVersion */
	param->hdr.length
		= htons ((count << 1) + 4);

	/* fill in parameter */
	for (versionPtr = param->Version, count = loVersion;
		count <= hiVersion;
		count++, versionPtr++)
	{
		*versionPtr = htons (count);
		if (count == fbmStationData.opVersionID)
			param->indexOpVersion = count - loVersion + 1;
	}

	/* if odd number of versions, pad last one to 0s */
	if (param->numVersions & 1)
		*versionPtr = 0;

	return (ntohs (param->hdr.length) + sizeof (ParamHdrType));
}

uInt16
ESFID_Param (pBuffer, esfID)
	uChar	*pBuffer;
	uChar	*esfID;
/*********************************************************************
Function:	Create the ESF ID parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		esfID	= 48-bit, IEEE-assigned ESF identifier.
Input:		Uses fbmStationData.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
ESFIDParamType	*param;

	/* cast pointer */
	param = (ESFIDParamType *) pBuffer;

	/* fill in header */
	param->hdr.type = htons (ESFID_PARAM_TYPE);
	param->hdr.length 
		= htons (sizeof (ESFIDParamType)
			- sizeof (ParamHdrType));

	/* fill in parameter */
	MEMCOPY (param->ESF_ID, esfID, 6);

	return (sizeof (ESFIDParamType));
}

uInt16
Authorization_Param (pBuffer)
	uChar	*pBuffer;
/*********************************************************************
Function:	Create the Authorization parameter.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
Input:		None.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Modification History:
*********************************************************************/
{
AuthorizationParamType	*param;
uInt16			aLen, extra;

	/* check if authorization is used */
	if (!fbmStationData.useAuthorization)
		return (0);

	/* cast pointer */
	param = (AuthorizationParamType *) pBuffer;

	/* fill in parameter */
	aLen = BuildAuthorization (param->Authorization);

	/* adjust length if not multiply of 4 */
	extra = aLen & 0x3;
	if (extra)
		aLen += (4 - extra);

	/* fill in header */
	param->hdr.type = htons (AUTHORIZATION_PARAM_TYPE);
	param->hdr.length
		= htons (aLen);

	return (aLen + sizeof (ParamHdrType));
}

uInt16
Attribute_Param (pBuffer, attr)
	uChar		*pBuffer;
	TLVParamType	*attr;
/*********************************************************************
Function:	Convert a MIB attribute TLV encoding into network order.
Parameters:	pBuffer	= pointer to location where parameter
			should start.
		attr	= pointer to attribute.
Input:		attr	= contains parameter in TLV MIB encoding.
Output:		pBuffer	= contains the SMT frame parameter.
Return:		Length in characters of the parameter including
		the type and length fields.
Notes:		If the buffer and the attribute have the same address,
		then the routine does not copy the attribute to the
		buffer and converts the attribute in situ.
Modification History:
*********************************************************************/
{
TLVParamType	*param;
uInt16		len;

	/* cast pointer */
	param = (TLVParamType *) pBuffer;

	/* get length */
	len = attr->paramLen + sizeof (TLVHdrType);

	/* copy if buffer is different location than attribute */
	if (pBuffer != (uChar *) attr)
		MEMCOPY (pBuffer, attr, len);

	/* convert attribute to network order */
	AttrNetOrder (param, NET_ORDER);

	/* return size of parameter */
	return (len);
}

