#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmq.c V6.2.3:cs.911022:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Services Process Module

	Action Queue Module

	File:		fbmq.c
	Created:	12/12/90

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91

	Copyright 1990,1991 XLNT Designs, Inc.

	This module implements a simple circular FIFO queue to store
	the actions to be processed by FSP. FSP handles three
	actions: frame received, message received, and timer tick.
	When a frame is received by the station, a frame received
	entry (FBM_Q_FRAME) is placed on the queue along with a pointer
	to the frame buffer. Upon receiving a message, a similar action
	takes place for the message action (FBM_Q_MESSAGE) and the
	message buffer. A timer action (FBM_Q_TIMER) does not have any
	data associated with it.

	There are four functions associated with the signal
	queue: InitFBMQueue(), PostFBMQueue(), ReadFBMQueue() and
	EmptyFBMQueue().

	The FIFO is implemented here as a circular queue in an array.
	The head is the index to the next entry to be added and the
	tail is the next entry to be read. The empty condition
	is denoted by the head and tail being equal.

	Modification History:
	910108-001	LJP
		When the FBM queue becomes full, it should not clear
		the queue as this could leave frame and message buffers
		allocated. Instead, the new event is not added and
		PostFBMQueue() returns an error code and sends and error
		message to the MAP.
	911022-005	LJP
		Added a timer event counter. For each timer event, instead
		of adding it to the queue, it is counted. This allows
		timer events to be accounted for even when the queue is
		full.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"smtmsg.h"	/* 910108-001 LJP error message */
#include	"fddihdr.h"
#include	"fbmmacro.h"
#include	"fbmhdr.h"

/*********************************************************************
	External Functions
*********************************************************************/

extern	void	SendFBMMessage ();


/*********************************************************************
	Queue Data Structures
*********************************************************************/

struct	QueueEntry {
	uInt16	type;
	uChar	*data;
	uInt16	len;
	uInt16	entity;
	uChar	eac;
};


/*********************************************************************
	Queue Macros
*********************************************************************/

/*
*	The NextQueueEntry(n) macro returns the next index with a
*	wrap to 0 if the array size will be exceeded.
*/
#define	NextQueueEntry(n)	((n == MAX_FBM_QUEUE_SIZE - 1) ? 0 : n + 1)

/*********************************************************************
	Signal Queue Global Data
*********************************************************************/

static	fbmQHead,			/* head index value */
	fbmQTail;			/* tail index value */

static	struct QueueEntry
	fbmQueue[MAX_FBM_QUEUE_SIZE];	/* FIFO array */

/*
*	911022-005	LJP
*	Timer event coiunter.
*/
Int16	fbmTimerCounter;


/*********************************************************************
	Queue Functions
*********************************************************************/

uInt32
InitFBMQueue ()
/*********************************************************************
Function:	Initialize the action queue structure.
Parameters:	None.
Input:		None.
Output:		Set fbmQHead and fbmQTail to 0.
Return:		0 if successful. Otherwise an error code is returned.
Modification History:
*********************************************************************/
{
ProcState	pState;
Int16		i;

	/*
	*	Disable interrupts while looking at the pointers.
	*/
	MDisableInterrupts (&pState);


	/*
	*	Clear pointers.
	*/
	fbmQHead = fbmQTail = 0;

	/*
	*	Clear queue.
	*/
	for (i = 0; i < MAX_FBM_QUEUE_SIZE; i++)
	{
		fbmQueue[i].type = 0;
		fbmQueue[i].data = NULL;
	}

	/*
	*	911022-005	LJP
	*	Clear timer count.
	*/
	fbmTimerCounter = 0;

	/*
	*	Enable interrupts.
	*/
	MRestoreInterrupts (&pState);

	return (0);
}

uInt32
EmptyFBMQueue ()
/*********************************************************************
Function:	Checks if signal queue is empty.
Parameters:	None.
Input:		fbmQHead and fbmQTail.
Output:		None.
Return:		FALSE if queue has items in it. TRUE if the queue is empty.
*********************************************************************/
{
uInt32		result;
ProcState	pState;

	/*
	*	Disable interrupts while looking at the pointers.
	*/
	MDisableInterrupts (&pState);

	/*
	*	Get queue empty status.
	*/
	result = (fbmQHead == fbmQTail);

	/*
	*	911022-005	LJP
	*	Check timer counter.
	*/
	result &= (fbmTimerCounter <= 0);

	/*
	*	Enable interrupts.
	*/
	MRestoreInterrupts (&pState);

	return (result);
}

uInt32
ReadFBMQueue (actionType, actionData, actionLen, actionEntity, actionEAC)
	uInt16	*actionType;
	uChar	**actionData;
	uInt16	*actionLen;
	uInt16	*actionEntity;
	uChar	*actionEAC;
/*********************************************************************
Function:	Reads next action from the queue.
Parameters:	actionType	= pointer to buffer for action type.
		actionData	= pointer to buffer for action data pointer.
		actionLen	= pointer to buffer for frame length.
		actionEntity	= pointer to buffer for entity ID.
		actionEAC	= pointer to buffer for EAC bits.
Input:		fbmQHead, fbmQTail, and fbmQueue.
Output:		fbmQHead incremented.
		actionType	= contains type of FBM action.
		actionData	= pointer to a buffer containing the
					action's associated data (i.e.,
					frame buffer for FBM_Q_FRAME and
					message buffer for FBM_Q_MESSAGE).
					Not used for FBM_Q_TIMER.
		actionLen	= length of frame in buffer for FBM_Q_FRAME.
					Not used for other actions.
		actionEntity	= index of entity for FBM_Q_FRAME.
					Not used for other actions.
		actionEAC	= EAC bits received with frame for
					FBM_Q_FRAME. Not used with other
					actions.
Return:		0		if signal is retured successfully.
		EFBM_Q_MT	if queue is empty.
*********************************************************************/
{
ProcState	pState;

	/*
	*	Disable interrupts while reading queue.
	*/
	MDisableInterrupts (&pState);

	/*
	*	911022-005	LJP
	*	Check for timer events.
	*/
	if (fbmTimerCounter > 0)
	{
		fbmTimerCounter--;
		*actionType = FBM_Q_TIMER;
		MRestoreInterrupts (&pState);
		return (0);
	}

	/*
	*	If queue is empty,
	*/
	if (fbmQHead == fbmQTail)
	{
		MRestoreInterrupts (&pState);
		return (EFBM_Q_MT);
	}

	/*
	*	Set return values.
	*/
	*actionType = fbmQueue[fbmQHead].type;
	switch (fbmQueue[fbmQHead].type)
	{
	case FBM_Q_FRAME:
		*actionData = fbmQueue[fbmQHead].data;
		*actionLen = fbmQueue[fbmQHead].len;
		*actionEntity = fbmQueue[fbmQHead].entity;
		*actionEAC = fbmQueue[fbmQHead].eac;
		break;

	case FBM_Q_MESSAGE:
		*actionData = fbmQueue[fbmQHead].data;
		break;
	}

	/*
	*	Increment head pointer.
	*/
	fbmQHead = NextQueueEntry (fbmQHead);

	/*
	*	Enable interrupts.
	*/
	MRestoreInterrupts (&pState);

	return (0);
}

uInt32
PostFBMQueue (actionType, actionData, actionLen, actionEntity, actionEAC)
	uInt16	actionType;
	uChar	*actionData;
	uInt16	actionLen;
	uInt16	actionEntity;
	uChar	actionEAC;
/*********************************************************************
Function:	Adds an action to the queue.
Parameters:	actionType	= action to store.
		actionData	= pointer to action data.
		actionLen	= length of frame for FBM_Q_FRAME type
		actionEntity	= index of entity for FBM_Q_FRAME type
		actionEAC	= EAC bits for FBM_Q_FRAME
Input:		fbmQHead, fbmQTail, and fbmQueue.
Output:		fbmQueue entry added. fbmQTail incremented.
Return:		0 if successfully added to queue.
		EFBM_Q_FULL if queue is full.
Note:		In the event that the queue becomes full, the requested
		event is NOT added to the queue. The calling routine is
		responsible for releasing the frame or message buffer
		if needed. A message is sent to the MAP indicating the
		error condition.
*********************************************************************/
{
ProcState	pState;			/* processor intr state */
SMTMessage	smtmsg;			/* error message 910108-001 LJP */

	/*
	*	Disable interrupts while updating queue.
	*/
	MDisableInterrupts (&pState);

	/*
	*	911022-005	LJP
	*	If timer event, add to counter and return.
	*/
	if (actionType == FBM_Q_TIMER)
	{
		fbmTimerCounter++;
		MRestoreInterrupts (&pState);
		return (0);
	}

	/*
	*	Check for queue full condition.
	*/
	if (NextQueueEntry (fbmQTail) == fbmQHead)
	{
		/*
		*	910108-001 LJP
		*	Report error condition to MAP.
		*	Do NOT clear the queue.
		*/
		smtmsg.destination = MAP_MSG_ID;
		smtmsg.source = FBM_MSG_ID;
		smtmsg.type = FBM_ERROR_MSG;
		smtmsg.typeInfo = EFBM_Q_FULL;
		smtmsg.localID = 0;
		smtmsg.len1 = 0;
		smtmsg.len2 = 0;
		SendFBMMessage (&smtmsg);

		MRestoreInterrupts (&pState);
		return (EFBM_Q_FULL);
	}

	/*
	*	Add action.
	*/
	fbmQueue[fbmQTail].type = actionType;
	switch (actionType)
	{
	case FBM_Q_FRAME:
		fbmQueue[fbmQTail].data = actionData;
		fbmQueue[fbmQTail].len = actionLen;
		fbmQueue[fbmQTail].entity = actionEntity;
		fbmQueue[fbmQTail].eac = actionEAC;
		break;

	case FBM_Q_MESSAGE:
		fbmQueue[fbmQTail].data = actionData;
		break;
	}

	fbmQTail = NextQueueEntry (fbmQTail);

	/*
	*	Enable interrupts.
	*/
	MRestoreInterrupts (&pState);

	return (0);
}
