#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmpmf.c V6.2.3:cs.911022:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Management Module

	Build PMFs Module

	File:		fbmpmf.c
	Created:	09/04/90

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	Frame Based Management for SMT.  This module contains routines
	to build the Parameter Management Frames.

	Modification History:

	*** Updated to SMT 6.2 ***

	910125-001	LJP
		The BuildPMFGetResponse() routines did not account for
		MIB attribute groups and was not including the multiple
		attributes in the length of the frame. Now, the routine
		loops through all attributes in the frame.
	910204-005	LJP
		BuildPMFGetResponse() needs to adjust the attribute
		pointer when it loops through all the attributes in
		the frame.
	911022-004	LJP
		Changed code that checks for errors and sets reason
		code so that codes are more logical. Still awaiting
		committee hierarchy on some tests.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	External Declarations
*********************************************************************/

extern	uInt32	FBMGetMIBAttr ();
extern	uInt16	htons ();
extern	uInt32	htonl ();
extern	uInt16	ntohs ();
extern	uInt32	ntohl ();
extern	uInt32	BuildSMTHeader ();
extern	uInt16	ReasonCode_Param ();
extern	uInt16	Attribute_Param ();
extern	uInt16	Authorization_Param ();
extern	uInt32	BuildRDFResponse ();
extern	void	AttrNetOrder ();


/*********************************************************************
	Parameter Management Frames
*********************************************************************/

uInt32
BuildPMFGetRequest (frameBuffer, destAddr, outputMAC, param)
	uChar		*frameBuffer;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds PMF Get Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		param		= buffer containing the MIB attribute to get.
Input:		destAddr	= 48-bit station ID of destination.
		param		= contains type-length-value encoding of 
				the MIB attribute to get.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */	
uInt16		paramLen;	/* Length of frame parameter */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	result = BuildSMTHeader (frameBuffer, SMT_INFO_FC, destAddr,
		outputMAC, PMFGET_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/*
	*	Set parameter type values to network order.
	*/
	/* first check if there is an index to convert */
	if ((param->paramType & MIB_OBJECT_MASK) != fddiSMT)
		param->MACINDEX = htons (param->MACINDEX);

	/* save length before converting */
	paramLen = param->paramLen;
	param->paramType = htons (param->paramType);
	param->paramLen = htons (param->paramLen);

	/* add on header size */
	paramLen += sizeof (TLVHdrType);
	MEMCOPY (memPtr, param, paramLen);
	frameLen += paramLen;

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFGetResponse (requestFrame, responseBuffer, outputMAC)
	uChar		*requestFrame;
	uChar		*responseBuffer;
	uInt16		outputMAC;
/*********************************************************************
Function:	This function builds a PMF Get Response.
Parameters:	requestFrame	= pointer to PMF Get Request frame buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
Input:		requestFrame	= contains ESF Request Frame to respond to.
Output:		responseBuffer	= contains the completed frame.
Return:		No value returned.
Modification History:
	910125-001	LJP
		This routine loops through all MIB attributes in
		the frame to account for attribute group requests.
*********************************************************************/
{
SMTFrameHdrType	*reqframeHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uInt16		frameLen;	/* total length of frame */
uInt16		frameLeft;	/* space left in INFO field for attr */
uInt16		paramLen;	/* length of a parameter */
uInt32		reasonCode;	/* MIB Get result */
uInt32		result;		/* return value */
uChar		*memPtr;	/* pointer into SMT INFO field */
TLVParamType	*getPtr;	/* MIB attribute to get pointer */
TLVParamType	*attrPtr;	/* MIB attribute value pointer */
TLVSetCountType	*setCount;	/* set count pointer */
uInt16		attrLen;	/* total attribute length 910125-001 LJP */

	/*
	*	Get frame headers.
	*/
	reqframeHdr = (SMTFrameHdrType *) requestFrame;
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqframeHdr->smtHdr.Transaction_ID);
	result = BuildSMTHeader (responseBuffer, SMT_INFO_FC,
		reqframeHdr->macHdr.SA, outputMAC, PMFGET_CLASS,
		SMTRESPONSE, result);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (rspframeHdr + 1);
	frameLen = 0;

	/*
	*	Get attribute ID.
	*/
	getPtr = (TLVParamType *) (reqframeHdr + 1);

	/*
	*	Fill in reason code as placeholder to find where MIB
	*	attribute value should go in response frame buffer.
	*/
	paramLen = ReasonCode_Param (memPtr, 0);
	frameLen += paramLen;
	reasonCode = RC_SUCCESS;

	/*
	*	Use frame buffer for set count buffer and for
	*	MIB attribute buffer. Save start location for
	*	set count buffer (current memPtr) and use first location
	*	after set count for MIB attribute buffer.
	*/
	setCount = (TLVSetCountType *) (memPtr + paramLen);
	attrPtr = (TLVParamType *) (memPtr + paramLen +
		sizeof (TLVHdrType) + sizeof (SetCountType));

	/*
	*	Calculate space left in SMT INFO field for MIB attribute.
	*/
	frameLeft = MAX_SMT_INFO_LEN - (paramLen + sizeof (TLVHdrType)
		+ sizeof (SetCountType));

	/*
	*	Validate request info.
	*/
	attrPtr->paramType = ntohs (getPtr->paramType);
	attrPtr->paramLen = ntohs (getPtr->paramLen);
	/*
	*	911022-004	LJP
	*	Validate object and request length for object.
	*/
	switch (attrPtr->paramType & MIB_OBJECT_MASK)
	{
	case fddiMAC:
	case fddiPATHClass:
	case fddiPATHClassPATH:
	case fddiPORT:
	case fddiATTACHMENT:
		/*
		*	At this point, it is possible to check the 
		*	parameter length. This check is commented
		*	out until the committee assigns a hierarchy
		*	to the reason codes.
		*/
#ifdef	COMMENTED_OUT
		/* verify length of request */
		if (attrPtr->paramLen != sizeof (TLV16BitType))
		{
			/* invalid length, send RDF */
			result = BuildRDFResponse (requestFrame,
				responseBuffer, outputMAC, RC_LENGTH_ERROR);
			reasonCode = RC_LENGTH_ERROR;
		}
		else
#endif
			attrPtr->MACINDEX = ntohs (getPtr->MACINDEX);
		break;

	case fddiSMT:
		/*
		*	This is commented out for the same reason
		*	as above.
		*/
#ifdef	COMMENTED_OUT
		/* verify length of request */
		if (attrPtr->paramLen != 0)
		{
			/* invalid length, send RDF */
			result = BuildRDFResponse (requestFrame,
				responseBuffer, outputMAC, RC_LENGTH_ERROR);
			reasonCode = RC_LENGTH_ERROR;
		}
#endif
		break;

	default:
		/* invalid parameter type, send RDF */
		result = BuildRDFResponse (requestFrame,
			responseBuffer, outputMAC, RC_ILLEGAL_PARAMETER);
		reasonCode = RC_ILLEGAL_PARAMETER;
	}

	/*
	*	Continue response frame if OK.
	*/
	if (reasonCode == RC_SUCCESS)
	{
		/*
		*	Get MIB attribute.
		*/
		reasonCode = FBMGetMIBAttr (frameLeft, attrPtr, setCount);

		/*
		*	Add remaining parameters if get was successful.
		*/
		if (reasonCode == RC_SUCCESS)
		{
			/* convert set count */
			paramLen = Attribute_Param (setCount, setCount);
			frameLen += paramLen;

			/* convert attribute */
			/*
			*	910125-001	LJP
			*	Starting with the first parameter, this
			*	code loops through all attributes in the
			*	frame. This code relies on the fact that
			*	the frame contents are cleared to 0 at
			*	the start of this routine and the
			*	FBMGetMIBAttr() routine only uses the
			*	buffer space required. By reaching the
			*	end of the buffer or finding an attribute
			*	ID of 0, the loop knows when to stop.
			*/
			attrLen = 0;
			while (attrPtr->paramType && (attrLen < frameLeft))
			{
				paramLen = Attribute_Param (attrPtr, attrPtr);
				frameLen += paramLen;
				attrLen += paramLen;
				/* 910204-005 LJP */
				attrPtr = (TLVParamType *) 
					(((uChar *) attrPtr) + paramLen);
			}
		}
	}

        /*
	 *	Fill in actual reason code.
	 */
        paramLen = ReasonCode_Param (memPtr, reasonCode);

	/*
	*	Set info length.
	*/
	rspframeHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFChangeRequest (frameBuffer, destAddr, outputMAC, setCount, param)
	uChar		*frameBuffer;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	TLVParamType	*setCount;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds a PMF Change Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		setCount	= buffer containing a Set Count.
		param		= buffer containing the MIB attribute to get.
Input:		destAddr	= 48-bit station ID of destination.
		setCount	= contains TLV encoding of the MIB
				Set Count attribute to use for
				the protocol.
		param		= contains TLV encoding of the MIB
				attribute to change.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */	
uInt16		paramLen;	/* Length of frame parameter */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	result = BuildSMTHeader (frameBuffer, SMT_INFO_FC, destAddr,
		outputMAC, PMFCHANGE_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/*
	*	Set authorization.
	*/
	paramLen = Authorization_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set Set Count.
	*/
	paramLen = Attribute_Param (memPtr, setCount);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set attribute.
	*/
	paramLen = Attribute_Param (memPtr, param);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFChangeResponse (requestFrame, responseBuffer, outputMAC,
		reasonCode, setCount, lastSetID, param)
	uChar		*requestFrame;
	uChar		*responseBuffer;
	uInt16		outputMAC;
	uInt32		reasonCode;
	TLVParamType	*setCount;
	TLVParamType	*lastSetID;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds a PMF Change Response.
Parameters:	requestFrame	= pointer to PMF Change Request frame buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
		reasonCode	= result of Change request.
		setCount	= buffer containing Set Count.
		lastSetID	= buffer containing Last Set Station ID
		param		= buffer containing current MIB attribute.
Input:		requestFrame	= contains ESF Request Frame to respond to.
		setCount	= contains the TLV encoded value of the Set
				Count attribute when the MIB attribute was
				changed.
				This parameter may be a NULL if reasonCode 
				is not Success and not Bad Set Count.
		lastSetID	= contains the TLV encoded address of the 
				fddiSMTLastSetStationId.
				This parameter may be a NULL if reasonCode 
				is not Success and not Bad Set Count.
		param		= contains the TLV encoded MIB attribute.
				This parameter may be a NULL if reasonCode
				is not Success and not Bad Set Count.
Output:		responseBuffer	= contains the completed frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqframeHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uInt16		frameLen;
uInt16		paramLen;
uInt32		result;
uChar		*memPtr;
	
	/*
	*	Get frame headers.
	*/
	reqframeHdr = (SMTFrameHdrType *) requestFrame;
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqframeHdr->smtHdr.Transaction_ID);
	result = BuildSMTHeader (responseBuffer, SMT_INFO_FC,
		reqframeHdr->macHdr.SA,
		outputMAC, PMFCHANGE_CLASS, SMTRESPONSE, result);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (rspframeHdr + 1);
	frameLen = 0;

	/* set reason code */
	paramLen = ReasonCode_Param (memPtr, reasonCode);
	frameLen += paramLen;
	memPtr += paramLen;

	/* add remaining parameters if get was successful or bad set count */
	if ((reasonCode == RC_SUCCESS) || (reasonCode == RC_BAD_SET_COUNT))
	{
		/* put in set count */
		paramLen = Attribute_Param (memPtr, setCount);
		frameLen += paramLen;
		memPtr += paramLen;

		/* put in last station id */
		paramLen = Attribute_Param (memPtr, lastSetID);
		frameLen += paramLen;
		memPtr += paramLen;

		/* put in attribute */
		paramLen = Attribute_Param (memPtr, param);
		frameLen += paramLen;
		memPtr += paramLen;
	}

	/*
	*	Set info length.
	*/
	rspframeHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFAddRequest (frameBuffer, destAddr, outputMAC, setCount, param)
	uChar		*frameBuffer;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	TLVParamType	*setCount;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds a PMF Add Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		setCount	= buffer containing a Set Count.
		param		= buffer containing the MIB attribute to get.
Input:		destAddr	= 48-bit station ID of destination.
		setCount	= contains TLV encoding of the MIB
				Set Count attribute to use for
				the protocol.
		param		= contains TLV encoding of the MIB
				attribute to change.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */
uInt16		paramLen;	/* Length of frame parameter */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	result = BuildSMTHeader (frameBuffer, SMT_INFO_FC, destAddr,
		outputMAC, PMFADD_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/*
	*	Set authorization.
	*/
	paramLen = Authorization_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set Set Count.
	*/
	paramLen = Attribute_Param (memPtr, setCount);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set attribute.
	*/
	paramLen = Attribute_Param (memPtr, param);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFAddResponse (requestFrame, responseBuffer, outputMAC, 
		reasonCode, setCount, lastSetID, param)
	uChar		*requestFrame;
	uChar		*responseBuffer;
	uInt16		outputMAC;
	uInt32		reasonCode;
	TLVParamType	*setCount;
	TLVParamType	*lastSetID;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds a PMF Add Response.
Parameters:	requestFrame	= pointer to PMF Add Request frame buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
		reasonCode	= result of Add request.
		setCount	= buffer containing Set Count.
		lastSetID	= buffer containing Last Set Station ID
		param		= buffer containing current MIB attribute.
Input:		requestFrame	= contains ESF Request Frame to respond to.
		setCount	= contains the TLV encoded value of the Set 
				Count attribute when the MIB attribute was
				changed.
				This parameter may be a NULL if reasonCode 
				is not Success and not Bad Set Count.
		lastSetID	= contains the TLV encoded address of the 
				fddiSMTLastSetStationId.
				This parameter may be a NULL if reasonCode
				is not Success and not Bad Set Count.
		param		= contains the TLV encoded MIB attribute.
				This parameter may be a NULL if reasonCode
				is not Success and not Bad Set Count.
Output:		responseBuffer	= contains the completed frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqframeHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uInt16		frameLen;
uInt16		paramLen;
uInt32		result;
uChar		*memPtr;
	
	/*
	*	Get frame headers.
	*/
	reqframeHdr = (SMTFrameHdrType *) requestFrame;
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;
                    	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqframeHdr->smtHdr.Transaction_ID);
	result = BuildSMTHeader (responseBuffer, SMT_INFO_FC,
		reqframeHdr->macHdr.SA, outputMAC, PMFADD_CLASS, 
		SMTRESPONSE, result);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (rspframeHdr + 1);
	frameLen = 0;

	/* set reason code */
	paramLen = ReasonCode_Param (memPtr, reasonCode);
	frameLen += paramLen;
	memPtr += paramLen;

	/* add remaining parameters if get was successful or bad set count */
	if ((reasonCode == RC_SUCCESS) || (reasonCode == RC_BAD_SET_COUNT))
	{
		/* put in set count */
		paramLen = Attribute_Param (memPtr, setCount);
		frameLen += paramLen;
		memPtr += paramLen;

		/* put in last station id */
		paramLen = Attribute_Param (memPtr, lastSetID);
		frameLen += paramLen;
		memPtr += paramLen;

		/* put in attribute */
		paramLen = Attribute_Param (memPtr, param);
		frameLen += paramLen;
		memPtr += paramLen;
	}

	/*
	*	Set info length.
	*/
	rspframeHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFRemoveRequest (frameBuffer, destAddr, outputMAC, setCount, param)
	uChar		*frameBuffer;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	TLVParamType	*setCount;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds a PMF Remove Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		setCount	= buffer containing a Set Count.
		param		= buffer containing the MIB attribute to get.
Input:		destAddr	= 48-bit station ID of destination.
		setCount	= contains TLV encoding of the MIB
				Set Count attribute to use for
				the protocol.
		param		= contains TLV encoding of the MIB
				attribute to change.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */
uInt16		paramLen;	/* Length of frame parameter */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	result = BuildSMTHeader (frameBuffer, SMT_INFO_FC, destAddr,
		outputMAC, PMFREMOVE_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/*
	*	Set authorization.
	*/
	paramLen = Authorization_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set Set Count.
	*/
	paramLen = Attribute_Param (memPtr, setCount);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set attribute.
	*/
	paramLen = Attribute_Param (memPtr, param);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildPMFRemoveResponse (requestFrame, responseBuffer, outputMAC,
		reasonCode, setCount, lastSetID, param)
	uChar		*requestFrame;
	uChar		*responseBuffer;
	uInt16		outputMAC;
	uInt32		reasonCode;
	TLVParamType	*setCount;
	TLVParamType	*lastSetID;
	TLVParamType	*param;
/*********************************************************************
Function:	This function builds a PMF Remove Response.
Parameters:	requestFrame	= pointer to PMF Remove Request frame buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
		reasonCode	= result of Add request.
		setCount	= buffer containing Set Count.
		lastSetID	= buffer containing Last Set Station ID
		param		= buffer containing current MIB attribute.
Input:		requestFrame	= contains ESF Request Frame to respond to.
		setCount	= contains the TLV encoded value of the Set 
				Count attribute when the MIB attribute was
				changed.
				This parameter may be a NULL if reasonCode 
				is not Success and not Bad Set Count.
		lastSetID	= contains the TLV encoded address of the 
				fddiSMTLastSetStationId.
				This parameter may be a NULL if reasonCode 
				is not Success and not Bad Set Count.
		param		= contains the TLV encoded MIB attribute.
				This parameter may be a NULL if reasonCode 
				is not Success and not Bad Set Count.
Output:		responseBuffer	= contains the completed frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqframeHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uInt16		frameLen;
uInt16		paramLen;
uInt32		result;
uChar		*memPtr;
	
	/*
	*	Get frame headers.
	*/
	reqframeHdr = (SMTFrameHdrType *) requestFrame;
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqframeHdr->smtHdr.Transaction_ID);
	result = BuildSMTHeader (responseBuffer, SMT_INFO_FC,
		reqframeHdr->macHdr.SA, outputMAC, PMFREMOVE_CLASS, 
		SMTRESPONSE, result);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (rspframeHdr + 1);
	frameLen = 0;

	/* set reason code */
	paramLen = ReasonCode_Param (memPtr, reasonCode);
	frameLen += paramLen;
	memPtr += paramLen;

	/* add remaining parameters if get was successful or bad set count */
	if ((reasonCode == RC_SUCCESS) || (reasonCode == RC_BAD_SET_COUNT))
	{
		/* put in set count */
		paramLen = Attribute_Param (memPtr, setCount);
		frameLen += paramLen;
		memPtr += paramLen;

		/* put in last station id */
		paramLen = Attribute_Param (memPtr, lastSetID);
		frameLen += paramLen;
		memPtr += paramLen;

		/* put in attribute */
		paramLen = Attribute_Param (memPtr, param);
		frameLen += paramLen;
		memPtr += paramLen;
	}

	/*
	*	Set info length.
	*/
	rspframeHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}
