#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) csptimer.c V6.2.3:cs.622c:5:5 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	Timer List Module

	File:		csptimer.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.622c	08/02/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the timer entry point used for the
	CSP timers. The CSP system requires at least one system
	timer to be available for use. However, CSP must multiplex
	this one timer among the various state machines runnings.
		
	To handle this multiplexing, this module provides an
	interface to a timer list. As intervals need to be timed,
	each state machine requests a timer to be set. This module
	returns an ID value to coordinate the request. When the
	interval expires, the timer module sends a timer signal
	to the state machine along with the ID value. This tells the
	state machine which timer expired.
		
	The timer module consists of three functions: ServiceCPSTimer(),
	SetCPSTimer(), and InitCPSTimer(). The InitCPSTimer() routine
	initializes the timer list and system timer. The SetCPSTimer()
	routine allows a state machine to add and/or remove a timer
	event. The ServiceCPSTimer() routine is called when the
	system timer expires.
		
	The timer list is implemented as an array. Each element
	in the array consists of the time amount (in microseconds),
	the state machine requesting the timer, and the state machine
	entity (PHY or MAC if needed). A busy flag indicates
	that the entry is in use by a timer event.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"smtmsg.h"
#include	"fddihdr.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include        "krnl.h"

/*********************************************************************
	External Function Declarations
*********************************************************************/

extern SEM Sema_CSP;
void	SetSystemTimer ();
uInt32	ReadSystemTimer ();
void	SMTSendSignal ();


/*********************************************************************
	Timer List Data Structures
*********************************************************************/

struct	TimerListEntry {
	Flag	busy;			/* SET if entry is in use */
	uInt32	time;			/* timer interval in microseconds */
	uInt16	machine;		/* state machine signal type */
	uInt16	entity;			/* state machine entity (if needed) */
	uInt16	ID;			/* time event ID */
};


/*********************************************************************
	Signal Queue Macros
*********************************************************************/

/*
*	The NextTimerID(n) macro returns the next timer event ID with a
*	wrap to 1 if the ID values wrap to negative values.
*/
#define	NextTimerID(n)	(((n + 1) < 1) ? 1 : n + 1)

/*********************************************************************
	Timer List Global Data
*********************************************************************/

static	uInt16
	currentID,			/* current event ID */
	timerCount,			/* # of active events in list */
	timerIndex;			/* index of event in system timer */

static	struct TimerListEntry
		timerList[MAX_TIMER_LIST_SIZE];	/* FIFO array */


/*********************************************************************
	Timer List Functions
*********************************************************************/

uInt32
InitCSPTimer ()
/*********************************************************************
Function:	Initialize the timer list structure.
Parameters:	None.
Input:		None.
Output:		Set timerCount to 0, timerIndex to -1, and currentID to 0.
Return:		0 if successful. Otherwise an error code is returned.
*********************************************************************/
{
ProcState	pState;
Int16		i;

	/*
	*	Disable interrupts while setting table.
	*/
	MDisableCSPInterrupts (&pState);

	/*
	*	Set initial values.
	*/
	timerCount = 0;
	timerIndex = -1;
	currentID = 0;
	
	/*
	*	Clear table.
	*/
	for (i = 0; i < MAX_TIMER_LIST_SIZE; i++)
		timerList[i].busy = CLEAR;
		
	/*
	*	Enable interrupts.
	*/
	MRestoreCSPInterrupts (&pState);
	
	return (0);
}

void
SetCSPTimer (interval, stateMachine, stateEntity, stateID)
	uInt32	interval;
	uInt16	stateMachine;
	uInt16	stateEntity;
	uInt16	*stateID;
/*********************************************************************
Function:	Adds and/or deletes a timer interval event.
Parameters:	interval	= time interval in MICROSECONDS
				if 0, no event is added.
		stateMachine	= state machine to signal on expiration.
		stateEntity	= state machine entity if required.
		stateID		= pointer to buffer with timer ID.
Input:		stateID buffer contains the ID of a timer event
		to remove from the list. If 0, then no deletion occurs.
		timerCount, timerIndex, and timerList.
Output:		timerCount, timerIndex, and timerList are changed accordingly.
		stateID buffer contains the ID of the added timer event.
		If no event is added, the buffer contains 0.
Return:		No value returned.
Modification History:
*********************************************************************/
{
ProcState	pState;				/* processor state */
uInt32		ticksLeft,			/* time left on system timer */
		ticksPassed,			/* time passed */
		minTime;			/* smallest time index value */
uInt16		deleteID;			/* ID to remove from list */
Int16		i;
Flag		added;				/* SET if event is added */

	/*
	*	Disable interrupts while updating list.
	*/
	MDisableCSPInterrupts (&pState);
	
	/*
	*	Stop timer
	*/
	if (timerCount > 0)
	{
		ticksLeft = ReadSystemTimer ();
		SetSystemTimer ((uInt32) 0);
		if (ticksLeft >= timerList[timerIndex].time)
			ticksPassed = 0;
		else
			ticksPassed = timerList[timerIndex].time - ticksLeft;
	}
	else
		ticksPassed = 0;
	
	/*
	*	Set added flag.
	*/
	added = (interval == 0);		/* if SET, then not adding */
	
	/*
	*	Set delete ID,
	*	and clear returned ID.
	*/
	deleteID = *stateID;
	*stateID = 0;
	
	/*
	*	If time value is too small for timer,
	*/
	if (!added && (interval < MIN_USECS))
	{
		/*
		*	Delay.
		*/
		while (interval > 0) interval--;

		/*
		*	Post signal directly.
		*/
		currentID = NextTimerID (currentID);
		*stateID = currentID;
		SMTSendSignal (stateMachine | TIMER_SIGNAL,
			stateEntity, (uInt32) currentID);
		added = SET;
	}

	/*
	*	Loop through list.
	*/	
	for (minTime = -1, i = 0; i < MAX_TIMER_LIST_SIZE; i++)
	{
		/*
		*	If event found with delete ID, then delete entry.
		*/
		if (deleteID && timerList[i].busy 
			&& (deleteID == timerList[i].ID))
		{
			/*
			*	If current event and interval is too
			*	small, then don't delete from list.
			*/
			if (i == timerIndex 
				&& (timerList[i].time - ticksPassed) 
					< MIN_USECS)
			{
				/* leave in list */
				timerList[i].machine = 0;
			}
			/*
			*	Otherwise remove entry.
			*/
			else
			{
				timerList[i].busy = CLEAR;/* free entry */
				--timerCount;		/* reduce event count */
			}
			deleteID = 0;		/* clear delete ID */
		}
			
		/*
		*	if entry is empty and adding, then add here
		*/
		if (!timerList[i].busy)
		{
			if (!added)
			{
				timerList[i].time = interval;
				timerList[i].busy = SET;
				timerList[i].machine = stateMachine;
				timerList[i].entity = stateEntity;
				currentID = NextTimerID (currentID);
				timerList[i].ID = currentID;
				*stateID = currentID;
				++timerCount;
				added = SET;
			}
		}
				
		/*
		*	otherwise reduce time value by amount passed
		*/
		else
			timerList[i].time -= ticksPassed;
				
		/*
		*	if entry is smallest interval in list
		*/
		if (timerList[i].busy && (timerList[i].time < minTime))
		{
			/* set new current index and minTime */
			timerIndex = i;
			minTime = timerList[i].time;
		}
	}
	
	/*
	*	Restart timer if events are being timed
	*/
	if (timerCount > 0)
	{
		/*
		*	Make sure minTime has a value to insure that
		*	the system timer is marked as ON.
		*/
		if (minTime < 1)
			minTime = 1;
		SetSystemTimer (minTime);
	}
		
	/*
	*	Check for list full condition.
	*/
	if ((interval > 0) && !added)
	{
		/* report error condition */
	SMTMessage smtmsg;

		smtmsg.destination = MAP_MSG_ID;
		smtmsg.source = CSP_MSG_ID;
		smtmsg.type = CSP_ERROR_MSG;
		smtmsg.typeInfo = ECSP_SIG_TIMER_FULL;
		smtmsg.localID = 0;
		smtmsg.len1 = 0;
		smtmsg.len2 = 0;
		SendCSPMessage (&smtmsg);
	        printf("Timer LIst Full\n");

		MRestoreCSPInterrupts (&pState);
	}
	
	/*
	*	Enable interrupts.
	*/
	MRestoreCSPInterrupts (&pState);
	
	return;
}

void
ServiceCSPTimer ()
/*********************************************************************
Function:	Process a system timer interrupt.
Parameters:	None.
Input:		timerIndex, timerCount, and timerList.
Output:		timerIndex, timerCount, and timerList adjusted accordingly.
Return:		No value returned.
Note:		It is assumed that interrupts are already disabled when
			this function is called.
Modification History:
*********************************************************************/
{
uInt32		ticksPassed,			/* time passed */
		minTime;			/* smallest time index value */
Int16		i, timeout;

        /*   jlin:
	 *   Flag to indicate if there is a timeout
	 */
        timeout = FALSE;
	/*
	*	If no items in the list, then exit
	*/
	if (timerCount <= 0)
		return;
		
	/*
	*	Get time passed
	*/
	ticksPassed = timerList[timerIndex].time;
	
	/*
	*	Loop through list.
	*/
	for (minTime = -1, i = 0; i < MAX_TIMER_LIST_SIZE; i++)
	{
		if (timerList[i].busy)
		{
			/*
			*	If time expired, then send a signal
			*/
			if (timerList[i].time <= ticksPassed)
			{
				/*
				*	Make sure state machine is valid,
				*	otherwise this is an unused, expired
				*	entry and should not be posted.
				*/
				if (timerList[i].machine)
				{
					SMTSendSignal 
					(timerList[i].machine | TIMER_SIGNAL,
					timerList[i].entity, 
					(uInt32) timerList[i].ID);
					/* jlin:
					 * Flag to indicate if there is a timeout
					 */
					timeout = TRUE;
				}
				timerList[i].busy = CLEAR;
				--timerCount;
			}
			
			/*
			*	Otherwise check for new minTime
			*/
			else 
			{
				/* reduce time value by amount passed */
				timerList[i].time -= ticksPassed;

				if (timerList[i].time < minTime)
				{
					/* set new current index and minTime */
					timerIndex = i;
					minTime = timerList[i].time;
				}
			}
		}
	}
	
	/*
	*	Restart timer if events are being timed
	*/
	if (timerCount > 0)
		SetSystemTimer (minTime);

        /*   jlin:
	 *   If there is a timeout, send the semaphore to the CSP task.
	 */
        if (timeout)
	        SendSignal (&Sema_CSP);
	return;
}

