#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) cspcfm.c V6.2.3:cs.911022:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module

	CFM State Machine

	File:		cspcfm.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91

	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the CFM state machine listed 
	in the ANSI X3T9.5 standard.

	Modification History:

	*** Updated to SMT 6.2 ***

	910419-001	LJP
		The port A CEM state machine does not provide for unique
		conditions on transitions CE(30) and CE(31). This makes
		them order dependent. Two fixes have been included. The
		first fix is a proposed change to the state machine
		conditions. The second fix simply changes the order of
		the current transitions.
	910419-003	LJP
		The state machines do not pair SET x:RM_Join with CLEAR
		from each port (see port A's CE(13) or CE(31)). To avoid
		multiple SETs, the RM_Join array in the port's CEM data
		contains the current setting for each MAC from this port.
		If the flag is set, no more RM_Join SETs are allowed.
		Multiple CLEARs are allowed and handled by RMT.
	910426-001	LJP
		The code to handle multiple SETs and CLEARS for RM_Join
		did not account for all possibilities. The RM_Join
		signal to RMT now tells RMT to count the number of
		RM_Joins SET in cemData[] and is sent whenever an
		RM_Join changes.
	911022-003	LJP
		Added code to Isolated_Actions(), Insert_P_Actions(), and
		Insert_X_Actions() for SMT-LBC-69 (avoid MAC-less wrap
		and rapid cycling of withheld A connection).
*********************************************************************/
#define         REDUNDANT_LINK
#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"

/*********************************************************************
	CFM External Functions
*********************************************************************/

extern	void	SMTSendSignal ();
extern	void	SendCSPEvent ();
extern	void	ConfigureAttach ();
extern	void	ConfigureMaster ();
extern	void	SetCSPTimer ();
extern	void	StartScrubbing ();

/*********************************************************************
	CEM Macros
*********************************************************************/

/*
*	910419-003	LJP
*	The following macros set and clear the RM_Join flags.
*/
#define	MRM_JOIN(macID, phyID, state)\
        CSPDPT3("$JOIN: Mac=%d  Phy=%d  State=%d $", macID, phyID, state);\
	cemData[phyID].RM_Join[macID] = state;\
	SMTSendSignal (SIG_RM_Join, macID, (uInt32) 0);

#define	MRM_LOOP(macID, phyID, state)\
        CSPDPT3("$LOOP: Mac=%d  Phy=%d  State=%d $", macID, phyID, state);\
	cemData[phyID].RM_Loop[macID] = state;\
	SMTSendSignal (SIG_RM_Loop, macID, (uInt32) 0);


/*********************************************************************
	CEM Static Variables
*********************************************************************/

/*
*	The CFM_State_Table array allows for quick conversion from the 
*	A and B port CEM states to a station's CFM state. The rows
*	represent the A Port CEM state and the columns are the
*	B Port CEM state.
*/
static	uChar
	CFM_State_Table[5][5] = {
	     /* ISOLATED,    INSERT_P,  INSERT_S,   INSERT_X,   LOCAL */
/* ISOLATED */	CF_ISOLATED, CF_WRAP_B, 0,          CF_WRAP_B,  CF_ISOLATED,
/* INSERT_P */	CF_WRAP_A,   CF_THRU,   0,          CF_THRU,    CF_WRAP_A,
/* INSERT_S */	0,           0,         0,          0,          0,
/* INSERT_X */	CF_WRAP_A,   0,         0,          CF_WRAP_AB, CF_WRAP_A,
/* LOCAL */	CF_ISOLATED, CF_WRAP_B, 0,          CF_WRAP_B,  CF_ISOLATED
		};

/*********************************************************************
	CEM State Actions
*********************************************************************/

static void
Isolated_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform Isolated_Actions for the CEM state machines.
Parameters:	sigEntity	= PHY causing configuration change.
Input:		Uses phyData, cemData and stationData.
Output:		Changes cemData and stationData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("Isolate port %d.",sigEntity);
	cemData[sigEntity].ceState = CE_ISOLATED;
	SendCSPEvent (fddiPORTCE_State, sigEntity);

	/*
	*	910419-003	LJP
	*	Use macros to set and clear RM_Join.
	*/
	MRM_JOIN (PRIMARY_MAC, sigEntity, CLEAR);
	MRM_JOIN (SECONDARY_MAC, sigEntity, CLEAR);
/*
	MRM_LOOP (LOCAL_MAC, sigEntity, CLEAR);
*/

	/*
	*	911022-003	LJP
	*	Add code to restart connection.
	*/
	if (phyData[sigEntity].PC_Type == PC_Type_B)
	{
		if (stationData.WA_Flag)
		{
			stationData.WA_Flag = CLEAR;
			SMTSendSignal (SIG_PC_Start, PHY_A, (uInt32) 0);
		}

		if (stationData.WAT_Flag)
		{
			stationData.WAT_Flag = CLEAR;
			if (phyData[PHY_A].PC_Mode != PC_Mode_Peer)
				SMTSendSignal (SIG_PC_Start, PHY_A, (uInt32) 0);
		}
	}

#ifdef REDUNDANT_LINK  /* jlin */
	if ((phyData[sigEntity].PC_Type == PC_Type_S) &&
	    (sigEntity == PHY_B))
	{	   
		if (stationData.WA_Flag)
		{
			stationData.WA_Flag = CLEAR;
			SMTSendSignal (SIG_PC_Start, PHY_A, (uInt32) 0);
		}
	}
#endif
	return;
}

static void
Insert_P_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform Insert_P_Actions for the CEM state machines.
Parameters:	sigEntity	= PHY causing configuration change.
Input:		Uses cemData.
Output:		Changes cemData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("Port %d:Insert_P.",sigEntity);
	cemData[sigEntity].ceState = CE_INSERT_P;
	SendCSPEvent (fddiPORTCE_State, sigEntity);
	
	/*
	*	910419-003	LJP
	*	Use macros to set and clear RM_Join.
	*/
	MRM_JOIN (PRIMARY_MAC, sigEntity, SET);
	MRM_JOIN (SECONDARY_MAC, sigEntity, CLEAR);

	/*
	*	911022-003	LJP
	*	Add code to identify Tree-Peer connection for
	*	SMT-LBC-69.
	*/
	if (phyData[sigEntity].PC_Type == PC_Type_B)
	{
		if (!MCF_WRAP_AB)
			if (phyData[PHY_B].PC_Mode == PC_Mode_Tree)
			{
				stationData.WA_Flag = SET;
				SMTSendSignal (SIG_PC_Start, PHY_A, (uInt32) 0);
			}

			else
			{
				stationData.WAT_Flag = SET;
				if (phyData[PHY_A].PC_Mode == PC_Mode_Tree)
					SMTSendSignal (SIG_PC_Start,
						PHY_A, (uInt32) 0);
			}
	}

#ifdef REDUNDANT_LINK  /* jlin */
	if ((phyData[sigEntity].PC_Type == PC_Type_S) &&
	    (sigEntity == PHY_B))
	{	   
	                stationData.WA_Flag = SET;
			SMTSendSignal (SIG_PC_Start, PHY_A, (uInt32) 0);
	}
#endif
	return;
}

static void
Insert_S_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform Insert_S_Actions for CEM state machines.
Parameters:	sigEntity	= PHY causing configuration change.
Input:		Uses cemData.
Output:		Changes cemData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("Port %d:Insert_S.",sigEntity);
	cemData[sigEntity].ceState = CE_INSERT_S;
	SendCSPEvent (fddiPORTCE_State, sigEntity);
	
	/*
	*	910419-003	LJP
	*	Use macros to set and clear RM_Join.
	*/
	MRM_JOIN (PRIMARY_MAC, sigEntity, CLEAR);
	MRM_JOIN (SECONDARY_MAC, sigEntity, SET);

	return;
}

static void
Insert_X_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform Insert_X_Actions for the CEM state machines.
Parameters:	sigEntity	= PHY causing configuration change.
Input:		Uses phyData, cemData and stationData.
Output:		Changes cemData and stationData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("Port %d:Insert_X.",sigEntity);
	cemData[sigEntity].ceState = CE_INSERT_X;
	SendCSPEvent (fddiPORTCE_State, sigEntity);

	/*
	*	910419-003	LJP
	*	Use macros to set and clear RM_Join.
	*/
	MRM_JOIN (PRIMARY_MAC, sigEntity, SET);
	MRM_JOIN (SECONDARY_MAC, sigEntity, SET);
	
	/*
	*	911022-003	LJP
	*	Add code to identify Tree-Peer connection for
	*	SMT-LBC-69.
	*/
	if (phyData[sigEntity].PC_Type == PC_Type_B)
	{
		if (!MCF_WRAP_AB)
			if (phyData[PHY_B].PC_Mode == PC_Mode_Tree)
			{
				stationData.WA_Flag = SET;
				SMTSendSignal (SIG_PC_Start, PHY_A, (uInt32) 0);
			}

			else
			{
				stationData.WAT_Flag = SET;
				if (phyData[PHY_A].PC_Mode == PC_Mode_Tree)
					SMTSendSignal (SIG_PC_Start,
						PHY_A, (uInt32) 0);
			}
	}

	return;
}

static void
Local_Actions (sigEntity)
	uInt16	sigEntity;
/*********************************************************************
Function:	Perform Local_Actions for the CEM state machines.
Parameters:	sigEntity	= PHY causing configuration change.
Input:		Uses cemData.
Output:		Changes cemData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
        CSPDPT1("Port %d:Local.",sigEntity);
	cemData[sigEntity].ceState = CE_LOCAL;
	SendCSPEvent (fddiPORTCE_State, sigEntity);

/*	MRM_LOOP (LOCAL_MAC, sigEntity, SET); */
	
	return;
}

/*********************************************************************
	CEM State Machines
*********************************************************************/

static void
SignalACEM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process a signal for the Port A CEM state machine
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses phyData, cemData and stationData.
Output:		None.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Select state.
	*/
	switch (cemData[PHY_A].ceState)
	{
	/*
	*	CE0:ISOLATED
	*/
	case CE_ISOLATED:
		/*
		*	CE(01)
		*/
		if (cemData[PHY_A].CF_Join && ((cemData[PHY_B].CF_Join
			&& (phyData[PHY_A].PC_Mode == PC_Mode_Peer)
			&& (phyData[PHY_B].PC_Mode == PC_Mode_Peer))
			|| (!MCF_PATH_S && !cemData[PHY_B].CF_Join)))
		{
			Insert_P_Actions (PHY_A);
		}

		/*
		*	CE(03)
		*/
		else if (MCF_PATH_S && cemData[PHY_A].CF_Join
			&& (!cemData[PHY_B].CF_Join
			|| (MCF_WRAP_AB
			&& ((phyData[PHY_A].PC_Mode == PC_Mode_Tree)
			|| (phyData[PHY_B].PC_Mode == PC_Mode_Tree)))))
		{
			Insert_X_Actions (PHY_A);
		}

		/*
		*	CE(04)
		*/
		else if (cemData[PHY_A].CF_Loop)
			Local_Actions (PHY_A);

		break;

	/*
	*	CE1:INSERT_P
	*/
	case CE_INSERT_P:
		/*
		*	CE(10)
		*/
		if (!cemData[PHY_A].CF_Join || (cemData[PHY_B].CF_Join
			&& ((phyData[PHY_A].PC_Mode == PC_Mode_Tree)
			|| (phyData[PHY_B].PC_Mode == PC_Mode_Tree))))
		{
			/*
			*	Scrub primary path.
			*/
			StartScrubbing (PATH_P);

			Isolated_Actions (PHY_A);
		}

		/*
		*	CE(13)
		*/
		else if (MCF_PATH_S && !cemData[PHY_B].CF_Join)
		{
			/*
			*	Scrub primary path.
			*/
			StartScrubbing (PATH_P);

			Insert_X_Actions (PHY_A);
		}

		break;

	/*
	*	CE3:INSERT_X
	*/
	case CE_INSERT_X:
/*
*	910419-001	LJP
*	This #ifdef tests a proposed change to the Port A CEM state machine.
*	The #else portion adjusts the code to make the current state
*	machine work. Both fixes work.
*/
/* #define	PROPOSED_CHANGES */
#ifdef	PROPOSED_CHANGES
		/*
		*	CE(30)
		*/
		if (!cemData[PHY_A].CF_Join || (cemData[PHY_B].CF_Join
			&& !MCF_WRAP_AB 
			&& (phyData[PHY_A].PC_Mode == PC_Mode_Tree
				|| phyData[PHY_B].PC_Mode == PC_Mode_Tree)))
		{
			/*
			*	Scrub primary and secondary paths.
			*/
			StartScrubbing (PATH_P);
			StartScrubbing (PATH_S);

			Isolated_Actions (PHY_A);
		}

		/*
		*	CE(31)
		*/
		else if (((phyData[PHY_A].PC_Mode == PC_Mode_Peer)
			&& (phyData[PHY_B].PC_Mode == PC_Mode_Peer)
			&& cemData[PHY_B].CF_Join)
			|| !MCF_PATH_S)
		{
			/*
			*	Scrub primary and secondary paths.
			*/
			StartScrubbing (PATH_P);
			StartScrubbing (PATH_S);

			Insert_P_Actions (PHY_A);
		}

#else

		/*
		*	CE(31)
		*/
		if (((phyData[PHY_A].PC_Mode == PC_Mode_Peer)
			&& (phyData[PHY_B].PC_Mode == PC_Mode_Peer)
			&& cemData[PHY_B].CF_Join)
			|| (!MCF_PATH_S && !cemData[PHY_B].CF_Join))
		{
			/*
			*	Scrub primary and secondary paths.
			*/
			StartScrubbing (PATH_P);
			StartScrubbing (PATH_S);

			Insert_P_Actions (PHY_A);
		}

		/*
		*	CE(30)
		*/
		else if (!cemData[PHY_A].CF_Join || (cemData[PHY_B].CF_Join
			&& (!MCF_PATH_S || !MCF_WRAP_AB)))
		{
			/*
			*	Scrub primary and secondary paths.
			*/
			StartScrubbing (PATH_P);
			StartScrubbing (PATH_S);

			Isolated_Actions (PHY_A);
		}
#endif

		break;

	/*
	*	CE4:LOCAL
	*/
	case CE_LOCAL:
		/*
		*	CE(40)
		*/
		if (!cemData[PHY_A].CF_Loop)
		{
			/*
			*	Scrub local path.
			*/
			StartScrubbing (PATH_L);

			Isolated_Actions (PHY_A);
		}
		break;
	}

	return;
}

static void
SignalBCEM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process a signal for the Port B CEM state machine.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses phyData, cemData and stationData.
Output:		None.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Select state.
	*/
	switch (cemData[PHY_B].ceState)
	{
	/*
	*	CE0:ISOLATED
	*/
	case CE_ISOLATED:
		/*
		*	CE(01)
		*/
		if (!MCF_PATH_S && cemData[PHY_B].CF_Join)
		{
			Insert_P_Actions (PHY_B);
		}

		/*
		*	CE(03)
		*/
		else if (MCF_PATH_S && cemData[PHY_B].CF_Join)
		{
			Insert_X_Actions (PHY_B);
		}

		/*
		*	CE(04)
		*/
		else if (cemData[PHY_B].CF_Loop)
		{
			Local_Actions (PHY_B);
		}

		break;

	/*
	*	CE1:INSERT_P
	*/
	case CE_INSERT_P:
		/*
		*	CE(10)
		*/
		if (!cemData[PHY_B].CF_Join)
		{
			/*
			*	Scrub_P
			*/
			StartScrubbing (PATH_P);

			Isolated_Actions (PHY_B);
		}

		/*
		*	CE(13)
		*/
		else if (MCF_PATH_S)
		{
			Insert_X_Actions (PHY_B);
		}

		break;

	/*
	*	CE3:INSERT_X
	*/
	case CE_INSERT_X:
		/*
		*	CE(30)
		*/
		if (!cemData[PHY_B].CF_Join)
		{
			/*
			*	Scrub primary and secondary paths.
			*/
			StartScrubbing (PATH_P);
			StartScrubbing (PATH_S);

			Isolated_Actions (PHY_B);
		}

		/*
		*	CE(31)
		*/
		else if (!MCF_PATH_S)
		{
			/*
			*	Scrub primary and secondary paths.
			*/
			StartScrubbing (PATH_P);
			StartScrubbing (PATH_S);

			Insert_P_Actions (PHY_B);
		}

		break;

	/*
	*	CE4:LOCAL
	*/
	case CE_LOCAL:
		/*
		*	CE(40)
		*/
		if (!cemData[PHY_B].CF_Join)
		{
			/*
			*	Scrub local path.
			*/
			StartScrubbing (PATH_L);

			Isolated_Actions (PHY_B);
		}

		break;
	}

	return;
}

static void
SignalMCEM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process a signal for the Master Port CEM state machine.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses cemData.
Output:		Changes cemData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Process signal.
	*/
	switch (sigType)
	{
	case SIG_CF_Join:
		cemData[sigEntity].CF_Join = sigData;
		break;

	case SIG_CF_Loop:
		cemData[sigEntity].CF_Loop = sigData;
		break;
	}

	/*
	*	Select state.
	*/
	switch (cemData[sigEntity].ceState)
	{
	/*
	*	CE0:ISOLATED
	*/
	case CE_ISOLATED:
		/*
		*	CE(01)
		*/
		if (!MCF_INSERT_S(sigEntity)
			&& cemData[sigEntity].CF_Join)
		{
			Insert_P_Actions (sigEntity);
		}

		/*
		*	CE(02)
		*/
		else if (MCF_INSERT_S(sigEntity)
			&& cemData[sigEntity].CF_Join)
		{
			Insert_S_Actions (sigEntity);
		}

		/*
		*	CE(04)
		*/
		else if (cemData[sigEntity].CF_Loop)
		{
			Local_Actions (sigEntity);
		}

		break;

	/*
	*	CE1:INSERT_P
	*/
	case CE_INSERT_P:
		/*
		*	CE(10)
		*/
		if (!cemData[sigEntity].CF_Join)
		{
			/*
			*	Scrub primary path.
			*/
			StartScrubbing (PATH_P);

			Isolated_Actions (sigEntity);
		}

		/*
		*	CE(12)
		*/
		else if (MCF_INSERT_S(sigEntity) && cemData[sigEntity].CF_Join)
		{
			/*
			*	Scrub primary path.
			*/
			StartScrubbing (PATH_P);

			Insert_S_Actions (sigEntity);
		}

		break;

	/*
	*	CE2:INSERT_S
	*/
	case CE_INSERT_S:
		/*
		*	CE(20)
		*/
		if (!cemData[sigEntity].CF_Join)
		{
			/*
			*	Scrub secondary path.
			*/
			StartScrubbing (PATH_S);

			Isolated_Actions (sigEntity);
		}

		/*
		*	CE(21)
		*/
		else if (!MCF_INSERT_S(sigEntity))
		{
			/*
			*	Scrub secondary path.
			*/
			StartScrubbing (PATH_S);

			Insert_P_Actions (sigEntity);
		}

		break;

	/*
	*	CE4:LOCAL
	*/
	case CE_LOCAL:
		/*
		*	CE(40)
		*/
		if (!cemData[sigEntity].CF_Loop)
		{
			/*
			*	Scrub local path.
			*/
			StartScrubbing (PATH_L);

			Isolated_Actions (sigEntity);
		}

		break;
	}

	return;
}

static void
SignalSCEM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process a signal for the S Port CEM state machine.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses cemData.
Output:		Changes cemData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Process signal.
	*/
	switch (sigType)
	{
	case SIG_CF_Join:
		cemData[sigEntity].CF_Join = sigData;
		break;

	case SIG_CF_Loop:
		cemData[sigEntity].CF_Loop = sigData;
		break;
	}

	/*
	*	Select state.
	*/
	switch (cemData[sigEntity].ceState)
	{
	/*
	*	CE0:ISOLATED
	*/
	case CE_ISOLATED:
		/*
		*	CE(01)
		*/
		if (cemData[sigEntity].CF_Join)
		{
			    Insert_P_Actions (sigEntity);
		}

		/*
		*	CE(04)
		*/
		else if (cemData[sigEntity].CF_Loop)
		{
			Local_Actions (sigEntity);
		}

		break;
	
	/*
	*	CE1:INSERT_P
	*/
	case CE_INSERT_P:
		/*
		*	CE(10)
		*/
		if (!cemData[sigEntity].CF_Join)
		{
			/*
			*	Scrub primary path.
			*/
			StartScrubbing (PATH_P);

			Isolated_Actions (sigEntity);
		}

		break;

	/*
	CE4:LOCAL
	*/
	case CE_LOCAL:
		/*
		*	CE(40)
		*/
		if (!cemData[sigEntity].CF_Loop)
		{
			/*
			*	Scrub local path.
			*/
			StartScrubbing (PATH_L);

			Isolated_Actions (sigEntity);
		}

		break;
	}

	return;
}

/*********************************************************************
	CFM State Machine
*********************************************************************/

void
SignalCFM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process a CFM signal.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses phyData, cemData and stationData.
Output:		Changes stationData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
Flag	oldJoin;			/* previous CF_Join setting */

	/*
	*	Get current CF_Join setting.
	*/
	oldJoin = cemData[sigEntity].CF_Join;

	/*
	*	Process through appropriate CEM state machine
	*	based on Port type and signal type.
	*/
	if (phyData[sigEntity].PC_Type == PC_Type_M)
	{
		/* Process the M Port CEM */
		SignalMCEM (sigType, sigEntity, sigData);

		/* Configure M Port */
		ConfigureMaster (sigEntity, cemData[sigEntity].ceState);
	}

	else if (phyData[sigEntity].PC_Type == PC_Type_S)
	{
		/* Process the S Port CEM */
		SignalSCEM (sigType, sigEntity, sigData);

		/* Determine station's CFM state */
		if (cemData[sigEntity].ceState == CE_INSERT_P)
			stationData.cfState = CF_WRAP_S;
		else
			stationData.cfState = CF_ISOLATED;
		SendCSPEvent (fddiSMTCF_State, 0);

		/* Configure station */
		ConfigureAttach (stationData.cfState);
	}

	else
	{
		/*
		*	Preprocess signal for handling by
		*	CEM state machine.
		*/
		switch (sigType)
		{
		case SIG_CF_Join:
			cemData[sigEntity].CF_Join = sigData;
			break;

		case SIG_CF_Loop:
			cemData[sigEntity].CF_Loop = sigData;
			break;
		}

		/*
		*	Send signal to both A and B port CEMs.
		*/
		SignalACEM (sigType, sigEntity, sigData);
		SignalBCEM (sigType, sigEntity, sigData);

		/* Set station's CFM state */
		stationData.cfState =
	CFM_State_Table[cemData[PHY_A].ceState][cemData[PHY_B].ceState];
		SendCSPEvent (fddiSMTCF_State, sigEntity);

		/* Configure attach ports */
		ConfigureAttach (stationData.cfState);
	}

	/*
	*	Notify MIB of configuration changes on change to CF_Join.
	*	This is the new behavior for fddiSMTConfigurationChgEvent
	*	as specified by committee after SMT 6.2 publication.
	*/
	if ((sigType == SIG_CF_Join)
			&& (cemData[sigEntity].CF_Join != oldJoin))
		SendCSPEvent (fddiSMTConfigurationChgEvent, sigEntity);

	return;
}
