/*****************************************************************************
 *  File        bmac.c
 *
 *  Description
 *      This file is the BMAC device driver.
 *      MAC A is numbered as MAC 1, and MAC B is numbered as MAC 0.
 *
 *  Copyright (c) Hughes Lan Systems, 1992-1993
 *
 *  Author: J. Jang
 *
 *  $Log:   /usr/shasta/pvcs/fddicon/drv/src/bmac.c_v  
 * 
 *    Rev 1.82   11 Nov 1993 13:21:12   jang
 * fixed the bug of late count incorrect
 * 
 *    Rev 1.81   10 Nov 1993 12:24:36   jlin
 * fixed the bug of TMAX and TVX
 * 
 *    Rev 1.80   12 Oct 1993 11:06:24   gregs
 * No change.
 * 
 *    Rev 1.79   29 Sep 1993 13:40:08   gregs
 * No change.
 * 
 *    Rev 1.78   13 Sep 1993 13:19:50   gregs
 * Initial Bridge SIT Release
 * 
 *    Rev 1.77   02 Aug 1993 17:45:38   jlin
 * Add macTREQ in drv_mib
 * 
 *    Rev 1.76   23 Jul 1993 18:10:36   jlin
 * polling  BMAC_RELR0
 * 
 *    Rev 1.75   09 Jul 1993 18:29:04   jang
 * added code to initialize ringop_set/ringop_clr
 * 
 *    Rev 1.74   08 Jul 1993 14:19:34   jang
 * clean  house to delete some compiler warning message
 * 
 *    Rev 1.73   17 Jun 1993 14:17:02   jang
 * fixed type error
 * 
 *    Rev 1.72   14 Jun 1993 19:18:22   jang
 * fixed frame status
 * y
 * 
 *    Rev 1.71   12 Jun 1993 10:44:30   jang
 * added a function to clear all the mac related counters
 * 
 *    Rev 1.70   08 Jun 1993 06:48:36   jang
 * moved variables to sram
 * 
 *    Rev 1.70   07 Jun 1993 17:12:28   jang
 * moved the variables to sram
 * 
 *    Rev 1.67   06 May 1993 14:28:26   jang
 * changed direct beacon
 * 
 *    Rev 1.66   06 May 1993 13:51:26   jlin
 * Fix another bug in the GetTimerRegMask()
 * Check if the timer value is 0, if it is, then set the timer to default value
 * also fix the for loop so that it will return the correct value
 * 
 *    Rev 1.65   06 May 1993 11:48:38   jlin
 * Fix a bug in GetTimerRegMask()
 * the pass in parameter is already in  80ns unit.
 * 
 *    Rev 1.64   05 May 1993 14:50:20   jang
 * fixed the bug of disable mac interrupt by the RMT
 * 
 *    Rev 1.63   29 Apr 1993 15:12:08   jang
 * called strobe_wdt() so wdt won't reset the system
 * 
 *    Rev 1.62   28 Apr 1993 15:19:06   jang
 * deleted the creationTime in Init_my_slot()
 * 
 *    Rev 1.61   28 Apr 1993 15:01:26   jang
 * changed the get_debug() to ifdef BMAC_DEBUG
 * 
 *    Rev 1.60   26 Apr 1993 10:21:32   jang
 * calling BMAC_EnableInt() in BMAC_EnableEvent()
 * 
 *    Rev 1.59   24 Apr 1993 17:12:30   jang
 * took out enable interrupt when bmac is set to run mode
 * 
 *    Rev 1.58   24 Apr 1993 17:07:38   jang
 * changed the function call of SM_MAC_CTRL_MAC_INT
 * 
 *    Rev 1.57   24 Apr 1993 10:44:56   jang
 * took out waiting for entering claim state when smt request to send claim
 * 
 *    Rev 1.56   22 Apr 1993 17:41:26   jang
 * took out the debug mac address string
 * 
 *    Rev 1.55   21 Apr 1993 15:32:16   jang
 * added code to send direct beacon
 * 
 *    Rev 1.54   20 Apr 1993 15:56:08   jang
 * only do init_my_slot() during application
 * 
 *    Rev 1.53   16 Apr 1993 11:56:02   jang
 * changed swRingOp from bool * to byte *
 * 
 *    Rev 1.52   15 Apr 1993 14:22:24   jang
 * added code to check the ring_op flag in the MIB
 * 
 *    Rev 1.51   14 Apr 1993 10:20:12   gregs
 * Fixed another un-terminated in BeaconTest.
 * 
 *    Rev 1.50   14 Apr 1993 10:04:56   gregs
 * Fixed unterminated comment in BEacontest.
 * 
 *    Rev 1.49   12 Apr 1993 14:05:20   jang
 * fixed the compiling warning
 * 
 *    Rev 1.48   10 Apr 1993 16:40:10   jang
 * 
 *    Rev 1.47   09 Apr 1993 14:32:16   jang
 * added more debug code
 * 
 *    Rev 1.46   05 Apr 1993 11:52:34   jang
 * took 'set mac in stop mode' out when reading TNEG and added code
 * to disable interrupt when set address/TVX etc
 * 
 *    Rev 1.45   05 Apr 1993 07:43:56   jang
 * 
 *    Rev 1.44   02 Apr 1993 11:54:12   jang
 * deleted the pstats initialization from BMAC_Init()
 * 
 *    Rev 1.43   31 Mar 1993 18:36:40   jang
 * chagned the self test 
 * 
 *    Rev 1.42   18 Mar 1993 17:58:30   jang
 * added an array to store mac addresses in FDDI order so we don't have to
 * swap address fields in ISR
 * 
 *    Rev 1.41   18 Mar 1993 15:00:32   jlin
 * Fix the bug in reading mac counter registers
 * 
 *    Rev 1.40   09 Mar 1993 13:26:16   jang
 * deleted descrment table size
 * 
 *    Rev 1.39   08 Mar 1993 18:42:24   jang
 * to decrement table size in fddiHost table
 * 
 *    Rev 1.38   08 Mar 1993 09:25:04   jang
 * deleted my_hosts[]
 * 
 *    Rev 1.37   05 Mar 1993 10:06:38   jang
 * added code to initialize hash table for the bootp download
 * 
 *    Rev 1.36   01 Mar 1993 15:21:04   jang
 * 
 *    Rev 1.35   24 Feb 1993 16:48:24   jang
 * for cmt  interop 
 * 
 *    Rev 1.34   24 Feb 1993 10:53:04   jang
 * changed the index of MyNid() by adding one to it
 * 
 *    Rev 1.33   21 Feb 1993 12:31:00   jang
 * deleted some printf
 * 
 *    Rev 1.32   19 Feb 1993 17:53:46   jang
 * 
 *    Rev 1.31   16 Feb 1993 10:55:40   jang
 * added a new function BMAC_GetBMACMode() and reset fgm0,1 during reset
 * 
 *    Rev 1.30   09 Feb 1993 17:22:38   jlin
 * disable cberr in disableevents()
 * 
 *    Rev 1.29   09 Feb 1993 12:47:28   jang
 * do the regular initialization
 * 
 *    Rev 1.28   08 Feb 1993 18:45:06   jang
 * added 2 new functions to enable/disable some events requested by smt
 * 
 *    Rev 1.27   26 Jan 1993 17:13:28   jang
 * fixed the bugs caused compileing error
 * 
 *    Rev 1.26   26 Jan 1993 15:11:34   jang
 * added function to return the mac related mibs
 * 
 *    Rev 1.25   26 Jan 1993 11:21:44   jang
 * put all the event registers to 0 temporarily for testing
 * 
 *    Rev 1.24   18 Jan 1993 07:50:32   jang
 * deleted mac_ptr in BSI_TYPE
 * 
 *    Rev 1.23   12 Jan 1993 09:31:50   jang
 * initialized fddi_if_pstats_tbl[]
 * 
 *    Rev 1.22   07 Jan 1993 09:30:06   jang
 * put rev 1.19 back
 * 
 *    Rev 1.19   06 Jan 1993 16:30:14   jang
 * added monp statistics
 ***************************************************************************/


#include <fddi.h>
#include <drv.h>
#include <bsi.h>
#include <bmac.h>
#include <smtmacro.h>
#include <cspmacro.h>
#include <fdrmonty.h>
#include <dbd.h>
#include <dips.h>
#include <fdrextrn.h>
#include <fddihdr.h>


#define RINGOP_DEBUG      0
#define BMAC_DEBUG        0
#define CLAIM_DEBUG       0
#define COUNT_DEBUG       0
#if COUNT_DEBUG
extern cp_cnt[], rx_cnt[], err_cnt[];
#endif

/*---------------------------------------------------------------------------
 *    Exported variables
 *---------------------------------------------------------------------------
 */

bool bmac_init_flag;
NID  MyFDDINid[2];


/*---------------------------------------------------------------------------
 *    Imported variables
 *---------------------------------------------------------------------------
 */

extern BMAC_TYPE bmacs[];
extern BSI_TYPE bsis[];
extern BMAC_INIT_TYPE bmacs_master_default[];
extern FDDI_PSTATS fddi_if_pstats_tbl[];
extern HOSTENTRY *MONP_hash();
extern BMAC_DEFAULT_TABLE bmaccon_default_table,bmacbr_default_table;

/*
 *---------------------------------------------------------------------------
 *      Local vairables
 *---------------------------------------------------------------------------
 */
static byte *DirBcnPtr;    /* pointed to the directed beacon frame */



/*
 *---------------------------------------------------------------------------
 *      Function Prototypes
 *---------------------------------------------------------------------------
 */
#ifdef DO_LINT
static void ResetBmacRegs(uint32);
static void ClearRelr0(BMAC_REG_TYPE *);
static void ClearRelr1(BMAC_REG_TYPE *);
static status_type BeaconTest(uint32,uint32,bool);
static status_type SetBeaconFrame(BMAC_REG_TYPE *);
static bool MacIsRun(BMAC_REG_TYPE *);
static void ResetMacAddrs (uint32);
static void GetMacDefault(uint32);
static void DisableGrpAddr(uint32,uint32);
static void WritePGM(volatile byte *,byte);
static bool ReadPGM(volatile byte *,byte);
static void ResetPGM(uint32);
#else
static void ResetBmacRegs();
static void ClearRelr0();
static void ClearRelr1();
static status_type BeaconTest();
static status_type ClaimTest();
static status_type SetBeaconFrame();
static bool MacIsRun();
static void ResetMacAddrs ();
static void ClearFRCT ();
static void ClearEICT();
static void ClearLFCT();
static void ClearFCCT ();
static void ClearFNCT ();
static void ClearFTCT ();
static void ClearTKCT ();
static void ClearRLCT();
static void ClearLTCT();
static void GetMacDefault();
static void DisableGrpAddr();
static void ClearTELR0();
static void ClearCILR();
static void ClearCOLR();
static void ClearIELR();
static void ClearESR();
static byte GetTimerRegMask();
static void show_bmac_regs();
static void WritePGM();
static bool ReadPGM();
static void ResetPGM();
static uint32 ReadFRCT();
static uint32 ReadEICT();
static uint32 ReadLFCT();
static uint32 ReadFCCT();
static uint32 ReadFNCT();
static uint32 ReadFTCT();
static uint32 ReadTKCT();
static uint32 ReadRLCT();
static uint32 ReadLTCT();
static void init_my_slot();
static void InitMACFrameStatus();

#endif



/***************************************************************************
 *  Function    BMAC_Init
 *
 *  Description
 *      This function initializes BMAC data structure, and the BMAC chip.
 *
 *  Parameter:
 *      uint16 mac_nums
 *
 *  Return: [OK | FDDI_ERROR]
 ***************************************************************************/
status_type BMAC_Init (mac_nums,source)
uint32 mac_nums;
int source;
{
  BMAC_CTRL_TYPE *i_ptr;
  int i;
  
  if (bmac_init_flag == TRUE)
    return (OK);

  if (mac_nums > fddi_board.bmac_num)
    return (FDDI_ERROR);

  DirBcnPtr = NULL;
  BMACISR_InitDebug();
  for (i = 0; i < mac_nums; i++) {
    strobe_wdt();
    GetMacDefault(i);
    if ((i_ptr = (BMAC_CTRL_TYPE *)lmalloc(sizeof(BMAC_CTRL_TYPE))) == NULL) {
#if BMAC_DEBUG
      printf("ERROR: no memory to initialize BMAC\n");
#endif
      return (FDDI_ERROR);
    }
    memset(i_ptr,0,sizeof(BMAC_CTRL_TYPE));
    bmacs[i].image_ptr = i_ptr;
    bmacs[i].revision = bmacs[i].reg_base->rr;

    ResetBmacRegs(i,source);
    ResetMacAddrs(i);
    InitMACFrameStatus(i);
  }       /* for */

  init_my_slot(1,bmacs[0].long_addr,0);
  
  if (source == DRV_FROM_APP)
    bmac_init_flag = TRUE;

  return (OK);
}
 
/***************************************************************************
 *  Function BMAC_CMTTestInit
 *
 *  Description
 *      This function initializes both BMACs to a dumb mac for the CMT
 *      release only. It only sets OR and MR.
 *
 *  Parameter:  none
 *
 *  Return: void
 **************************************************************************/
void BMAC_CMTTestInit ()
{
    int i;
    volatile BMAC_REG_TYPE *r_ptr;
/*    NID *my_addr; */

    for (i=0; i < 2; i++) {
        GetMacDefault(i);
	r_ptr = bmacs[i].reg_base;
	r_ptr->fr = BMAC_FR_MARST;
	r_ptr->mr = 0;
	r_ptr->or = BMAC_OR_ELA | BMAC_OR_IRR;
	r_ptr->mr = BMAC_MR_RUN;
    }
}


/***************************************************************************
 *  Function    BMAC_Reset
 *
 *  Description
 *      This function sets BMAC to default value.
 *
 *  Parameter:
 *      uint32 mac_num - the mac to reset
 *
 *  Return: void
 *************************************************************************/
void BMAC_Reset (mac_num)
uint32 mac_num;
{
#if RINGOP_DEBUG
  printf("reset bmac %d\n",mac_num);
#endif
  (bmacs[mac_num].reg_base)->fr = BMAC_FR_MCRST;
}
#if 0
/***************************************************************************
 *  Function    BMAC_ClearBMACs
 *
 *  Description
 *    This function clears all the BMAC interrupts, put bmac to stop mode,
 *    and release all the memory.
 *
 *  Parameters:
 *    uint32 num_macs - # of BMACs on the board
 *    uint32 type - FDDICON_MASTER |
 ***************************************************************************/
void BMAC_ClearBMACs (num_macs,type)
uint32 num_macs;
uint32 type;
{
  int i;
  volatile BMAC_REG_TYPE *r_ptr;

  if (type == FDDICON_MASTER) {
    for (i=0; i < num_macs; i++) {
      BMAC_StopBmac(i);
      r_ptr = bmacs[i].reg_base;
      free(bmacs[i].image_ptr);
    }
  }
  bmac_init_flag = FALSE;
}
#endif
/***************************************************************************
 *  Function    BMAC_SetLoopback
 *
 *  Description
 *    This function sets the BAMC to internal/external loopback mode
 *
 *  Parameters:
 *    uint32 mac_num
 *    uint32 mode - [DRV_START | DRV_STOP]
 *
 *  Return:  void
 **************************************************************************/
void BMAC_SetLoopback (mac_num,mode)
uint32 mac_num;
uint32 mode;
{
  volatile BMAC_REG_TYPE *r_ptr;

  r_ptr = bmacs[mac_num].reg_base;
  if (mode == DRV_START)
    r_ptr->mr |= BMAC_MR_ILB;
  else
    r_ptr->mr &= ~BMAC_MR_ILB;
}

/***************************************************************************
 *  Function    BMAC_SetExtMatch
 *
 *  Description
 *    This function sets the OR_EMIND
 *
 *  Parameters:
 *    uint32 mac_num
 *    uint32 mode - [MAC_AC_NONE | MAC_AC_COPIED]
 *
 *  Return:  void
 **************************************************************************/
void BMAC_SetExtMatch (mac_num,mode)
uint32 mac_num;
uint32 mode;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

  if (mode == MAC_AC_NONE)
    r_ptr->or &= ~BMAC_OR_EMIND;
  else
    r_ptr->or |= BMAC_OR_EMIND;
}


/***************************************************************************
 *  Function    BMAC_ChkDupAddr
 *
 *  Description
 *      This function checks duplicate address.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: [TRUE | FALSE]
 **************************************************************************/
bool BMAC_ChkDupAddr (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    byte reg;
    bool retn;
    long s; 

    r_ptr = bmacs[mac_num].reg_base;
    s = crit_on();

    reg = r_ptr->relr0;
    if (reg & BMAC_RELR0_DUPADD) {
        retn = TRUE;
        r_ptr->relr0 = reg & ~BMAC_RELR0_DUPADD;
#if 0
   printf("BMAC_ChkDupAddr(), DUPADDR bit is set in bsi %d\n",mac_num);
#endif
    }
    else
        retn = FALSE;

    crit_off(s);

    return (retn);
}
 
/***************************************************************************
 *  Function    BMAC_SetLongAddr
 *
 *  Description
 *      This function sets the BMAC's MLA register. NOTE that this function
 *      assumes that the BMAC is in stop mode. Therefore the calling function
 *      has to put the BMAC in STOP mode otherwise command error will occur.
 *
 *  Parameters:
 *      uint32 mac_num
 *      byte *addr - it is assumed to have 6 bytes
 *
 *  Return: void
 **************************************************************************/
void BMAC_SetLongAddr (mac_num,addr)
uint32 mac_num;
byte *addr;
{
    volatile BMAC_REG_TYPE *r_ptr;
    byte mr;
/*    uint32 port_num; */
    long s;

    s = crit_on();
#if 0
{
    int i;
    printf("mac %d:     ",mac_num);
    for (i=0; i < 6; i++)
      printf("%x.",*(addr+i));
    printf("\n");
}
#endif      /* if 1 */

    r_ptr = bmacs[mac_num].reg_base;
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;      /* stop BMAC */
    r_ptr->mla0 = *addr;
    r_ptr->mla1 = *(addr+1);
    r_ptr->mla2 = *(addr+2);
    r_ptr->mla3 = *(addr+3);
    r_ptr->mla4 = *(addr+4);
    r_ptr->mla5 = *(addr+5);

    memcpy(bmacs[mac_num].long_addr,addr,6);
    r_ptr->mr = mr;     /* restore mr */
    crit_off(s);
}


/***************************************************************************
 *  BMAC_ClearAllMacCounters
 *
 *  Description
 *    This function clears all the mac related counters
 *
 *  Parameter:
 *    uint32 mac_num
 *
 *  Return: void
 **************************************************************************/
void BMAC_ClearAllMacCounters (mac_num)
uint32 mac_num;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
  DRV_FDDI_MIBS *m_ptr = bmacs[mac_num].mib_ptr;

  ClearFRCT(r_ptr,m_ptr->macFrameRxCount);
  ClearEICT(r_ptr,m_ptr->macErrIsolateCount);
  ClearLFCT(r_ptr,m_ptr->macLostFrameCount);
  ClearFCCT(r_ptr,m_ptr->macFrameCopiedCount);
  ClearFNCT(r_ptr,m_ptr->macFrameNotCopiedCount);
  ClearFTCT(r_ptr,m_ptr->macFrameTxCount);
  ClearTKCT(r_ptr,m_ptr->macTokenRxCount);
  ClearRLCT(r_ptr,m_ptr->macRingLatencyCount);
  ClearLTCT(r_ptr,m_ptr->macLateCount);
}

/***************************************************************************
 *  Functrion  BMAC_ReadTMAX
 *
 *  Description
 *    This function reads TMAX register and returns whatever value in
 *    the register without remapping
 *
 *  Parameters:
 *      uint32 mac_num
 *      byte *val
 *
 *  Return: OK
 **************************************************************************/
int BMAC_ReadTMAX (mac_num,val)
uint32 mac_num;
byte *val;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

  *val = r_ptr->tmax;
  return OK;
}

/***************************************************************************
 *  Functrion  BMAC_SetTMAX
 *
 *  Description
 *    This function writes a value to the TMAX register.
 *
 *  Parameters:
 *      uint32 mac_num
 *      byte *val
 *
 *  Return: OK
 **************************************************************************/
int BMAC_SetTMAX (mac_num,val)
uint32 mac_num;
byte val;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

  r_ptr->tmax = val;
  return OK;
}

/***************************************************************************
 *  Function    BMAC_SetShortAddr
 *
 *  Description
 *      This function sets the BMAC's MSA register.
 *
 *  Parameters:
 *      BMAC_REG_TYPE *r_ptr -
 *      byte *addr - it is assumed to have 2 bytes
 *
 *  Return: void
 **************************************************************************/
void BMAC_SetShortAddr (mac_num,addr)
uint32 mac_num;
byte *addr;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
    byte mr;
    long s;

    s = crit_on();
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;      /* stop BMAC */

    r_ptr->msa0 = *addr;
    r_ptr->msa1 = *(addr+1);

    memcpy(bmacs[mac_num].short_addr,addr,2);
    r_ptr->mr = mr;     /* restore mr */
    crit_off(s);
}
#if 0
/***************************************************************************
 *  Function    BMAC_StopRepeat
 *
 *  Description
 *    This function stops the mac to repeat frame
 *
 *  Parameter:
 *    uint32 mac_num
 *
 *  Return: void
 **************************************************************************/
void BMAC_StopRepeat (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

    r_ptr->or |= BMAC_OR_IRPT;
}

/***************************************************************************
 *  Function    BMAC_StartRepeat
 *
 *  Description
 *    This function starts the mac to repeat frame
 *
 *  Parameter:
 *    uint32 mac_num
 *
 *  Return: void
 **************************************************************************/
void BMAC_StartRepeat (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

    r_ptr->or &= ~BMAC_OR_IRPT;
}
#endif
/***************************************************************************
 *  Function    BMAC_SetTreq
 *
 *  Description
 *      This function sets the TREQ registers
 *
 *  Parameters:
 *      uint32 mac_num
 *      uint32 val - 80 ns as an uint
 *
 *  Return: [OK | FDDI_ERROR]
 *************************************************************************/
status_type BMAC_SetTreq (mac_num,val)
uint32 mac_num;
uint32 val;
{
    volatile BMAC_REG_TYPE *r_ptr;
    BMAC_CTRL_TYPE *i_ptr;
    byte reg;
    uint32 tneg;

    r_ptr = bmacs[mac_num].reg_base;
    i_ptr = bmacs[mac_num].image_ptr;

    reg = r_ptr->mr;
    r_ptr->mr = 0;   /* stop bmac */

    val = (val | 0xff000000) & 0xffffff00;
    r_ptr->treq0 = 0xff;
    i_ptr->treq1 = (byte) (val >> 16);
    i_ptr->treq2 = (byte) (val >> 8);
    r_ptr->treq3 = 0;
    r_ptr->treq1 = (byte) i_ptr->treq1;
    r_ptr->treq2 = (byte) i_ptr->treq2;
    *(init_mibs.d_mib[mac_num].macTREQ) = val;
    r_ptr->mr = reg;    
    return (OK);
}

/***************************************************************************
 *  Function    BMAC_READ_RFLAG
 *
 *  Description
 *      This function reads CRS0 RFLAG bit. It retures TRUE if it is a
 *      restricted flag, FALSE otherwise.
 *
 *  parameter:
 *
 *  Return: [TRUE | FALSE]
 **************************************************************************/
bool BMAC_Read_Rflag (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    byte reg;
    
    r_ptr = bmacs[mac_num].reg_base;
    reg = r_ptr->crs0;
    if (reg & BMAC_CRS0_RFLAG)
        return (TRUE);

    return (FALSE);
}

/***************************************************************************
 *  Function    BMAC_Set_AsynThreshold
 *
 *  Description
 *      This function sets the THSH1 timer register.
 *
 *  Parameters:
 *      uint32 mac_num
 *
 *  Return: [OK | FDDI_ERROR]
 *************************************************************************/
status_type BMAC_Set_AsyncThreshold (mac_num,val)
uint32 mac_num;
uint32 val;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
/*    BMAC_CTRL_TYPE *i_ptr = bmacs[mac_num].image_ptr; */
    byte reg;
/*    uint32 tneg; */
    long s;
    
    s = crit_on();
    reg = r_ptr->mr;
    r_ptr->mr = reg & ~BMAC_MR_RUN;      /* stop BMAC */

    r_ptr->thsh1 = val & BMAC_THSH1_MASK;

    r_ptr->mr = reg;     /* restore mr */    
    crit_off(s);
    
    return (OK);
}

/***************************************************************************
 *  Function    BMAC_SetTimerRegs
 *
 *  Description
 *      This function sets the TMAX timer register.
 *
 *  Parameters:
 *      uint32 mac_num
 *      uint32 val - a 2's complement number of # of 80 ns unit
 *      byte reg - [BMAC_TMR_TMAX | BMAC_TMR_TVX]
 *
 *  Return: [OK | FDDI_ERROR]
 *************************************************************************/
status_type BMAC_SetTimerRegs (mac_num,val,reg)
uint32 mac_num;
uint32 val;
byte reg;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
    byte mr, mask;
/*    uint32 time; */
    long s;

    /* it looks weird because it  is a 2's complement number */
    if (val > BMAC_TREQ_MIN || val < BMAC_TREQ_MAX)
        return (FDDI_ERROR);
        
    s = crit_on();
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;      /* stop BMAC */

    mask = GetTimerRegMask(val);
    if (reg == BMAC_TMR_TMAX)
        r_ptr->tmax = mask & BMAC_TMAX_MASK;
    else if (reg == BMAC_TMR_TVX)
        r_ptr->tvx = mask & BMAC_TVX_MASK;

    r_ptr->mr = mr;     /* restore mr */        

    crit_off(s);
    
    return (OK);
}

/***************************************************************************
 *  Function    BMAC_Set_TVX
 *
 *  Description
 *      This function sets the TVX timer register.
 *
 *  Parameters:
 *      uint32 mac_num
 *
 *  Return: [OK | FDDI_ERROR]
 *************************************************************************/
status_type BMAC_Set_TVX (mac_num,val)
uint32 mac_num;
uint32 val;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
/*    BMAC_CTRL_TYPE *i_ptr = bmacs[mac_num].image_ptr;  */
    byte reg;
/*     uint32 tneg;  */
    long s;

    s = crit_on();
    reg = r_ptr->mr;
    r_ptr->mr = reg & ~BMAC_MR_RUN;      /* stop BMAC */

    r_ptr->tvx = val & BMAC_TVX_MASK;
    
    r_ptr->mr = reg;     /* restore mr */
    crit_off(s);

    return (OK);
}


/***************************************************************************
 *  Function    BMAC_Get_TTRT
 *
 *  Description
 *      This function gets the current value of TTRT stored in TNEGs.
 *
 *  Parameters:
 *      uint32 mac_num
 *
 *  Return: uint32
 *************************************************************************/
uint32 BMAC_Get_TTRT (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
    uint32 val;
    long s;

    s = crit_on();
    val = r_ptr->tneg0 << 24;
    val |= r_ptr->tneg1 << 16;
    val |= r_ptr->tneg2 << 8;
    val |= r_ptr->tneg3;
    crit_off(s);
    
    return (val);
}
 
/***************************************************************************
 *  Function    BMAC_GetShortAddr
 *
 *  Description
 *      This function gets the MAC short address. The MAC address is
 *      supposed to store in the MIB, therefore this function is designed to
 *      be used by the driver itself.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: *addr
 **************************************************************************/
byte *BMAC_GetShortAddr (mac_num)
uint32 mac_num;
{
    return ((byte *) &(bmacs[mac_num].short_addr));
}

 
/***************************************************************************
 *  Function    BMAC_GetLongAddr
 *
 *  Description
 *      This function gets the MAC short address. The MAC address is
 *      supposed to store in the MIB, therefore this function is designed to
 *      be used by the driver itself.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: *addr
 **************************************************************************/
byte *BMAC_GetLongAddr (mac_num)
uint32 mac_num;
{
    return ((byte *) &(bmacs[mac_num].long_addr));
}

/***************************************************************************
 *  Function    BMAC_SetGrpLongAddr
 *
 *  Description
 *      This function sets BMAC group address. It returns OK if group address
 *      is set accordingly. If the address is already in the list, an
 *      ERROR is returned.
 *
 *  Parameters:
 *      uint32 mac_num
 *      byte *addr - the group address to set
 *
 *  Return: [OK | FDDI_ERROR]
 **************************************************************************/
status_type BMAC_SetGrpLongAddr (mac_num,addr)
uint32 mac_num;
byte *addr;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
    byte mr;
    long s;

    s = crit_on();
    mr = r_ptr->mr;
    r_ptr->mr = 0;	    
    if (memcmp(addr,&(bmacs[mac_num].grp_long_addr),5) == 0) {  
      WritePGM(r_ptr,*(addr+5));
    }
    else {    /* reset all the group address */
      memcpy(bmacs[mac_num].grp_long_addr,addr,6);
      r_ptr->gla0 = *addr | 0x80;  /* gla(47) has to be 1 */
      r_ptr->gla1 = *(addr+1);
      r_ptr->gla2 = *(addr+2);
      r_ptr->gla3 = *(addr+3);
      r_ptr->gla4 = *(addr+4);
      WritePGM(r_ptr,*(addr+5));
    }

    r_ptr->mr = mr;    /* restore MR */
    crit_off(s);
    return (OK);
}
 
/***************************************************************************
 *  Function    BMAC_GetBMACMode
 *
 *  Description
 *    This function finds out the current mode of the bmac. MAC is in the 
 *    run mode iff 1. mode reg in RUN mode, 2. h/w ring_op is up, 3.
 *    s/w RingOp is also up.
 *
 *  Parameter
 *    uint32 mac_num
 *
 *  Return: [DRV_START | DRV_STOP]
 **************************************************************************/
int BMAC_GetBMACMode (mac_num)
uint32 mac_num;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
  unsigned char *ptr = bmacs[mac_num].swRingOp;
  volatile byte rop = r_ptr->cts0 & BMAC_CTS0_ROP;

  if (!(r_ptr->mr & BMAC_MR_RUN) || (rop != BMAC_CTS0_ROP) ||
      (*ptr != RM_RING_OP))  {
    return (DRV_STOP);
  }

  return (DRV_START);
}
/***************************************************************************
 *  Function    BMAC_DebugGetBMACMode
 *
 *  Description
 *    This function finds out the current mode of the bmac. MAC is in the 
 *    run mode iff 1. mode reg in RUN mode, 2. h/w ring_op is up, 3.
 *    s/w RingOp is also up.
 *
 *  Parameter
 *    uint32 mac_num
 *
 *  Return: void
 **************************************************************************/
int BMAC_DebugGetBMACMode (mac_num)
uint32 mac_num;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
  unsigned char *ptr = bmacs[mac_num].swRingOp;

#ifndef __FEBRIDGE
  if (!(r_ptr->mr & BMAC_MR_RUN) || !(bmacs[mac_num].ring_op) ||
      (*ptr != RM_RING_OP))  {
    printf("\nbmac %d not in ready state\n",mac_num);
    if (!(r_ptr->mr & BMAC_MR_RUN))
      printf("bmac in stop mode\n");
    if (!(bmacs[mac_num].ring_op))
      printf("bmac h/w ring_op is clear\n");
    if (*ptr != RM_RING_OP)
      printf("bmac s/w ring_op is clear\n");
    return (DRV_STOP);
  }
    printf("\nbmac %d is ready\n",mac_num);
  return (DRV_START);
#else
  if (!(r_ptr->mr & BMAC_MR_RUN) || !(bmacs[mac_num].ring_op) ||
      (*ptr != RM_RING_OP))  {
    dprintf("\nbmac %d not in ready state\n",mac_num);
    if (!(r_ptr->mr & BMAC_MR_RUN))
      dprintf("bmac in stop mode\n");
    if (!(bmacs[mac_num].ring_op))
      dprintf("bmac h/w ring_op is clear\n");
    if (*ptr != RM_RING_OP)
      dprintf("bmac s/w ring_op is clear\n");
    return (DRV_STOP);
  }
    dprintf("\nbmac %d is ready\n",mac_num);
  return (DRV_START);

#endif
}

/***************************************************************************
 *  Function    BMAC_StopBmac
 *
 *  Description
 *      This function sets BMAC to STOP mode. When BMAC is in STOP mode,
 *      it won't generate any interrupt. Therefore it is unnecessary
 *      to mask off the IMR.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 *************************************************************************/
void BMAC_StopBmac (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    r_ptr->mr &= ~BMAC_MR_RUN;
}

/***************************************************************************
 *  Function    BMAC_StartBmac
 *
 *  Description
 *      This function sets BMAC to RUN mode
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 *************************************************************************/
void BMAC_StartBmac (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
#if RINGOP_DEBUG
    printf("set bmac %d to run mode\n",mac_num);
#endif /* RINGOP_DEBUG */

/*    BMAC_EnableInt(mac_num); */
    r_ptr->mr |= BMAC_MR_RUN;
}
 
/*****************************************************************************
 *  Function  BMAC_BeaconRequest
 *
 *  Description
 *      This function sends out a beacon frame from the specified MAC
 *
 *  Parameters:
 *      uint32 mac_num
 *      byte *f_ptr - pointer to beacon frame. only for external frame
 *
 *  Return:     void
 *****************************************************************************/
int BMAC_BeaconRequest (mac_num)
uint32 mac_num;
{
/*    byte reg; */
    volatile BMAC_REG_TYPE *r_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    
    /* generate external beacon frame */
    if (bmacs[mac_num].BeaconType == EXTERNAL_BEACON) { 
      if (DirBcnPtr == NULL)
	return FDDI_ERROR;
      return (BSI_SendDirectBeacon(mac_num,DirBcnPtr));
    }
    else 
      /* force to beacon state and send a beacon frame */
      r_ptr->fr |= BMAC_FR_BCN;

    return OK;
}

/*****************************************************************************
 *  Function  BMAC_ClaimRequest
 *
 *  Description
 *      This function sends out a claim frame from the specified MAC. It
 *      returns FDDI_ERROR if not enter claim state after timeout.
 *
 *  Parameters:
 *      uint32 mac_num
 *
 *  Return: [OK | FDDI_ERROR]
 *****************************************************************************/
status_type BMAC_ClaimRequest (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    long s;

    s = crit_on();
    r_ptr = bmacs[mac_num].reg_base;
    r_ptr->fr &= ~BMAC_FR_MCRST;
    r_ptr->fr |= BMAC_FR_CLM;
#if CLAIM_DEBUG
	printf("mac %d sends claim\n",mac_num);
#endif

    crit_off(s);
    return (OK);
}

/***************************************************************************
 *  Function    BMAC_EnableParity
 *
 *  Description
 *      This function enables the BMAC parity checking.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ***************************************************************************/
void BMAC_EnableParity (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    
    r_ptr = bmacs[mac_num].reg_base;

    /* set up Exception Mask Register */
    r_ptr->emr |= (BMAC_EMR_PPE | BMAC_EMR_CPE);

    /* enable the corresponding bits in the MODE register */
    r_ptr->mr |= (BMAC_MR_CBP | BMAC_MR_MRP | BMAC_MR_PIP);
}

/***************************************************************************
 *  Function    BMAC_DisableParity
 *
 *  Description
 *      This function disables the BMAC parity checking.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ***************************************************************************/
void BMAC_DisableParity (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    
    r_ptr = bmacs[mac_num].reg_base;

    /* mask off Exception Mask Register */
    r_ptr->emr &= (~BMAC_EMR_PPE & ~BMAC_EMR_CPE);
    r_ptr->mr &= (~BMAC_MR_CBP & ~BMAC_MR_MRP & ~BMAC_MR_PIP);
}

/***************************************************************************
 *  Function    BMAC_DisableInt
 *
 *  Description
 *      This function disables the BMAC interrupts. The image pointer
 *      is not changed.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ***************************************************************************/
void BMAC_DisableInt (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
#if  CLAIM_DEBUG
  printf("BMAC_DisableInt()\n");
#endif
  r_ptr->imr = 0;
}

 
/***************************************************************************
 *  Function    BMAC_EnableInt
 *
 *  Description
 *      This function enables the BMAC interrupts by reload the image
 *      pointer.
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr
 *      BMAC_CTRL_TYPE *i_ptr;
 *
 *  Return: void
 ***************************************************************************/
void BMAC_EnableInt (mac_num)
uint32 mac_num;
{
  volatile BMAC_REG_TYPE *r_ptr;
  BMAC_CTRL_TYPE *i_ptr;

  r_ptr = bmacs[mac_num].reg_base;
  i_ptr = bmacs[mac_num].image_ptr;
  /* clear all the latch registers */

  r_ptr->relr0 = 0;
  r_ptr->relr1 = 0;
  r_ptr->telr0 = 0;
  r_ptr->colr = 0;
  r_ptr->ielr = 0;
  r_ptr->imr = i_ptr->imr;
#if RINGOP_DEBUG
   printf("enable mac %d interrupt, mr %x\n",mac_num,r_ptr->mr);
#endif
}






/***************************************************************************
 *  Function    BMAC_DisableEvents
 *
 *  Description
 *      This function turns off some events which RMT doesn't want
 *      to see it. They are: my_beacon, my_claim, T4T5, other_beacon.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ***************************************************************************/
void BMAC_DisableEvents (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

    /* disable REMR1 because RMT doesn't need beacon/claim interrupt */
    r_ptr->remr1 = 0;

    /* disable TEMR0.CBERR for T4T5 */
    r_ptr->temr0 &= ~BMAC_TEMR0_CBERR;
}

 
/***************************************************************************
 *  Function    BMAC_EnableInt
 *
 *  Description
 *      This function enables the BMAC events latch by reload the image
 *      pointer.
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr
 *      BMAC_CTRL_TYPE *i_ptr;
 *
 *  Return: void
 ***************************************************************************/
void BMAC_EnableEvents (mac_num)
uint32 mac_num;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
  BMAC_DEFAULT_TABLE *tbl_ptr;
 
  if (fddi_board.type == FDDI_BRIDGE)
    tbl_ptr = &bmacbr_default_table;
  else /*  if (fddi_board.type == FDDICON_MASTER) */
    tbl_ptr = &bmaccon_default_table;
 
  r_ptr->relr1 = 0;
  r_ptr->telr0 = 0;
  r_ptr->remr1 = tbl_ptr->remr1;
  r_ptr->temr0 |= BMAC_TEMR0_CBERR;
}

/*****************************************************************************
 *  Functiion   BMAC_TestClaim
 *
 *  Description
 *      This function sets BMAC_FR_CLM bit to issue a claim frame, waiting
 *      'msecs' for the BMAC to enter the claim/beacon state, and then
 *      waiting for the BMAC_RELR0_ROP bit to be set. If the RingOp is never
 *      set, the test is failed and FALSE will be returned. Note that this
 *      function will not reschudle, so the calling function can not put
 *      msec too big. Also NOTE that the calling routine is responsible for
 *      setup the test path and the corresponding registers it wants to test.
 *
 *  Parameters
 *      uint32 mac_num
 *      uint32  msecs - mini seconds time out
 *
 *  Return: [OK | FDDI_ERROR]
 *****************************************************************************/
status_type BMAC_TestClaim (mac_num,msecs)
uint32 mac_num;
uint32 msecs;
{
    return (ClaimTest(mac_num,msecs,FALSE));
}

/***************************************************************************
 *  Function BMAC_TestBeacon
 *
 *  Description
 *      This function does the self beacon test.
 *
 *  Parameters:
 *      uint32 mac_num
 *      uint32  msecs - mini seconds time out
 *
 *  Return: [OK | FDDI_ERROR]
 *****************************************************************************/
status_type BMAC_TestBeacon (mac_num,msecs)
uint32 mac_num;
uint32 msecs;
{
    return (BeaconTest(mac_num,msecs,FALSE));
}

/***************************************************************************
 *  Function    BMAC_InternalLBTest
 *
 *  Description
 *      This function does the BMAC internal loopback test.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: [OK | FDDI_ERROR]
 ***************************************************************************/
status_type BMAC_InternalLBTest (mac_num)
uint32 mac_num;
{
    if (BeaconTest(mac_num,DRV_10MSEC,TRUE) != OK)
        return (FDDI_ERROR);

    return (ClaimTest(mac_num,DRV_10MSEC,TRUE));
}

/***************************************************************************
 *  Function    BMAC_SetBeaconType
 *
 *  Description
 *      This function set the way of generating beacon frame. The beacon
 *      frame can be generated by BMAC itself or by put a beacon frame
 *      to the BSI request channel.
 *
 *  Parameter:
 *      uint32 mac_num
 *      uint32 beacon_type - [SM_MA_CTRL_EXTER_BEACON | SM_MA_INTER_BEACON]
 *      
 *  Return:  void
 **************************************************************************/
void BMAC_SetBeaconType (mac_num,beacon_type, pkt_ptr)
uint32 mac_num;
uint32 beacon_type;
byte *pkt_ptr;
{
/*  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;  */

  if (beacon_type == SM_MA_CTRL_EXTER_BEACON) {
    DirBcnPtr = pkt_ptr;
    bmacs[mac_num].BeaconType = EXTERNAL_BEACON;
    
  }
  else {
    bmacs[mac_num].BeaconType = INTERNAL_BEACON;
    if (bsis[mac_num].direct_beacon) {
      bsis[mac_num].direct_beacon = FALSE;
      BSI_StopDirectBeacon(mac_num);
    }
  }
}

/***************************************************************************
 *  Function  BMAC_ConfigRecovery
 *
 *  Description
 *   This function configures the bmac with regards to the automatic 
 *   recovery ability.
 *
 *  Parameter:
 *   uint32 bmac_num
 *   act 
 *
 *  Return: void
 **************************************************************************/
void BMAC_ConfigRecovery (mac_num,act)
uint32 mac_num, act;
{
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

  if (act == BMAC_INHIBIT_RECOVERY)
    r_ptr->or |= BMAC_OR_IRR;
  else
    r_ptr->or &= ~BMAC_OR_IRR;
}
 
/***************************************************************************
 *  Function BMAC_GetMacCounters
 *
 *  Description:
 *    This function returns the mac counters MIBs.
 *
 *  Parameters:
 *    uint32 mac_no
 *    uint32 type
 *
 *  Return: uint32 - mib value
 **************************************************************************/
uint32 BMAC_GetMacCounters (mac_num,type)
uint32 mac_num;
uint32 type;
{
  volatile register BMAC_REG_TYPE *r_ptr;
  uint32 reg_val, counts;
  DRV_FDDI_MIBS *m_ptr;
  long s;
#if 0
  printf("\n.....GetMacCounts(), mac %d, type %d\n",mac_num,type);
#endif
  s = crit_on();
  r_ptr = bmacs[mac_num].reg_base;
  m_ptr = &(init_mibs.d_mib[mac_num]);

  switch(type) {
  case SM_MA_STATUS_FRAME_CT:
    reg_val = ReadFRCT(r_ptr);
#if COUNT_DEBUG
    printf("frame cnt, reg %x, mib %d, int %d\n",reg_val,*(m_ptr->macFrameRxCount),rx_cnt[mac_num]);
#endif
    counts = reg_val + *(m_ptr->macFrameRxCount);
    break;

  case SM_MA_STATUS_COPIED_CT:
    reg_val = ReadFCCT(r_ptr);
#if COUNT_DEBUG
    printf("copied cnt, reg %x, mib %d, %x, int %d\n",reg_val,*(m_ptr->macFrameCopiedCount),m_ptr->macFrameCopiedCount,cp_cnt[mac_num]);
#endif
    counts = reg_val + *(m_ptr->macFrameCopiedCount);
    break;

  case SM_MA_STATUS_ERROR_CT:
    reg_val = ReadEICT(r_ptr);
#if COUNT_DEBUG
    printf("mac err count, reg %x, mib %d, int %d\n",reg_val,*(m_ptr->macErrIsolateCount),err_cnt[mac_num]);
#endif
    counts = reg_val + *(m_ptr->macErrIsolateCount);
    break;

  case SM_MA_STATUS_LOST_CT:
    reg_val = ReadLFCT(r_ptr);
    counts = reg_val + *(m_ptr->macLostFrameCount);
    break;

  case SM_MA_STATUS_NOT_COPIED_CT:
    reg_val = ReadFNCT(r_ptr);
    counts = reg_val + *(m_ptr->macFrameNotCopiedCount);
    break;

  case SM_MA_STATUS_LATE_CT:
    /* everytime when Late Counter is incremented, the mib is updated */
    counts =  *(m_ptr->macLateCount);
    break;

  case SM_MA_STATUS_TX_CT:
    reg_val = ReadFTCT(r_ptr);
#if COUNT_DEBUG
    printf("tx cnt, reg %x, mib %d\n",reg_val,*(m_ptr->macFrameTxCount));
#endif
    counts = reg_val + *(m_ptr->macFrameTxCount);
    break;

  case SM_MA_STATUS_TOKEN_CT:
    reg_val = ReadTKCT(r_ptr);
#if COUNT_DEBUG
    printf("token cnt, reg %x, mib %d\n",reg_val,*(m_ptr->macTokenRxCount));
#endif
    counts = reg_val + *(m_ptr->macTokenRxCount);
    break;

  case SM_MA_STATUS_TVX_EXPIRED_CT:
    counts =  *(m_ptr->macTVXExpiredCount);
    break;

  default:
    counts = 0;
	break;
  }   /* switch */

  crit_off(s);
  return (counts);
}
 
/***************************************************************************
 *  Function    BMAC_SetFixGrpAddr
 *
 *  Description
 *      This function sets the fix group address to FGM0 & FGM1. It assumes
 *      that the first 44 bits are 1.
 *
 *  Parameter:
 *      uint32 mac_num
 *      byte val
 *
 *  Return: [OK | FDDI_ERROR]
 ***************************************************************************/
status_type BMAC_SetFixGrpAddr (mac_num,val)
uint32  mac_num;
byte val;
{
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
/*    BMAC_CTRL_TYPE *i_ptr;  */
    byte mr, index = 1;
    long s;

    s = crit_on();
    mr = r_ptr->mr;
    r_ptr->mr = 0;
    val = val & BMAC_FGM_MASK;
    if (val > 7) {
        val -= 8;
        index = index << val;
        r_ptr->fgm1 |= index;
    }
    else {
        index = index << val;
        r_ptr->fgm0 |= index;
    }

    r_ptr->mr = mr;   /* restore BMAC */
    crit_off(s);
    
    return (OK);
}

/***************************************************************************
 *  Function BMAC_ShowMacCounts
 *
 *  Description
 *    This function shows the mac releated counts
 *
 *  Parameter:
 *   uint32 mac_num
 *
 *  Return: void
 *************************************************************************/
void BMAC_ShowMacCounts (mac_num)
uint32 mac_num;
{
/*  DRV_FDDI_MIBS *m_ptr = &(init_mibs.d_mib[mac_num]);  */
  volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;

  if (mac_num > 1) {
    printf("mac num %d wrong\n",mac_num);
    return;
  }

  printf("FRCT %d, mib %d\n",ReadFRCT(r_ptr),*(init_mibs.d_mib[mac_num].macFrameRxCount));
  printf("FCCT %d, mib %d, %x\n",ReadFCCT(r_ptr),*(init_mibs.d_mib[mac_num].macFrameCopiedCount),init_mibs.d_mib[mac_num].macFrameCopiedCount);
  printf("EICT %d, mib %d\n",ReadEICT(r_ptr),*(init_mibs.d_mib[mac_num].macErrIsolateCount));
  printf("LFCT %d, mib %d\n",ReadLFCT(r_ptr),*(init_mibs.d_mib[mac_num].macLostFrameCount));
  printf("FNCT %d, mib %d\n",ReadFNCT(r_ptr),*(init_mibs.d_mib[mac_num].macFrameNotCopiedCount));
  printf("LTCT %d, mib %d\n",ReadLTCT(r_ptr),*(init_mibs.d_mib[mac_num].macLateCount));
  printf("FTCT %d, mib %d\n",ReadFTCT(r_ptr),*(init_mibs.d_mib[mac_num].macFrameTxCount));


  PrintIntCounts(mac_num);
}

/*
 *-------------------------------------------------------------------------
 *              Local Functions
 *-------------------------------------------------------------------------
 */
 
/***************************************************************************
 *  Function    ResetBmacRegs
 *
 *  Description
 *      This function sets BMAC registers. Note that THE ORDER IS VERY
 *      IMPORTANT.
 *
 *  Parameter:
 *      BMAC_TYPE *mac_ptr;
 *      source - 1 from application, 0 from booter
 *
 *  Return: [OK | FDDI_ERROR]
 *************************************************************************/
static void ResetBmacRegs (mac_num,source)
uint32 mac_num;
int source;
{
    volatile BMAC_REG_TYPE *r_ptr;
    BMAC_CTRL_TYPE *i_ptr;
    DRV_FDDI_MIBS *m_ptr;
    BMAC_DEFAULT_TABLE *tbl_ptr;

    if (fddi_board.type == FDDI_BRIDGE)
        tbl_ptr = &bmacbr_default_table;
    else  /*  if (fddi_board.type == FDDICON_MASTER)  */
        tbl_ptr = &bmaccon_default_table;
 
    r_ptr = bmacs[mac_num].reg_base;
    m_ptr = bmacs[mac_num].mib_ptr;
    i_ptr = bmacs[mac_num].image_ptr;
    memset(i_ptr,0,sizeof(BMAC_CTRL_TYPE));
    
    /*---- set up Function Register ----*/
    r_ptr->fr = tbl_ptr->fr;
    r_ptr->mr = 0;   /* stop bmac */

    /*---
        load the MAC parameter RAM, MAC address is done before this function
    ---*/
#if 0
    ResetPGM(mac_num);
#endif  /* if 0 */
    DisableGrpAddr(mac_num,BMAC_LGROUP);
    DisableGrpAddr(mac_num,BMAC_SGROUP);
    BMAC_SetTreq(mac_num,*(init_mibs.d_mib[mac_num].macTREQ));   /* setup TREQs */
    r_ptr->thsh1 = 0x0c;
    SetBeaconFrame(r_ptr);             /* set up TBT registers. */
    /* fgm registers will not be cleared during reset, so we have to
       clear it manually at initialization */
    r_ptr->fgm0 = 0;
    r_ptr->fgm1 = 0;
    BMAC_SetFixGrpAddr(mac_num,0x0f);       /* setup broadcast address */
#if 0
    /* set up proup address for bootp */
    if (mac_num == 0)
      BMAC_SetGrpLongAddr(mac_num,(byte *)MyNid(1));
    else
      BMAC_SetGrpLongAddr(mac_num,(byte *)MyNid(0));
#endif
    /*---- clear all the event counters ----*/
    BMAC_ClearAllMacCounters(mac_num);
#if COUNT_DEBUG
    printf("frct %d\n",ReadFRCT(r_ptr));
    printf("eict %d\n",ReadEICT(r_ptr));
    printf("lfct %d\n",ReadLFCT(r_ptr));
    printf("fcct %d\n",ReadFCCT(r_ptr));
    printf("fnct %d\n",ReadFNCT(r_ptr));
    printf("ftct %d\n",ReadFTCT(r_ptr));
    printf("tkct %d\n",ReadTKCT(r_ptr));
    printf("rlct %d\n",ReadRLCT(r_ptr));
    printf("ltct %d\n",ReadLTCT(r_ptr));
#endif
    /*---- clear all the latch registers -----*/
    ClearRelr0(r_ptr);          /* clear events latch register /    
    ClearRelr1(r_ptr);
    ClearTELR0(r_ptr);
    ClearCILR(r_ptr);    
    ClearCOLR(r_ptr);    
    ClearIELR(r_ptr);
    ClearESR(r_ptr);
    
    /*----
        set up ring event registers. 'beacon frame received', 'claim frame
        received' are read from RELR1. 'other mac frame received' is not
        processed by SMT. 'Duplicate address' will be polled. In relr1,
        'higher/lower claims are not processed by SMT.
    ----*/
    r_ptr->remr0 = tbl_ptr->remr0;

    r_ptr->remr1 = tbl_ptr->remr1;

    /*---- setup token and timer event register 0 ----*/
    r_ptr->temr0 = tbl_ptr->temr0;

    /***
        setup Counter Increment Mask Register. Always 0. We don't use it.
    ***/
    r_ptr->cimr = tbl_ptr->cimr;

    /***
        setup Counter Overflow Mask Register
    ***/
    if (source == DRV_FROM_BOOT)
      r_ptr->comr = 0;
    else
      r_ptr->comr = tbl_ptr->comr;

    /***
        set up Exception Mask Register: clear this register
    ***/
    r_ptr->emr = tbl_ptr->emr;

    /***
        set up Interrupt Mask Register
    ***/
    if (source == DRV_FROM_APP)
      i_ptr->imr = tbl_ptr->imr;
    else
      i_ptr->imr = 0x43;
    r_ptr->imr = i_ptr->imr;

    /*------ set the timer threshold register --------*/
    BMAC_SetTimerRegs(mac_num,init_mibs.s_mib[mac_num].tmax,BMAC_TMR_TMAX);
    BMAC_SetTimerRegs(mac_num,init_mibs.s_mib[mac_num].tvx,BMAC_TMR_TVX);

    /*---- set up Option Register ----*/
    r_ptr->or = tbl_ptr->or;

    /*---- set up Mode Register ----*/
    r_ptr->mr = tbl_ptr->mr;

    r_ptr->fr = 0;
}


/****************************************************************************
 *  Function ClearRelr0
 *
 *  Description
 *      This function clears Ring Event and Latch Register.
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearRelr0 (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte val, i=0;
    
    do {
        val = r_ptr->relr0;
        r_ptr->relr0 = 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (i++ < CWI_REPEAT_MAX));
}


/****************************************************************************
 *  Function ClearRelr1
 *
 *  Description
 *      This function clears Ring Event and Latch Register.
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearRelr1 (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
/*    byte val;  */
    register int i=0;

    do {
/*        val = r_ptr->relr1;  */
        r_ptr->relr1 &= 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/****************************************************************************
 *  Function ClearTELR0
 *
 *  Description
 *      This function clears Token and Timer Event and Latch Register.
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearTELR0 (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte val;
    register int i=0;

    do {
        val = r_ptr->telr0;
        r_ptr->telr0 = 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}
 
/****************************************************************************
 *  Function ClearCILR
 *
 *  Description
 *      This function clears Counter Increment Latch Register
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearCILR (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte val;
    register int i=0;

    do {
        val = r_ptr->cilr;
        r_ptr->cilr = 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/****************************************************************************
 *  Function ClearCOLR
 *
 *  Description
 *      This function clears Counter Overflow Latch Register
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearCOLR (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte val;
    register int i=0;

    do {
        val = r_ptr->colr;
        r_ptr->colr = 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/****************************************************************************
 *  Function ClearIELR
 *
 *  Description
 *      This function clears Internal Event Latch Register
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearIELR (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte val;
    register int i=0;

    do {
        val = r_ptr->ielr;
        r_ptr->ielr = 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/****************************************************************************
 *  Function ClearESR
 *
 *  Description
 *      This function clears Exception Status Register
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: void
 ****************************************************************************/
static void ClearESR (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte val;
    register int i=0;

    do {
        val = r_ptr->esr;
        r_ptr->esr = 0;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}
 

/****************************************************************************
 *  Function    ResetMacAddrs
 *
 *  Description
 *      This function rewrite the addressess (long and short) to the
 *      MLA and MSA.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ****************************************************************************/
static void ResetMacAddrs (mac_num)
uint32 mac_num;
{
    byte *addr;
    byte *fddi_nid;
    
    /* set up MAC long address and short address */
    addr = init_mibs.s_mib[mac_num].mac_addr;
    BMAC_SetLongAddr(mac_num,addr);
    fddi_nid = (byte *) &(MyFDDINid[mac_num]);
    memcpy(fddi_nid,addr,6);

#if 0
    /* not support short mac address */   
    addr = (byte *)GetMacShortAddr(mac_num);
    BMAC_SetShortAddr(&bmacs[mac_num],addr);
#endif    
}


/****************************************************************************
 *  Function    BeaconTest
 *
 *  Description
 *      This function force BMAC to beacon state and verify that the BMAC
 *      receives beacon frame. The calling function should disable the MAC
 *      interrupt.
 *
 *  Parameter:
 *      uint32 mac_num
 *      bool loopback - [TRUE | FALSE], true if it is for loopback test.
 *
 *  Return: [OK | FDDI_ERROR]
 ****************************************************************************/
static status_type BeaconTest (mac_num,wait_time,loopback)
uint32 mac_num;
uint32 wait_time;
bool loopback;
{
    byte val, mr;
    uint32 time_out, retn_code = OK;
    volatile BMAC_REG_TYPE *r_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    mr = r_ptr->mr;

    BMAC_StopBmac(mac_num);

    if (loopback)
      r_ptr->mr = BMAC_MR_ILB  | BMAC_MR_RUN;
    else
      r_ptr->mr = BMAC_MR_RUN;

    ClearRelr1(r_ptr);          /* clear events */
    time_out = RealTimeTicks() + wait_time;
    r_ptr->fr = BMAC_FR_BCN;

    while (TRUE) {
        val = r_ptr->relr1;
        if (val & BMAC_RELR1_MYBCN)
            break;
        if (time_out < RealTimeTicks()) {
            retn_code = FDDI_ERROR;
	    break;
	}
    }       

    ClearRelr1(r_ptr);          /* clear events */
    r_ptr->fr = BMAC_FR_MCRST;  /* reset receiver and transmitter */
    r_ptr->mr = mr;

    return (retn_code);
}


/****************************************************************************
 *  Function    ClaimTest
 *
 *  Description
 *      This function force BMAC to claim state and verify that the BMAC
 *      receives claim frame. It assumes that the BMAC is in stop mode.
 *
 *  Parameter:
 *      uint32 mac_num
 *      uint32 wait_time - # of msec waiting for entering claim state
 *      bool loopback - [TRUE | FALSE], true if it is for loopback test.
 *
 *  Return: [OK | FDDI_ERROR]
 ****************************************************************************/
static status_type ClaimTest (mac_num,wait_time,loopback)
uint32 mac_num;
uint32 wait_time;
bool loopback;
{
    byte val,mr;
    uint32 time_out, r_code=OK;
    volatile BMAC_REG_TYPE *r_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    mr = r_ptr->mr;

    BMAC_StopBmac(mac_num);

    if (loopback) {
        r_ptr->mr = BMAC_MR_ILB  | BMAC_MR_RUN;
    }
    else
      r_ptr->mr = BMAC_MR_RUN;
        
    ClearRelr1(r_ptr);          /* clear events latch registers */
    time_out = RealTimeTicks() + wait_time;
    r_ptr->fr = BMAC_FR_CLM;      /* force entering claim state */

    while (TRUE) {
        val = r_ptr->relr1;
        if (val & BMAC_RELR1_MYCLM)      /* receive claim frame */ 
            break;
        if (time_out < RealTimeTicks()) {
            r_code = FDDI_ERROR;
	    break;
	}
    }       

    ClearRelr1(r_ptr);          /* clear events latch registers */
    r_ptr->fr = BMAC_FR_MCRST;
    r_ptr->mr = mr;

    return (r_code);
}

/****************************************************************************
 *  Function    SetBeaconFrame
 *
 *  Description
 *      This function sets Transmit Beacon Type registers. According to
 *      the FDDI standard, the beacon frame info field should be at least
 *      4 octets in lenfth. We currently sets all 4 octets to 0. NOTE
 *      that the calling function has to STOP the BMAC before calling this
 *      function.
 *
 *  Parameters:
 *      uint32  mac_num
 *
 *  Return: [OK | FDDI_ERROR]
 ****************************************************************************/
static status_type SetBeaconFrame (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
    byte mr;

    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = 0;

    r_ptr->tbt0 = r_ptr->tbt1 = r_ptr->tbt2 = r_ptr->tbt3 = 0;

    r_ptr->mr = mr;     /* restore mr */
    return (OK);
}

/*************************************************************************
 *  Function    MacIsRun
 *
 *  Description
 *      This function checks whether mac is in RUN mode. It returns TRUE
 *      if it is in RUN mode.
 *
 *  Parameter:
 *      BMAC_REG_TYPE *r_ptr;
 *
 *  Return: [TRUE | FALSE]
 ***************************************************************************/   
static bool MacIsRun (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  if ((r_ptr->mr & BMAC_MR_RUN) == 0)
    return (FALSE);

  return (TRUE);
}

/***************************************************************************
 *  Function    ClearFRCT
 *
 *  Description
 *      This function clears Frame Receive Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: void
 ***************************************************************************/
static void ClearFRCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = 0;
    *m_ptr = 0;    
    r_ptr->frct1 = 0;
    r_ptr->frct2 = 0;
    r_ptr->frct3 = 0;

    r_ptr->mr = mr;     /* restore mr */
}

/***************************************************************************
 *  Function    ClearEICT
 *
 *  Description
 *      This function clears Error Isolate Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearEICT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;

    *m_ptr = 0;    
    r_ptr->eict1 = 0;
    r_ptr->eict2 = 0;
    r_ptr->eict3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}

/***************************************************************************
 *  Function    ClearLFCT
 *
 *  Description
 *      This function clears Lost Frame Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearLFCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;

    *m_ptr = 0;    
    r_ptr->lfct1 = 0;
    r_ptr->lfct2 = 0;
    r_ptr->lfct3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}

/***************************************************************************
 *  Function    ClearFCCT
 *
 *  Description
 *      This function clears Frame Copied Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearFCCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;
    *m_ptr = 0;    
    
    r_ptr->fcct1 = 0;
    r_ptr->fcct2 = 0;
    r_ptr->fcct3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}


/***************************************************************************
 *  Function    ClearFNCT
 *
 *  Description
 *      This function clears Frame Not Copied Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearFNCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;

    *m_ptr = 0;    
    r_ptr->fnct1 = 0;
    r_ptr->fnct2 = 0;
    r_ptr->fnct3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}


/***************************************************************************
 *  Function    ClearFTCT
 *
 *  Description
 *      This function clears Frame Transmitted Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearFTCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;

    *m_ptr = 0;    
    r_ptr->ftct1 = 0;
    r_ptr->ftct2 = 0;
    r_ptr->ftct3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}



/***************************************************************************
 *  Function    ClearTKCT
 *
 *  Description
 *      This function clears Token Rx Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearTKCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;

    *m_ptr = 0;    
    r_ptr->tkct1 = 0;
    r_ptr->tkct2 = 0;
    r_ptr->tkct3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}


/***************************************************************************
 *  Function    ClearRLCT
 *
 *  Description
 *      This function clears Ring Latency Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearRLCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint32 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;
    *m_ptr = 0;    
    r_ptr->rlct1 = 0;
    r_ptr->rlct2 = 0;
    r_ptr->rlct3 = 0;
    r_ptr->mr = mr;     /* restore mr */
}


/***************************************************************************
 *  Function    ClearLTCT
 *
 *  Description
 *      This function clears Late Counter
 *
 *  Parameters:
 *      BMAC_REG_TYPE   *r_ptr
 *      uint32   *m_ptr;
 *
 *  Return: void
 ***************************************************************************/
static void ClearLTCT (r_ptr,m_ptr)
volatile BMAC_REG_TYPE *r_ptr;
uint16 *m_ptr;
{
    byte mr;
    
    /* make sure that the BMAC is in stop mode */
    mr = r_ptr->mr;
    r_ptr->mr = mr & ~BMAC_MR_RUN;

    *m_ptr = 0;    
    r_ptr->ltct = 0;
    r_ptr->mr = mr;     /* restore mr */
}


/***************************************************************************
 *  Function    GetMacDefault
 *
 *  Description
 *      This function copies default value from bmacs_master_default[] table.
 *
 *  Parameter
 *      uint32 mac_num
 *
 *  Re[220zturn: void
 ***************************************************************************/
static void GetMacDefault (mac_num)
uint32 mac_num;
{
    bmacs[mac_num].reg_base = bmacs_master_default[mac_num].reg_base; 
    bmacs[mac_num].BeaconType = bmacs_master_default[mac_num].BeaconType;
    bmacs[mac_num].mac_num = bmacs_master_default[mac_num].mac_num;
    bmacs[mac_num].status = bmacs_master_default[mac_num].status;
    bmacs[mac_num].mib_ptr = &(init_mibs.d_mib[mac_num]);
    bmacs[mac_num].swRingOp = init_mibs.s_mib[mac_num].swRingOp;
    bmacs[mac_num].ringop_set = bmacs[mac_num].ringop_clr = 0;
}

/***************************************************************************
 *  Function    DisableGrpAddr
 *
 *  Description
 *      This function disable group addressing by setting GLAs/GSAs to all 1s
 *
 *  Parameter:
 *      uint32 mac_num
 *      uint32 type - [BMAC_LGROUP | BMAC_SGROUP]
 *
 *  Return: void
 **************************************************************************/
static void DisableGrpAddr (mac_num,type)
uint32 mac_num;
uint32 type;
{
    byte reg;
    volatile BMAC_REG_TYPE *r_ptr;
    byte *start_ptr, *end_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    reg = r_ptr->mr;
    r_ptr->mr = reg & ~BMAC_MR_RUN;     /* stop BMAC */
    if (type == BMAC_LGROUP)
        r_ptr->gla0 = r_ptr->gla1=r_ptr->gla2=r_ptr->gla3 = r_ptr->gla4 = 0xff;
    else
        r_ptr->gsa0 = 0xff;

    /* clear all the PGMx */
    start_ptr = (byte *) &(r_ptr->pgm10);
    end_ptr = (byte *) &(r_ptr->pgmf);
    for (;start_ptr <= end_ptr; start_ptr++)
         *start_ptr = 0;

    r_ptr->mr = reg;        /* restore MR */
}


/****************************************************************************
 *  Function    GetTimerRegMask
 *
 *  Description
 *      This function converts a 80 ns unit 2's complement number to the
 *      special mask for the TMAX and TVX registers.
 *
 *  Parameter
 *      uint32 val - 2's complement number in 80 ns unit
 *
 *  Return: mask to write to TMAX and TVX registers
 ****************************************************************************/
uint32 timemask[16] = {4096,8192,16384,32768,65536,131070,262140,524290,1048600,2097200,4194300,8388600,16777000,33554000,67109000,134220000};
static byte GetTimerRegMask (val)
uint32 val;
{
    uInt16 i, k;
    uInt32 j;
    
    val = ((MFddi2Usecs(-val)) * 100);

    if (val <= timemask[0])
      return 0;
 
    for (i=1; i < 16; i++)       { /* jlin: 05/06/93: fix the loop so that 
				     it will return the right value */       

      if (val > timemask[i])
      {
	 continue;
      }

      if (val < (timemask[i] - timemask[i-1])/ 2 + timemask[i-1])
      {
	return (i-1);
      }
      else
	return i;
    }

    return i; 
}

/***********************************************************************
 *  Function WritePGM
 *
 *  Description
 *      This function sets the corresponding bit for PGMs
 *
 *  Parameters:
 *    byte *ptr - pointer to PGM0, the base address for PGMs
 *    byte addr - the 6th byte of the address
 *
 *  Return:  void
 **********************************************************************/
static void WritePGM (r_ptr,addr)
volatile BMAC_REG_TYPE *r_ptr;
byte addr;
{
    uint i;
    byte pgm = 1;
    volatile byte *ptr;

    if (addr > 0x7f) {
        ptr = &(r_ptr->pgm10);
        addr &= 0x7f;    /* mask off the msb */
    }
    else
        ptr = &(r_ptr->pgm0);

    ptr += (addr >> 3);
    i = addr % 8;
    *ptr |= (pgm << i);
}


/***********************************************************************
 *  Function ReadPGM
 *
 *  Description
 *      This function reads the corresponding bit for PGMs. It returns
 *      TRUE if the corresponding bit is set, FALSE otherwise.
 *
 *  Parameters:
 *    byte *ptr - pointer to PGM0, the base address for PGMs
 *    byte addr - the 6th byte of the address
 *
 *  Return:  [TRUE | FALSE]
 **********************************************************************/
static bool ReadPGM (ptr,addr)
volatile byte *ptr;
byte addr;
{
    byte val, mask=1;

    ptr += (addr >> 3);
    mask = mask << (addr % 8);
    val = *ptr;
    return (val & mask);
}

/***********************************************************************
 *  Function  ResetPGM
 *
 *  Description
 *      This function resets all the PGMs to 0. The PGMs won't be 
 *      cleared during reset.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 **********************************************************************/
static void ResetPGM (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    byte *start_ptr, *end_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    start_ptr = (byte *) &(r_ptr->pgm10);
    end_ptr = (byte *) &(r_ptr->pgmf);
    for (;start_ptr <= end_ptr; start_ptr++)
         *start_ptr = 0;
}

/***************************************************************************
 *  Function ReadFRCT
 *
 *  Description
 *    THis function reads FRCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in FRCT
 ***************************************************************************/
static uint32 ReadFRCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  register uint32 frct, tmp = 0;
#if 0
  uint32 t1, t2, t3, t0;
  t3 = r_ptr->frct3;
  t2 = r_ptr->frct2;
  t1 = r_ptr->frct1;
  t0 = r_ptr->frct0;
  printf("frct0: %x, %x, %x, %x\n",t0, t1, t2, t3);

  t0 = r_ptr->frct0;
  t1 = r_ptr->frct1;
  t2 = r_ptr->frct2;
  t3 = r_ptr->frct3;
  printf("frct0: %x, %x, %x, %x\n",t0, t1, t2, t3);
#endif

  frct = r_ptr->frct3 & 0xff;
  tmp = ((uInt32) r_ptr->frct2) << 8;
  frct |= tmp;
  tmp = ((uInt32) r_ptr->frct1) << 16;
  frct |= tmp;

  return (frct);
}

/***************************************************************************
 *  Function ReadEICT
 *
 *  Description
 *    THis function reads EICT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in EICT
 ***************************************************************************/
static uint32 ReadEICT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 eict = 0, tmp = 0;
#if 0
  uint32 t1, t2, t3, t0;
  t3 = r_ptr->eict3;
  t2 = r_ptr->eict2;
  t1 = r_ptr->eict1;
  t0 = r_ptr->eict0;
  printf("eict0: %x, %x, %x, %x\n",t0, t1, t2, t3);
#endif

  eict = r_ptr->eict3 & 0xff;
  tmp = ((uint32) r_ptr->eict2) << 8;
  eict |= tmp;
  tmp = ((uint32) r_ptr->eict1) << 16;
  eict |= tmp;

  return (eict);
}

/***************************************************************************
 *  Function ReadLFCT
 *
 *  Description
 *    THis function reads LFCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in LFCT
 ***************************************************************************/
static uint32 ReadLFCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 lfct = 0, tmp;

  lfct = r_ptr->lfct3 & 0xff;
  tmp = ((uInt32) r_ptr->lfct2) << 8;
  lfct |= tmp;
  tmp = ((uInt32) r_ptr->lfct1) << 16;
  lfct |= tmp;
 
  return (lfct);
}

/***************************************************************************
 *  Function ReadFCCT
 *
 *  Description
 *    THis function reads FCCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in FCCT
 ***************************************************************************/
static uint32 ReadFCCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 fcct = 0, tmp = 0;

  fcct = r_ptr->fcct3 & 0xff;
  tmp = ((uInt32) r_ptr->fcct2) << 8;
  fcct |= tmp;
  tmp = ((uInt32) r_ptr->fcct1) << 16;
  fcct |= tmp;
#if 0
  printf("fcct %x\n",fcct);
#endif
  return (fcct);
}

/***************************************************************************
 *  Function ReadFNCT
 *
 *  Description
 *    THis function reads FNCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in FNCT
 ***************************************************************************/
static uint32 ReadFNCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 fnct = 0, tmp = 0;

  fnct = r_ptr->fnct3 & 0xff;
  tmp = ((uInt32) r_ptr->fnct2) << 8;
  fnct |= tmp;
  tmp = ((uInt32) r_ptr->fnct1) << 16;
  fnct |= tmp;

  return (fnct);
}

/***************************************************************************
 *  Function ReadFTCT
 *
 *  Description
 *    THis function reads FTCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in FTCT
 ***************************************************************************/
static uint32 ReadFTCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 ftct = 0, tmp = 0;

  ftct = (r_ptr->ftct3 & 0xff);
  tmp= ((uInt32) r_ptr->ftct2) << 8;
  ftct |= tmp;
  tmp = ((uInt32) r_ptr->ftct1) << 16;
  ftct |= tmp;

  return (ftct);
}

/***************************************************************************
 *  Function ReadTKCT
 *
 *  Description
 *    THis function reads TKCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in TKCT
 ***************************************************************************/
static uint32 ReadTKCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 tkct = 0, tmp = 0;
#if 0
  uint32 t1, t2, t3, t0;
  t3 = r_ptr->tkct3;
  t2 = r_ptr->tkct2;
  t1 = r_ptr->tkct1;
  t0 = r_ptr->tkct0;
  printf("tkct0: %x, %x, %x, %x\n",t0, t1, t2, t3);
#endif

  tkct = r_ptr->tkct3 & 0xff;
  tmp = ((uInt32) r_ptr->tkct2) << 8;
  tkct |= tmp;
  tmp = ((uInt32) r_ptr->tkct1) << 16;
  tkct |= tmp;

  return (tkct);
}

/***************************************************************************
 *  Function ReadRLCT
 *
 *  Description
 *    THis function reads RLCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in RLCT
 ***************************************************************************/
static uint32 ReadRLCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
  uint32 rlct = 0, tmp = 0;

  rlct = r_ptr->rlct3 & 0xff;
  tmp = ((uInt32) r_ptr->rlct2) << 8;
  rlct |= tmp;
  tmp = ((uInt32) r_ptr->rlct1) << 16;
  rlct |= tmp;

  return (rlct);
}


/***************************************************************************
 *  Function ReadLTCT
 *
 *  Description
 *    THis function reads LTCT and returns the value
 *
 *  Parameter:
 *      BMAC_REG_TYPE   *r_ptr
 *
 *  Return: value in LTCT
 ***************************************************************************/
static uint32 ReadLTCT (r_ptr)
volatile BMAC_REG_TYPE *r_ptr;
{
#if 0
  enter_debug(0x1234,r_ptr->ltct,&(r_ptr->ltct));
#endif
  return ((uint32)r_ptr->ltct);
}


/***********************************************************************
 *  Function SetMACFrameStatus
 *
 *  Description
 *    This function sets the CAM to perform frame status function
 *
 *  Parameter: 
 *    uint32 mac_num
 *
 *  Return: void
 **********************************************************************/
static void InitMACFrameStatus (mac_num)
uint32 mac_num;
{

}

/***********************************************************************
 *  Function init_my_slots
 *
 *  Description
 *    This function initializes HOSTENTRY with our slot mac address.
 *
 *  Parameter:
 *   uint32 mac_num
 *
 *  Return: void
 **********************************************************************/
static void init_my_slot (mac_num,addr,port_num)
uint32 mac_num;
byte *addr;
int port_num;
{
  HOSTENTRY *host, *other_host;
  byte *ptr;

  host = MONP_hash(addr,mac_num,TRUE);  /* add to hash table */
  if (host == NULL) {
#if BMAC_DEBUG
      printf("Error: Can't get host entry for the concentrator\n");
#endif
    return;
  }

  host->MySlotNid = TRUE;

  host->macAddress[0] = 6;  /* len of the mac addr */
  ptr = &(host->macAddress[1]);   /* write in mac address */
  memcpy(ptr,addr,6);
#if 0
  {
    extern HOSTENTRY fddihosts[];
    int i;
    for (i=0; i < 2; i++) {
      host = &(fddihosts[i]);
      printf("i %d: host %x, myslotid %d\n",i,host,host->MySlotNid);
    }
  }
#endif
}



/*
 *-----------------------------------------------------------------------
 *        bmac diagnos.c
 *-----------------------------------------------------------------------
 */


/***********************************************************************
 *  Function bmac_diag
 *
 *  Description
 *      This is the entry point of bmac diagnostics
 *
 *  Parameters:
 *      uint32 bmac_num
 *      uint32 act
 *
 *  Return: [OK | ERROR
 **********************************************************************/
int bmac_diag (mac_num,act)
uint32 mac_num;
uint32 act;
{
    char buf_1[] = {0x01,0x02,0x03,0x4,0x5,0x6};
    char buf_2[] = {0x1,0x2,0x3,0x4,0x5,0x9};
    volatile BMAC_REG_TYPE *r_ptr = bmacs[mac_num].reg_base;
    int i, val;
#if 0
    if (bmac_init_flag == FALSE) {
      DRV_ResetLANChips();
      DRV_InitFddiCon();
      if (BMAC_Init(fddi_board.bmac_num,DRV_FROM_BOOT) == OK)  {
	printf("bmac init succeed\n");
      }
      else {
	printf("BMAC_Init() failed\n");
      }
    }
#endif
    switch (act) {
      case 'b':    /* send beacon frame */
        if (BeaconTest(mac_num,DRV_1SEC,TRUE) == OK)
	break;

      case 'c':    /* send claim frame */
        if (ClaimTest(mac_num,DRV_1SEC,TRUE) == OK)
	  printf("Claim Test succeed\n");
	break;

      case 'd':
	if (bmac_init_flag == FALSE) {
	    printf("MAC has not initialized, abort\n");
	    return (1);
	}
	show_bmac_regs(mac_num);
	break;

      case 'i':
	val = QueryUnsigned("fix grp addr: ",0,0,15);
	BMAC_SetFixGrpAddr(mac_num,val);
	break;

      case 'l':    /* set long addr */
        BMAC_SetLongAddr(mac_num,(byte *)MyNid(mac_num));
	break;

      case 's':    /* set short address */
#if 0
	BMAC_SetShortAddr(mac_num,(byte *)MyNid(mac_num));
#else
	ReadFRCT(r_ptr);
	ReadEICT(r_ptr);
#endif
	break;

      case 't':    /* LCT test */
	BMAC_ShowMacCounts(mac_num);
	break;

      case 'g':    /* set group long address */
#if 0
	BMAC_SetGrpLongAddr(mac_num,(byte *)buf_1);
	BMAC_SetGrpLongAddr(mac_num,(byte *)buf_2);
#else
	BMAC_GetMacCounters(mac_num,SM_MA_STATUS_COPIED_CT);
	BMAC_GetMacCounters(mac_num,SM_MA_STATUS_FRAME_CT);
	BMAC_GetMacCounters(mac_num,SM_MA_STATUS_ERROR_CT);
#endif
	break;

      case 'e':    /* exit from bmac test */
/*	BMAC_ClearBMACs(2,FDDICON_MASTER);  */
	break;
    }
    return OK;
}




/************************************************************************
 *  Function show_bmac_regs
 *
 *  Description
 *      This function displays all the register in the given bmac
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ***********************************************************************/
static void show_bmac_regs (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;

    r_ptr = bmacs[mac_num].reg_base;
    printf("MAC # %d addr is %lx:, rev is %x\n",mac_num,r_ptr,bmacs[mac_num].revision);
    printf("mr %x, or %x, fr %x, rr %x, crs0 %x\n",r_ptr->mr,r_ptr->or,r_ptr->fr,r_ptr->rr,r_ptr->crs0);
    printf("cts0 %x, relr0 %x, remr0 %x, relr1 %x, remr1 %x\n",r_ptr->cts0,r_ptr->relr0,r_ptr->remr0,r_ptr->relr1,r_ptr->remr1);
    printf("telr0 %x, temr0 %x, cilr %x, cimr %x, colr %x\n",r_ptr->telr0,r_ptr->temr0,r_ptr->cilr,r_ptr->cimr,r_ptr->colr);
    printf("comr %x, ielr %x, esr %x, emr %x, icr %x, imr %x\n",r_ptr->comr,r_ptr->ielr,r_ptr->esr,r_ptr->emr,r_ptr->icr,r_ptr->imr);

PrintIntCounts(mac_num);
}



