/**************************************************************************
 *  Copyright (c) 1992 Hughes LAN Systems                                 *
 *  All rights Reserved                                                   *
 *                                                                        *
 *  This file contains unpublished proprietary source code of Hughes LAN  *
 *  Systems.                                                              *
 **************************************************************************
                                                                           
   This file contains the service routines for HLS SYSTEM-MIB
   It is directly derived from the MIB specification using the AWK program 
   'MIBSVC.AWK'
   Should significant changes be needed in this file that are NOT          
   reflected back into the MIB specification, then modify this header to   
   indicate that this is the case.                                         

	$Log:   /b/gregs/bridge/snmp/routines/hlssystem.c_v  $
 * 
 *    Rev 1.7   15 Nov 1993 13:45:34   vinay
 * removed the include file and put in defines for CF-STATE
 * 
 *    Rev 1.6   12 Oct 1993 10:38:28   vinay
 * fixed ledStatus (hopefully)
 * 
 *    Rev 1.5   08 Oct 1993 15:30:20   vinay
 * fixed the invalid ip address bugs
 * .
 * 
 *    Rev 1.4   24 Sep 1993 11:12:16   vinay
 * fixed the netinterfaceseg
 * 
 *    Rev 1.3   03 Sep 1993 12:03:20   vinay
 * fixed the netinterfacesegment svc
 * 
 *    Rev 1.2   19 Aug 1993 17:17:40   vinay
 * changed the prroductname , fixed the led status and changed the 
 * netInterfaceSegment to return the right value.
 * 
 *    Rev 1.1   30 Jul 1993 13:28:00   vinay
 * Added some code to netInterfaceSegment
 * 
 *    Rev 1.0   20 Jul 1993 09:37:08   franks
 * Initial revision.
 * 
 *    Rev 1.4   05 Mar 1993 15:04:08   kwok
 * FOIRL card type can be either 0x8000 or 0xA000
 * 
 *    Rev 1.3   28 Jul 1992 11:05:56   kwok
 * return 0 at the end of the function trapEnterprise.
 * 
 *    Rev 1.2   11 May 1992 09:29:18   franks
 * No change.
 * 
 *    Rev 1.1   10 Apr 1992 16:46:56   kwok
 * Change the productName object such that it returns "834" 
 * "834 301A", "834 301TN", "834 301F" or "834 301T" for stanley
 * without a transceiver, with AUI, with Thin net, with FOIRL and
 * with 10 base T respectively.
 * 
 *    Rev 1.0   30 Mar 1992 17:37:36   pvcs
 * Initial revision.

 *************************************************************************/

#define HLS-MIB_SVC

#include <types.h> 
#include <target.h>
#include <krnl.h>
#include <dbd.h>
#include <snmp.h>
#include <asn1.h>
#include <syteksnm.h>
#include <prcctl.h>
#include <tcpip.h>
#include <eeprecs.h>
#include <nim960h.h>
#include <nvrecs.h>
#include <sys.h>
#include <dips.h>
#include <led.h>
#include <udp.h>
#include <tftpboot.h>
#include <memory.h>
#include <bridges.h>




#define	TRAP_SET_MANAGER	1
#define	TRAP_SET_GENERIC	2
#define	TRAP_SET_SPECIFIC	3
#define	TRAP_DELETE_ENTRY	4
#define PORTA			1
#define PORTB			2

/*
*       CFM States
*/
#define CF_ISOLATED     0
#define CF_WRAP_S       1
#define CF_WRAP_A       2
#define CF_WRAP_B       3
#define CF_WRAP_AB      4
#define CF_THRU         5


/*
 *	Operations on updateSoftware hlsSystem 27.
 */
#define	UPDATE_USE_DIPS		1
#define	UPDATE_ON_RESET		2
#define	UPDATE_NO_UPDATE	3
#define	UPDATE_NO_FLASH		4
#define	UPDATE_IMMEDIATE	5	

/*
 *	Offset for netInterfaceSegment
 */
#define FEBRGOFFSET		18

extern	NIM960_HDR	Nim960Header;
extern	unsigned int	ClearCounterTicks;
extern	SEM	SwUpdateFromSnmp;
extern	int	UpdateMode;
static	char *TransTypeToStr(int xcvr_type);

long	get_newvalue();

/**************************************************************************
*  Procedure    :   svc_snmpAuth
*  Path         :   1.3.6.1.4.1.26.20.1
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   " Method which the agent used to authenticate  SNMP
*                    requests it receives."
*
**************************************************************************/

int svc_snmpAuth(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)(BridgeStatus->SNMPAuthentication);
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		BridgeStatus->SNMPAuthentication = *value;
		SaveBStatus();
		AdmSetSnmpAuth(*value);
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value != SNMP_ACC_NONE && 
				*value != SNMP_ACC_COMMUNITY)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_trapManagerNumber
*  Path         :   1.3.6.1.4.1.26.20.7
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Maximum number of trap managers in trap manager
*                    table."
*
**************************************************************************/

int svc_trapManagerNumber(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = TRAP_MANAGER_NO; 
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/*
 *	Trap manager related utilities.
 */

chk_trapindex(compc, compl, index)
byte  	*compc;
unsigned int  *compl;
int	*index;
{
	in_name	dest;
	int	i;

		
	SnmpCopyToIp(&dest, compl);
	if(*compc != 4 || dest == 0)
		return ERR_INSTANCE;
	for (i = 0; i < TRAP_MANAGER_NO; i++)
	{
		if (BridgeStatus->SnmpMgr[i] == dest)
			{
			*index = i + 1;
			break;
			}
	}
	if (i > (TRAP_MANAGER_NO - 1))
		return NO_SUCH_NAME;
	return 0;
}
get_trapnxtidx(compc, compl, index)
byte	*compc;
int	*compl;
int	*index;

{
	in_name	dest;
	int	i;

	if(*compc == 0)
	{
		for (i = 0; i < TRAP_MANAGER_NO; i++)
		{
			if(BridgeStatus->SnmpMgr[i] != 0)
				{
				*index = i + 1;
				break;
				}
		}
		if (i > (TRAP_MANAGER_NO - 1))
		{
			*compc = 0;
			*compl = 0;
			return NO_SUCH_NAME;
		}
	}
	else
	{
		if(chk_trapindex(compc, compl, index) == 0 && *index < TRAP_MANAGER_NO)
		{
			for (i = *index; i < TRAP_MANAGER_NO; i++)
			{
				if (BridgeStatus->SnmpMgr[i] != 0)
				{
					*index = i + 1;
					break;
				}
			}
			if (i >= TRAP_MANAGER_NO)
			{
				*compl = 0;
				*compc = 0;
				return NO_SUCH_NAME;
			}
		}
		else
		{
			*compl = 0;
			*compc = 0;
			return NO_SUCH_NAME;
		}
	}
	IpCopyToSnmp(compl, &BridgeStatus->SnmpMgr[*index - 1]);
	*compc = 4;
	return 0;
}

chk_snmptrapdest(compc, compl, index)
byte	*compc;
int	*compl;
int	*index;
{
	int	i;

	if(chk_trapindex(compc, compl, index) == 0)
		return(0);
	for (i = 0; i < TRAP_MANAGER_NO; i++)
	{
		if(BridgeStatus->SnmpMgr[i] == 0)
			{
			*index = i + 1;
			break;
			}
	}
	if (i > (TRAP_MANAGER_NO - 1))
		return(NO_SUCH_NAME);
	return(0);
}
/**************************************************************************
*  Procedure    :   svc_trapManagerDest
*  Path         :   1.3.6.1.4.1.26.20.8.1.1
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The IP address of this trap manager."
*
**************************************************************************/

int svc_trapManagerDest(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	tindex;

	rtn_code = chk_trapindex(compc, compl, &tindex);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_trapnxtidx((byte  *)compc, 
		    (unsigned int  *)compl, &tindex);

	case SNMP_GET:
		if ( rtn_code == 0 )
		{
			if(1 <= tindex && tindex <= TRAP_MANAGER_NO)
				*value = BridgeStatus->SnmpMgr[tindex - 1];
		}
		else
			return NO_SUCH_NAME;
		break;

	case SNMP_SET:
		set_snmptrap(TRAP_SET_MANAGER, compc, compl, value, lenp);
		break;

	case SNMP_TEST:
		if(chk_snmptrapdest(compc, compl, &tindex) != 0)
			return BAD_VALUE;
		break;

	}

	return 0;
}

/**************************************************************************
*  Procedure    :   svc_trapGeneric
*  Path         :   1.3.6.1.4.1.26.20.8.1.2
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "Generic traps which should  be sent to this trap
*                    manager by this agent. Each bit represent a trap. If
*                    the bit is set, the trap will be sent to this trap
*                    manager. Bit I of Jth octet represents (I+(8*J))th
*                    trap number."
*
**************************************************************************/

int svc_trapGeneric(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	tindex;
	long	newvalue;

  	rtn_code = chk_trapindex(compc, compl, &tindex);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_trapnxtidx((byte  *)compc, 
		    (unsigned int  *)compl, &tindex);
		/* FALL TRHROUGH */

	case SNMP_GET:
		if ( rtn_code == 0 )
		{
			*value = BridgeStatus->GenericTrap[tindex - 1];
			*lenp = sizeof(BridgeStatus->GenericTrap[tindex - 1]);;
		}
		else
			return NO_SUCH_NAME;
		break;

	case SNMP_SET:
		set_snmptrap(TRAP_SET_GENERIC, compc, compl, value, lenp);
		break;

	case SNMP_TEST:
		if(chk_snmptrapdest(compc, compl, &tindex) != 0)
			return BAD_VALUE;
		if(*lenp > sizeof(BridgeStatus->GenericTrap[0]))
			return BAD_VALUE;
		newvalue = (long)get_newvalue(*lenp, value);
		if(newvalue > 0x1f)
			return BAD_VALUE;
		break;

	}
	return 0;
}


long
get_newvalue(len, value)
int	len;
OCTET_P value;
{
	char ip[4];
	int	i;

	*(long *)ip = 0;
	for(i = 0; i < len; i++)
		ip[i] = *value++;
	return (*(long *)ip);
}


set_snmptrap(type, compc, compl, value, lenp)
int		type;
byte		*compc;
unsigned int	*compl;
UINT_32_T	*value;
int		*lenp;
{
	int	i;
	in_name	dest;
	long	newvalue;
	int	index;

	if(chk_snmptrapdest(compc, compl, &index) != 0)
		return BAD_VALUE;
	SnmpCopyToIp(&dest, compl);
	if (BridgeStatus->SnmpMgr[index - 1] == 0)
		set_deftrap(index,(long)dest);
	switch(type)
	{
	case TRAP_SET_MANAGER:
		set_deftrap(index, *value);
		break;

	case TRAP_SET_GENERIC:
		newvalue = (long)get_newvalue(*lenp, value);
		BridgeStatus->GenericTrap[index - 1] = newvalue;
		break;

	case TRAP_SET_SPECIFIC:
		newvalue = (long)get_newvalue(*lenp, value);
		BridgeStatus->SpecificTrap[index - 1] = newvalue;
		break;

	case TRAP_DELETE_ENTRY:
		if(*value == INVALID)
		{
			BridgeStatus->SnmpMgr[index - 1] = 0;
			BridgeStatus->GenericTrap[index - 1] = 0;
			BridgeStatus->SpecificTrap[index - 1] = 0;
		}
		break;
	}
	SaveBStatus();
}

set_deftrap(index,ipadr)
int	index;
long	ipadr;
{

	index--;
	BridgeStatus->SnmpMgr[index] = ipadr;
	BridgeStatus->GenericTrap[index] = 0x1f;
	BridgeStatus->SpecificTrap[index] = SESCONFAIL|SESCONNECTED|SESCLOSE|
		    SESABORT|NVRAMERR;
}
/**************************************************************************
*  Procedure    :   svc_trapEnterprise
*  Path         :   1.3.6.1.4.1.26.20.8.1.3
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "Enterprise-specific trap which should be sent to
*                    this  trap manager by the agent. Each bit represents
*                    a trap. If the bit is set, the trap will be sent to
*                    this trap manager. Bit I of Jth  octet represents
*                    HLS enterprise (i+(8*j)) th trap number."
*
**************************************************************************/

int svc_trapEnterprise(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	tindex;
	long	newvalue;

	rtn_code = chk_trapindex(compc, compl, &tindex);
	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_trapnxtidx((byte  *)compc, 
		    (unsigned int  *)compl, &tindex);

	case SNMP_GET:
		if ( rtn_code == 0 )
		{
			*value = BridgeStatus->SpecificTrap[tindex - 1]; 
			*lenp = 4;
		}
		else
			return NO_SUCH_NAME;
		break;

	case SNMP_SET:
		set_snmptrap(TRAP_SET_SPECIFIC, compc, compl, value, lenp);
		break;

	case SNMP_TEST:
		if(chk_snmptrapdest(compc, compl, &tindex) != 0)
			return BAD_VALUE;
		if(*lenp > sizeof(BridgeStatus->GenericTrap[0]))
			return BAD_VALUE;
		newvalue = (long)get_newvalue(*lenp, value);
		if(newvalue & ~(SESCONFAIL|SESCONNECTED|SESCLOSE|
		    SESABORT|NVRAMERR))
			return BAD_VALUE;
		break;

	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_trapEntryType
*  Path         :   1.3.6.1.4.1.26.20.8.1.4
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Writing invalid(2) to this object deletes this
*                    entry from the table."
*
**************************************************************************/

int svc_trapEntryType(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	tindex;

	rtn_code = chk_trapindex(compc, compl, &tindex);
	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_trapnxtidx((byte  *)compc, 
		    (unsigned int  *)compl, &tindex);

	case SNMP_GET:
		if ( rtn_code == 0 )
		{
			if (0 < tindex && tindex <= TRAP_MANAGER_NO  && 
				BridgeStatus->SnmpMgr[tindex - 1] != 0)
			{
				*value = VALID;
			}
			else
				*value = INVALID;
		}
		else
			return NO_SUCH_NAME;
		break;

	case SNMP_SET:
		set_snmptrap(TRAP_DELETE_ENTRY, compc, compl, value, lenp);
		break;

	case SNMP_TEST:
		if(*value == VALID)
		{
			if(chk_snmptrapdest(compc, compl, &tindex) != 0)
				return BAD_VALUE;
			else
				break;
		}
		else if(*value != INVALID)
			return BAD_VALUE;
		return(rtn_code);
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_slotId
*  Path         :   1.3.6.1.4.1.26.20.9
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "If the agent is executing on a function card that
*                    is plugged into a HLS Enterprise HUB Chassis then
*                    the value of slotId indicates the physical slot
*                    number that the card is in."
*
**************************************************************************/

int svc_slotId(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = IsStandalone() ? 0 : GetSlotID();
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_hubId
*  Path         :   1.3.6.1.4.1.26.20.10
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "If the agent is executing on a function card that
*                    is plugged into a HLS Enterprise HUB Chassis then
*                    the value of hubId indicates the unique numeric
*                    identifier of the hub that the card is in."
*
**************************************************************************/

int svc_hubId(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = IsStandalone() ? 0 : GetHubID();
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_tftpServer
*  Path         :   1.3.6.1.4.1.26.20.11
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The IP address of TFTP server from which the agent
*                    gets the configuration information and downloads the
*                    software image file. If this is a zero IP address,
*                    then the BOOTP server will be used as the TFTP
*                    server. This server will be used as default SNMP
*                    manager if no SNMP managers are defined."
*
**************************************************************************/

int svc_tftpServer(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	char *ipAddr;
	char ipa[20];
	IP ipdevice;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) eep_boot_rec.eep_ServerIP;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		eep_boot_rec.eep_ServerIP = *value;
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		break;

	case SNMP_TEST:
		if(*value != 0)
		{
			ipAddr = (char *)inet_ntoa(*value);
			if(GetIPAddress(&ipdevice,ipAddr,&ipa))
			return BAD_VALUE;
		}
		return rtn_code;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_netInterfaceSeg
*  Path         :   1.3.6.1.4.1.26.20.13
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "The backplane interface segment which the agent
*                    currently connects to. Each octet contains a value
*                    which represents a segment. A value 0 means no
*                    connection to any segments. a value 1 means
*                    connecting to Ether-1. A value 2 means connecting to
*                    Ether-2.  A value 3 means  connecting to Ether-3.  A
*                    value 4 means connecting to TOKENRING-1. A value 5
*                    means connecting to TOKENRING-2. A value 6 means
*                    connecting to TOKENRING-3 A value 7 means connecting
*                    to FDDI-1. A value 8 means connecting to FDDI-2.A
*                    value 9 means connecting to HSB.  A value 255 means
*                    connecting to the segment indicated by dipSwitches."
*
**************************************************************************/

int svc_netInterfaceSeg(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
byte		*value;
int		index;
{
	int rtn_code;
#ifdef 0
	return svc_snmpnosupport(service, compc, compl, lenp, value, 0);
#endif

	rtn_code = chk_index(*compc, *compl);
	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			if(eep_boot_rec.eep_ring_number != 0xffff && 
			   eep_boot_rec.eep_ring_number != 0 )
				*value = (OCTET_T)(eep_boot_rec.eep_ring_number+FEBRGOFFSET);
			else if(eep_boot_rec.eep_ring_number == 0)
				*value = (OCTET_T)0;
			else	
				*value = (OCTET_T)255;
			
			*lenp = 1;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		if (*value == 255)
			eep_boot_rec.eep_ring_number = 0xffff;
		else if(*value == 0)
			eep_boot_rec.eep_ring_number = 0;
		else
			eep_boot_rec.eep_ring_number = *value - FEBRGOFFSET;
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		insert_hub_ring(0, get_ring_number());
		break;

	case SNMP_TEST:
		if((*value < 19 || *value > 33) && *value != 0 && *value != 255)
			return BAD_VALUE;
		else
			return 0;
	}
	return 0;

}

/**************************************************************************
*  Procedure    :   svc_timeServer
*  Path         :   1.3.6.1.4.1.26.20.14
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The IP address of the time server from which the
*                    system get the time."
*
**************************************************************************/

int svc_timeServer(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	char *ipAddr;
	char ipa[20];
	IP ipdevice;
	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) eep_boot_rec.eep_TimeServer;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		eep_boot_rec.eep_TimeServer = *value;
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
			/* FILL IN ACTUAL SET OPERATIONS HERE */
		break;

	case SNMP_TEST:
		if(*value != 0)
		{
			ipAddr = (char *)inet_ntoa(*value);
			if(GetIPAddress(&ipdevice,ipAddr,&ipa))
				return BAD_VALUE;
		}
		return(rtn_code);
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_systemPassword
*  Path         :   1.3.6.1.4.1.26.20.15
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "The password which is required in order to access
*                    to administrative port of the card. The card will
*                    return empty value when  read."
*
**************************************************************************/

int svc_systemPassword(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	char	buffer[14];

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*lenp = 0;
			*value = (UINT_32_T) 0;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		if (*lenp < 12)
		{
			memcpy(buffer, value, *lenp);
			buffer[*lenp] = '\0';
			InitPassword(buffer);
			SaveBStatus();
		}
		else
			return BAD_VALUE;
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*lenp >= 12)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_initAdminPort
*  Path         :   1.3.6.1.4.1.26.20.16
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Writing valid to this object initializes the
*                    administrative port.  The current session of the
*                    administrative port will be disconnected."
*
**************************************************************************/

int svc_initAdminPort(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) IsTelnetActive() ? 1 : 0;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		if (*value != 1)
			return BAD_VALUE;
		if (IsTelnetActive())
		{
			printf("\n\nTelnet session is terminated by the network manager.\n\n");
			CloseTelnet();
		}
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if (*value != 1)
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_tftpTimer
*  Path         :   1.3.6.1.4.1.26.20.18
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Retransmission timer of TFTP downloading in 10
*                    millisecond."
*
**************************************************************************/

int svc_tftpTimer(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	return svc_snmpnosupport(service, compc, compl, lenp, value, 0);
}

/**************************************************************************
*  Procedure    :   svc_powerRequirement
*  Path         :   1.3.6.1.4.1.26.20.20
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "The maximum static power requirement of the device,
*                    specified in milliAmperes(mA)."
*
**************************************************************************/

int svc_powerRequirement(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			 *	11000 mA 
			 */

			*value = (UINT_32_T) 11000;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_lastClearSystemCountersTime
*  Path         :   1.3.6.1.4.1.26.20.21
*  Access       :   RO
*  Syntax       :   TimeTicks VT_TIMETICKS
*  Description  :   
*                   " The TimeStamp at which agent's counters were last
*                    cleared. If no reset has occurred then the TimeStamp
*                    will be 0."
*
**************************************************************************/

int svc_lastClearSystemCountersTime(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{	/*
		 	 *	in hundredth of a second.
			 */
			*value = (UINT_32_T) ClearCounterTicks / 10;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_systemInitCause
*  Path         :   1.3.6.1.4.1.26.20.22
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "The cause of the last system reboot."
*
**************************************************************************/

int svc_systemInitCause(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	count;
	int	level;	/* reset level	*/
	/*
	 *	Power failure(0) 	->	poweron (1)
	 *	Button Reset(1)	->	pushbutton (2)
	 *	SNMP Reset(2)	->	No match (0)
	 *	Software Crash(3)	->	pushbutton (2)
	 *	Self test failure(4) ->	pushbutton (2)
	 *	sw reset 0(5)	->	No match (0)
	 *	sw reset 1(6)	->	sw reset1 (4)
	 *	sw reset 3(7)	->	sw reset3(6)
	 *	sw reset 4(8)	->	sw reset4(7)
	 *	extended reset (9) ->	pushbutton (2)
	 *	software update ->	software update (9)
	 */
	static	int	ResetLog2InitCause[] = {1, 2, 0, 2, 2, 0, 4, 6, 7, 2, 9};
	RESETLOG	*resetlog = &RstLog->nvr_resetlog;


	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			 *	Get the last reset record.
			 */
			count = resetlog->rstlog_count;
			if (count >= RSTLOG_COUNT)
				count = count % RSTLOG_COUNT;
			level = resetlog->rstlog[count - 1].rstlog_msg_no;
			*value = (UINT_32_T)ResetLog2InitCause[level]; 
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_productName
*  Path         :   1.3.6.1.4.1.26.20.23
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The name of the product.  This name should be able
*                    to identify the hardware configuration even the
*                    products have the same model number, but may have
*                    minor hardware configuration differences.  For
*                    example, 10BaseT concentrators with different
*                    ethernet transceiver adapter card have the same
*                    model number.
*		     for stanley:
*		     "834 " no transceiver
*                    "834 301A" 	AUI
*                    "834 301TN"	Thin net
*                    "834 301F"		FOIRL
*                    "834 301T"		10 base T"
*
**************************************************************************/

int svc_productName(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int 	rtn_code;
	int	xcvr_type; /* transceiver type	*/

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			strcpy(value, "836");	/* model for stanley */
#ifdef 0
			xcvr_type = TransceiverType();
			strcat(value, TransTypeToStr(xcvr_type));
#endif
			*lenp = strlen(value);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_bootPServer
*  Path         :   1.3.6.1.4.1.26.20.24
*  Access       :   RO
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The ip address of the BOOTP server from which the
*                    system learned its IP address."
*
**************************************************************************/

int svc_bootPServer(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) eep_boot_rec.eep_BootpServer;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_updateSoftware
*  Path         :   1.3.6.1.4.1.26.20.27
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   " Specifying if the application software in the
*                    Flash EPROM will  be updated at next system
*                    rebooting. usedipswitch(1) means the system will use
*                    dipswitch 2 and 3 to decide the application software
*                    should be updated or not.  update(2) means that
*                    system will update the application software.
*                    noupdate(3) means that the system will not update
*                    the application software.  After the system is
*                    rebooted, this object is set to usedipswitch by the
*                    agent."
*
**************************************************************************/

int svc_updateSoftware(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			 *	All software update happens 
			 *	immediately. There is no software
			 *	update on the next reset.
			 */
			*value = (UINT_32_T) UPDATE_NO_UPDATE;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		/*
		 *	Tell the command task that we want to
		 *	do a software update.
		 */
#ifdef 0
		SendSignal(&SwUpdateFromSnmp);
		UpdateMode = TFTP_MODE_REMOTE;
#endif
		printf("\nSoftware will be updated due to a request from the network manager\n\n"); 
		eep_boot_rec.eep_ThinNet = UPDATE_FLASH;
		if(PutEeprom(EEP_BOOT_ADDR,&eep_boot_rec,EEP_BOOT_SIZE)	!= EEPROM_AOK)
			printf("Error writing to EEPROM.\n");
			UpdateSoftwareReset(1);

		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			/*
			 *	Stanley supports immediate update
			 *	only.
			 */
			if (*value != UPDATE_IMMEDIATE) /* ||
				*value != UPDATE_ON_RESET) */
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_updateSWDate
*  Path         :   1.3.6.1.4.1.26.20.28
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The date which the application software in the
*                    Flash EPROM was updated in ASCII string MM-DD-YY."
*
**************************************************************************/

int svc_updateSWDate(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			memcpy(value, eep_boot_rec.eep_BurnDate, 8);
			*lenp = 8;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_updateSWTime
*  Path         :   1.3.6.1.4.1.26.20.29
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The time of updateSWDate which the application
*                    software in the Flash EPROM was updated, formatted
*                    in ASCI HH:MM:SS."
*
**************************************************************************/

int svc_updateSWTime(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			memcpy(value, eep_boot_rec.eep_BurnTime, 8);
			*lenp = 8;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_dipSwitches
*  Path         :   1.3.6.1.4.1.26.20.32
*  Access       :   RO
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "The dipswitches setting of the agent.  Each octet
*                    contains value ON(1) or OFF(0).  Each octet
*                    represents a dipswitch. First octet is dipswitch 1 ,
*                    second octet is dipswitch 2 ...etc."
*
**************************************************************************/

int svc_dipSwitches(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	unsigned int mask;	/* for which dip switch	*/
	char	*ret_dip;
	unsigned int i;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			mask = Dip_reset();
			/* DIP SW 1 */
			for (i = 0x8000, ret_dip = (char *)value; 
				i > 0xff ; i /= 2 )
			{
				*ret_dip++ = (mask & i) ? 1 : 0;
			}
			/* DIP SW 2 */
			for (i = 0x80 ; i > 0 ; i /= 2 )
			{
				*ret_dip++ = (mask & i) ? 1:0;
			}
			
			*lenp = DIP_SWITCH_COUNT;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}
/**************************************************************************
*  Procedure    :   svc_bootPConfigRequest
*  Path         :   1.3.6.1.4.1.26.20.33
*  Access       :   RW
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The file name of the configuration file which will
*                    be requested in the BootP request. The filename
*                    returned by the responding BootP server is stored in
*                    tftpConfigFile. Depending on the configuration of
*                    the BootP server, a substitute configuration file
*                    may actually be returned for the agent to download."
*
**************************************************************************/

int svc_bootPConfigRequest(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			strcpy(value, eep_boot_rec.eep_request_file);
			*lenp = strlen(value);
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		memcpy(eep_boot_rec.eep_request_file, value, *lenp);
		eep_boot_rec.eep_request_file[*lenp] = '\0';
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		break;
	case SNMP_TEST:
		if (rtn_code == 0)
		{
			if (*lenp > 63)
				return BAD_VALUE;
		}
		return rtn_code;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_tftpConfigFile
*  Path         :   1.3.6.1.4.1.26.20.34
*  Access       :   RW
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The file name of the Configuration File that was
*                    actually downloaded from the TFTP server."
*
**************************************************************************/

int svc_tftpConfigFile(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			strcpy(value, eep_boot_rec.eep_config_file);
			*lenp = strlen(value);
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		memcpy(eep_boot_rec.eep_config_file, value, *lenp);
		eep_boot_rec.eep_config_file[*lenp] = '\0';
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		break;
	case SNMP_TEST:
		if (rtn_code == 0)
		{
			if (*lenp > 63)
				return BAD_VALUE;
		}
		return rtn_code;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_tftpImageFile
*  Path         :   1.3.6.1.4.1.26.20.35
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The file name of the software image which was
*                    downloaded from the TFTP server."
*
**************************************************************************/

int svc_tftpImageFile(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			strcpy(value, eep_boot_rec.eep_image_file);
			*lenp = strlen(value);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_ledStatus
*  Path         :   1.3.6.1.4.1.26.20.36
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The status of the LEDs on the front panel of the
*                    function card. Each octet in the string corresponds
*                    with one LED. The order of the LEDs in the string is
*                    defined by the function card, with the exception of
*                    the first two (2), which are the Ready and Active
*                    LEDs, respectively. Each octet represents an LED as
*                    follows: '0' OFF, '1' GREEN, '2' RED, '3' Flashing
*                    GREEN-OFF, '4' Flashing RED-OFF, '5' Flashing GREEN-
*                    RED. Further values are undefined, but reserved."
*
**************************************************************************/

int svc_ledStatus(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
OCTET_P		value;
int		index;
{
	int rtn_code;
	uint	ledmask1;
	char Fddi_Rx_Led();
	char Fddi_Tx_Led();

	/*
	 *	translate from the internal format to the snmp format.
	 *	off(0)	->	off(0)
	 *	red(1)	->	red(2)
	 *	green(2) ->	green(1)
	 */
	static	char	LedColour2snmp[] = {'0', '2', '1', '3', '4', '5'}; 

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{	
			uint ledmask2,ledStat;
			int i;

			/*
			 *	The first led is the ready, if there is an
			 *	adm session, send flashing green
			 */

			if(IsTelnetActive())
				*value = '3';
			else
				*value = '1';
			value++;
			/*
			 *	The second led is the active.  If the
			 * 	flicker LED is set send a Flashing green else
			 *	send the corresponding state.
			 */
			ledmask2 = sys.sys_ledb;
			/****
			if(sys.sys_flk_ledb & LED_ACT)
				*value = '3';
			else if(sys.sys_ledb & LED_ACT))
				*value = '1';
			****/
			*value = (sys.sys_flk_ledb & LED_ACT) ?
				'3':LedColour2snmp[(ledmask2 & LED_ACT) >> 2];
			value++;
			/*
			 *	The mono colour leds follow
			 */
			/*printf("led flk = %x\n", sys.sys_flk_ledm);*/
			for(ledmask1 = LED_RX1, ledmask2 = LED_TX1, i = 0; 
				i < 3;
				ledmask1 <<= 1, ledmask2 <<= 1, i++)
			{
				/*
				 *	Receive LED
				 */
				if((sys.sys_flk_ledm & ledmask1) )
					*value = '3';
				else if(sys.sys_ledm & ledmask1)
					*value = '1';
				else
					*value = '0';
				value++;

				/*
				 *	Transmit LED
				 */
				if((sys.sys_flk_ledm & ledmask2) )
					*value = '3';
				else if(sys.sys_ledm & ledmask2)
					*value = '1';
				else
					*value = '0';
				value++;

			}
			/*
			 *	FDDI LEDS 
			 */

			*value = Fddi_Rx_Led(PORTA, sys.sys_ledb);
			value++;

			*value = Fddi_Rx_Led(PORTB, sys.sys_ledb);
			value++;

			*value = Fddi_Tx_Led(PORTA, sys.sys_ledm);
			value++;

			*value = Fddi_Tx_Led(PORTB, sys.sys_ledm);
			value++;

			/* P/T LEDS */

			*value = (sys.sys_ctrl1 & PEER_TREEA_ON) ? '1': '0';
			value++;

			*value = (sys.sys_ctrl1 & PEER_TREEB_ON) ? '1': '0';
			value++;

			/* LE LEDS for A and B */
			*value = (sys.sys_ledb & LED_LEA)?'1': '0';
			value++;
			*value = (sys.sys_ledb & LED_LEB)?'1': '0';
			value++;
			
			*lenp = 16;	/* 
					   active, ready, 3 x Tx and 3 x Rx
					   FDDI A and B RX,
					   FDDI A and B TX,
					   P/T and LE for A and B
					*/
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

char Fddi_Rx_Led(int port, uint ledstat)
{
	switch( GetFddiCFState() )  /* switch on fddi config */
    {
    case CF_ISOLATED:       /* not connected */
	break;
    case CF_WRAP_A:         /* only port A is active */
        /* rx is from port A, and tx is to port A */
	if(port == 1)
	{
		if((ledstat & LED_RXA) )
			return '3';
		else if(ledstat & LED_RXA)
			return '1';
	}
	break;
    case CF_WRAP_B:         /* only port B is active */
        /* rx is from port B, and tx is to port B */
	if(port == 2)
	{
		if((ledstat & LED_RXB) )
			return '3';
		else if(ledstat & LED_RXB)
			return '1';
	}
	break;
    case CF_THRU:           /* normal dual-attach case */
        /* rx is from port A, and tx is to port B */
	if(port == 1)
	{
		if((ledstat & LED_RXA) )
			return '3';
		else if(ledstat & LED_RXA)
			return '1';
	}
	else if(port == 2)
	{
		if((ledstat & LED_RXB) )
			return '3';
		else if(ledstat & LED_RXB)
			return '1';
	}
	break;
    } /* end switch */
	return '0';
}

char Fddi_Tx_Led(int port, uint ledstat)
{
	switch( GetFddiCFState() )  /* switch on fddi config */
    {
    case CF_ISOLATED:       /* not connected */
	break;
    case CF_WRAP_A:         /* only port A is active */
        /* rx is from port A, and tx is to port A */
	if(port == 1)
	{
		if((ledstat & LED_TXA) )
			return '3';
		else if(ledstat & LED_TXA)
			return '1';
	}
	break;
    case CF_WRAP_B:         /* only port B is active */
        /* rx is from port B, and tx is to port B */
	if(port == 2)
	{
		if((ledstat & LED_TXB) )
			return '3';
		else if(ledstat & LED_TXB)
			return '1';
	}
	break;
    case CF_THRU:           /* normal dual-attach case */
        /* rx is from port A, and tx is to port B */
	if(port == 1)
	{
		if((ledstat & LED_TXA) )
			return '3';
		else if(ledstat & LED_TXA)
			return '1';
	}
	else if(port == 2)
	{
		if((ledstat & LED_TXB) )
			return '3';
		else if(ledstat & LED_TXB)
			return '1';
	}
	break;
    } /* end switch */
	return '0';
}

/**************************************************************************
*  Procedure    :   svc_secondaryRouter
*  Path         :   1.3.6.1.4.1.26.20.37
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The IP address of the secondary router which the
*                    system forwards IP datagram destined to remote
*                    networks if the primary router fails or is not
*                    reachable."
*
**************************************************************************/

int svc_secondaryRouter(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	char *ipAddr;
	char ipa[20];
	IP ipdevice;
	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = _initp->net_gway2;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		_initp->net_gway2 = *value;
		eep_boot_rec.eep_RouterIP[1] = *value;
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		break;
	case SNMP_TEST:
		if(*value != 0)
		{
			ipAddr = (char *)inet_ntoa(*value);
			if(GetIPAddress(&ipdevice,ipAddr,&ipa))
				return BAD_VALUE;
		}
		return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_burnCycles
*  Path         :   1.3.6.1.4.1.26.20.38
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "The number of times that the FLASH EPROM has been
*                    erased and programmed. The FLASH EPROM should be
*                    replaced after a certain number of cycles, defined
*                    in the function card specifications. It is nominally
*                    10,000."
*
**************************************************************************/

int svc_burnCycles(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = eep_boot_rec.eep_BurnCycles;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}



SnmpCopyToIp(byte *to, uint from[])

	{
	int	i;

	for (i = 0; i < 4; i++)
		*to++ = from[i];
	}

IpCopyToSnmp(uint to[], byte *from)

	{
	int	i;

	for (i = 0; i < 4; i++)
		to[i] = *from++;
	}


static char *TransTypeToStr(int type)

	{
	char	*s;

	switch (type)
		{
		case TRANSCEIVER_AUI:
			s = " 301A";
			break;
			
		case TRANSCEIVER_THIN_NET:
			s = " 301TN";
			break;

		case TRANSCEIVER_10BASET:
			s = " 301T";
			break;

#ifdef FSFIX
		case TRANSCEIVER_FOIRL:
		case TRANSCEIVER_FOIRL_ALT:
			s = " 301F";
			break;
#endif

		case TRANSCEIVER_BLANK:
			s = "";
			break;

		default:
			s = " UNKNOWN";
			break;
		}
	return s;
	}

