/**************************************************************************
 *  Copyright (c) 1992 Hughes LAN Systems                                 *
 *  All rights Reserved                                                   *
 *                                                                        *
 *  This file contains unpublished proprietary source code of Hughes LAN  *
 *  Systems.                                                              *
 **************************************************************************
                                                                           
   This file contains the service routines for HLSBRIDGES-MIB
   It is directly derived from the MIB specification using the AWK program 
   'MIBSVC.AWK'
   Should significant changes be needed in this file that are NOT          
   reflected back into the MIB specification, then modify this header to   
   indicate that this is the case.                                         
                                                                           
 *************************************************************************/

#define HLSBRIDGES-MIB_SVC
#define ENABLE 2
#define DISABLE 1

#include <types.h> 
#include <target.h>
#include <krnl.h>
#include <dbd.h>
#include <snmp.h>
#include <asn1.h>
#include <syteksnm.h>
#include <prcctl.h>

/* Additional include files */
#include <bridges.h>
#include <prcadr.h>
#include <nvrecs.h>
#include <sncvar.h>
#include "834CmdSnmp.h"


/* External Definitions */
extern NVR_BSTATUS  *BridgeStatus;
extern PRCCTL       prc;
extern SV           sv_var[];
extern NVR_RESETLOG *RstLog;
extern int NumberOfSonicPort;

register PRCCTL *prcctl = &prc;

/**************************************************************************
*  Procedure    :   svc_bridgeModel
*  Path         :   1.3.6.1.4.1.26.2.4
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The HLS model number of the bridge."
*
**************************************************************************/

int svc_bridgeModel(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)  NIM960_834;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_bridgeMode
*  Path         :   1.3.6.1.4.1.26.2.15
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The current mode of the bridge."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_bridgeMode(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	return NO_SUCH_NAME;
#ifdef 0
	return(svc_snmpcommon(service,compc,compl,lenp,value,0));
#endif
}

/**************************************************************************
*  Procedure    :   svc_ageingOption
*  Path         :   1.3.6.1.4.1.26.2.16
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The status of the Ageing option in the bridge."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_ageingOption(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	return NO_SUCH_NAME;
#ifdef 0
	return(svc_snmpcommon(service,compc,compl,lenp,value,0));
#endif
}

/**************************************************************************
*  Procedure    :   svc_chLearnMode
*  Path         :   1.3.6.1.4.1.26.2.17.1.1
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The status of learning mode for this channel."
*
**************************************************************************/

int svc_chLearnMode(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Learning is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->Learning & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;
        }
		else
			return NO_SUCH_NAME;
		break;

    case SNMP_SET:
        {
            if( *compc != 1 || *compl > 4 )
                return(rtn_code = NO_SUCH_NAME);

            PortMask = Indx2Mask( *compl );
            switch( *value )
            {
                 case ENABLE:
                     BridgeStatus->Learning |= PortMask;
                     prcctl->Prc_Learning   |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->Learning  &= ~PortMask;
                     prcctl->Prc_Learning    &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(rtn_code = BAD_VALUE);
                    break;
            } /* end of *value switch   */
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
		if(*value != ENABLE && *value != DISABLE)
			return BAD_VALUE;
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chRestrictMode
*  Path         :   1.3.6.1.4.1.26.2.17.1.2
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The status of Restrict mode for this channel.
*                          A channel's Restrict mode must be enabled in
*                          order for the channel to have filtering by
*                          destination address enabled for any address."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_chRestrictMode(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chOriginMode
*  Path         :   1.3.6.1.4.1.26.2.17.1.3
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The status of the Origin mode for this channel.
*                          A channel's Origin mode must be enabled in order
*                          for the channel to have filtering of non-unicast
*                          destination addresses enabled for any individual
*                          source addresses."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_chOriginMode(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chGroupFwdMode
*  Path         :   1.3.6.1.4.1.26.2.17.1.4
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The value of this object determines whether
*                          frames destined for unknown group addresses
*                          (i.e. multicast addresses which have not been
*                          statically configured) are to be forwarded on
*                          this channel."
*
**************************************************************************/

int svc_chGroupFwdMode(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Parameter is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->UnkMCast & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;
        }
		else
			return NO_SUCH_NAME;
		break;

    case SNMP_SET:
        {
            if( *compc != 1 || *compl > 4 )
                return(rtn_code = NO_SUCH_NAME);

            PortMask = Indx2Mask( *compl );
            switch( *value )
            {
                 case ENABLE:
                     BridgeStatus->UnkMCast |= PortMask;
                     prcctl->Prc_UnknMcast  |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->UnkMCast  &= ~PortMask;
                     prcctl->Prc_UnknMcast    &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(rtn_code = BAD_VALUE);
                    break;
            } /* end of *value switch   */
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
		if(*value != ENABLE && *value != DISABLE)
			return BAD_VALUE;
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chIndFwdMode
*  Path         :   1.3.6.1.4.1.26.2.17.1.5
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The value of this object determines whether
*                          frames destined for unknown individual addresses
*                          (i.e. unicast addresses which are not present
*                          in the bridge's forwarding tables) are to be
*                          forwarded on this channel."
*
**************************************************************************/

int svc_chIndFwdMode(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Parameter is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->UnkUCast & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;
        }
		else
			return NO_SUCH_NAME;
		break;

    case SNMP_SET:
        {
            if( *compc != 1 || *compl > 4 )
                return(rtn_code = NO_SUCH_NAME);

            PortMask = Indx2Mask( *compl );
            switch( *value )
            {
                 case ENABLE:
                     BridgeStatus->UnkUCast |= PortMask;
                     prcctl->Prc_UnknUcast  |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->UnkUCast  &= ~PortMask;
                     prcctl->Prc_UnknUcast    &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(rtn_code = BAD_VALUE);
                    break;
            } /* end of *value switch   */
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
		if(*value != ENABLE && *value != DISABLE)
			return BAD_VALUE;
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chDefRestrictFlag
*  Path         :   1.3.6.1.4.1.26.2.17.1.6
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The value to which the Restrict flag is set for
*                          subsequently (re-)learnt addresses on this
*                          channel.  When Restrict mode is enabled, frames
*                          to adestination address which has the Restrict
*                          flag set are not forwarded."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_chDefRestrictFlag(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chDefStaticFlag
*  Path         :   1.3.6.1.4.1.26.2.17.1.7
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The value to which the Static flag is set for
*                          subsequently learnt addresses on this channel.
*                          Addresses with the Static flag set do not age."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_chDefStaticFlag(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chDefOriginFlag
*  Path         :   1.3.6.1.4.1.26.2.17.1.8
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The value to which the Origin flag is set for
*                          subsequently (re-)learnt addresses on this
*                          channel.  When Origin mode is enabled, frames
*                          destined for non-unicast addresses are not
*                          forwarded if their source address has the
*                          Origin flag set."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_chDefOriginFlag(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chInCRCs
*  Path         :   1.3.6.1.4.1.26.2.17.1.9
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of packets for which a CRC error has
*                          been detected.  For channels on which CRC errors
*                          can not occur, this object always has a value
*                          of 0."
*
**************************************************************************/

int svc_chInCRCs(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code = 0;

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtportidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
	    rtn_code = chk_portidx(*compc, *compl);
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) sv_vars[*compl-1].sv_crc ;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chInOverruns
*  Path         :   1.3.6.1.4.1.26.2.17.1.10
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of received packets for which an
*                          overrun error occurred.  For channels on which
*                          overruns can not occur, this object always has
*                          a value of 0."
*
**************************************************************************/

int svc_chInOverruns(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int	         *lenp;
UINT_32_T    *value;
int	         index;
{
	int rtn_code = 0;


	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtportidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
	    rtn_code = chk_portidx(*compc, *compl);
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) sv_vars[*compl-1].sv_rxfifo ;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chOutColls
*  Path         :   1.3.6.1.4.1.26.2.17.1.11
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "On a CSMA/CD interface, the number of packets
*                          for which a collision occurred during
*                          transmission.  On any other type of interface,
*                          this counter is always zero."
*
**************************************************************************/

int svc_chOutColls(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int	         *lenp;
UINT_32_T    *value;
int	         index;
{
	int rtn_code = 0;


	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtportidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
	    rtn_code = chk_portidx(*compc, *compl);
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) sv_vars[*compl-1].sv_clsn; 
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chWhos
*  Path         :   1.3.6.1.4.1.26.2.17.1.12
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "On a token-bus interface, the number of times
*                          the bridge had to ask who its successor is.  On
*                          any other type of interface, the value of this
*                          counter is always zero."
*
**************************************************************************/

int svc_chWhos(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chInRunts
*  Path         :   1.3.6.1.4.1.26.2.17.1.13
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of undersize packets received on
*                          this channel."
*
**************************************************************************/

int svc_chInRunts(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) (prccnt.Cnt_TooBig[*compl - 1] + prccnt.Cnt_TooSmall[*compl - 1]);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chInGiants
*  Path         :   1.3.6.1.4.1.26.2.17.1.14
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of oversize packets received on
*                          this channel."
*
**************************************************************************/

int svc_chInGiants(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) prccnt.Cnt_TooBig[*compl - 1];
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chInLenErrors
*  Path         :   1.3.6.1.4.1.26.2.17.1.15
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of packets received on this channel,
*                          for which the number of bytes received was less
*                          than the value in the length field of the
*                          packet's header."
*
**************************************************************************/

int svc_chInLenErrors(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chInLocks
*  Path         :   1.3.6.1.4.1.26.2.17.1.16
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of times an interface was reset
*                          because the interface's receive deadman timer
*                          expired."
*
**************************************************************************/

int svc_chInLocks(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chOutLocks
*  Path         :   1.3.6.1.4.1.26.2.17.1.17
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of times an interface was reset
*                          because the interface's transmit deadman timer
*                          expired."
*
**************************************************************************/

int svc_chOutLocks(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_prtidx(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtprtidx((unsigned int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T)0;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_chNoBuffers
*  Path         :   1.3.6.1.4.1.26.2.17.1.18
*  Access       :   RO
*  Syntax       :   Counter VT_COUNTER
*  Description  :   
*                          "The number of packets not received because the
*                          system was temporarily out of buffers, and so
*                          could not transfer the data into memory."
*
**************************************************************************/

int svc_chNoBuffers(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code = 0;


	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtportidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
	    rtn_code = chk_portidx(*compc, *compl);
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) sv_vars[*compl-1].sv_lst;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}
/**************************************************************************
*  Procedure    :   svc_channelNumber
*  Path         :   1.3.6.1.4.1.26.2.17.1.19
*  Access       :   RO
*  Syntax       :   VT_NUMBER
*  Description  :   
*                          "The number of channels "
*
**************************************************************************/

int svc_channelNumber(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code = 0;
	int port;

	rtn_code = chk_portinst(*compc, compl,&port);
	switch(service)
	{
	case SNMP_GET:
		if(rtn_code == 0)
			*value = (UINT_32_T) port;
		break;
	case SNMP_GETNXT:
		if((rtn_code = get_nxtportinst((int *)compc,
					(unsigned int *)compl, &port)) == 0)
			*value = (UINT_32_T) port+1;
		break;
	}
	return rtn_code;
}


/**************************************************************************
*  Procedure    :   svc_protocolBlocking
*  Path         :   1.3.6.1.4.1.26.2.18
*  Access       :   RO
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                          "The set of protocols for which the bridge will
*                          not forward any packets.  Each octet contains a
*                          non-zero value if the corresponding protocol is
*                          to be blocked.  The octets correspond to the
*                          protocols, as follows:
*  
*                                 1st octet:   IP
*                                 2nd octet:   ARP
*                                 3rd octet:   RARP
*                                 4th octet:   reserved (must be zero)
*                                 5th octet:   DECNET
*                                 6th octet:   IPX
*                                 7th octet:   V2
*                                 8th octet:   XNS
*                                 9th octet:   TOPS
*                                 10th octet:  BridgeID (3Com T/Servers)
*                                 11th octet:  LAT
*                                 12th octet:  AppleTalk"
*                                 "Not supported by HLS834"
*
**************************************************************************/

int svc_protocolBlocking(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int	         *lenp;
UINT_32_T    *value;
int	         index;
{
	return(svc_snmpcommon(service,compc,compl,lenp,value,0));
}

/**************************************************************************
*  Procedure    :   svc_reason
*  Path         :   1.3.6.1.4.1.26.2.19.1.1
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The reason for the reset."
*
**************************************************************************/
int svc_reason(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int	         *lenp;
UINT_32_T    *value;
int	         index;
{
	int rtn_code = 0;
	uint rstindx;
	RESETLOG *resetlog = &RstLog->nvr_resetlog;
    RST_ENTRY *rstl;


	switch(service)
	{

	case SNMP_GETNXT:

		rtn_code = get_nxtportidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */

	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			if(*compl > RSTLOG_COUNT || *compl > resetlog->rstlog_count ||
			   *compc != 1 )
			    return(NO_SUCH_NAME);
			*/
			
			if(*compl > RSTLOG_COUNT || *compl > resetlog->rstlog_count) 
			    return(NO_SUCH_NAME);
            
			/* There RSTLOG_COUNT(8) possible entries in the reset log     */
			/* The get instance 1 == Most Recent, 2 == Second Most Recent  */
			/* and so on. This conversion will be handled by the following */
			/* function.                                                   */
			rstindx = Instance2Indx( *compl );
            rstl = &(resetlog->rstlog[rstindx]);

			switch(rstl->rstlog_msg_no) {

				case RSTMSG0: /* Power Fail */ 
			                  *value = (UINT_32_T) 4; 
                              break;

				case RSTMSG1: /* Button Reset */
			                  *value = (UINT_32_T) 2; 
                              break;

				case RSTMSG2: /* SNMP Reset */
			                  *value = (UINT_32_T) 8; 
                              break;

				case RSTMSG3: /* Software Crash */
			                  *value = (UINT_32_T) 3; 
                              break;

				case RSTMSG4: /* Self test Failure */
			                  *value = (UINT_32_T) 5; 
                              break;

				case RSTMSG5: /* Soft Level 0 */
			                  *value = (UINT_32_T) 9; 
                              break;

				case RSTMSG6:/* Soft Level 1 */
			                  *value = (UINT_32_T) 10; 
                              break;

				case RSTMSG7:/* Soft Level 3 */
			                  *value = (UINT_32_T) 11; 
                              break;

				case RSTMSG8: /* Soft Level 4 */
			                  *value = (UINT_32_T) 12; 
                              break;

				case RSTMSG9: /* Extended Button Reset */
			                  *value = (UINT_32_T) 13; 
                              break;

				case RSTMSG10: /* Software Update */
			                  *value = (UINT_32_T) 14; 
                              break;
				 
                default:    /* Unknown */
			                  *value = (UINT_32_T) 8; 
                              break;
			}
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_uptime
*  Path         :   1.3.6.1.4.1.26.2.19.1.2
*  Access       :   RO
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                          "The length of time, in seconds, the bridge had
*                          been up prior to the reset represented by this
*                          entry."
*
**************************************************************************/
int svc_uptime(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int	         *lenp;
UINT_32_T    *value;
int	         index;
{
	int rtn_code = 0;
	uint rstindx;
	uint seconds = 0;
	RESETLOG *resetlog = &RstLog->nvr_resetlog;
    RST_ENTRY *rstl;


	switch(service)
	{

	case SNMP_GETNXT:

		rtn_code = get_nxtportidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */

	case SNMP_GET:
		if (rtn_code == 0)
		{
			if(*compl > RSTLOG_COUNT || *compl > resetlog->rstlog_count ||
			   *compc != 1 )
			    return(NO_SUCH_NAME);
            
			/* There RSTLOG_COUNT(8) possible entries in the reset log     */
			/* The get instance 1 == Most Recent, 2 == Second Most Recent  */
			/* and so on. This conversion will be handled by the following */
			/* function.                                                   */
			rstindx = Instance2Indx( *compl );
            rstl = &(resetlog->rstlog[rstindx]);

			/* Calculate the uptime in seconds */ 
		    seconds += (rstl->rstlog_up_hour * 3600);
			seconds += (rstl->rstlog_up_min * 60);
       
            *value = (UINT_32_T) seconds; 
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;

}

/**************************************************************************
*  Procedure    :   svc_packetSize
*  Path         :   1.3.6.1.4.1.26.2.20
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The range of frame sizes to be forwarded onto an
*                          Ethernet/802.3 network by this bridge.  The value
*                          standard(1) represents the standard range of 60
*                          through 1536 bytes. The value extended(2)
*                          represents the (non-standard) range of 16 through
*                          2100 bytes."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_packetSize(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	return NO_SUCH_NAME;
#ifdef 0
	return(svc_snmpnosupport(service,compc,compl,lenp,value,0));
#endif
}

/**************************************************************************
*  Procedure    :   svc_transmitDeadmanTimer
*  Path         :   1.3.6.1.4.1.26.2.21
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The deadman timer for frame transmission on an
*                          interface.  The value is the maximum length of
*                          time, in units of 160 milliseconds, that a frame
*                          will await transmission before the interface is
*                          reset."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_transmitDeadmanTimer(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	return NO_SUCH_NAME;
#ifdef 0
	return(svc_snmpnosupport(service,compc,compl,lenp,value,0));
#endif
}

/**************************************************************************
*  Procedure    :   svc_receiveDeadmanTimer
*  Path         :   1.3.6.1.4.1.26.2.22
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The deadman timer for frame reception on an
*                          interface.  The value is not given in units of
*                          time, but rather as a count of frames transmitted
*                          on the interface.  If no frames are received on
*                          the interface during the interval taken to transmit
*                          this count of frames, then the interface is reset."
*                          "Not supported by HLS834"
*
**************************************************************************/

int svc_receiveDeadmanTimer(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	return NO_SUCH_NAME;
#ifdef 0
	return(svc_snmpnosupport(service,compc,compl,lenp,value,0));
#endif
}

/**************************************************************************
*  Procedure    :   svc_bridgeStpMode
*  Path         :   1.3.6.1.4.1.26.2.26
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "Place the bridge into or out of spanning tree mode."
*
**************************************************************************/

int svc_bridgeStpMode(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;

	rtn_code = chk_index(*compc, *compl);

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        if (rtn_code == 0)
        {
			if( *compl != 0 )
				return(NO_SUCH_NAME);

            if( BridgeStatus->StpMode == STP_MODE_ENABLE )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;
        }
		else
			return NO_SUCH_NAME;
		break;

    case SNMP_SET:
        {
            if( *compc != 1 || *compl > 4 )
                return(rtn_code = NO_SUCH_NAME);

            switch( *value )
            {
                 case ENABLE:
					 /* Add the STP address to the working database if stp */
					 /* was previously disabled */
					 if(BridgeStatus->StpMode != STP_MODE_ENABLE)
						 AddStpAddr();
                     BridgeStatus->StpMode = STP_MODE_ENABLE;
                     prcctl->Prc_StpMode   = (byte)STP_MODE_ENABLE;
					 enable_stp();
                     SaveBStatus();
                     break;

                 case DISABLE:      
					 /* Delete the STP address from working database if stp */
					 /* was previously disabled */
					 if(BridgeStatus->StpMode != STP_MODE_DISABLE)
						 DelStpAddr();
                     BridgeStatus->StpMode  = STP_MODE_DISABLE;
                     prcctl->Prc_StpMode    = (byte)STP_MODE_DISABLE;
					 disable_stp();
                     SaveBStatus();
                     break;
  
                default:
                    return(rtn_code = BAD_VALUE);
                    break;
            } /* end of *value switch   */
            break;
        }

    case SNMP_TEST:
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
		if(*value != ENABLE && *value != DISABLE)
			return BAD_VALUE;
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_bridgeConsoleTimeout
*  Path         :   1.3.6.1.4.1.26.2.27
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                           "Integer value represents the number of minutes that 
*                            the bridge will wait until the console session is 
*                            logged out, due to inactivity. A value of 0 will 
*                            disable the timeout feature. Range 0-60 minutes."
*
**************************************************************************/

int svc_bridgeConsoleTimeout(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			if( *compl != 0 )
				return(NO_SUCH_NAME);

			*value = (UINT_32_T)BridgeStatus->TimeOut ;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		{
	            if( *value > 60  || *compc != 1 )
			return( BAD_VALUE );

                    BridgeStatus->TimeOut = *value;
		    SaveBStatus();

		}
		break;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
	            if( *value > 60  || *compc != 1 )
			return( BAD_VALUE );
			
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}


chk_portinst(inst_cnt,instp,port)
unsigned int inst_cnt;
unsigned int  *instp;
unsigned int *port;
{
        if ( inst_cnt == 1 && (*instp > 0 && *instp <= NumberOfSonicPort + 1) )
        {
                *port = *instp;
                return 0;
        }
        return NO_SUCH_NAME;
}

get_nxtportinst(compc,compl,port)
int  *compc;
unsigned int  *compl;
int *port;
{
        if (*compc == 0)
        {
                *port = 0;
                *compl = 1;
                *compc = 1;
                return 0;
        }
        if (*compc > 1)
                return ERR_INSTANCE;
        else
        {
                if ( *compl < NumberOfSonicPort + 1 )
                {
                        *port = *compl;
                        *compl += 1;
                        *compc = 1;
                        return 0;
                }
                else
                        return NO_SUCH_NAME;
        }
}


