/**************************************************************************
 *  Copyright (c) 1992 Hughes LAN Systems                                 *
 *  All rights Reserved                                                   *
 *                                                                        *
 *  This file contains unpublished proprietary source code of Hughes LAN  *
 *  Systems.                                                              *
 **************************************************************************
                                                                           
   This file contains the service routines for HLS-MIB
   It is directly derived from the MIB specification using the AWK program 
   'MIBSVC.AWK'
   Should significant changes be needed in this file that are NOT          
   reflected back into the MIB specification, then modify this header to   
   indicate that this is the case.                                         

   $Log:   /b/gregs/bridge/snmp/routines/hls.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:37:48   vinay
 * fixed the invalid ip address problem
 * 
 *    Rev 1.1   18 Aug 1993 18:20:26   vinay
 * fixed bootpromversion and systemresetlevel functions
 * 
 *    Rev 1.0   20 Jul 1993 09:37:06   franks
 * Initial revision.
 * 
 *    Rev 1.1   11 May 1992 09:29:08   franks
 * No change.
 * 
 *    Rev 1.0   30 Mar 1992 17:37:34   pvcs
 * Initial revision.
                                                                           
 *************************************************************************/

#define HLS-MIB_SVC

#include <types.h> 
#include <target.h>
#include <krnl.h>
#include <dbd.h>
#include <snmp.h>
#include <asn1.h>
#include <syteksnm.h>
#include <prcctl.h>
#include <tcpip.h>
#include <eeprecs.h>
#include <nim960h.h>
#include <nvrecs.h>
#include <sys.h>
#include <dips.h>
#include <bridges.h>

extern	NIM960_HDR	Nim960Header;

extern	ulong	InitTimeSeconds;
extern	ulong	InitDateSeconds;

extern char *chk_time();
extern char *chk_date();

/*
 *	when the counters was cleared (in multiple of 10ms)
 */
extern	unsigned int	ClearCounterTicks;

static	in_name	new_ip;

/**************************************************************************
*  Procedure    :   svc_sysType
*  Path         :   1.3.6.1.4.1.26.3
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   " The type of the system."
*
**************************************************************************/

int svc_sysType(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) 1;/* bridge */
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_serialNumber
*  Path         :   1.3.6.1.4.1.26.4
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "The serial number of the system being managed."
*
**************************************************************************/

int svc_serialNumber(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (long )eep_mfg_rec.eep_srlno;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_bootPromVersion
*  Path         :   1.3.6.1.4.1.26.5
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The boot prom version number of the system being
*                    managed."
*
**************************************************************************/

int svc_bootPromVersion(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	NIM960_HDR	*BootPromHeader = (NIM960_HDR *)BEPROM512;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			 *	2.24.92 K Kong
			 *	The boot prom size may be 128K or 64K for 
			 *	the rev. 2 hardware. We have to check the
			 *	nim960 header to see what the size of
			 *	the boot prom is.
			 *	This code should be taken out one everybody
			 *	move to the new hardware.
			 */
			if (BootPromHeader->signature != NIM960_MAGIC ||
				BootPromHeader->Entry == NULL ||
				cksum(BootPromHeader, BootPromHeader->FileLength >> 1) != 0xffff)
				{
				BootPromHeader = (NIM960_HDR *)BEPROM512;
				}
    			*lenp = strlen(BootPromHeader->version);
			memcpy(value, BootPromHeader->version, *lenp);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_sysAdmAccess
*  Path         :   1.3.6.1.4.1.26.6
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "The method being used to authenticate a user
*                    accessing the administration console of the system
*                    being managed."
*
**************************************************************************/

int svc_sysAdmAccess(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			 *	password unless switch off by the
			 *	DIP switch.
			 */
			*value = (UINT_32_T)(sys.sys_switch & DIP_PASSWD_OFF 
				? 1 : 2);
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		if (*value == 1 && (sys.sys_switch & DIP_PASSWD_OFF))
			break;
		else if (*value == 2 && !(sys.sys_switch & DIP_PASSWD_OFF))
			break;
		return BAD_VALUE;
	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if ((*value == 1 && (sys.sys_switch & DIP_PASSWD_OFF)) ||
				(*value == 2 && !(sys.sys_switch & DIP_PASSWD_OFF)))
				return rtn_code;
			else
				return BAD_VALUE;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_appSwVersion
*  Path         :   1.3.6.1.4.1.26.7
*  Access       :   RO
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The application software  version number of the
*                    system  being managed."
*
**************************************************************************/

int svc_appSwVersion(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*lenp = strlen(Nim960Header.version);
			memcpy((char  *)value, Nim960Header.version, *lenp);
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_hlsMibVersion
*  Path         :   1.3.6.1.4.1.26.9
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "The HLS Mib  version number supported by the system
*                    being managed. The version of MIB defined by this
*                    document is version3."
*
**************************************************************************/

int svc_hlsMibVersion(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = SNMP_MIB_VERSION;
		}
		else
			return NO_SUCH_NAME;
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_systemReset
*  Path         :   1.3.6.1.4.1.26.11
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Writing to this object causes the agent to be
*                    rebooted.  The value written to the object
*                    determines default system  configuration after
*                    system is rebooted. The read value is the last reset
*                    level of the system.  Hardware failure, power
*                    failure, and software failure are represented by
*                    value level1."
*
**************************************************************************/

int svc_systemReset(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	int	count;
	int	level;	/* reset level	*/
	RESETLOG	*resetlog = &RstLog->nvr_resetlog;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			/*
			 *	Get the last reset record.
			 */
			count = resetlog->rstlog_count;
			if (count > RSTLOG_COUNT)
				count = count % RSTLOG_COUNT;
			level = resetlog->rstlog[count - 1].rstlog_msg_no;

			/*
			 *	Power failure, button reset snmp reset,
			 *	software crash and self test failure are considered
			 *	to be level 0 reset.
			 */
			if (level < RSTMSG5)
				level = 1;
			else
				{
				level -= RSTMSG5;
				/*
				 *	there is no level 2,
				 *	level 2 -> 3
				 *	level 3 -> 4.
				if (level >= 2)
					level++;
				 */
				}

			*value = (UINT_32_T)level; 

		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		if (*value < 1 || *value > 4)
			return NO_SUCH_NAME;
		printf("\nWarning: This module is being reset by the network manager ...\n");
		UpdateSoftwareReset(*value);
		sys_flag = 2;
		break;

	case SNMP_TEST:
		if(rtn_code == 0 )
		{
			if (*value < 1 || *value > 4 )
				return BAD_VALUE;
			break;
		}
		else
			return rtn_code;

	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_clearCounter
*  Path         :   1.3.6.1.4.1.26.12
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Writing any value to this object will clear
*                    counters of the agent. The agent always returns 0.
*                    In the case where multiple SNMP managers may be
*                    managing the agent, it is inadvisable to clear the
*                    counters."
*
**************************************************************************/

int svc_clearCounter(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (UINT_32_T) 0;
		}
		else
			return NO_SUCH_NAME;
		break;

	case SNMP_SET:
		ClearMainTrScrn();	/* clear all the counters */
		SnmpClearCounters();	/* snmp counters	*/
		TcpClearCounters();	/* tcp counters		*/
		UdpClearCounters();	/* udp counters		*/
		IpClearCounters();	/* ip counters		*/
		IcmpClearCounters();	/* icmp counters	*/
		StpClearCounters();
		IfClearCounters();
		ClearCounterTicks = RealTimeTicks();
		break;

	case SNMP_TEST:
		return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_systemDate
*  Path         :   1.3.6.1.4.1.26.13
*  Access       :   RW
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The current system date of the system being
*                    managed, formatted in ASCII string MM-DD-YY"
*
**************************************************************************/

int svc_systemDate(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	char	buf[10];
	uint	yy;
	uint	mm;
	uint	dd;
	char	c;


	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
		        GetDate(&dd, &mm, &yy);
			if (yy > 1990)
				yy -= 1900;
			sprintf((char *)buf,"%02d-%02d-%02d", mm, dd, yy);
			*lenp = strlen(buf);
			memcpy((char  *)value, (char *)buf, *lenp);
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		memcpy((char  *)buf, (char *)value, *lenp);
		buf[*lenp] = '\0';
		/*
		 *	Check if the string is valid.
		 *	We should have checked the date(dd) is ok for a 
		 *	particular month and year.
		 *	i.e. if mm is 1, max value of mm is 31
		 *		mm is 2, max value of mm is 29 in a leap year
		 *		or 28 otherwise
		 *		... so on.
		 */
		if (sscanf(buf, "%02d-%02d-%02d%c",&mm, &dd, &yy, &c) != 3
			|| 1 > mm || mm > 12 || 1 > dd || dd > 31)
			{
			return BAD_VALUE;
			}
		/*
		 *	The year will be in 199x or 2xxx
		 */
		if (yy > 90)
			yy += 1900;
		else
			yy += 2000;
		/*
		 *	adjust mm as  0 - 11 is expected.
		 */
		InitDateSeconds = DateToSeconds(dd, mm - 1, yy);
		break;

	case SNMP_TEST:
		if(*lenp > 8)
			{
			/* value to long */
			return BAD_VALUE;
			}
		if ( rtn_code == 0 )
			{
				memcpy((char  *)buf, (char *)value, *lenp);
				buf[*lenp] = '\0';
				if (sscanf(buf, "%02d-%02d-%02d%c",&mm, &dd, &yy, &c) != 3
					|| 1 > mm || mm > 12 || 1 > dd || dd > 31)
					{
					return BAD_VALUE;
					}
				if(mm == 2 && dd >29)
					return BAD_VALUE;
			break;
			}
		return rtn_code;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_systemTime
*  Path         :   1.3.6.1.4.1.26.14
*  Access       :   RW
*  Syntax       :   DisplayString VT_STRING
*  Description  :   
*                   "The current system time  of the system being
*                    managed, formatted in ASCII HH:MM:SS."
*
**************************************************************************/

int svc_systemTime(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int	rtn_code;
	char	buf[10];
	uint	hour;
	uint	min;
	uint	sec;
	uchar	c;


	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			GetTime(&hour, &min, &sec);
			sprintf((char *)buf,"%02d:%02d:%02d", hour, min, sec);
			*lenp = strlen(buf);
			memcpy((char  *)value, buf, *lenp);
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		memcpy(buf, (char  *)value, *lenp);
		buf[*lenp] = '\0';
		if (sscanf(buf, "%02d:%02d:%02d%c", &hour, &min, &sec, &c) != 3 ||
			hour > 24 || min > 60 || sec > 60)
			return BAD_VALUE;
		InitTimeSeconds = (hour*3600) + (min*60) + sec;
		InitTimeSeconds -= RealTimeTicks()/100;
		break;

	case SNMP_TEST:
		if(rtn_code == 0)
		{
			if(*lenp > 8)
				return BAD_VALUE;
			memcpy(buf, (char  *)value, *lenp);
			buf[*lenp] = '\0';
			if (sscanf(buf, "%02d:%02d:%02d%c",&hour,&min,&sec,&c) != 3 ||
				hour > 24 || min > 60 || sec > 60)
				return BAD_VALUE;
			break;
		}
		else
			return(rtn_code);
		break;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_sysIpAddress
*  Path         :   1.3.6.1.4.1.26.17
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The IP address  of  the system being managed by
*                    SNMP managers."
*
**************************************************************************/

int svc_sysIpAddress(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	char *ipAddr;
	char ipa[20];
	IP ipdevice;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (INT_32_T)_initp->in_me;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		/*
		 *	Don't change the ip yet.
		 *	We want to reply to the net manager using
		 *	our old ip before we change the ip.
		 */
		new_ip = *value;
		sys_flag = 1;
		break;

	case SNMP_TEST:
		if(*value != 0)
		{
			ipAddr = (char *)inet_ntoa(*value);
			if(GetIPAddress(&ipdevice,ipAddr,&ipa))
				return BAD_VALUE;
		}
		return(rtn_code);
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_primaryRouter
*  Path         :   1.3.6.1.4.1.26.18
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The IP address of the primary default router which
*                    the system forwards IP datagram destined to remote
*                    networks."
*
**************************************************************************/

int svc_primaryRouter(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;
	char *ipAddr;
	char ipa[20];
	IP ipdevice;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (INT_32_T)_initp->net_gway;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		eep_boot_rec.eep_RouterIP[0]= *((INT_32_T  *)value);
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		_initp->net_gway = *value;
		/*
		 *	Inform the adminbus driver that the router
		 *	address has been changed.
		 */
		AdmSetRouter(*value);
		break;

	case SNMP_TEST:
		if(*value != 0)
		{
			ipAddr = (char *)inet_ntoa(*value);
			if(GetIPAddress(&ipdevice,ipAddr,&ipa))
				return BAD_VALUE;
		}
		return rtn_code;
	}
	return 0;
}

/**************************************************************************
*  Procedure    :   svc_sysNetMask
*  Path         :   1.3.6.1.4.1.26.19
*  Access       :   RW
*  Syntax       :   IpAddress VT_IPADDRESS
*  Description  :   
*                   "The net mask which the system uses to determine if
*                    a packet is destined to a remote network or local
*                    network when the system forwards IP datagrams."
*
**************************************************************************/

int svc_sysNetMask(service, compc, compl, lenp, value, index)
unsigned int	service;
byte		*compc;
unsigned int	*compl;
int		*lenp;
UINT_32_T	*value;
int		index;
{
	int rtn_code;

	rtn_code = chk_index(*compc, *compl);

	switch(service)
	{
	case SNMP_GETNXT:
		rtn_code = get_nxtidx((int *)compc,
					(unsigned int *)compl, rtn_code);
		/* FALL TRHROUGH */
	case SNMP_GET:
		if (rtn_code == 0)
		{
			*value = (INT_32_T)_initp->net_mask;
		}
		else
			return NO_SUCH_NAME;
		break;
	case SNMP_SET:
		eep_boot_rec.eep_NetMask= *value;
		PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
		_initp->net_mask = *value;
		/*
		 *	Tell the Adminbus driver that the net mask has been
		 *	changed.
		 */
		AdmSetNetMask(*value);
		break;

	case SNMP_TEST:
		return rtn_code;
	}
	return 0;
}
/*
 *	The snmp response has been sent to the network manager,
 *	we can now change the IP.
 */
SetNewIP()
	{
	int	CurrentTime;

	/*
	 *	Tell the Adminbus driver that the my ip address has been
	 *	changed.
	 */
	AdmSetIP(new_ip);
	/*
	 *	Close telnet if it is active as it cannot talk
	 *	to me anymore.
	 */
	if (IsTelnetActive())
		{
		printf("\nWarning: The IP address of this module has been\n");
		printf("changed by the Network manager from %s to ",
			inet_ntoa(_initp->in_me)); 
		printf("%s.\n", inet_ntoa(new_ip));
		printf("The current session is disconnected\n\n");
		CurrentTime = RealTimeTicks();
		/*
		 *	Wait 1/2 second for the messages to go out before
		 *	we disconnect.
		 */
		while((RealTimeTicks() - CurrentTime) < 50)
			ReSchedule();
		CloseTelnet();
		}
	_initp->in_me = new_ip;
	eep_boot_rec.eep_IP = new_ip;
	PutEeprom(EEP_BOOT_ADDR, &eep_boot_rec, EEP_BOOT_SIZE);
	}
