/**			       
*
*	Program Name:	TESTER
*
*	Filename:	ramtest.c
*
*	$Log:   /h/pvcs/bridge/tester/ramtest.c_v  $
 * 
 *    Rev 1.1   14 Apr 1992 10:42:22   kwok
 * If it is SRAM, reduce the size to be tested by 64 K.
 * 
 *    Rev 1.0   30 Mar 1992 17:42:28   pvcs
 * Initial revision.

*	Creation Date:	6.14.91
*
*
*	Date:
*
*	Version:	1.0
*
*	Programmers:	K Kong
*
*	Modifications:
*
*	Comments:	It contains functions to test memories.
*			This test is suitable for eprom based only as
*			some of the tests are destructive.
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/




#include <types.h>
#include <memory.h>


#define	TEST_BYTE_PATTERN1	0x55
#define	TEST_BYTE_PATTERN2	0xaa
#define	TEST_WORD_PATTERN	0x5a5a5a5a
extern	unsigned * QuadTest();
int	TestBits(volatile unsigned *address, int size, char *RamType);
void 	*RamByteTest(unsigned char *address, int size);
void 	*RamWordTest(unsigned *address, int size);
void 	*RamWordAddressTest(unsigned *address, int size);

/*
 * name		RamTest	- test ram
 *
 * synopsis	RamTest(address, size, RamType)
 *		byte	*address; <<	start address of ram to be tested
 *		int	size; <<	of ram to be tested in number of 
 *					bytes
 *		char	*RamType; <<	what sort of ram i.e.
 *					"DRAM", "SRAM", "SHRAM" ...etc
 *
 * description	It verifies that the ram is working properly.
 *		It does the following test:-
 *		1	write a byte to one location will not modify other
 *			locations.
 *		3	write the word pattern of 0x5a5a5a5a to all 
 *			locations and read them back.
 *		4	write quad word pattern of 
 *			0x55555555aaaaaaaa9999999966666666 to all locations
 *			and read them back.
 *		5	use the address as the word pattern.
 *		6	Every bit of the memory is set and cleared.
 *
 * returns	0		test passed
 *		otherwise	test failed
 */

RamTest(byte *address, int size, char *RamType)

	{
	void	*ram = NULL;
	int	ret;

	/*
	 *	Do not touch the first 64 k as they are used
	 *	by this program.
	 */
	if (address == (byte *)SRAM)
		{
		address += 0x10000;
		size -= 0x10000;
		}
	/*
	 *	Test 1
	 */
	Di();
	ram = RamByteTest(address, size);
	Ei();
	if (ram != NULL)
		{
		return 1;
		}

	/*
	 *	Test 3
	 */
	Di();
	ram = RamWordTest((unsigned *)address, size);
	Ei();
	if (ram != NULL)
		{
		return 1;
		}
	/*
	 *	Test 5
	 */
	Di();
	ram = RamWordAddressTest((unsigned *)address, size);
	Ei();
	if (ram != NULL)
		{
		return 1;
		}
	/*
	 *	Test 4
	 */
	Di();
	ram = QuadTest(address, size);
	Ei();
	if (ram != NULL)
		{
		return 1;
		}
	Di();
	ret = TestBits((volatile unsigned *)address, size, RamType);
	Ei();
	return ret; 
	}


/*
 * name		TestBits
 *
 * synopsis	TestBits(address, size, *RamType)
 *		volatile unsigned	*address; <<	of ram to be tested.
 *		int	size; <<	of ram in number of
 *					bytes to be tested
 *		char	*RamType; <<	i.e. "DRAM", "SRAM" ...etc
 *
 * description	It sets and clears every bits of the ram a word at 
 *		a time to check that the ram is ok.  This test is 
 *		non-destructive.  i.e  the data in ram will not be
 *		modified.
 *
 * returns	0		test passed
 *		otherwise	test failed
 */
TestBits(volatile unsigned *address, int size, char *RamType)

	{
	/*
	 *	We choose test pattern1 and pattern2 such that
	 *	pattern1 xor pattern2 gives 0xffffffff.
	 */
	register unsigned	pattern1 = 0x55555555;
	register unsigned	pattern2 = 0xaaaaaaaa;
	register unsigned	data;
	register unsigned	pattern1_data;
	register unsigned	pattern2_data;
	/*
	 *	size in number of unsigned.
	 */
	size = size / sizeof(unsigned);
	while(size--)
		{
		/*
		 *	Save the original data in address.
		 */
		data = *address;
		*address = pattern1;
		/*
		 *	pattern1_data = pattern1.
		 */
		pattern1_data = *address;	
		*address = pattern2;
		/*
		 *	pattern1 xor pattern2 should be 0xffffffff
		 */
		pattern1_data ^= *address;
		*address = data;
		/*
		 *	data xor  itself should be 0
		 */
		data ^= *address;
		/*
		 *	pattern1_data xor 0xffffffff should be 0
		 */
		pattern1_data ^= 0xffffffff;
		/*
		 *	if data and pattern1_data are not zero, we have
		 *	an error.
		 */
		if ((data |= pattern1_data) != 0)
			{
			return 1;
			}
		address++;
		}
	return 0;
	}

/*
 * name		RamByteTest	
 *
 * synopsis	void *RamByteTest(address, size)
 *		unsigned char *	address; <<	of ram to be tested.
 *		int	size; <<		of ram in number of
 *						bytes to be tested
 *
 * description	It writes the byte pattern of 0 - x, where x is 231, 197 and 
 *		83  to
 *		all locations of the ram starting at "address" with
 *		size "size".  It then reads them back a byte at a time to
 *		verify that the pattern has been stored in ram properly.
 *
 * returns	NULL		test passed
 *		otherwise	location at which the test failed
 */

void *RamByteTest(unsigned char *address, int size)

	{
	unsigned char	pattern;
	unsigned char	*start = address;
	unsigned char	*end = address + size;	
	int		i;
	unsigned 	RepeatCount;;


	for (i = 0; i < 3; i++)
		{
		if (i == 0)
			RepeatCount = 231;
		else if (i == 1)
			RepeatCount = 197;
		else
			RepeatCount = 83;
		pattern = 0;
		for (address = start; address < end; address++)
			{
			*address = pattern++;
			if (pattern >= RepeatCount)
				pattern = 0;
			}
		pattern = 0;
		for (address = start; address < end; address++)
			{
			if (*address != pattern++)
				return address;
			if (pattern >= RepeatCount)
				pattern = 0;
			}
		}
	return NULL;
	}

/*
 * name		RamWordTest	
 *
 * synopsis	void *RamWordTest(address, size)
 *		unsigned char *	address; <<	of ram to be tested.
 *		int	size; <<		of ram in number of
 *						bytes to be tested
 *
 * description	It writes the word pattern of "TEST_WORD_PATTERN" to
 *		all locations of the ram starting at "address" with
 *		size "size".  It then reads them back a word at a time to
 *		verify that the pattern has been stored in ram properly.
 *		It assumes that the address is properly aligned.
 *
 * returns	NULL		test passed
 *		otherwise	location at which the test failed
 */

void *RamWordTest(unsigned *address, int size)

	{
	unsigned	pattern = TEST_WORD_PATTERN;
	register unsigned	data;	/* save original data in here */
	register unsigned	SaveNext;
	int	WriteNext = 1;

	size = size / sizeof(unsigned);	/* in number of unsigned	*/
	/*
	 *	write to the ram
	 */
	while(size--)
		{
		if (size == 0)
			WriteNext = 0;
		/*
		 *	Save the original data in address.
		 */
		data = *address;
		/*
		 *	Write a 0 to the bus to let the bus
		 *	to settle down before we try to read back
		 *	the previous word from memory.
		 */
		if (WriteNext != 0)
			SaveNext = *(address + 1);
		*address = pattern;
		if (WriteNext != 0)
			*(address + 1) = 0;
		/*
		 *	read from ram to verify that it's ok
		 */
		if (*address != pattern)
			{
			return (void *)address;
			}
		/*
		 *	restore the data
		 */
		if (WriteNext != 0)
			*(address + 1) = SaveNext;
		*address++ = data;
		}
	return NULL;
	}

/*
 * name		RamWordAddressTest	
 *
 * synopsis	void *RamWordAddressTest(address, size)
 *		unsigned char *	address; <<	of ram to be tested.
 *		int	size; <<		of ram in number of
 *						bytes to be tested
 *
 * description	It writes the word pattern which is the current address 
 *		being tested to	all locations of the ram starting at 
 *		"address" with	size "size".  Every word location contains
 *		different value. It then reads them back a word at a time to
 *		verify that the pattern has been stored in ram properly.
 *
 * returns	NULL		test passed
 *		otherwise	location at which the test failed
 */

void *RamWordAddressTest(unsigned *address, int size)

	{

	volatile unsigned	*ram = (volatile unsigned *)address;
	int		RamSize;

	size = size / sizeof(unsigned);	/* in number of unsigned	*/
	RamSize = size;
	/*
	 *	write to ram
	 */
	while(RamSize--)
		{
		/*
		 *	Use the current address as the testing pattern.
		 */
		*ram = (unsigned)ram;
		ram++;
		}
	ram = address;
	RamSize = size;
	while(RamSize--)
		{
		if (*ram != (unsigned)ram)
			{
			return (unsigned *)ram;
			}
		ram++;
		}

	ram = address;
	RamSize = size;
	/*
	 *	write to ram
	 */
	while(RamSize--)
		{
		/*
		 *	Use the current address with hi 2 bytes and low 2 bytes
		 *	swapped as the testing pattern.
		 */
		*ram = (unsigned)(((unsigned)ram << 16) | ((unsigned)ram >> 16));
		ram++;
		}
	ram = address;
	RamSize = size;
	while(RamSize--)
		{
		if (*ram != (unsigned)((unsigned)ram << 16 | (unsigned)ram >> 16))
			{
			return (unsigned *)ram;
			}
		ram++;
		}
	return NULL;
	}
