/**			       
*
*	Program Name:	Stanely
*
*	Filename:	stdio.c
*
*	$Log:   /b/gregs/bridge/netio/stdio.c_v  $
 * 
 *    Rev 1.4   12 Oct 1993 09:13:48   franks
 * No change.
 * 
 *    Rev 1.3   29 Sep 1993 09:36:44   franks
 * No change.
 * 
 *    Rev 1.2   10 Sep 1993 15:17:54   franks
 * No change.
 * 
 *    Rev 1.1   08 Sep 1993 10:54:38   franks
 * No change.
 * 
 *    Rev 1.0   30 Jul 1993 13:09:24   franks
 * Initial revision.
 * 
 *    Rev 1.1   15 Sep 1992 10:28:24   kwok
 * Return 0 for the function AdmCloseUpcall().
 * 
 *    Rev 1.0   30 Mar 1992 17:32:22   pvcs
 * Initial revision.
*
*	Creation Date:	7.22.91
*
*	Date:
*
*	Version:	1.0
*
*	Programmers:	K Kong
*
*	Modifications:	9.15.92	K Kong
*			AdmCloseUpcall() should return 0.
*
*	Comments:	This file contains the stdio support functions.
*			-	putch()
*			-	getch()
*
*			The stdio port can be one of the following:-
*			-	uart port
*			-	telnet session
*			-	admin bus
*			At any one time, only one of the port will be 
*			active and be the stdio port for the system.
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/



#include <types.h>
#include <krnl.h>
#include <dbd.h>
#include <tcpip.h>
#include <telnet.h>
#include <sys.h>
#include "netio.h"

extern	int 	tnc_valid;	/* telnet session active indicator */
extern	telnet	*tnc;
extern	int	AdmConState[];
extern	int	ConsoleActive;

int	IOpath;		/* where is the current IO path, 
					-	IO_UART
					-	IO_TELNET
					-	IO_ADMIN
		 	*/
/*
 *	To inform the command task that the Remote console has
 *	been disconnected.
 **/
SEM	RemoteDisconnected;

static void UartMonitor(void);
static int TerminateTelnet(void);
static int TerminateAdmin(void);

#define	UART_MONITOR_STACK_SIZE	1024	
uint	UartMonitorStack[UART_MONITOR_STACK_SIZE / sizeof(int)];

/*
 * name		netio_init	init the netion module
 *
 * synopsis	netio_init(void)
 *
 * description	It initializes the netion moudle in a ready state.  It is
 *		assumed that when this function is called, the tcpip stack
 *		has been initialized.
 *
 * returns	0		ok
 *		error number	otherwise
 *
 */
netio_init()

	{
	IOpath = IO_UART;
	sys.sys_remote_console = 0;
	CreatSemaphore(&RemoteDisconnected, 0);
	CreatTask(UartMonitor, UartMonitorStack, UART_MONITOR_STACK_SIZE, (int)0);
	return 0;
	}
/*
 * name		getch
 *
 * synopsis	int getch(void)
 *
 * description	Get a character from the currently active stdio port.
 *
 * returns	next character available
 *		-1	if there is no character 
 */
 
getch()
	{
	int	c;

	switch (IOpath)
		{
		case IO_UART:
		default:
			/*
			 *	The default is the uart
			 */
			return UartGetch();

		case IO_TELNET:
			if ((c = tn_getc(tnc)) == -1 && !IsTelnetActive())
				{
				IOpath = IO_UART;
				sys.sys_remote_console = 0;
				/*
				 *	Tell the command task that the
				 *	remote console connection has been
				 *	disconnected.
				 */
				SendSignal(&RemoteDisconnected);
				}
			return c;

		case IO_ADMIN:
			if ((c = AdmGetch(0)) == -1 && !IsAdmConOpened(0))
				{
				IOpath = IO_UART;
				sys.sys_remote_console = 0;
				/*
				 *	Tell the command task that the
				 *	remote console connection has been
				 *	disconnected.
				 */
				SendSignal(&RemoteDisconnected);
				}
			return c;
		}
	}

/*
 * name		putch
 *
 * synopsis	int putch(c)
 *		int	c; <<		the character to be output
 *
 * description	It prints a character to the currently active stdio port.
 *
 * returns	-1	for error
 *		otherwise ok
 */

putch(int c)

	{
	int	ret;

	switch (IOpath)
		{
		case IO_UART:
		default:
			/*
			 *	The default is the uart
			 */
			return UartPutch(c);

		case IO_TELNET:
			if ((ret = NetPutch(c)) == -1 && !IsTelnetActive())
				{
				IOpath = IO_UART;
				sys.sys_remote_console = 0;
				/*
				 *	Tell the command task that the
				 *	remote console connection has been
				 *	disconnected.
				 */
				SendSignal(&RemoteDisconnected);
				}
			return ret;


		case IO_ADMIN:
			if ((ret = AdmPutch(0, c)) == -1 && !IsAdmConOpened(0))
				{
				IOpath = IO_UART;
				sys.sys_remote_console = 0;
				/*
				 *	Tell the command task that the
				 *	remote console connection has been
				 *	disconnected.
				 */
				SendSignal(&RemoteDisconnected);
				}
			return ret;
		}
	}

/*
 * name		UartMonitor
 *
 * synopsis	void UartMonitor(void)
 *
 * description	This is a standalone task which reads from the local
 *		console port if there is an active remote session.
 *		If a key has been pressed, it askes the user if the
 *		user wants to force the remtoe session to close down
 *		and returns to the local console.  If the user wants
 *		to return to the local console, the remote session will
 *		be closed down.
 *
 * returns	it never returns
 */

static void UartMonitor()

	{
	int	c;
	int	closing = 0;

	for (;;)
		{
		if (IOpath != IO_UART && (c = UartGetch()) != -1 
			&& closing == 0)
			{
			/*
			 *	It the uart session is inactive,
			 *	we monitor the uart port.  If a key
			 *	has been pressed, we ask the user if the
			 *	user wants to terminate the telnet session
			 *	and force the system to return to the local
			 *	console port.
			 */
			if (IOpath == IO_TELNET && TerminateTelnet() != 0)
				{
				NetPuts("\r\n\tLocal console activated. Session will be closed.\r\n");
				tn_close(tnc, 100);	/* 1 seconds	*/
				UartPuts("\r\nRemote telnet session terminated\r\nControl is returned to the local console\r\n");
				/*
				 *	Flush the uart input
				 *	buffer
				 */
				while (UartGetch() != -1)
					continue;
				closing = 1;
				}
			if (IOpath == IO_ADMIN && TerminateAdmin() != 0)
				{
				AdmPuts(0, "\r\n\tLocal console activated. Session will be closed.\r\n");
				UartPuts("\r\nAdmin. bus console terminated\r\nControl is returned to the local console\r\n");
				/*
				 *	Flush the uart input
				 *	buffer
				 */
				while (UartGetch() != -1)
					continue;
				closing = 1;
				AdmConClose(0);
				IOpath = IO_UART;
				sys.sys_remote_console = 0;
				}
			if (closing == 1)
				{
				/*
				 *	Tell the command task that the
				 *	remote console connection has been
				 *	disconnected.
				 */
				SendSignal(&RemoteDisconnected);
				}
			}
		else
			closing = 0;
		/*
		 *	Do nothing if there is no telnet session
		 */
		ReSchedule();
		}
	}
/*
 * name		TerminateTelnet
 *
 * synopsis	TerminateTelenet()
 *
 * description	It asks the user if the users wants to force the active
 *		telnet session to close and returns to the local console.
 *
 * returns	0	- do not terminate the telnet session
 *		otherwise	- terminate the telnet session
 */

static int TerminateTelnet()
	{
	int	c;

	UartPuts("\r\nThere is an active remote telnet session from ");
	UartPuts(inet_ntoa((byte *)tnc->tn_fhst));
	UartPuts("\r\nDo you want to force the remote telnet to disconnect\r\n");
	UartPuts("and return to the local console ? [y/(n)] ==> ");
	while ((c = UartGetch()) == -1)
		ReSchedule();
	UartPutch(c);
	UartPuts("\r\n\n");
	return (c == 'y') || (c == 'Y');
	}	       

/*
 *	Put a string to the uart port
 */
UartPuts(char *message)

	{
	while (*message != NULL)
		{
		while (UartPutch(*message) == -1)
			ReSchedule();
		message++;
		}
	}
/*
 * name		TerminateAdmin
 *
 * synopsis	TerminateAdmin()
 *
 * description	It asks the user if the users wants to force the active
 *		Admin. bus console to close and returns to the local console.
 *
 * returns	0	- do not terminate
 *		otherwise	- terminate 
 */

static int TerminateAdmin()
	{
	int	c;

	UartPuts("\r\nThere is an active remote session from the admin. bus\r\n");
	UartPuts("\r\nDo you want to force the admin bus console to disconnect\r\n");
	UartPuts("and return to the local console ? [y/(n)] ==> ");
	while ((c = UartGetch()) == -1)
		ReSchedule();
	UartPutch(c);
	UartPuts("\r\n\n");
	return (c == 'y') || (c == 'Y');
	}	       

/*
 *	CON open request from the admin. bus
 */
AdmOpenUpcall(int slot)

	{
	if (IOpath == IO_TELNET || ConsoleActive)
		{
		/*
		 *	Do not accept open call when the telnet
		 *	session or the local console is active.
		 */
		return 1;
		}
	IOpath = IO_ADMIN;
	sys.sys_remote_console = 1;
	return 0;
	}

AdmCloseUpcall(int slot)

	{
	if (IOpath == IO_ADMIN)
		{
		IOpath = IO_UART;
		sys.sys_remote_console = 0;
		/*
		 *	Tell the command task to ask for the
		 *	password.
		 */
		SendSignal(&RemoteDisconnected);
		}
	return 0;
	}

InactiveClose()
	
	{
	switch (IOpath)
		{
		case IO_TELNET:
			CloseTelnet();
			break;

		case IO_ADMIN:
			AdmConClose(0);
			IOpath = IO_UART;
			sys.sys_remote_console = 0;
			break;

		}
	}
