#include <types.h>
#include <eeprecs.h>
#include "flash.h"
#include <nvram.h>
#include <nvrecs.h>
#include <nim960h.h>
#include <time.h>
#include <led.h>

/*
NVR_BOOT nvr_boot_rec;
*/
extern int exit_stat;
extern char *to;
extern int out_length;
extern int program_id;
extern word aplLocation;

int program_flash(void *, byte *, int);
void *down_load_flash();
int copy_to_flash(void *, byte *, int , int);

struct flash_info flash_info;

/************************************************************************
*
*	Function:	program_flash
*
*	Description:	Write a block of data into the flash memory
*			if enough memory is available, compress the data first
*			if not, if there is enough flash memory, do a
*			straight memory to flash copy
*
*	Inputs:		feprom = address of start of flash memory
*			from   = address of data to be copied
*			size   = size of data to be copied in bytes
*
*	Outputs:	None
*
*	Returns:	0 = error
*			1 = OK
*
**************************************************************************/

/*--------------------------------------------------------------------------
 *	Write a block of data into the flash memory
 * Return: 1 -- OK.
 *	   0 -- error.
 */
int
program_flash(void *feprom, byte *from, int size)
{
	register int retcode;
	register NIM960_HDR *header;
	register unsigned long csum;
	register char *cp,*loadaddr;
	register int size_x;

	/*
	* if size is zero - good return
	*/
	if (size == 0)
		return(1);

	/*
	* can not handle negative size
	*/
	if (size < 0)
		{
		printf("Error: Flash write size (%d) is negative\n",
			size);
		return(0);
		}

	/*
	* clear flash record
	*/
	memset(&flash_info, 0, sizeof(struct flash_info));

	/*
	* build the new flash descriptor
	build_flash_record(&flash_info, feprom);
	*/
	build_flash_record();

	/*
	* copy the fields from the code to the descriptor
	*/
	header = (NIM960_HDR *)from;
	strcpy(flash_info.board_name, header->name); /*, sizeof(flash_info.board_name));*/
	strcpy(flash_info.copyright, header->CopyRightMessage); /*, sizeof(flash_info.copyright));*/
	strcpy(flash_info.version, header->version);/*, sizeof(flash_info.version));*/

	/*
	* save the decompressed size
	*/
	flash_info.decomp_size = size;

	/*
	* show the file in flash is compressed
	*/
	flash_info.load_type = UNIX_COMPRESSION;
	/*
	* erase the flash
	* we are erasing the uncompressed size
	* assumption is that it is bigger than the compressed size
	*/
	size_x = size;
	if (size_x > (eep_mfg_rec.eep_fprom_size - aplLocation))
		size_x = eep_mfg_rec.eep_fprom_size - aplLocation;

	set_leds(LED_BI, LED_BI_COLOR, LED_RDY_GRN | LED_ACT_RED);
	erase_flash(feprom, size_x, 1);

	/*
	* open the flash for writing
	*/
	open_flash();

	/*
	* compress and write the data to the flash
	*/
	retcode = compress(from, size, (feprom + sizeof(struct flash_info)));
	/*
	* close the flash for writing so the read will work properly
	*/
	close_flash();

	if (retcode < 0)
		{
		if (exit_stat != NOMEM)
			{
			printf("Error: writing of the compressed data to the flash failed: retcode = %x exit_stat = %x\n", retcode,exit_stat);
			return(0);
			}
		else if ((sizeof(struct flash_info) + size + aplLocation) < eep_mfg_rec.eep_fprom_size)
			{
			cp = (char *)(feprom + sizeof(struct flash_info));
			retcode = copy_to_flash(cp, from, size, 1);

			if (retcode == 0)
				{
				/*
				* check the code to see if it is good
				*/
				if (check_code((NIM960_HDR *)cp, 1) == 0)
					/*
					* no good return error
					*/
					return(0);
	
				flash_info.load_type = UNCOMPRESSED;
				retcode = size;
				}
			else
				return(0);

			}
		/* jlin: Should include NOMEM case. 06/09/93 */
		else
		  {
		    printf("Error: Running out of memory when compressing the program.\n");
		     return (0);
		  }
		}

	/*
	* save the compressed size and the total size
	*/
	flash_info.comp_size = retcode;
	flash_info.info_size = sizeof(struct flash_info);
	flash_info.checksum  = 0L;

	/*
	* cannot handle a size bigger than the flash size
	*/
	if (aplLocation + flash_info.info_size + flash_info.comp_size > eep_mfg_rec.eep_fprom_size)
		{
		printf("Error: Flash write size (%d) is bigger than the flash size (%d)\n",
			size, eep_mfg_rec.eep_fprom_size);
		return(0);
		}
	/*
	* calculate the checksum for the flash info record plus the data
	*/

	csum  = cksum((ushort *)&flash_info, (sizeof(struct flash_info) >> 1));
	delay(200); /* This delay is needed as there is a flash read problem */


	csum += cksum((ushort *)(feprom + sizeof(struct flash_info)), (flash_info.comp_size >> 1));
	while (csum >> 16)
		csum = (csum & 0xFFFFl) + (csum >> 16);
	flash_info.checksum  = ~((ushort)csum);
	delay(200); /* This delay is needed as there is a flash read problem */
	/*
	* write the flash info record to the flash
	*/
	Copy_flash_info(feprom, sizeof(struct flash_info), 0);


	/*
	* check the flash
	*/
	printf("Checking flash eprom \n");

	if (check_flash((struct flash_info *)feprom, 1) != 0)
		return(0);
	/*
	* if the data was written compressed, we need to try to uncompress
	* the data and compare it to the original
	* The uncompressed data was compared when it was copied
	*/
	if (flash_info.load_type != UNCOMPRESSED)
		{
		/*
		* compare downloaded code with the original
		*/
		if ((cp = down_load_flash((struct flash_info *)(FEPROM+aplLocation),1)) == 0)
			return(0);

		printf("Comparing decompressed code\n");
		if (memcmp(cp, from, size))
			{
			printf("Error: Compare of original code and decompressed code failed\n");
			/*
			* free the decompressed test copy of flash
			lmfree(cp, flash_info.decomp_size);
			*/
			free(cp);
			ErrorAbort(4); /* Display Error Code E4 */

			return(0);
			}

		/*
		* free the decompressed test copy of flash
		lmfree(cp, flash_info.decomp_size);
		*/
		free(cp);
		}
	/*
	* write of flash was successfull
	* now copy necessary information to the nvram also
	copy_flash_info(&nvr_boot_rec, &flash_info);
	*/

	return(1);
}

/************************************************************************
*
*	Function:	down_load_flash
*
*	Description:	Return the address of the code to be loaded into DRAM.
*			If flash is compressed, uncompress it into DRAM.
*			if flash is uncompressed, just return the address
*			of the code.  It is assumed that the code in the
*			flash immediately follows the flash record
*
*	Inputs:		flash_info = address of flash record in flash
*			
*	Outputs:	None
*
*	Returns:	0  = error
*			!0 = address of code to be moved and executed
*
**************************************************************************/

void *
down_load_flash(struct flash_info *flashInfo, int cmp_mem_flag)
{
	register int retcode;
	register byte *cp, *loadaddr;
	NIM960_HDR  *header;


	/*
	* check how the code was written to the flash
	*/
	if (flashInfo->load_type == UNCOMPRESSED)
		{
		/*
		* flash is uncompressed
		* return flash address of code
		*/
		cp = (byte *)(FEPROM + aplLocation + flashInfo->info_size);
		retcode = flashInfo->decomp_size;
		}

	else if (flashInfo->load_type == UNIX_COMPRESSION)
		{
		/*
		* flash was compressed using the unix compression
		* if there is enough memory - decompress the flash
		*/
		cp = (byte *) lmalloc(flashInfo->decomp_size);
		if (cp)
			{
			printf("Uncompressing and Downloading the code from flash ");
			retcode = decompress((byte *)(flashInfo + sizeof(struct flash_info)), flashInfo->comp_size, cp);
			if (retcode < 0)
				{
				if (exit_stat == NOMEM)
					{
					printf("\nNot enough memory to Uncompress flash\n");
					printf("Can not do download and compare\n");
					}
				else
					{
					printf("Error while decompressing = %x\n", exit_stat);
					return(0);
					}
				}

			else if (retcode != flashInfo->decomp_size)
				{
				printf("decompress: file size error - is %X should be %X\n",
					retcode, flashInfo->decomp_size);
				return(0);
				}
			}
		else
			{
			printf("\nNot enough memory to Uncompress flash\n");
			printf("Can not do download and compare\n");
			}
		}
	else
		cp = (byte *)(FEPROM + aplLocation);

	/*
	* check the code to see if it is good
	*/
	if (check_code((NIM960_HDR *)cp, 1) == 0)
		/*
		* no good return error
		*/
		return(0);

	/*
	** if the cmp_mem_flag is set, return the pointer without doing memcpy
	*/
	if(cmp_mem_flag)
		return((char *)cp);
	/*
	* Copy the code to the text area specified 
	* in the text area.
	*/
	header = (NIM960_HDR *)cp;
	loadaddr = (byte *)header->TextLoadAddress;
	memcpy(loadaddr,cp,retcode);
	free(cp);

	return(loadaddr);
}


Copy_flash_info(void *to,int size, int print)
{
	register byte *src;
	register byte *dst;
	register int retcode;
	register int count;
	
	/*
	* Burn the flash EPROM
	*/
	open_flash();

	if (print)
		printf("Writing flash info to flash eprom ");

	src = (byte *)&flash_info;
	dst = (byte *)to;
	count = size;
	while (count--)
		{
		if (retcode=write_flash(dst, *src))
			{
			close_flash();
			return(retcode);
			}

		src++;
		dst++;

		if ((count & 0xff) == 0)
			ReSchedule();

		if (((count & 0xffff) == 0) && print)
			printf("-");
		}

	if (print)
		printf("\n");

	close_flash();

	/*
	* read back and compare 
	*/
	if (print)
		printf("Comparing flash info ");
	src = (byte *)&flash_info;
	dst = (byte *)to;
	count = size;

	ResetFlashForRead();
	delay(300); 	/* 
			Since there is no pull down resistor for FLASH
			it takes a while for the voltage to come down and the 
			Flash is in an unknown state. This delay takes care 
			of that. Test the Flash programming by taking out
			this dealy and all the other delays when a pull down
			resistor is added.
			*/
			
	while (count--)
		{
		register byte sb;
		register byte db;
		FlashDelay10Us();
		sb = *src;
		db = *dst;
		if (sb != db)
			{
			printf("compare error source %X (%x)  destination %X (%x)\n",
				src, sb, dst, db);
			return(1);
			}

		src++;
		dst++;

		if ((count & 0xfff) == 0)
			ReSchedule();

		if (((count & 0xffff) == 0) && print)
			printf("-");
		}

	if (print)
		printf("-\n");
	ResetFlashForRead();
	return(0);
}
/************************************************************************
*
*	Function:	copy_to_flash
*
*	Description:	Write a block of data into the flash memory
*			After the write of flash, read the data back
*			and compare it to the original data
*			the is a straight copy routine. move from
*			memory to flash memory
*
*	Inputs:		to    = address of where to write in flash
*			from  = address of data to be written to flash
*			size  = size of data in bytes
*			print = 0 = do NOT print information messages
*			      = 1 = print information messages
*			
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
**************************************************************************/
int
copy_to_flash(void *to, byte *from, int size, int print)
{
	register byte *src;
	register byte *dst;
	register int retcode;
	register int count;

	/*
	* Burn the flash EPROM
	*/
	open_flash();

	if (print)
		printf("Writing to flash eprom ");

	src = (byte *)from;
	dst = (byte *)to;
	count = size;
	while (count--)
		{
		if (retcode=write_flash(dst, *src))
			{
			close_flash();
			return(retcode);
			}

		src++;
		dst++;

		if ((count & 0xff) == 0)
			ReSchedule();

		if (((count & 0xffff) == 0) && print)
			printf("-");
		}

	if (print)
		printf("\n");

	close_flash();

	/*
	* read back and compare 
	*/
	if (print)
		printf("Comparing flash eprom ");
	src = (byte *)from;
	dst = (byte *)to;
	count = size;
	ResetFlashForRead();
	delay(200);
	while (count--)
		{
		register byte sb;
		register byte db;
		FlashDelay10Us();
		sb = *src;
		db = *dst;
		if (sb != db)
			{
			printf("compare error source %X (%x)  destination %X (%x)\n",
				src, sb, dst, db);
			return(1);
			}

		src++;
		dst++;

		if ((count & 0xfff) == 0)
			ReSchedule();

		if (((count & 0xffff) == 0) && print)
			printf("-");
		}

	if (print)
		printf("-\n");

	return(0);
}
#ifdef 0
PrintFlashInfo(struct flash_info *ptr)
{
	delay(100);
	printf("MARKER = %s\n",ptr->marker);
	printf("CHKSUM = %x\n",ptr->checksum);
	printf("INFOSIZE = %d\n",ptr->info_size);
	printf("COMPSIZE = %d\n",ptr->comp_size);
	printf("DECOMP_SIZE = %d\n",ptr->decomp_size);
	printf("BURN_CYCLES = %d\n",ptr->burn_cycles);
	printf("LOAD_TYPE = %d\n",ptr->load_type);
	printf("FILE_NAME = %s\n",ptr->file_name);
	printf("VERSION = %s\n",ptr->version);
}
#endif
	


build_flash_record()
{
	register struct tm *tm;
	time_t time;
	char burn_date[12];
	char burn_time[12];
	register ulong seventy = 0x83aa7e80;
	register ushort csum;
	struct flash_info *new_flash_record = &flash_info;
	/*
	* get the time from the server
	*/
#ifdef vinay
	time = GetTimeFromServer(nvr_boot_rec.nvr_TimeServer);
#endif

	/*
	* convert the time to an ascii format
	* if no time returned from server - use zero as default
	*/
#ifdef vinay
	if (time)
		{
		time += nvr_boot_rec.nvr_OffsetGMT - seventy;
		tm = gmtime(&time);
		sprintf(burn_date, "%02d/%02d/%02d", tm->tm_mon + 1, tm->tm_mday, tm->tm_year);
		sprintf(burn_time, "%02d:%02d:%02d", tm->tm_hour, tm->tm_min, tm->tm_sec);
		}
	else
#endif
		{
		strcpy(burn_date, "00/00/00");/*, sizeof(burn_date));*/
		strcpy(burn_time, "00:00:00");/*, sizeof(burn_time));*/
		}

	/*
	* clear entire new flash record
	*/
	memset(new_flash_record, 0, sizeof(struct flash_info));

	/*
	* set marker
	*/
	strcpy(new_flash_record->marker, "FLASH   ");/*, sizeof(new_flash_record->marker));*/

	/*
	* set name of file written to the flash
	*/
#ifdef vinay
	strncpy(new_flash_record->file_name, nvr_boot_rec.nvr_image_file, sizeof(new_flash_record->file_name));
#endif
	strcpy(new_flash_record->file_name, eep_boot_rec.eep_image_file);/*, sizeof(new_flash_record->file_name));*/

	/*
	* set time information
	*/
	memcpy(new_flash_record->burn_date, burn_date, 8);
	memcpy(new_flash_record->burn_time, burn_time, 8);
	new_flash_record->seconds = time;

#ifdef vinay
	/*
	* get the number of flash burns from the old record if it is good
	* otherwise use the NVRAM burn count
	*/

	/*
	* Bridge has the boot record in EEPROM so no record will be stored in
	* NVRAM.
	*/
	if (check_flash(old_flash_record, 0) == 0)
		/*
		* flash record is good
		*/
		new_flash_record->burn_cycles = old_flash_record->burn_cycles;
	else
		/*
		* flash record is bad - use info from nvram boot record
		*/
		new_flash_record->burn_cycles = nvr_boot_rec.nvr_BurnCycles;

	/*
	* increment the burn count
	*/
	new_flash_record->burn_cycles++;
#endif

	/*
	* the different size fields, the board name, copy right notice
	* and checksum must be filled in by the calling program
	*/
	return;
}

/*
* returns:	0 = OK
*		1 = bad marker
*		2 = bad checksum
*/
int
check_flash(struct flash_info *flash_hdr, int print_error)
{
	register unsigned short	csum;
	register unsigned short	csumx;

	delay(200); /* This delay is needed due to flash read problem */

/*
	ResetFlashForRead();
*/
	if(strncmp(flash_hdr->marker, "FLASH   ",8/*strlen(flash_hdr->marker)*/) != 0)
		{
			printf("Flash marker is bad\n");
		/*
		* flash record is bad
		*/
		if (print_error)
			printf("Flash marker is bad\n");
		return(1);
		}
	/*
	* checksum entire flash data
	*/
	csum = cksum((ushort *)flash_hdr, ((flash_hdr->info_size + flash_hdr->comp_size) >> 1));
	if ((csum != 0) && (csum != 0xffff))
		{
		/*
		* flash record is bad
		*/
		if (print_error)
			printf("Flash checksum is bad - %X\n", csum);
		return(2);
		}

	/*
	* flash record is good
	*/
	return(0);
}

check_code(NIM960_HDR *header, int print_errors)
{
	register u_short csum;
	register int ret;

	ret = check_header(header, 1);

	if (ret)
		{
		csum = cksum(header, header->FileLength >> 1);
		if ((csum != 0xFFFF) && (csum != 0))
			{
			if (1)
				printf("Error: Image did not checksum - checksum = 0x%x\n", csum);
			ret = 0;
			}
		}

	return(ret);
}

check_header(NIM960_HDR *header, int print_errors)
{
	register int ret;

	ret = 1;
	if (header->signature != NIM960_MAGIC)
		{
		if (print_errors)
			{
			printf("Error: Not a 960 image format\n");
			printf("signature is bad - signature is 0x%x it should be 0x%x\n",
				header->signature, NIM960_MAGIC);
			}
		ret = 0;
		}

	if (header->Entry == NULL)
		{
		if (print_errors)
			printf("Error: No program entry point\n");
		ret = 0;
		}

	if (header->ProgramId != program_id)
		{
		if (print_errors)
			{
			printf("Error: Program not for this board\n");
			printf("program id is bad - program id is 0x%X it should be 0x%X\n",
				header->ProgramId, program_id);
			}
		ret = 0;
		}

	return(ret);
}
