#include <types.h>
#include <memory.h>
#include <eeprecs.h>
#include <nvrecs.h>
#include <newflash.h>

#define VPP		0x900000c0	/*Vpp enable for FLASH PROM */
#define	SYSADDR		FEPROM		/* This constant can be initialized */
					/* to any address within the memory */
					/* map of the target 28F008SA and is */
					/* alterable depending on the */
					/* system architecture */

/*
* common commands
*/
#define	IDRDCOMM	0X90		/* Inteligent Identifier command */

/*
* Bit Masks for 28f008
*/
#define	ESEQMASK	0X30		/* Mask to isolate the erase and byte */
					/* write status bits of the Status */
					/* Register */
#define	ESEQFAIL	0X30		/* Status Register value after */
					/* masking if block erase command */
					/* sequence error has been detected */
#define	EERRMSK		0X20		/* Mask to isolate the erase status */
					/* bit of the Status Register */
#define	ERASERR		0X20		/* Status Register value after */
					/* masking if block erase error has */
					/* been detected */
#define	WERRMSK		0X10		/* Mask to isolate the byte write */
					/* error bit of the Status Register */
#define	WRITERR		0X10		/* Status Register value after */
					/* masking if byte write error has */
					/* been detected	*/
#define	RDYMASK		0X80		/* Mask to isolate the WSM Status */
					/* bit of the Status Register */
#define	WSMRDY		0X80		/* Status Register value after */
					/* masking, signifying that the WSM */
					/* is no longer busy */
#define	VLOWMASK	0X08		/* Mask to isolate the Vpp status bit */
					/* of the Status Register */
#define	VPPLOW		0X08		/* Status Register value after */
					/* masking if Vpp low has been */
					/* detected */

/*
* Misc.
*/
#define INTELID	0X89			/* Manufacturer ID for Intel devices */
#define	DVCID1		0Xa1		/* device id for 28f008SA   */
#define DVCID2		0Xa2		/* device id for 28F008SA-L */
#define	DVCID3		0Xbd		/* device id for 28f020     */
#define AMDID	0X01			/* Manufacturer ID for Intel devices */
#define	DVCID4		0X2a		/* device id for Am28f020     */

#ifdef redefined
#define MAX_RETRIES_WRITE	25;	/* retry count for F020 type devices */
#endif

/*
* device types
*/
#define	F008		1
#define	F020		2


/*
* command structure
*/
struct commands
	{
	byte read_command;		/* read */
	byte write_command;		/* byte write setup */
	byte write_verify_command;	/* byte write verify */
	byte erase_command;		/* erase setup */
	byte erase_verify_command;	/* erase verify */
	byte reset_command;		/* reset */
	byte read_status_command;	/* read status */
	byte clear_status_command;	/* clear status */
	};

/*
* commands for 28f0008 type devices
*/
struct commands f008_commands =
	{
	0xff,			/* read */
	0x40,			/* byte write setup */
	0x00,			/* null */
	0x20,			/* erase setup */
	0xd0,			/* erase confirg */
	0xff,			/* reset */
	0x70,			/* status read */
	0x50			/* status clear */
	};

/*
* commands for 28f0020 type devices
*/
struct commands f020_commands =
	{
	0x00,			/* read */
	0x40,			/* byte write setup */
	0xc0,			/* write verify */
	0x20,			/* erase setup */
	0xa0,			/* erase confirg */
	0xff,			/* reset */
	0x00,			/* null */
	0x00			/* null */
	};

/*
* flash device descriptor structure
*/
struct device
	{
	byte manufacture_id;
	byte device_id;

	byte device_type;

	int sector_size;
	int number_of_sectors;

	struct commands *commands;
	};

/*
* flash device descriptor table
*/
struct device device_table[] =
	{
	{INTELID, DVCID1, F008,  64 * 1024, 16, &f008_commands},
	{INTELID, DVCID2, F008,  64 * 1024, 16, &f008_commands},
	{INTELID, DVCID3, F020, 256 * 1024,  3, &f020_commands},
	{AMDID,   DVCID4, F020, 256 * 1024,  3, &f020_commands},
	{0}
	};

struct device   *fdp;		/* pointer to description for flash device */
struct commands *fdc;		/* pointer to commands for flash device */

int write_flash(byte *, byte);

/************************************************************************
*
*	Function:	VppUp
*
*	Description:	Turn on the Vpp for programing flash EPROM
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	Nothing
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

#ifdef 0
static int
VppUp(void)
{
	register byte	*vppctrl;

	vppctrl = (byte *)VPP;
	*vppctrl = 0x0200;	/* 3 raise the VPP to flash EPROM */
	*vppctrl = 1;			/* raise the VPP to flash EPROM */


        FlashDelay10Ms();
	return(0);
}

/************************************************************************
*
*	Function:	VppDown
*
*	Description:	Turn off the Vpp for programing flash EPROM
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	Nothing
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

static int
VppDown(void)
{
	register byte	*vppctrl;

	vppctrl = (byte *)VPP;
	*vppctrl = 0;			/* lower the VPP to flash EPROM */

        FlashDelay10Ms();

	return(0);
}
#endif
/******************************************************************************
*
*	Function:	idread
*
*	Description:	Reads the manufacturer and device IDs
*
*	Inputs:		address  = address of where to put the base address of
*				   part
*			mfgrid   = address of where to put the Manufactures ID
*				   of part
*			deviceid = address of where to put the device ID of part
*
*	Outputs:	address  = base address of part
*			mfgrid   = Manufactures ID of part
*			deviceid = device ID of part
*
*	Returns:	0 = ID read correct
*
*	Device Read Mode on Return: inteligent Identifier
*
******************************************************************************/
extern int fault_cnt, TimeTicksInt;
extern void usecInt();

static int
idread(byte *address, byte *mfgrid, byte *deviceid)
{

	register volatile byte *tempaddr;	/* Pointer address variable */
						/* used to read IDs */
	VppUp();
/*
	FlashDelay10Ms();
*/
	tempaddr	= address;
	*tempaddr	= IDRDCOMM;		/* Write inteligent */
						/* Identifier command to an */
						/* address within the */
						/* memory map */
	*mfgrid	  = *tempaddr;			/* Read mfgr ID */
	tempaddr += 3;
	*deviceid = *tempaddr;			/* Read device ID */

	VppDown();

	return(0);
}

/************************************************************************
*
*	Function:	rdmode
*
*	Description:	Puts the target 28F008SA in Read Array Mode.
*			This function might be used, for example, to prepare
*			the system for return to code execution out of the
*			Flash memory after byte write or block erase algorithms
*			have been executed off-chip
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	Nothing
*
*	Device Read Mode on Return: Read Array
*
**************************************************************************/

static int
rdmode(void)
{
	register byte *tempaddr;		/* Pointer variable used to */
						/* write commands to the */
						/* device */

	tempaddr	= (byte *)SYSADDR;
	*tempaddr	= fdc->read_command;	/* Write Read Array command */
						/* to 28F008SA */
	return;
}
int ResetFlashForRead()
{
	reset_f020();
}
/****************************************************************************
*
*	Function:	reset_f020
*
*	Description:	Clears the 28F020
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	Nothing
*
*	Device Read Mode on Return:	Read Array
*
*****************************************************************************/

static int
reset_f020(void)
{
	register int i;
	register byte *tempaddr;		/* Pointer variable used to */
						/* write commands to device */

	for (i = 0; i < fdp->number_of_sectors; i++)
		{
		tempaddr	= (byte *)FEPROM + (i * fdp->sector_size);
		*tempaddr	= fdc->reset_command;
		*tempaddr	= fdc->reset_command;
		*tempaddr	= fdc->read_command;
		tempaddr	= (byte *)FEPROM + (i * fdp->sector_size); 
		*tempaddr	= fdc->reset_command;
		*tempaddr	= fdc->reset_command;
		*tempaddr	= fdc->read_command;
		tempaddr	= (byte *)FEPROM + (i * fdp->sector_size); 
		*tempaddr	= fdc->reset_command;
		*tempaddr	= fdc->reset_command;
		*tempaddr	= fdc->read_command;
		}

	return;
}

/****************************************************************************
*
*	Function:	Statclr
*
*	Description:	Clears the 28F008SA Status Register
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	Nothing
*
*	Device Read Mode on Return:	Array
*
*****************************************************************************/

static int
statclr(void)
{
	register byte *stataddr;		/* Pointer variable used to */
						/* write commands to device */

	stataddr	= (byte *)SYSADDR;
	*stataddr	= fdc->clear_status_command;	/* Write Clear Status */
						/* Register command */
	return;
}

/*****************************************************************************
*
*	Function:	End
*
*	Description:	Checks to see if the WSM is busy (is byte write/block
*			erase completed?)
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	statdata = Status Register data read from device
*
*	Device Read Mode on Return: Status Register
*
*****************************************************************************/

static byte
end(void)
{
	register volatile byte *stataddr;	/* Pointer variable used to */
						/* write commands to device */
	register byte statdata;			/* Status Register data to be */
						/* passed back to the main */
						/* program for analysis */

	stataddr	= (byte *)SYSADDR;
	*stataddr	= fdc->read_status_command; /* Write Read Status Register */
						/* command to 28F008SA */
	while (((statdata = *stataddr) & RDYMASK) != WSMRDY)
		{
		/*
		* Byte write/block erasure still in progress
		*/
		ReSchedule();
		}

	/*
	* Byte write/block erase attempt completed
	* return operation status
	*/
	return(statdata);
}

/****************************************************************************
*
*	Function:	Erasgbn
*
*	Description:	Begins erase of a block.
*
*	Inputs:		blckaddr = System address within the block to be erased
*
*	Outputs:	None
*
*	Returns:	0 = Block erase successfully initiated
*			1 = Block erase not initiated (ID check error)
*
*	Device Read Mode on Return:	Status Register (ID if returns 1)
*
*****************************************************************************/

static int
erasbgn(volatile byte *blckaddr)
{

	*blckaddr	= fdc->erase_command;	/* Write Erase Setup command */
						/* to block address */
	*blckaddr	= fdc->erase_verify_command;	/* Write Erase Confirm */
						/* command to block address */
	return(0);
}


/****************************************************************************
*
*	Function:	Eraschk
*
*	Description:	Completes full Status Register check for block erase
*			(proper command sequence, Vpp low detect, block erase
*			success).  This routine assumes that block erase
*			completion has already been checked in function end(),
*			and therefore does not check the WSM Status bit of
*			the Status Register
*
*	Inputs:		statdata = Status Register data read in function end
*
*	Outputs:	None
*
*	Returns:	0 = Block erase completed successfully
*			1 = Error; Vpp low detect
*			2 = Error; Block erase error
*			3 = Error; Improper command sequencing
*
*	Device Read Mode on Return:	Same as when entered
*
*****************************************************************************/

static int
eraschk(byte statdata)
{
	if ((statdata & VLOWMASK) == VPPLOW)
		return(1);			/* Vpp low detect error, */
						/* return code "1" */
	if ((statdata & EERRMSK) == ERASERR)
		return(2);			/* Block erase error detect, */
						/* return code "2" */
	if ((statdata & ESEQMASK) == ESEQFAIL)
		return(3);			/* Block erase command */
						/* sequence error, */
						/* return code "3" */
	return(0);				/* Block erase success, */
						/* return code "0" */
}


/************************************************************************
*
*	Function:	EraseFlashBlock
*
*	Description:	Erase a 64K block of the 28F008SA flash EPROM,
*			and check the erased content
*
*	Inputs:		blockaddr = block address to be erased
*				    should be on a 64k boundary
*
*	Outputs:	None
*
*	Returns:	0 = Block erase completed successfully
*			1 = Vpp low detect
*			2 = Block erase error
*			3 = Improper command sequencing
*			4 = erase read back error
*
*	Device Read Mode on Return:	Read Array
*
**************************************************************************/

static int
EraseFlashBlock(byte *blockaddr)
{
	register byte flashStatus;
	register byte *pAddr;
	register int i;
	register int j;
	register int error;

	/*
	* issue erase block command
	*/
	erasbgn(blockaddr);

	/*
	* wait for completion
	*/
	flashStatus = end();

	/*
	* check status of erase operation
	*/
	error = eraschk(flashStatus);

	/*
	* if erase successful read each byte - should be 0xff
	*/
	if (error == 0)
		{
		for (j = 0, pAddr = blockaddr; j < fdp->sector_size; j++, pAddr++)
			if (*pAddr != 0xFF)	/* check the erased content */
				{
				error = 4;	/* erase readback error */
				break;
				}
		}

}

/************************************************************************
*
*	Function:	erase_f020
*
*	Description:	Erase the f020 flash eprom
*
*	Inputs:		address = starting address for erase
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
*	Device Read Mode on Return:	Read Array
*
**************************************************************************/

static
erase_f020(byte* address)
{
	register volatile byte *erase_address;
	register byte *write_address;
	register byte data;
	register int size;
	register int i;
	register int retries;

	/*
	* write zeroes to all locations
	*/
	size = fdp->sector_size;

	write_address = address;
	for (i = 0; i < size; i++)
		{
		if (write_flash(write_address, 0) != 0)
			{
			/*
			* error
			*/
			printf("Error: Cannot Program flash eprom to 0 at address(es): = %X\n", write_address);
			reset_f020();
			return(1);
			}
		else
			write_address++;

		/*
		* output a period to show test still running
		*/
		if (!(i%0x2000))
			printf(".");

		/*
		* let something else run
		*/
		if (!(i%0x400))
			ReSchedule();

		}

	/*
	* reset
	*/
	reset_f020();
	ReSchedule();

	/*
	* begin erasing
	*/
	erase_address = address;
	retries = 3000;
	i = 0;
	while ((i < size) && (retries > 0))
		{
		*erase_address = fdc->erase_command;
		*erase_address = fdc->erase_command;
/*
		Wait(2);
*/
		FlashDelay10Ms();
		

		*erase_address = fdc->erase_verify_command;
	/*
		Wait_us(6);
	*/
	FlashDelay10Us();

		while ((data = *erase_address) == 0xff)
			{
			erase_address++;
			i++;
			/*
			 *	If we reach the end of flash eprom,
			 *	then we are done.
			 */
			if (i >= size)
				{
				reset_f020();
				return(0);
				}

			/*
			* output a period to show test still running
			*/
			if (!(i%0x2000))
				printf(".");

			/*
			* let something else run
			*/
			if (!(i%0x400))
				ReSchedule();

			*erase_address = fdc->erase_verify_command;
/***
			Wait_us(6);
	*/
	FlashDelay10Us();
			}

		if (i < size)
			retries--;
		}

	/*
	* if we get here the erase failed
	*/

	/*
	* reset for reading
	*/
	reset_f020();

	printf("\nError: Cannot erase the flash eprom at address(es): = %X\n", erase_address);

	return(1);
}

/************************************************************************
*
*	Function:	TestFlashWriteAddress
*
*	Description:	Write the address of each long in the block
*			into itself
*
*	Inputs:		pBlkAddr = address of 64k block to be tested
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

static int
TestFlashWriteAddress(byte *pBlkAddr)
{
	register int i;
	register int j;
	register byte pattern;
	register byte *bp;
	byte *pAddr;
	register int error;

	printf("Flash EPROM address test writing at %X ", pBlkAddr);

	/*
	* erase the flash EPROM  block
	*/
	if (erase_flash(pBlkAddr, fdp->sector_size, 1))
		{
		printf("error, test aborted\n");
		return(1);
		}

	ReSchedule();
	/*
	* open flash for write operation
	*/
	if (open_flash())
		{
		printf("TestFlashWriteAddress: open flash failed\n");
		return(1);
		}

	if (fdp->device_type == F020)
		printf("\n");

	/*
	* write the address of each long in itself
	*/
	bp = (byte *)&pAddr;
	for (i=0, pAddr = pBlkAddr; i < fdp->sector_size/4; i++, pAddr += 4)
		{
		for (j = 0; j < 4; j++)
			{
			pattern = bp[j];
			if (error = write_flash(pAddr + j, pattern))
				{
				printf("\nError: %D - Address test, writing FEPROM at %x\n", error, pAddr);
				close_flash();
				return(1);
				}
			}

		/*
		* output a period to show test still running
		*/
		if (!(i%(0x1000/4)))
			printf(".");

		/*
		* let something else run
		*/
		if (!(i%0x400/4))
			ReSchedule();
		}

	printf("\n");

	/*
	* close flash for write/erase operation and return to read mode
	*/
	close_flash();

	/*
	* increment burn count information
	* only save it in the nvram record as we have just destroyed
	* the flash record
	*/
#ifdef changelater
	nvr_boot_rec.nvr_BurnCycles++;
	PutBootRec(&nvr_boot_rec);
#endif

	return(0);
}

/************************************************************************
*
*	Function:	TestFlashReadAddress
*
*	Description:	read each long in the block and check to see if
*			it points to itself
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

static int
TestFlashReadAddress(byte *pBlkAddr)
{
	register int i;
	register long *lAddr;
	register long lpattern;
	register long ldata;

	printf("Flash EPROM address test reading at %X ", pBlkAddr);

	/*
	* read each long in the block and check to see if it points to itself
	*/
	for (i = 0, lAddr = (long *)pBlkAddr; i < fdp->sector_size/4; i++, lAddr++)
		{
		lpattern = (long)lAddr;
		ldata = *lAddr;
		if (ldata != lpattern)
			{
			printf("\nError: FEPROM address test at %X data is %X should be %X\n", lAddr, ldata, lpattern);
			return(1);
			}

		/*
		* output a period to show test still running
		*/
		if (!(i%(0x1000/4)))
			printf(".");

		/*
		* let something else run
		*/
		if (!(i%0x400/4))
			ReSchedule();
		}

	printf("OK\n");

	return(0);
}

/************************************************************************
*
*	Function:	TestFlashBlockData
*
*	Description:	Test memory by writing the given byte into all
*			of the bytes of the 64k sector.  Read the bytes
*			back and check to see if the same as the byte
*			written
*
*	Inputs:		pBlkAddr = address of 64k block to test
*			pattern  = data byte to write in each location
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

static int
TestFlashBlockData(byte *pBlkAddr, byte pattern,int block)
{
	register int i;
	register byte *pAddr;
	register int error;
	register byte b;

	/*
	* erase the flash EPROM  block
	*/
	printf("\nErase flash EPROM block %x ...\n",block);
	if (erase_flash(pBlkAddr , fdp->sector_size, 1))
		{
		printf("\nerase error, test aborted.\n");
		return(1);
		}

	ReSchedule();

	/*
	* open flash for write operation
	*/
	if (open_flash())
		{
		printf("TestFlashBlockData: open flash failed\n");
		return(1);
		}

	/*
	/*
	* write the test pattern
	*/
	printf("Flash EPROM write data test (%x) ", pattern);
	for (i = 0, pAddr = pBlkAddr; i < fdp->sector_size; i++, pAddr++)
		{
		if (error = write_flash(pAddr, pattern))
			{
			printf("\nError: %D - Writing FEPROM at %x\n", error, pAddr);
			close_flash();
			return(1);
			}

		/*
		* output a period to show test still running
		*/
		if (!(i%0x1000))
			printf(".");

		/*
		* let something else run
		*/
		if (!(i%0x400))
			ReSchedule();
		}

	printf("\n");

	/*
	* close flash for write operation and return to read mode
	*/
	close_flash();

	/*
	* read back and compare against the pattern
	*/
	error = 0;
	printf("Flash EPROM read data test  (%x) ", pattern);
	for (i = 0, pAddr = pBlkAddr; i < fdp->sector_size; i++, pAddr++)
		{
		b = *pAddr;
		if (b != pattern)
			{
			printf("\nError: Reading FEPROM at %X is %X should be %X\n", pAddr, b, pattern);
			if (error++ > 10)
				return(1);
			}

		/*
		* output a period to show test still running
		*/
		if (!(i%0x1000))
			printf(".");

		/*
		* let something else run
		*/
		if (!(i%0x400))
			ReSchedule();
		}

	printf("OK\n");

	return(0);
}

/************************************************************************
*
*	Function:	TestProgramFlash
*
*	Description:	Test the flash eprom memory
*			1. test data pattern 0x5a
*			2. test data pattern 0xa5
*			3. test addressing pattern
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

int
TestFlash()
{
	register byte *pBlkAddr;
	register int block;
	register int start_block;
	register int end_block;
	register int error;

	/*
	*	Test the flash eprom.
	*/

	/*
	* the open and clode routines are called so the pointer to
	* the device descriptor is set up porperly
	*/

	error = open_flash();
	close_flash();

	if (error)
		{
		printf("TestFlash: open flash failed\n");
		return(1);
		}
#ifdef 0
	else
		printf("Flash EPROM manufacture ID = %X, device ID = %X\n",
			fdp->manufacture_id, fdp->device_id);
#endif

	start_block = 0;
	end_block   = fdp->number_of_sectors;

	/*
	* data test block by block
	*/
	for (block = start_block; block < end_block; block++)
		{
		pBlkAddr = (byte*)FEPROM + block * fdp->sector_size;

		/*
		* write 0x5a to all bytes
		*/
		if (TestFlashBlockData(pBlkAddr, 0x5a,block))
			return(1);

#ifdef 0
		/*
		* write 0xa5 to all bytes
		*/
		if (TestFlashBlockData(pBlkAddr, 0xa5,block))
			return(1);
#endif

		}

	/*
	* increment burn count information once for each pattern
	* only save it in the nvram record as we have just destroyed
	* the flash record
	*/
#ifdef temp
	nvr_boot_rec.nvr_BurnCycles++;
	nvr_boot_rec.nvr_BurnCycles++;
	PutBootRec(&nvr_boot_rec);
#endif

	/*
	* write the address of each long in each long
	*/
	printf("\n");
#ifdef 0
	for (block = start_block; block < end_block; block++)
		{
		pBlkAddr = (byte*)FEPROM + block * fdp->sector_size;
		if (TestFlashWriteAddress(pBlkAddr))
			printf("\n");
		}


	/*
	* read back each long and see if it still contains its address
	*/
	printf("\n");
	for (block = start_block; block < end_block; block++)
		{
		pBlkAddr = (byte*)FEPROM + block * fdp->sector_size;
		if (TestFlashReadAddress(pBlkAddr))
			return(1);
		}
#endif

	printf("\n");

	return(0);
}

/************************************************************************
*
*	Function:	open_flash
*
*	Description:	Set up flash for writing
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

int
open_flash(void)
{
	byte manufacture_id;
	byte device_id;
	register i;

	idread((byte *)SYSADDR, &manufacture_id, &device_id);

	/*
	* find device descriptor entry
	*/

	fdp = device_table;
	for (i = 0; i < (sizeof(device_table) / sizeof(struct device)); i++)
		{
		if ((manufacture_id == fdp->manufacture_id) &&
		    (device_id      == fdp->device_id))
			break;
		fdp++;
		}

	if ((manufacture_id != fdp->manufacture_id) || 
	    (device_id      != fdp->device_id))
		{
		printf("Flash device is not in the table of known flash types\n");
		printf("Flash EPROM manufacture ID = %X, device ID = %X\n",
			manufacture_id, device_id);
		return(1);
		}

	fdc = fdp->commands;

	/*
	*
	* compute number of sectors in the flash
	*/
	fdp->number_of_sectors = eep_mfg_rec.eep_fprom_size / fdp->sector_size;

	/*
	* for F020 devices verify they are all the same
	*/
	for (i = 1; i < fdp->number_of_sectors; i++)
		{
		idread((byte *)(SYSADDR + (i * fdp->sector_size)), &manufacture_id, &device_id);
		if ((manufacture_id != fdp->manufacture_id) ||
		    (device_id      != fdp->device_id))
			{
			printf("Flash parts are not all the same type\n");
#ifdef 0
			printf("Flash EPROM 0 manufacture ID = %X, device ID = %X\n",
				fdp->manufacture_id, fdp->device_id);
#endif
			printf("Flash EPROM %d manufacture ID = %X, device ID = %X\n",
				i, manufacture_id, device_id);
			return(1);
			}
		}

	if (fdp->device_type == F008)
		statclr();	/* clear the status register */

	else if (fdp->device_type == F020)
		reset_f020();
	VppUp();

	return(0);
}
/************************************************************************
*
*	Function:	close_flash
*
*	Description:	close flash for writing and put flash in read mode
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	Nothing
*
*	Device Read Mode on Return:	Read Array
*
**************************************************************************/

int
close_flash(void)
{
	VppDown();

	if (fdp->device_type == F008)
		rdmode();

	else if (fdp->device_type == F020)
		reset_f020();

	return(0);
}

/************************************************************************
*
*	Function:	erase_flash
*
*	Description:	errase a block of flash
*
*	Inputs:		address = starting address of block
*			size    = size of block in bytes
*			print   = 0 = do not print
*				= 1 = print
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			3 = unknown device type
*
*	Device Read Mode on Return:	Same as when entered
*
**************************************************************************/

int
erase_flash(void *address, int size, int print)
{
	register byte *pBlkAddr;
	register int block;
	register int start_block;
	register int end_block;
	register int ret;
	register ulong temp;
	/*
	* open flash for write/erase operation
	*/
	if (open_flash())
		{
		printf("erase_flash: open flash failed\n");
		return(1);
		}

	/*
	* Erase the flash EPROM
	*/
	if (print)
		printf("Erasing flash eprom ");
	

	/*
	* is this an F008 type device
	*/
	if (fdp->device_type == F008)
		{
		temp = ((ulong)address - (ulong)FEPROM);
		start_block = temp >> 16;
		end_block = ((temp + size + 0xffff) >> 16);
		for (block = start_block; block < end_block; block++)
			{
			pBlkAddr = (byte *) FEPROM + (block << 16);
			if (EraseFlashBlock(pBlkAddr))
				{
				/*
				* close flash for write/erase operation
				* and return to read mode
				*/
				close_flash();

				return(1);
				}
			if (print)
				printf("-");

			/*
			* let something else run
			*/
			ReSchedule();
			}

		if (print)
			printf("\n");

		/*
		* close flash for write/erase operation and return to read mode
		*/
		close_flash();

		return(0);
		}

	/*
	* is this an F020 type device
	*/
	else if (fdp->device_type == F020)
		{
		temp = ((ulong)address - (ulong)FEPROM);
		start_block = temp >> 18;
#ifdef 0
		end_block = ((temp + size + 0x3ffff) >> 18);
		for (block = start_block; block < end_block; block++)
#endif
		for (block = start_block; block < 3; block++)
			{
			pBlkAddr = (byte *)FEPROM + (block * fdp->sector_size);
			if (erase_f020(pBlkAddr))
				{
				/*
				* close flash for write/erase operation
				* and return to read mode
				*/
				close_flash();

				return(1);
				}
			if (print)
				printf("-");

			/*
			* let something else run
			*/
			ReSchedule();
			}

		if (print)
			printf("\n");

		/*
		* close flash for write/erase operation and return to read mode
		*/
		close_flash();

		return(0);
		}

	/*
	* unknown device
	*/
	else
		{
		/*
		* close flash for write/erase operation and return to read mode
		*/
		close_flash();

		return(3);
		}

	return;
}

/************************************************************************
*
*	Function:	write_flash
*
*	Description:	Write one byte data into the flash memory and
*			return the result of the operation
*
*	Inputs:		address = address of where to write data byte
*			data    = data to be written at address
*
*	Outputs:	None
*
*	Returns:	status of write
*			0 = Byte write completed successfully
*			1 = Error; Vpp low detect
*			2 = Error; Byte write error
*			3 = Error; unknown device
*			4 = Error; read back and compare failed
*
*	Device Read Mode on Return:	Status Register
*
**************************************************************************/

int
write_flash(byte *address, byte data)
{
	register byte status;
	/*
	* is this an F008 type device
	*/
	if (fdp->device_type == F008)
		{
		/*
		* write byte to flash
		*/
		*address = fdc->write_command;	/* Byte Write Setup command */
						/* to the destination address */
		*address = data;		/* Write data to the */
						/* destination address */
		/*
		* wait for write to complete
		*/
		status = end();

		/*
		* check and return the result of the write
		*/
		if ((status & VLOWMASK) == VPPLOW)
			return(1);		/* Vpp low detect error, */

		if ((status & WERRMSK) == WRITERR)
			return(2);		/* Byte write error detect, */

		*address = fdc->read_command;	/* Write Read Array command */
		if (*address != data)		/* verify data */
			return(4);

		return(0);			/* success */
		}

	/*
	* is this an F020 type device
	*/
	else if (fdp->device_type == F020)
		{
		int	retries;
		retries = MAX_RETRIES_WRITE;
		while (retries--)
			{
			*address = fdc->write_command;	/* send write cmd */
			FlashDelay10Us();
			*address = data;		/* write data */
			FlashDelay10Us();
			*address = fdc->write_verify_command; /* send verify */
			FlashDelay10Us();
			if (*address == data)		/* verify data */
				return 0;
			}

		return(2);
		}

	/*
	* unknown device
	*/
	else
		return(3);
}
