/************************************************************************
*
*	The following routine do the compressing and uncompressing of the
*	code written to the flash.  The code transferred over the network
*	is uncompressed.  These routines will compress the code and write
*	it into the flash memory.  The will also uncompress the code in the
*	flash memory and write it into dram memory.
*
*	The compression algorithm used is the standard unix compression.
*
**************************************************************************/

#include "types.h"
#include "flash.h"
#include <memory.h>
#include <eeprecs.h>

/*
* And now for some typedefs
*/
typedef unsigned short CODE;
typedef unsigned char  UCHAR;
typedef unsigned int   INTCODE;
typedef unsigned int   HASH;
typedef int FLAG;

#define NULLPTR(type)   ((type *) NULL)

#define UNUSED      ((CODE)0)   /* Indicates hash table value unused    */
#define CLEAR       ((CODE)256) /* Code requesting table to be cleared  */
#define FIRSTFREE   ((CODE)257) /* First free code for token encoding */
#define MAXTOKLEN   512         /* Max chars in token; size of buffer   */
#define YES         1
#define NO          0
#define NULL        0

#define FALSE 0
#ifdef TRUE
#undef TRUE
#endif
#define TRUE !FALSE

#define CHECK_GAP 10000L     /* ratio check interval */
#define INITBITS    9
#define MAXBITS  16
#define BIT_MASK    0x1f
#define BLOCK_MASK  0x80
#define EOF -1

/*
* The following two parameter are the hash table size and
* maximum code value for 16 bit code bit-lengths.  The requirements
* are that Hashsize must be a prime number and Maxcode must be less
* than Hashsize.  Table occupancy factor is (Maxcode - 256)/Hashsize.
* Note:  a lower Maxcode for 16-bit codes is used in order to
* keep the hash table size less than 64k entries.
*/

#define Hashsize 0xffd9

#define Maxcode 0xefff

#define allocx(type,ptr,size) \
    (((ptr) = (type *) lmalloc((unsigned int)(size)*sizeof(type))) == NULLPTR(type) \
    ? NOMEM : OK \
    )


/*
* Macro to allocate new memory to a pointer with an offset value.
*/
#define alloc_array(type, ptr, size, offset) \
    ( allocx(type, ptr, (size) - (offset)) != OK \
      ? NOMEM \
      : (((ptr) -= (offset)), OK) \
    )

int exit_stat = 0;
int RetriesErase;
int TimeTicksInt;
extern void usecInt();
extern word aplLocation;
static int block_compress = BLOCK_MASK;
static long checkpoint = CHECK_GAP;
static long int ratio;
static UCHAR magic_header[] = { 0x1F,0x9D };  /* 1F 9D */

static int offset;
static long int in_count ;         /* length of input */
static long int bytes_out;         /* length of compressed output */

static INTCODE prefxcode;
static INTCODE nextfree;
static INTCODE highcode;
static INTCODE maxcode;
static HASH hashsize;
static int  bits;

static HASH hashf[256];

static char *suffix = NULLPTR(char) ;

static CODE *prefix = NULLPTR(CODE);

static CODE *ht = NULLPTR(CODE);
#define probe(hash) (ht[hash])

static byte *to;
static byte *from;
static int in_length;
static int out_length;

void putbyte(char);
void putcode(INTCODE, int);
void assert(int, int);

/************************************************************************
*
*	Function:	alloc_tables
*
*	Description:	allocate prefix, suffix and hash tables.
*			if the pointer to a table is nonzero, leave it alone
*			if the size of a table is zero, do not allocate
*
*	Inputs:		maxcode  = size of code tables
*			hashsize = size of hash table
*
*	Outputs:	suffix = pointer to suffix table
*			prefix = pointer to prefix table
*			ht     = pointer to hash table
*
*	Returns:	OK = OK = 0
*			NOMEM = 2 = no memory available
*			error code is also put in exit_stat
*
**************************************************************************/

int
alloc_tables(INTCODE maxcode, HASH hashsize)
{
	if ((suffix == 0) && maxcode)
		{
		alloc_array(char, suffix, (maxcode + 1), 256);
		if (suffix == 0)
			{
			exit_stat = NOMEM;
			return(NOMEM);
			}
		}

	if ((prefix == 0) && maxcode)
		{
		alloc_array(CODE, prefix, (maxcode + 1), 256);
		if (prefix == 0)
			{
			exit_stat = NOMEM;
			return(NOMEM);
			}
		}

	if ((ht == 0) && hashsize)
		{
		alloc_array(CODE, ht, hashsize, 0);
		if (ht == 0)
			{
			exit_stat = NOMEM;
			return(NOMEM);
			}
		}

	return (OK);
}

/************************************************************************
*
*	Function:	free_tables
*
*	Description:	free the tables allocated in allocate_tables
*			The tables are freed in the reverse order
*			they were allocated because of the limitions
*			of the memory allocation routines
*			the table pointers are only freed if they are non-zero
*
*	Inputs:		maxcode  = size of code tables
*			hashsize = size of hash table
*
*	Outputs:	clears hash table pointer ht
*			clears prefix table pointer prefix
*			clears suffix table pointer suffix
*
*	Returns:	OK = OK = 0
*
**************************************************************************/

int
free_tables(INTCODE maxcode, HASH hashsize)
{
	if (ht != 0)
		{
#ifdef 0
		lmfree(ht, hashsize * sizeof(CODE));
#endif
		free(ht);
		ht = 0;
		}

	if (prefix != 0)
		{
#ifdef 0
		lmfree(prefix + 256, ((maxcode + 1) - 256) * sizeof(CODE));
#endif
		free(prefix+256);
		ht = 0;
		prefix = 0;
		}

	if (suffix != 0)
		{
#ifdef 0
		lmfree(suffix + 256, (maxcode + 1) - 256);
#endif
		free(suffix+256);
		suffix = 0;
		}

	return(OK);
}

/************************************************************************
*
*	Function:	init_hash_tables
*
*	Description:	initialize the hash table
*
*	Inputs:		None
*
*	Outputs:	hash table contains zeroes
*
*	Returns:	None
*
**************************************************************************/

void
init_hash_table()
{
	register HASH hash;

	hash = hashsize;
	while (hash--)
		ht[hash] = 0;

	highcode = ~(~(INTCODE)0 << (bits = INITBITS));
	nextfree = (block_compress ? FIRSTFREE : 256);

	return;
}

/************************************************************************
*
*	Function:	cl_block
*
*	Description:	
*
*	Inputs:		
*
*	Outputs:	None
*
*	Returns:	TRUE  = do compress
*			FALSE = do not compress
*
**************************************************************************/

/* table clear for block compress */
/* this is for adaptive reset present in version 4.0 joe release */
/* DjG, sets it up and returns TRUE to compress and FALSE to not compress */

int
cl_block ()     
{
	register long int rat;

	checkpoint = in_count + CHECK_GAP;

	if(in_count > 0x007fffff)
		{	/* shift will overflow */
		rat = bytes_out >> 8;
		if(rat == 0)       /* Don't divide by zero */
			rat = 0x7fffffff;
		else
			rat = in_count / rat;
		}
	else
		rat = (in_count << 8) / bytes_out;  /* 8 fractional bits */

	if ( rat > ratio )
		{
		ratio = rat;
		return FALSE;
		}
	else
		{
		ratio = 0;
		return TRUE;    /* clear the table */
		}
}

/************************************************************************
*
*	Function:	compress
*
*	Description:	compress the given data has been compressed using
*			the unix compression method
*
*	Inputs:		c_from   = pointer to data to be compressed
*			c_length = length of data to be compressed
*			c_to     = where to put the compressed data
*
*	Outputs:	None
*
*	Returns:	-1 = error - the error number can be found in exit_stat
*			(!= -1) = the number of bytes in the uncompressed data
*
**************************************************************************/

int
compress(byte *c_from, int c_length, byte *c_to)
{
	register int c;
	register int adjbits;
	register HASH hash;
	register INTCODE code;

	char temp;

	to         = c_to;
	from       = c_from;
	in_length  = c_length;
	out_length = 0;

	maxcode = Maxcode;
	hashsize = Hashsize;

	/*
	* Compress the data only if it does not fit into the flash
	*/
	if(c_length < (eep_mfg_rec.eep_fprom_size - ((int)c_to - (int)FEPROM)))
	{
		exit_stat = NOMEM;
		return -1;
	}

	/*
	* Only needed for adaptive reset
	*/
	checkpoint = CHECK_GAP;
	ratio = 0;

	adjbits = MAXBITS - 10;
	for (c = 256; --c >= 0; )
		{
		hashf[c] = ((( c &0x7) << 7) ^ c) << adjbits;
		}

	exit_stat = OK;
	if (alloc_tables(maxcode, hashsize))  /* exit_stat already set */
		return(-1);

	init_hash_table();

	/*
	* Check the input stream for previously seen strings.  We keep
	* adding characters to the previously seen prefix string until we
	* get a character which forms a new (unseen) string.  We then send
	* the code for the previously seen prefix string, and add the new
	* string to our tables.  The check for previous strings is done by
	* hashing.  If the code for the hash value is unused, then we have
	* a new string.  If the code is used, we check to see if the prefix
	* and suffix values match the current input; if so, we have found
	* a previously seen string.  Otherwise, we have a hash collision,
	* and we try secondary hash probes until we either find the current
	* string, or we find an unused entry (which indicates a new string).
	*/
	write_flash_bytes(magic_header, 2);
	temp = ((char)(MAXBITS | block_compress));
	write_flash_bytes(&temp, 1);
	bytes_out = 3L;     /* includes 3-byte header mojo */
	out_length = bytes_out;

	in_count = 1L;
	offset = 0;

	if ((c = getbyte()) == EOF)
		return(out_length);

	printf("Compressing code and writing to flash ");

	prefxcode = (INTCODE)c;

	while ((c = getbyte()) != EOF)
		{
		in_count++;

		if ((in_count & 0xff) == 0)
			ReSchedule();

		if ((in_count & 0xffff) == 0)
			printf("-");

		hash = prefxcode ^ hashf[c];
		/*
		* I need to check that my hash value is within range
		* because my 16-bit hash table is smaller than 64k.
		*/
		if (hash >= hashsize)
			hash -= hashsize;
		if ((code = (INTCODE)probe(hash)) != UNUSED)
			{
			assert(10, code >= FIRSTFREE);
			assert(12, code <= maxcode);
			if (suffix[code] != (char)c || (INTCODE)prefix[code] != prefxcode)
				{
				/* hashdelta is subtracted from hash on each
				* iteration of the following hash table search
				* loop.  I compute it once here to remove it
				* from the loop.
				*/
				HASH hashdelta = (0x120 - c) << (adjbits);
				do 
					{
					/* rehash and keep looking */
					assert(1, code >= FIRSTFREE);
					assert(11, code <= maxcode);
					if (hash >= hashdelta)
						hash -= hashdelta;
					else
						hash += (hashsize - hashdelta);
					assert(2, hash < hashsize);
					if ((code = (INTCODE)probe(hash)) == UNUSED)
						goto newcode;
					} while (suffix[code] != (char)c || (INTCODE)prefix[code] != prefxcode);
				}
			prefxcode = code;
			}
		else
			{
newcode:

			putcode(prefxcode, bits);
			code = nextfree;
			assert(3, hash < hashsize);
			assert(4, code >= FIRSTFREE);
			assert(5, code <= maxcode + 1);
			if (code <= maxcode)
				{
				probe(hash) = (CODE)code;
				prefix[code] = (CODE)prefxcode;
				suffix[code] = (char)c;
				if (code > highcode)
					{
					highcode += code;
					++bits;
					}
				nextfree = code + 1;
				}
			else if ((in_count >= checkpoint) && block_compress )
				{
				if (cl_block())
					{
					putcode((INTCODE)c, bits);
					putcode(CLEAR, bits);
					init_hash_table();
					if ((c = getbyte()) == EOF)
						break;
					in_count++;
					}
				}
			prefxcode = (INTCODE)c;
			}
		}

	putcode(prefxcode, bits);
	putcode(CLEAR, 0);

	free_tables(maxcode, hashsize);
	printf("-\n");

	return(out_length);
}

UCHAR rmask[9] = {0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff};

/************************************************************************
*
*	Function:	putcode
*
*	Description:	
*
*	Inputs:		
*
*	Outputs:	None
*
*	Returns:	None
*
**************************************************************************/

void
putcode(INTCODE code, register int bits)
{
	static int oldbits = 0;
	static UCHAR outbuf[MAXBITS];

	register UCHAR *buf;
	register int shift;

	if (bits != oldbits)
		{
		if (bits == 0)
			{
			/*
			* bits == 0 means EOF, write the rest of the buffer.
			*/
			if (offset > 0)
				{
				write_flash_bytes(outbuf, (offset + 7) >> 3);
				bytes_out += ((offset + 7) >> 3);
				}
			offset = 0;
			oldbits = 0;
			return;
			}
		else
			{
			/*
			* Change the code size.  We must write the whole buffer,
			* because the expand side won't discover the size change
			* until after it has read a buffer full.
			*/
			if (offset > 0)
				{
				write_flash_bytes(outbuf, oldbits);
				bytes_out += oldbits;
				offset = 0;
				}
			oldbits = bits;
			}
		}

	/*
	* Get to the first byte.
	*/
	buf = outbuf + ((shift = offset) >> 3);
	if ((shift &= 7) != 0)
		{
		*(buf) |= (*buf & rmask[shift]) | (UCHAR)(code << shift);
		*(++buf) = (UCHAR)(code >> (8 - shift));
		if (bits + shift > 16)
			*(++buf) = (UCHAR)(code >> (16 - shift));
		}
	else
		{
		/*
		* Special case for fast execution
		*/
		*(buf) = (UCHAR)code;
		*(++buf) = (UCHAR)(code >> 8);
		}

	if ((offset += bits) == (bits << 3))
		{
		bytes_out += bits;
		write_flash_bytes(outbuf, bits);
		offset = 0;
		}

	return;
}


/************************************************************************
*
*	Function:	nextcode
*
*	Description:	Get the next code from input
*
*	Inputs:		codeptr = address of where to put the next code
*
*	Outputs:	None
*
*	Returns:	TRUE  = success
*			FALSE = end-of-file
*
**************************************************************************/

int
nextcode(INTCODE *codeptr)
{
	static int prevbits = 0;
	static int size;
	static UCHAR inbuf[MAXBITS];

	register INTCODE code;
	register int shift;
	register UCHAR *bp;

	/*
	* If the next entry is a different bit-size than the preceeding one
	* then we must adjust the size and scrap the old buffer.
	*/
	if (prevbits != bits)
		{
		prevbits = bits;
		size = 0;
		}

	/*
	* If we can't read another code from the buffer, then refill it.
	*/
	if (size - (shift = offset) < bits)
		{
		/*
		* Read more input and convert size from # of bytes to # of bits
		*/
		if ((size = (getbytes(inbuf, bits) << 3)) <= 0)
			return(NO);

		offset = shift = 0;
		}

	/*
	* Get to the first byte.
	*/
	bp = inbuf + (shift >> 3);

	/*
	* Get first part (low order bits)
	*/
	code = (*bp++ >> (shift &= 7));

	/*
	* high order bits.
	*/
	code |= *bp++ << (shift = 8 - shift);
	if ((shift += 8) < bits)
		code |= *bp << shift;
	*codeptr = code & highcode;
	offset += bits;

	return (TRUE);
}

/************************************************************************
*
*	Function:	decompress
*
*	Description:	uncompress the given data
*			the data has been compressed using the unix
*			compression method
*
*	Inputs:		d_from   = pointer to data to be uncompressed
*			d_length = length of data to be uncompressed
*			d_to     = where to put the uncompressed data
*
*	Outputs:	None
*
*	Returns:	-1 = error - the error number can be found in exit_stat
*			(!= -1) = the number of bytes in the uncompressed data
*
**************************************************************************/

int
decompress(byte *d_from, int d_length, byte *d_to)
{
	static char *token = NULL;         /* String buffer to build token */
	static int maxtoklen = MAXTOKLEN;

	register int i;
	register INTCODE code;
	register char sufxchar;
	INTCODE savecode;
	register FLAG fulltable;
	register FLAG cleartable;
	register int maxbits = 0;

	if((int)d_from != FEPROM + aplLocation + sizeof(struct flash_info))
		d_from = (byte *)(FEPROM + aplLocation+sizeof(struct flash_info));

	/*
	* set up pointers
	*/
	to         = d_to;
	from       = d_from;
	in_length  = d_length;
	out_length = 0;
	/*
	* check to see if really compressed data
	*/
	if ((getbyte() != (magic_header[0] & 0xff)) ||
	    (getbyte() != (magic_header[1] & 0xff)))
		{
		exit_stat = INFILEBAD;
		return(-1);
		}

	maxbits = getbyte();    /* set -b from file */
	block_compress = maxbits & BLOCK_MASK;
	maxbits &= BIT_MASK;
	if(maxbits != MAXBITS) 
		{
		exit_stat = INFILEBAD;
		return(-1);
		}

	exit_stat = OK;

	if (alloc_tables(maxcode = ~(~(INTCODE)0 << MAXBITS),0))
		{
		/*
		* exit_stat already set
		*/
		return(-1);
		}

	/*
	* Initialze the token buffer.
	*/
	if ((token == NULL) && ((token = (char *)lmalloc(maxtoklen)) == NULL))
		{
		exit_stat = NOMEM;
		return(-1);
		}

	cleartable = TRUE;
	savecode = CLEAR;
	offset = 0;
	do
		{
		if ((code = savecode) == CLEAR && cleartable)
			{
			highcode = ~(~(INTCODE)0 << (bits = INITBITS));
			fulltable = FALSE;
			nextfree = (cleartable = block_compress) == FALSE ? 256 : FIRSTFREE;
			if (!nextcode(&prefxcode))
				break;
			putbyte((sufxchar = (char)prefxcode));
			continue;
			}
		i = 0;
		if (code >= nextfree && !fulltable)
			{
			if (code != nextfree)
				{
				exit_stat = CODEBAD;
				printf("\n");
				return(-1);
				}
			/*
			* Special case for sequence KwKwK (see text of article)
			*/
			code = prefxcode;
			token[i++] = sufxchar;
			}
		/*
		* Build the token string in reverse order by chasing down
		* through successive prefix tokens of the current token.
		* Then output it.
		*/
		while (code >= 256)
			{
			if (i >= maxtoklen)
				{
				int old_length;

				old_length = maxtoklen;
				maxtoklen *= 2;   /* double the size of the token buffer */
				free(token);
				if((token = (char *)lmalloc(maxtoklen)) == NULL)
#ifdef 0
				if ((token = (char *)realloc(token, old_length, maxtoklen)) == NULL)
#endif
					{
					exit_stat = TOKTOOBIG;
					printf("\n");
					return(-1);
					}
				}
			token[i++] = suffix[code];
			code = (INTCODE)prefix[code];
			}
		putbyte(sufxchar = (char)code);
		while (--i >= 0)
			putbyte(token[i]);

		/*
		* If table isn't full, add new token code to the table with
		* codeprefix and codesuffix, and remember current code.
		*/
		if (!fulltable)
			{
			code = nextfree;
			assert(6, 256 <= code && code <= maxcode);
			prefix[code] = (CODE)prefxcode;
			suffix[code] = sufxchar;
			prefxcode = savecode;
			if (code++ == highcode)
				{
				if (highcode >= maxcode)
					{
					fulltable = TRUE;
					--code;
					}
				else
					{
					++bits;
					highcode += code;           /* nextfree == highcode + 1 */
					}
				}
			nextfree = code;
			}

		} while (nextcode(&savecode));

#ifdef 0
	lmfree(token, maxtoklen);
#endif
	free(token);

	free_tables(maxcode, hashsize);

	printf("-\n");
	return(out_length);
}

/************************************************************************
*
*	Function:	putbyte
*
*	Description:	write the given byte into the address pointed to
*			by the global 'to'
*			let system run after 0x7ff bytes
*			print still alive marker every 0xffff bytes
*
*	Inputs:		b = byte to be written
*
*	Outputs:	None
*
*	Returns:	None
*
**************************************************************************/

void
putbyte(char b)
{
	*to++ = b;
	out_length++;

	if ((out_length & 0x7ff) == 0)
		ReSchedule();

	if ((out_length & 0xffff) == 0)
		printf("-");

	return;
}

/************************************************************************
*
*	Function:	getbyte
*
*	Description:	return the byte pointed to by the global 'from'
*			increment the global 'from' and decrement its
*			length
*
*	Inputs:		None
*
*	Outputs:	None
*
*	Returns:	byte pointed to as an integer
*			EOF = end of buffer
*
**************************************************************************/

int
getbyte()
{
	register unsigned char c;
	register int i;

	if (in_length == 0)
		return(EOF);

	c = (unsigned char)*from++;
	in_length--;
	i = c;

	return(i);
}

/************************************************************************
*
*	Function:	write_flash_bytes
*
*	Description:	write the given byte stream into flash
*
*	Inputs:		bp = address of bytes to be written
*			n  = number of bytes to be written
*
*	Outputs:	None
*
*	Returns:	0 = OK
*			1 = error
*
**************************************************************************/

int
write_flash_bytes(byte *bp, int n)
{
	register int i;

	for (i = 0; i < n; i++)
		{
		if (write_flash(to, bp[i]))
			{
			printf("Error writing byte to flash memory\n");
			return(1);
			}
		to++;
		out_length++;
		}

	return(0);
}

/************************************************************************
*
*	Function:	getbytes
*
*	Description:	read a number of bytes using the getbyte routine
*			into a buffer
*
*	Inputs:		bp = pointer of where to put bytes
*			n  = number of bytes to read
*
*	Outputs:	None
*
*	Returns:	size = number of bytes actually read
*
**************************************************************************/

int
getbytes(byte *bp, int n)
{
	register int i;
	register int size;
	register int c;

	size = 0;
	for (i = 0; i < n; i++)
		{
		c = getbyte();
		if (c == EOF)
			break;
		bp[i] = c;
		size++;
		}

	return(size);
}

/************************************************************************
*
*	Function:	assert
*
*	Description:	indicates that x should be true at this point.
*			If it is not true, a message is written to the console.
*
*	Inputs:		n = tag for where assert is called from
*			x = expression which should be true
*
*	Outputs:	None
*
*	Returns:	None
*
**************************************************************************/

void
assert(int n, int x)
{
	if (!x)
		printf("assert failed %D\n", n);

	return;
}
