;	8080 CBIOS for Z80-Simulator
;
;	Copyright (C) 2007 by Udo Munk
;
MSIZE	EQU	63		;cp/m version memory size in kilobytes
;
;	"bias" is address offset from 2900H for memory systems
;	than 16K (referred to as "b" throughout the text).
;
BIAS	EQU	(MSIZE-16)*1024
CCP	EQU	BIAS+2900H	;base of ccp
BDOS	EQU	CCP+806H	;base of bdos
BIOS	EQU	CCP+1500H	;base of bios
NSECTS	EQU	(BIOS-CCP)/128	;warm start sector count
CDISK	EQU	0004H		;current disk number 0=A,...,15=P
IOBYTE	EQU	0003H		;intel i/o byte
;
;	I/O ports
;
CONSTA	EQU	0		;console status port
CONDAT	EQU	1		;console data port
PRTSTA	EQU	2		;printer status port
PRTDAT	EQU	3		;printer data port
AUXDAT	EQU	5		;auxiliary data port
FDCD	EQU	10		;fdc-port: # of drive
FDCT	EQU	11		;fdc-port: # of track
FDCS	EQU	12		;fdc-port: # of sector
FDCOP	EQU	13		;fdc-port: command
FDCST	EQU	14		;fdc-port: status
DMAL	EQU	15		;dma-port: dma address low
DMAH	EQU	16		;dma-port: dma address high
;
	ORG	BIOS		;origin of this program
;
;	jump vector for individual subroutines
;
	JMP	BOOT		;cold start
WBOOTE: JMP	WBOOT		;warm start
	JMP	CONST		;console status
	JMP	CONIN		;console character in
	JMP	CONOUT		;console character out
	JMP	LIST		;list character out
	JMP	PUNCH		;punch character out
	JMP	READER		;reader character in
	JMP	HOME		;move head to home position
	JMP	SELDSK		;select disk
	JMP	SETTRK		;set track number
	JMP	SETSEC		;set sector number
	JMP	SETDMA		;set dma address
	JMP	READ		;read disk
	JMP	WRITE		;write disk
;
;	messages
;
SIGNON: DB	'63K CP/M Vers. 1.4 (8080 CBIOS V1.0 for Z80SIM, '
	DB	'Copyright 2007 by Udo Munk)'
	DB	13,10,0
;
LDERR:	DB	'Error booting'
	DB	13,10,0
;
;	end of messages
;
;	utility functions
;
;	print a 0 terminated string to console device
;	pointer to string in HL
;
PRTMSG:	XCHG
	LDAX	D
	XCHG
	ORA	A
	RZ
	MOV	C,A
	CALL	CONOUT
	INX	H
	JMP	PRTMSG
;
;	individual subroutines to perform each function
;	simplest case is to just perform parameter initialization
;
BOOT:   LXI	SP,80H		;use space below buffer for stack
	LXI	H,SIGNON	;print message
	CALL	PRTMSG
	XRA	A		;zero in the accum
	STA	IOBYTE		;clear the iobyte
	STA	CDISK		;select disk zero
	JMP	GOCPM		;initialize and go to cp/m
;
;	simplest case is to read the disk until all sectors loaded
;
WBOOT:  LXI	SP,80H		;use space below buffer for stack
	MVI	C,0		;select disk 0
	CALL	SELDSK
	CALL	HOME		;go to track 00
;
	MVI	B,NSECTS	;b counts # of sectors to load
	MVI	C,0		;c has the current track number
	MVI	D,2		;d has the next sector to read
;	note that we begin by reading track 0, sector 2 since sector 1
;	contains the cold start loader, which is skipped in a warm start
	LXI	H,CCP		;base of cp/m (initial load point)
LOAD1:				;load one more sector
	PUSH	B		;save sector count, current track
	PUSH	D		;save next sector to read
	PUSH	H		;save dma address
	MOV	C,D		;get sector address to register c
	CALL	SETSEC		;set sector address from register c
	POP	B		;recall dma address to b,c
	PUSH	B		;replace on stack for later recall
	CALL	SETDMA		;set dma address from b,c
;	drive set to 0, track set, sector set, dma address set
	CALL	READ
	ORA	A		;any errors?
	JZ	LOAD2		;no, continue
	LXI	H,LDERR		;error, print message
	CALL	PRTMSG
	DI			;and halt the machine
	HLT
;	no error, move to next sector
LOAD2:	POP	H		;recall dma address
	LXI	D,128		;dma=dma+128
	DAD	D		;new dma address is in h,l
	POP	D		;recall sector address
	POP	B		;recall number of sectors remaining,
				;and current trk
	DCR	B		;sectors=sectors-1
	JZ	GOCPM		;transfer to cp/m if all have been loaded
;	more sectors remain to load, check for track change
	INR	D
	MOV	A,D		;sector=27?, if so, change tracks
	CPI	27
	JC	LOAD1		;carry generated if sector<27
;	end of current track, go to next track
	MVI	D,1		;begin with first sector of next track
	INR	C		;track=track+1
;	save register state, and change tracks
	CALL	SETTRK		;track address set from register c
	JMP	LOAD1		;for another sector
;	end of load operation, set parameters and go to cp/m
GOCPM:
	MVI	A,0C3H		;c3 is a jmp instruction
	STA	0		;for jmp to wboot
	LXI	H,WBOOTE	;wboot entry point
	SHLD	1		;set address field for jmp at 0
;
	STA	5		;for jmp to bdos
	LXI	H,BDOS		;bdos entry point
	SHLD	6		;address field of jump at 5 to bdos
;
	LXI	B,80H		;default dma address is 80h
	CALL	SETDMA
;
	LDA	CDISK		;get current disk number
	MOV	C,A		;send to the ccp
	JMP	CCP		;go to cp/m for further processing
;
;
;	simple i/o handlers
;
;	console status, return 0ffh if character ready, 00h if not
;
CONST:	IN	CONSTA		;get console status
	RET
;
;	console character into register a
;
CONIN:	IN	CONDAT		;get character from console
	RET
;
;	console character output from register c
;
CONOUT: MOV	A,C		;get to accumulator
	OUT	CONDAT		;send character to console
	RET
;
;	list character from register c
;
LIST:	MOV	A,C		;character to register a
	OUT	PRTDAT
	RET
;
;	punch character from register c
;
PUNCH:	MOV	A,C		;character to register a
	OUT	AUXDAT
	RET
;
;	read character into register a from reader device
;
READER: IN	AUXDAT
	RET
;
;
;	i/o drivers for the disk follow
;
;	move to the track 00 position of current drive
;	translate this call into a settrk call with parameter 00
;
HOME:	MVI	C,0		;select track 0
	JMP	SETTRK		;we will move to 00 on first read/write
;
;	select disk given by register C
;
SELDSK: MOV	A,C
	OUT	FDCD		;selekt disk drive
	RET
;
;	set track given by register c
;
SETTRK: MOV	A,C
	OUT	FDCT
	RET
;
;	set sector given by register c
;
SETSEC: MOV	A,C
	OUT	FDCS
	RET
;
;	set dma address given by registers b and c
;
SETDMA: MOV	A,C		;low order address
	OUT	DMAL
	MOV	A,B		;high order address
	OUT	DMAH		;in dma
	RET
;
;	perform read operation
;
READ:	XRA	A		;read command -> A
	JMP	WAITIO		;to perform the actual i/o
;
;	perform a write operation
;
WRITE:	MVI	A,1		;write command -> A
;
;	enter here from read and write to perform the actual i/o
;	operation.  return a 00h in register a if the operation completes
;	properly, and 01h if an error occurs during the read or write
;
WAITIO: OUT	FDCOP		;start i/o operation
	IN	FDCST		;status of i/o operation -> A
	RET
;
	END			;of BIOS
