/*
 * SessionServer.c
 *
 * Created Spring 1985 by Michael Baldwin
 *
 * Ported to 4.3BSD (Sklower's XNS implementation)
 *
 * Modified Fall 1985 to:
 *   process .bridgerc
 *   check quotas
 *   write wtmp and lastlog entries
 *   use SIGSTOP to hold sessions
 *
 * Last edited by AJD on Thu Jan  2 21:49:55 PST 1986
 *
 * Session servers (not just initiation daemon) must run with effective
 * uid of root to kill suid root stubs.  Sigh.
 *
 */
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <lastlog.h>
#include <netinet/in.h>
#include <pwd.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/quota.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sysexits.h>
#include <utmp.h>

#include "CommandOps.h"
#include "NetworkStream.h"
#include "StubOps.h"

extern int errno;
extern char * index();
extern char * malloc();

#define NSPORT_SESSION 3001 /* well-known session initiation socket */

#define SUPERUSER 0

#define LISTENER (LAST_NONSTD)	/* standard file descriptor */
#define LOG	(LISTENER - 1)	/* standard file descriptor */

#define MAX_SESSIONS 64		/* max # simultaneous sessions */

#define LOGIN_TIMEOUT (60)	/* login timeout 1 minute */
#define	DEFAULT_TIMEOUT (5*60)	/* default keepalive timeout */
#define TIMEOUT_GRACE 60	/* grace period before killing session */

#define MAX_ARGC 20		/* max # args for session initial command */
#define CMD_BUFSIZE 1024	/* max length of session cmd */

#define BRIDGERC ".bridgerc"	/* session init file */

#define WTMPNAME "/usr/adm/wtmp"		/* accounting file */
#define LASTLOGNAME "/usr/adm/lastlog"		/* "last login" file */
#define NOLOGINNAME "/etc/nologin"		/* inhibits logins */

#define ENV_INITIALCOMMAND "BRIDGE_INITIALCOMMAND"

#define DEFAULT_COMMAND	"RTTY LoginSh"		/* default initial command */

#define DEFAULT_SHELL "/bin/sh"			/* default shell */


/*
 * Logging
 *
 * (This may be useful enough to move to a separate file.)
 *
 * Log must be an existing file writable by real uid.
 */

static int logFildes = -1;
static char logBuf[1024];


OpenLog(logName, fildes)
    char *logName;
    int fildes;
{
    if( logFildes < 0 ) {
        if( access(logName, W_OK) < 0 )
	    return;
	if( (logFildes = open(logName, (O_WRONLY|O_APPEND), 0666)) < 0 )
	    return;
    }
    /* assert: logFildes >= 0 */
    if( (fildes >= 0) && (logFildes != fildes) )
        { dup2(logFildes, fildes); close(logFildes); logFildes = fildes; }
    ioctl(logFildes, FIOCLEX, 0);
}

CloseLog()
{
    if( logFildes >= 0 ) close(logFildes);
    logFildes = -1;
}


LogMsg(fmt, x, y, z)
{
    if( logFildes >= 0 ) {
        sprintf(logBuf, "Process %d: ", getpid());
	write(logFildes, logBuf, strlen(logBuf));
        sprintf(logBuf, fmt, x, y, z);
        write(logFildes, logBuf, strlen(logBuf));
    }
}

/*
 * Signal handling.
 *
 * This only works in VAX implementation, because of the assumption
 * that the signal number is passed as an argument to the handler.
 *
 * It's structured this way to make the logging stuff easier,
 * and because it was helpful for debugging.
 *
 * Any UNIX that doesn't support this is STUPID.  Sigh.
 */

SetSigs(func, sig /*, sig ... 0 */ )
    int (*func)();
    int sig;
{
    register int *p = &sig;
    for(p = &sig; *p; p++) {
	LogMsg("catch signal(%d, 0x%x)\n", *p, func);
	signal(*p, func);
    }
}


/*
 *
 * DL Session Initiation Server
 *
 * A system daemon.
 *
 */

/*
 * map: (session manager pid) --> slot
 */

int sessionMap[MAX_SESSIONS];

int
ReserveSessionSlot()
{
    int slot;

    for( slot = 0; slot < MAX_SESSIONS; slot++ )
        if( sessionMap[slot] == 0 )
	    { sessionMap[slot] = -1; return( slot ); }
    return( -1 );
}


SetSessionSlotPid(slot, pid)
    int slot;
    int pid;
{
    if( slot >= 0 ) sessionMap[slot] = pid;
}


int
GetSessionSlotFromPid(pid)
    int pid;
{
    int slot;

    for( slot = 0; slot < MAX_SESSIONS; slot++ )
        if( sessionMap[slot] == pid ) return( slot );
    return( -1 );
}


FreeSessionSlot(slot)
    int slot;
{
    if( slot >= 0 ) sessionMap[slot] = 0;
}


/*
 * clean up after session manager process dies
 */

ReapSession()
{
    union wait status;
    int pid;
    int slot;

    LogMsg("collecting terminated sessions\n");
    while( (pid = wait3(&status, WNOHANG, NULL)) > 0 ) {
	LogMsg("termsig %d retcode %d coredump %d\n", status.w_termsig,
		status.w_retcode, status.w_coredump);
	slot = GetSessionSlotFromPid(pid);
	LogMsg("pid %d session slot %d\n", pid, slot); 
	FreeSessionSlot(slot);
    }
    LogMsg("no more children\n");
}

/*
 * Signal handler
 */
 
ServerOnSig(i)
{
    LogMsg("received signal %d\n", i);
    switch(i) {
	case SIGCHLD:
	    ReapSession();
	    break;
	case SIGHUP:
	    RestartServer(1);
	    /*NOTREACHED*/
	case SIGTERM:
	    LogMsg("Exiting\n");
	    exit(EX_OK);
	    /*NOTREACHED*/
	default:
	    LogMsg("unexpected signal ignored.\n");
	    break;
    }
}

/*
 * Session server restart mechanism
 */
jmp_buf restartEnv;

RestartServer(why)
    int why;
{
    longjmp(restartEnv, why);
    /*NOTREACHED*/
}


/*
 * Session server main program
 */

main(argc, argv)
    int argc;
    char **argv;
{
    int socketNo = NSPORT_SESSION;
    short port;	/* socketNo with correct type for CreateListener() */
    char *logName = NULL;
    int restartReason = 0;
        
    if( ParseArgs( argc, argv,
            "[-s socketNumber] [-l logFileName]",
	    "-sd",	&socketNo,
	    "-ls",	&logName,
	    0 ) < 0 )
	exit( EX_USAGE );

    port = socketNo;
    if( logName != NULL ) (void)OpenLog(logName, LOG);

    /* create and bind listener socket */
    if( MoveDescriptor(SPPCreateListener(port), LISTENER) < 0 ) {
	fprintf(stderr, "Error %d creating socket\n", errno);
	exit(EX_CANTCREAT);
    }

    /* fork off child to run as daemon, exit from parent */
    { int child = fork();
        if( child < 0 )
	    { fprintf(stderr, "Can't fork"); exit(EX_OSERR); }
        else if( child > 0 )
	    _exit(0);
    }

    /* detach from terminal */
    if( ReplaceStdTTYs("/dev/null", /*setControl=*/TRUE) < 0 )
        { fprintf(stderr, "Can't detach TTYs"); exit(EX_UNAVAILABLE); }


    /* prepare for restart */
    if( (restartReason = setjmp(restartEnv)) != 0 ) /* restart */ {
        LogMsg("Restarting server %d\n", restartReason);
        close( LISTENER );
	if( MoveDescriptor(SPPCreateListener(port), LISTENER) < 0 ) {
	    LogMsg("Error %d creating socket for restart\n", errno);
	    exit(EX_CANTCREAT);
	}
    }


    LogMsg("Session initiator listening on port %d (%#x)\n", port, port );

    /* catch signals */
    { int sig;
        for( sig = 1; sig < NSIG; sig++ ) signal(sig, SIG_IGN);
        SetSigs(ServerOnSig, SIGCHLD, SIGHUP, SIGTERM, 0);
	SetSigs(SIG_DFL, SIGQUIT, 0);
    }

    /* service requests forever */
    for (;;) {
        int child;
	int slot;
	int tempStream;

	close( STDSTREAM ); /* may not be open, but that's okay */

	if( (tempStream = SPPAccept(LISTENER,NULL)) < 0 ) {
	    LogMsg("SPPAccept(...) failed %d\n", sppResult);
	    sleep(5);
	    /* KLUDGE -- but the server has been known to loop here! */
	    if( sppResult = -ECONNABORTED ) RestartServer(2);
	    continue;
	} 
	if( MoveDescriptor( tempStream, STDSTREAM ) < 0 ) {
	    LogMsg("MoveDescriptor(...) failed\n");
	    sleep(5); continue;
	}

	LogMsg("received session request\n");

	if( (slot = ReserveSessionSlot()) < 0 ) {
	    LogMsg("out of session slots\n");
	    pause(); /* wait for child to terminate */
	    continue;
	}

	if( (child = fork()) == 0 ) /* child */ {

	    close(LISTENER);
	    strncpy(argv[0], "[Bridge Session]", strlen(argv[0]));
	    ManageSession(slot);
	    /*NOTREACHED*/

	} else if( child > 0 ) /* parent */ {

	    SetSessionSlotPid(slot, child);

	} else if( child < 0 ) /* fork error */ {

	    LogMsg("connect request fork error %d\n", errno);
	    FreeSessionSlot(slot);

	}

    }
    /*NOTREACHED*/
}


/*
 *
 * Session Manager
 *
 * One per Bridge session.
 *
 */
 
char sessionLine[20];	/* fake tty line identifying session */
char *sessionUser;	/* logged-in session user */
char *sessionHost;	/* host initiating session */

int sessionID;		/* process group for all command stubs */
int sessionuid;		/* uid of logged-in session user */
int sessiongid;		/* gid of logged-in session user */

int sessionHeld = 0;	/* true => SIGSTOP has been sent to stubs */
			/* (so workstation can world-swap)	  */

int wtmpFile = -1;	/* file descriptor for wtmp */
#define WTMPWRITTEN() (wtmpFile >= 0)

/*
 * Write an accounting entry in wtmp
 */
WriteWtmp(user, host)
    char *user;
    char *host;
{
    struct utmp wtmpEntry;

    if( wtmpFile < 0 ) wtmpFile = open(WTMPNAME, O_WRONLY|O_APPEND);
    if( wtmpFile < 0 ) return;

    strncpy(wtmpEntry.ut_line, sessionLine, (sizeof wtmpEntry.ut_line));
    strncpy(wtmpEntry.ut_name, user, (sizeof wtmpEntry.ut_name));
    strncpy(wtmpEntry.ut_host, host, (sizeof wtmpEntry.ut_host));
    time( &wtmpEntry.ut_time );

    write(wtmpFile, (char *)&wtmpEntry, (sizeof wtmpEntry));
}


/*
 * Write lastlog entry
 */
WriteLastLog()
{
    struct lastlog llEntry;
    int f;

    if( (f = open(LASTLOGNAME, O_RDWR)) < 0) return;

    time( &llEntry.ll_time );
    strncpy(llEntry.ll_line, sessionLine, (sizeof llEntry.ll_line));
    strncpy(llEntry.ll_host, sessionHost, (sizeof llEntry.ll_host));

    lseek(f, (long)(sessionuid) * sizeof (struct lastlog), 0);
    write(f, (char *) &llEntry, (sizeof llEntry));
    close(f);
}


/*
 * Get message from /etc/nologin if there is one.
 * Return it in a string, with \n converted to \r
 */
char *
GetNoLoginMsg()
{
    int f = -1;
    static char buf[400];
    char *answer = NULL;
    register char *p;
    int n;

    if( (f = open(NOLOGINNAME, O_RDONLY)) >= 0 ) {
        if( (n = read(f, buf, (sizeof buf)-1)) >= 0 ) {
	    buf[n] = 0;
	    for( p = buf; *p; p++ )
	        if( *p == '\n' ) *p = '\r';
	    answer = buf;
	}
	close(f);
    }
    return( answer );
}


/*
 * Hold/Release a session
 * (by STOPping or restarting all the stubs)
 */

HoldSession()
{
    killpg(sessionID, SIGSTOP);
    sessionHeld = 1;
}

ReleaseSession()
{
    if( !sessionHeld ) return;
    killpg(sessionID, SIGCONT);
    sessionHeld = 0;
}


/*
 * Either the remote exec server has died or told us to go away,
 * or someone sent us a hangup from UNIX.  Close the stream (it
 * may already be dead), hangup all stubs in this session, write
 * a logout accounting entry.
 */
KillSession()
{
    LogMsg("killpg(%d,%d)\n", sessionID, SIGHUP);
    signal(SIGHUP, SIG_IGN);
    if( sessionHeld ) { ReleaseSession(); sleep(5); }
    killpg(sessionID, SIGHUP);
    SPPClose(STDSTREAM, TRUE);
    if( WTMPWRITTEN() ) WriteWtmp("", "");
    exit(EX_OK);
}

/*
 * Clean up after terminated commands.
 * N.B. This also gets entered when stubs are stopped by HoldSession().
 */
ReapCommand()
{
    union wait status;

    LogMsg("collecting children\n");
    while( wait3(&status, WNOHANG, NULL) > 0 ) 
	LogMsg("termsig %d retcode %d coredump %d\n", status.w_termsig,
		status.w_retcode, status.w_coredump);
    LogMsg("no more children\n");
}

/*
 * session manager signal handler
 */
SessionOnSig(i)
{
    LogMsg("received signal %d\n", i);
    switch(i) {
        case SIGHUP:
	case SIGALRM:
	    KillSession();
	    /*NOTREACHED*/
	case SIGCHLD:
	    ReapCommand();
	    break;
	default:
	    LogMsg("unexpected signal ignored.\n");
	    break;
    }
}


/*
 * Process .bridgerc file
 *
 * For now, just create environment entries from lines in file.
 */
 
extern char **environ;
char *(nullEnviron[]) = { NULL };

ReadBridgerc()
{
    FILE *rcFile;
    int len;
    char lineBuf[1024];
    register char *p;

    if( (rcFile = fopen(BRIDGERC, "r")) == NULL )
        return;

    while( fgets(lineBuf, (sizeof lineBuf), rcFile) != NULL ) {
        if( (len = strlen(lineBuf)) == 0 )
	    continue;
	if( lineBuf[len-1] == '\n' )
	    lineBuf[--len] = 0; /* clobber the newline */
	for( p = lineBuf; (*p != 0) && (*p != '='); p++ ) ;
	if( *p == '=' ) {
	    *p++ = 0;
	    while( *p == ' ' ) p++;
	}
	if( *p == 0 )
	    continue;
	SetEnv(lineBuf, p);
    }

    fclose(rcFile);
}


/*
 * Session manager entry point
 */
ManageSession(slot)
    int slot;
{
    char *xpass;
    char *xws;
    register struct passwd *pw;
    extern struct ns_addr *GetXNSAddr();
    char authBuf[CMD_BUFSIZE];
    char *noLoginMsg;
    static int on = 1;


	sprintf(sessionLine, "bty%d", slot);

        sessionID = getpid();
	LogMsg("new session slot %d ID %d\n", slot, sessionID );

	SetSigs(SessionOnSig, SIGHUP, SIGALRM, SIGCHLD, 0);

        alarm(LOGIN_TIMEOUT);

	for(;;) {

	    { int cc;
	        cc = SPPRead( STDSTREAM,
	            authBuf, (sizeof authBuf) - 1, /*readToEM=*/1 );
	        if( cc <= 0 )
	            { sleep(5); continue; }
	        authBuf[cc] = 0;
	    }

	    /* EVENTUALLY get XNS authentication data here */

	    { char *argp;
	        argp = authBuf;
		xws = GetToken(&argp);
		sessionUser = GetToken(&argp);
		xpass = GetToken(&argp);
	    }
	    
	    if( (GetXNSAddr(xws) != NULL) /* parses okay */
		    && ((pw = getpwnam(sessionUser)) != NULL)
		    && !strcmp(crypt(xpass, pw->pw_passwd), pw->pw_passwd) )
		break;

	    LogMsg("login incorrect\n");
	    SendCmdf(STDSTREAM, NACK, "login incorrect");

	}

	alarm( 0 ); /* cancel outstanding request */

	while(*xpass) *xpass++ = 0; /* smash password */

	/*  Compute host name.  EVENTUALLY do this better. */
	{ char *temp;
	    sessionHost = malloc( strlen(xws)+1 );
	    strcpy(sessionHost, xws);
	    temp = index(sessionHost, '.');
	    if( temp != NULL ) temp = index(temp+1, '.');
	    if( temp != NULL ) *temp = 0;
	}

	sessionuid = pw->pw_uid;
	sessiongid = pw->pw_gid;

	LogMsg("workstation %s\n", xws );
	LogMsg("authenticated as %d, %d\n", sessionuid, sessiongid);

	if( (sessionuid != SUPERUSER)
	        && ((noLoginMsg = GetNoLoginMsg()) != NULL) ) {
	    SendCmdf(STDSTREAM, 'Q', "%s", noLoginMsg);
	    LogMsg("logins inhibited\n");
	    KillSession();
	    /*NOTREACHED*/
	}
	
	if (quota(Q_SETUID, sessionuid, 0, 0) < 0) {
	    char * errMsg;
	    if (errno == EUSERS)
	        errMsg = "Too many users logged on already";
	    else if (errno == EPROCLIM)
	        errMsg = "You have too many processes running";
	    else
	        errMsg = "setuid error";
	    SendCmdf(STDSTREAM, 'Q', "%s", errMsg);
	    LogMsg("quota err %s\n", errMsg);
	    KillSession();
	    /*NOTREACHED*/
	}

	setreuid( sessionuid, SUPERUSER );
	setgid( sessiongid );
	initgroups(sessionUser, sessiongid);

	LogMsg("server real uid (gid) set to %d, (%d)\n", getuid(), getgid() );
	LogMsg("server eff uid (gid) set to %d, (%d)\n", geteuid(), getegid() );

	environ = nullEnviron;
	SetEnv("USER", sessionUser);
	SetEnv("HOME", pw->pw_dir);
	SetEnv("SHELL", ((*pw->pw_shell) ? pw->pw_shell : DEFAULT_SHELL) );
	SetEnv("WORKSTATION", xws);
	SetEnv("HOST", sessionHost);
	SetEnvI("SESSION", sessionID);
	/* EVENTUALLY add session conversation key to environment */
	SetEnv("PATH", DEF_PATH );

	if( chdir(pw->pw_dir) < 0 ) {
	    LogMsg("no HOME");
	} else {
	    ReadBridgerc();
	}

	WriteWtmp(sessionUser, sessionHost);
	WriteLastLog();
	LogMsg("ACK start session %d\n", sessionID);
	SendACK( STDSTREAM );
	RunSession();
	KillSession();
	/*NOTREACHED*/

}



int
CollectArgs(argp, argv, spaceIsDelim)
    register char *argp;
    char **argv;
    int spaceIsDelim;
{
    register int i;
    for( i = 0; i < MAX_ARGC; i++ ) {
        while( *argp == ' ' ) argp++;
	if( *argp == NULL ) break;
        argv[i] = argp;
	for(;;) {
	    if( (*argp == NULL) || (*argp == '\r') ) break;
	    if( spaceIsDelim && (*argp == ' ') ) break;
	    argp++;
	}
	if( *argp ) *argp++ = NULL;
    }
    argv[i] = NULL;
    return(i);
}



RunSession()
{
    int timeout = DEFAULT_TIMEOUT;
    int cmd;
    int child;
    char cmdBuf[CMD_BUFSIZE];

	for (;;) {

	    alarm( timeout + TIMEOUT_GRACE );
	    cmd = RcvCmd( STDSTREAM, cmdBuf, (sizeof cmdBuf) - 1 );
	    alarm( 0 );

	    switch( cmd ) {

		case 'E':	/* Execute command */ {
		    char *(childArgv[MAX_ARGC + 1]);
		    int nArgs;
		    char *temp;
		    int i;

		    nArgs = CollectArgs(cmdBuf, childArgv,
		            /*spaceIsDelim=*/ 0);
		    if( (nArgs == 0) &&
		            ((temp = GetEnv(ENV_INITIALCOMMAND)) != NULL) ) {
		        strncpy(cmdBuf, temp, (sizeof cmdBuf));
			cmdBuf[(sizeof cmdBuf)-1] = 0;
			nArgs = CollectArgs(cmdBuf, childArgv,
			        /*spaceIsDelim=*/ 1);
		    }
		    if( nArgs == 0 ) {
		        strcpy(cmdBuf, DEFAULT_COMMAND);
			nArgs = CollectArgs(cmdBuf, childArgv,
			        /*spaceIsDelim=*/ 1);
		    }

		    LogMsg("command %s ...\n", childArgv[0]);
		    
		    if ((child = fork()) == 0) {
		        if( (setregid( sessiongid, sessiongid ) < 0)
			        || (setreuid( sessionuid, sessionuid ) < 0) ) {
			    LogMsg("can't set ids to %d, %d\n",
			            sessionuid, sessiongid);
			    exit(EX_NOPERM);
			}
			for( i = STDERR+1; i < NOFILE; i++)
			    if( i != logFildes ) close(i);
			for( i = 1; i < NSIG; i++ )
			    signal(i, SIG_DFL);
			execvp(childArgv[0], childArgv);
			/*NOTREACHED*/
			LogMsg("exec %s failed %d\n",childArgv[0], errno);
			exit(EX_UNAVAILABLE);
		    }
		    if( child < 0 ) {
		        LogMsg("can't fork\n");
			SendCmdf(STDSTREAM, NACK, "Can't fork %d", errno);
		    } else {
		        SendACK(STDSTREAM);
		    }
		    break; }

		case 'H':	/* Hold session */
		    LogMsg("holdsession %s\n", cmdBuf);
		    HoldSession();
		    goto getTimeout;

		case 'Z':	/* keepalive / release session */
		    LogMsg("keepalive %s\n", cmdBuf);
		    ReleaseSession();
		    goto getTimeout;

		getTimeout:
		    if( (cmdBuf[0] != NULL) && (cmdBuf[0] != '\r') ) {
			if( (timeout = atoi(cmdBuf)) <= 0 ) {
			    SendCmdf(STDSTREAM, NACK, "Can't parse timeout");
			    timeout = DEFAULT_TIMEOUT;
			    break;
			}
		    } else {
		        timeout = DEFAULT_TIMEOUT;
		    }
		    SendACK(STDSTREAM);
		    break;

		case 'Q':	/* Quit */
		case 0: 	/* end of stream */
		    LogMsg("quit command %d\n", cmd);
		    goto out;

		default:	/* Oops */
		    LogMsg("bad command %d\n", cmd);
		    sleep(10);
		    break;
	    }
	}
  out:
    LogMsg("return from RunSession()\n");
}

