/*
 * RMailOps.c
 *
 * Edited by AJD on Mon Jul 22 00:01:30 EDT 1985
 *
 * Routines for reading Unix mail spool into message sets.
 *
 * TODO: This code needs more error checking.
 *       In particular, it completely ignores write errors
 *       in the heap or table of contents.
 *
 * FILE FORMAT:
 * 
 * The beginning of a mail message is delimited by
 *
 *   1. a null line or beginning of file, followed by
 *   2. the word "From " (including the space).
 *
 * It seems acceptable to consider the null line as part of the
 * previous message.
 *
 * Interesting fields follow.
 *
 * I. The (Unix-generated) From field.
 *
 * Examples:
 *
 *   From crow.pa@Xerox.ARPA Wed May  8 13:33:40 1985
 *   From susan Fri May 17 17:15:33 1985
 *   From MEYER@MIT-MC Tue May 21 22:12:01 1985
 *   From MAILER-DAEMON@rocky Thu May 23 17:20:30 1985
 *   From SATTERTHWAITE@TL-20B.ARPA Thu Jun  6 12:11:37 1985
 *
 * Format:
 *
 *   From senderAddress dayOfWeek monthOfYear dd hh:mm:ss yyyy
 *
 * II. The Subject field.
 *
 * Format:
 *
 *   Subject: arbitraryTextTerminatedByANewLine
 *
 */

#include <sys/types.h>

#include <errno.h>
#include <sys/stat.h>
#include <stdio.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <sysexits.h>

#include "MsgHeap.h"
#include "RMailOps.h"

#define LINE_LENGTH 1024
#define FILENAME_LENGTH 100

#define NUMERIC_MSGID yes

#define TRUE 1
#define FALSE 0

static char lineBuf[LINE_LENGTH];
static int lineLen;

static char firstLineBuf[LINE_LENGTH];
static char subjectBuf[LINE_LENGTH];

static FILE *usrSpoolFile = NULL;
static char *usrSpoolFileName = NULL;
static int usrSpoolFileLen;
static time_t usrSpoolFileMTime;
static int usrSpoolFileEoF;

static int usingLockFile = 0;

static FILE *toCFile = NULL;
static int currentMsgID;

static int msgHeap = -1;

extern int errno;

char *getenv();
char *rindex();
char *index();

/*
 * Utility for setting mail spool file name.
 * Defaults correctly if both arguments are NULL.
 */

static char usrSpoolFileNameBuf[FILENAME_LENGTH];

char *
SetSpoolFileName(myName, spoolDir)
    char *myName;
    char *spoolDir;
{
    register char *p;
    int len;

    if( usrSpoolFile != NULL ) return(NULL);
    usrSpoolFileName = NULL; /* forget the old one */

    if( spoolDir == NULL ) spoolDir = SPOOL_DIR;
    if( myName == NULL ) myName = getenv("USER");
    if( myName == NULL ) return( NULL );

    strncpy( usrSpoolFileNameBuf,
            spoolDir, (sizeof usrSpoolFileNameBuf)-2 );

    len = strlen(usrSpoolFileNameBuf);
    p = usrSpoolFileNameBuf + len;
    if( p[-1] != '/' ) { *p++ = '/'; len++; }

    strncat( usrSpoolFileNameBuf,
            myName, (sizeof usrSpoolFileNameBuf)-len-1 );

    return( usrSpoolFileName = usrSpoolFileNameBuf );
}


/*
 * Check for new mail.
 * Side effect: set usrSpoolFileLen, usrSpoolFileMTime.
 * Return 0 (no new mail), 1 (new mail), -1, ... (error)
 */
int
CheckMail( newMailOnly )
    int newMailOnly;
{
    struct stat b;

    if( (usrSpoolFileName == NULL)
            || (stat(usrSpoolFileName, &b) < 0) )
        return(-1);

    if( (b.st_mtime == usrSpoolFileMTime) && newMailOnly )
        return(0);

    usrSpoolFileMTime = b.st_mtime;
    usrSpoolFileLen = b.st_size;

    return( (usrSpoolFileLen > 0) ? 1 : 0 );
}


/*
 * Utilities for retrieving mail
 */
 
static int
IsPrefix(pattern, string)
    register char *pattern;
    register char *string;
{
    while( *pattern )
        if( *pattern++ != *string++ ) return(FALSE);
    return(TRUE);
}


#define IsNullLine() (lineBuf[0] == '\n')


static void
GetLine()
{
    char *s;
    if( ! usrSpoolFileEoF ) {
        s = fgets( lineBuf, (sizeof lineBuf), usrSpoolFile );
        usrSpoolFileEoF = (s == NULL);
    }
    lineLen = ( usrSpoolFileEoF ? 0 : strlen(lineBuf) );
}


#define PutLine() WriteMsg( msgHeap, lineBuf, lineLen )


static void
StartNewMsg()
{
    currentMsgID = StartMsg( msgHeap, 0 );
    /* strcpy( firstLineBuf, "From ?@? day mon dd hh:mm:ss yyyy\n" ); */
    strcpy( subjectBuf, "Subject: (no subject)\n" );
}


#define DoneWithMsg() EndMsg( msgHeap )


static void
AddToToC()
{
    register char *p;

    if( (p = rindex(firstLineBuf, ':')) != NULL ) *p = 0;
#   ifdef NUMERIC_MSGID
#	define MSG_FMT "U%d %s   %s"
#   else
#	define MSG_FMT "Umsg.%d %s   %s"
#   endif

    fprintf( toCFile, MSG_FMT,
        currentMsgID,
	firstLineBuf + 5 /* strlen("From ") */,
	subjectBuf + 9 /* strlen("Subject: ") */ );

#   undef MSG_FMT
}


/*
 * Retrieve mail.
 * Return number of messages read, or <0 (error)
 */
int
RetrieveMail( aToCFile, aMsgHeap, deleteFromSpool )
    FILE *aToCFile;
    int aMsgHeap;
    int deleteFromSpool;
{
    int sawNullLine;
    int nMsgs = 0;
    struct stat b;

	if( (usrSpoolFile == NULL)
		|| (fstat(fileno(usrSpoolFile),&b) < 0)
	        || ((toCFile = aToCFile) == NULL)
	        || (CheckWritableMsgHeap(msgHeap = aMsgHeap) != 0) )
	    return(-1);

	usrSpoolFileMTime = b.st_mtime;
	usrSpoolFileLen = b.st_size;

	if( fseek(usrSpoolFile,0L,0) < 0)
	    return(-2);
	usrSpoolFileEoF = FALSE;

	if( fseek(toCFile,0L,2) < 0 )
	    return(-3);

    	do {
	    GetLine();
	} while( (!usrSpoolFileEoF) && (!IsPrefix("From ", lineBuf)) );

	while( !usrSpoolFileEoF ) {

	    StartNewMsg();

	    strcpy( firstLineBuf, lineBuf );
	    PutLine(); GetLine();

	    while( (!usrSpoolFileEoF) && (!IsNullLine()) ) {

	        if( IsPrefix("Subject: ", lineBuf) ) {
		    strcpy(subjectBuf, lineBuf );
		}

		PutLine(); GetLine();
	    }

	    sprintf(lineBuf, "X-MessageID: %d\n", currentMsgID);
	    lineLen = strlen(lineBuf);
	    PutLine();

	    strcpy(lineBuf, "\n"); lineLen = 1;

	    while( !usrSpoolFileEoF ) {
	        if( IsNullLine() ) {
		    sawNullLine = TRUE;
		} else {
		   if( sawNullLine && IsPrefix("From ", lineBuf) ) break;
		    sawNullLine = FALSE;
		}
		PutLine(); GetLine();
	    }

	    AddToToC();
	    DoneWithMsg();
	    nMsgs += 1;

        }
	
	if( CheckDirtyMsgHeap(msgHeap) == 0 )
	    if( WriteMsgHeap(msgHeap) < 0 ) return(-4);
	    
	if( ftell(usrSpoolFile) != usrSpoolFileLen )
	    return(-5);

	if( deleteFromSpool )
	    ftruncate( fileno(usrSpoolFile), 0 );

	return( nMsgs );
}


/*
 * Bill Nesheim's "put mail back in the spool file" command.
 *
 * Return a single message to mail service.
 * This doesn't delete the X-MessageID field; thus, a message that
 * has been returned to mail service will get multiple X-MessageID
 * fields.  The last one is the effective one.
 */
int
UnRetrieveMsg(aMsgID, aMsgHeap)
    int aMsgID;
    int aMsgHeap;
{
    int result = 0;
    int cc;
    struct stat b;

	if( (usrSpoolFile == NULL)
		|| (fstat(fileno(usrSpoolFile),&b) < 0)
	        || (CheckMsgHeap(msgHeap = aMsgHeap) != 0) )
	    return(-1);

	usrSpoolFileMTime = b.st_mtime;
	usrSpoolFileLen = b.st_size;

	if( fseek(usrSpoolFile,0L,L_XTND) < 0 ) return(-2);

	if( OpenMsg(msgHeap, currentMsgID = aMsgID) < 0 ) {
	    result = -3;
	} else {
	    for(;;) {
	        cc = ReadMsg(msgHeap,lineBuf,LINE_LENGTH);
		if( cc < 0 )
		    { result = -4; break; }
		if( cc == 0 )
		    break;
		if( fwrite(lineBuf, 1, cc, usrSpoolFile) != cc )
		    { result = -5; break; }
	    }
	    (void)(CloseMsg(msgHeap));
	}

	fflush(usrSpoolFile);
	if( result < 0 ) {
	    /* truncate spool file to its original length */
	    int tempFildes = dup( fileno(usrSpoolFile) );
	    fclose(usrSpoolFile);  usrSpoolFile = NULL;
	    ftruncate( tempFildes, usrSpoolFileLen );
	    usrSpoolFile = fdopen( tempFildes, "w+" );
	}
	return( result );
}



/*
 * Lock / unlock mail service.
 *
 * At Cornell with 4.2BSD, /usr/spool/mail was write-protected but we still
 * used crufty old V7-style lock files.  Thus, locking and unlocking mail
 * spool files required an exec of a suid-root command.  This is no longer
 * true in 4.3BSD, and the code probably should be removed.  Sigh.
 */
int
LockMailService(useLockFile)
{
    int answer;

    if( (usrSpoolFileName == NULL)
            || ((usrSpoolFile = fopen(usrSpoolFileName, "a+")) == NULL) )
        return(-2);

    usingLockFile = useLockFile;

    if( usingLockFile ) {
        int childResult = RunChild( LOCK_MAIL, "-l", 0 );
	answer = ((childResult == EX_OK) ? 1 :
			((childResult == EX_TEMPFAIL) ? 0 : -childResult ));
    } else {
        if( flock(fileno(usrSpoolFile),(LOCK_EX|LOCK_NB)) == 0 )
	    answer = 1;
	else
	    answer = ((errno == EWOULDBLOCK) ? 0 : -1 );
    }

    if( answer <= 0 ) {
	fclose(usrSpoolFile);
	usrSpoolFile = NULL;
    }

    return( answer );
}



int
UnlockMailService()
{
    if( usrSpoolFile == NULL )
        return(-1);

    if( usingLockFile )
	(void)RunChild( LOCK_MAIL, "-u", 0 );
    else
        (void)flock( fileno(usrSpoolFile), LOCK_UN );

    fclose(usrSpoolFile);
    usrSpoolFile = NULL;
    return(0);
}


static int
RunChild( cmd, argv0, argv1 )
    char *cmd, *argv0, *argv1;
{
    int child, waitedChild;
    union wait status;

    if( (child = fork()) < 0 ) return(EX_OSERR);
    if( child == 0 ) /* this is child */ {
        execv( cmd, &cmd );
	/*NOTREACHED*/
	exit(EX_UNAVAILABLE);
    }
    do {
        waitedChild = wait(&status);
    } while( (waitedChild >= 0) && (waitedChild != child) );
    return( (waitedChild < 0) ? EX_OSERR : status.w_retcode );
}

