/*
 * PTYOps.c -- $Header: PTYOps.c,v 1.3 86/12/12 16:19:58 jqj Exp $
 *
 * Common to RTTY, ...
 *
 * Must be suid root for chown's to work.
 *
 * $Log:	PTYOps.c,v $
 * Revision 1.3  86/12/12  16:19:58  jqj
 * typo in line 180 -- add missing comma.
 * 
 * Revision 1.2  86/11/03  15:51:21  jqj
 * Set /dev/tty? to group "tty" and mode 0620 to correspond to new login.c
 * security standards.
 * 
 * Edited by AJD on Sat Dec  7 12:06:41 PST 1985
 *
 */

#include <sys/types.h>

#include <errno.h>
#include <sgtty.h>
#include <signal.h>
#include <stdio.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <utmp.h>
#include <grp.h>

#include "NetworkStream.h"
#include "PTYOps.h"
#include "StubOps.h"

extern int errno;
extern char * rindex();

#define UTMPNAME "/etc/utmp"

/* NOTE: (16*NPTYGROUPS) should be at least as large as the	*/
/* number of pty's in the system.  More costs a little running	*/
/* time but doesn't break anything.				*/

#define NPTYGROUPS 4
#define PTYGROUPNAMES "pqrstuvw"

#define LINESPERPTYGROUP 16
#define PTYLINENAMES "0123456789abcdef"


/*
 * utmp entry management
 *
 * Whenever a pty replaces the control tty, any existing utmp entry is deleted
 * and a new utmp entry is created.
 */


WriteUtmp(slot, line, name, host)
    int slot;		/* slot in utmp file, -1 ==> erase prev slot */
    char *line;		/* line name (NULL okay if slot == -1) */
    char *name;		/* user name (NULL okay if slot == -1) */
    char *host;		/* host name (NULL okay if slot == -1) */
{
    static int oldSlot = -1;
    static struct utmp utmpEntry;
    int f;

    if( slot >= 0 ) {
        char *temp = rindex(line, '/');
        strncpy( utmpEntry.ut_line, ((temp != NULL) ? temp+1 : line),
	        (sizeof utmpEntry.ut_line) );
	strncpy( utmpEntry.ut_name, ((name != NULL) ? name : ""),
	        (sizeof utmpEntry.ut_name) );
	strncpy( utmpEntry.ut_host, ((host != NULL) ? host : ""),
		(sizeof utmpEntry.ut_host) );
	oldSlot = slot;
    } else {
        if( oldSlot < 0 ) return;
        strncpy( utmpEntry.ut_name, "", (sizeof utmpEntry.ut_name) );
	strncpy( utmpEntry.ut_host, "", (sizeof utmpEntry.ut_host) );
	slot = oldSlot; oldSlot = -1;
    }
    
    time( &utmpEntry.ut_time );

    if( (f = open(UTMPNAME, O_WRONLY)) < 0 ) return;
    lseek(f, (long)(slot*(sizeof utmpEntry)), 0);
    write(f, (char *)(&utmpEntry), (sizeof utmpEntry));
    close(f);
}


/*
 * Get TTY end of PTY, set modes like ctrl TTY.
 *
 * If d >= 0, that's the descriptor for TTY; otherwise, new TTY replaces
 * STDIN, STDOUT, STDERR and becomes control TTY.
 *
 * NOTE: the pgroup of the new tty is getpid(); this is probably not
 * what the caller eventually wants.
 */

int
PTYSetup(ptyName, ttyName, d)
    char *ptyName;
    char *ttyName;
    int d;
{
    int t;
    struct sgttyb b;
    struct tchars tc;
    int lmode;
    struct ltchars ltc;
    int ldisc;
    int gotModes = FALSE;
    int mypgrp = getpgrp(0);
    int mypid = getpid();
    static int zero = 0;

    t = open("/dev/tty", O_RDWR, 0666);
    if (t >= 0) {
	gotModes = TRUE;
	ioctl( t, TIOCGETD, &ldisc);
	ioctl( t, TIOCGETP, &b);
	ioctl( t, TIOCGETC, &tc);
	ioctl( t, TIOCLGET, &lmode);
	ioctl( t, TIOCGLTC, &ltc);
	close(t);
    }
    if( d < 0 ) {
        ReplaceStdTTYs(ttyName, TRUE);
        ioctl(STDIN, TIOCSPGRP, &mypid ); /* Don't want it to kill session! */
	d = STDIN;
	WriteUtmp(-1, NULL, NULL, NULL); /* erase any prev entry */
	WriteUtmp(ttyslot(), ttyName, GetEnv("USER"), GetEnv("HOST"));
    } else {
        (void)MoveDescriptor( open(ttyName,O_RDWR,0666), d );
    }

    ioctl(d, TIOCNXCL, 0);
    ioctl(d, FIONBIO, &zero);
    ioctl(d, FIOASYNC, &zero);
    if( gotModes ) {
	ioctl( d, TIOCSETD, &ldisc);
	ioctl( d, TIOCLSET, &lmode);
	/* ?? b.sg_flags |= CRMOD|XTABS|ANYP; ?? */
	ioctl( d, TIOCSETP, &b);
	ioctl( d, TIOCSETC, &tc);
	ioctl( d, TIOCSLTC, &ltc);
    } else {
        ioctl( d, TIOCSETD, &zero);
	ioctl( d, TIOCLSET, &zero);
	/* ?? Should do explicit default TIOCSETP, TIOCSETC, TIOCSLTC ?? */
    }

    /* ?? If this can use d rather than PTY, hoist to preceding code ?? */
    ioctl(PTY, TIOCGETP, &b);
    b.sg_flags |= (ECHO|CRMOD|XTABS|ANYP);
    ioctl(PTY, TIOCSETP, &b);

    { int myuid = getuid();
      int mygid = getgid();
        fchown(PTY, myuid, mygid);
        fchmod(PTY, 0600);
        fchown(d, myuid, TTYGID(mygid));
	fchmod(d, 0620);
    }

    return( 0 );
}

/* get the group assumed to own /dev/tty* files */
static
int
TTYGID(default_gid)
	int default_gid;
{
	struct group *getgrnam(), *gr;
	int gid = default_gid;
	
	setgrent();
	gr = getgrnam("tty");		/* default group id for ttys */
	if (gr != (struct group *) 0)
		gid = gr->gr_gid;
	endgrent();
	return( gid );
}

/*
 * Release named pty/tty pair.
 *
 * If d < 0, this is control tty, so remove utmp entry.
 *
 * NOTE: PTYShutdown does NOT close the file descriptors (they may
 * already have been closed, and in any event PTYShutdown can't
 * know whether it needs to flush stdio buffers).  It's okay
 * to let exit() close the descriptors, or do it manually after
 * calling PTYShutdown.  The slave (tty) end should be closed first.
 */
int
PTYShutdown(ptyName, ttyName, d)
    char *ptyName;
    char *ttyName;
    int d;
{
    chmod(ttyName, 0666);
    chown(ttyName, 0, 0);
    chmod(ptyName, 0666);
    chown(ptyName, 0, 0);
    if( d < 0 ) {
	WriteUtmp(-1, NULL, NULL, NULL);
	vhangup();
    }
    return( 0 );
}


/*
 * Select from pty and network.
 * Hand data to telnet receiver finite state machine.
 */

/* TODO: I think the +- 2's are artifacts of old Telnet code -- */
/* CHECK THIS OUT! */


char ptyibuf[XNS_MAXDATA - 2]; /* ?? */
char *ptyip = ptyibuf;

char ptyobuf[XNS_MAXDATA + 2]; /* ?? */
char *pfrontp = ptyobuf;
char *pbackp = ptyobuf;

char netibuf[XNS_MAXDATA];
char *netip = netibuf;

char netobuf[XNS_MAXDATA];
char *nfrontp = netobuf;
char *nbackp = netobuf;

int	pcc, ncc;

static void
CopyFromPTY()
{
    register int cc;
    register char *from;
    register char *to;
    
    if( pcc <= 0 ) return;
    to = nfrontp;
    from = ptyip;
    cc = &netobuf[(sizeof netobuf) - 2] - to; /* ?? */
    if( cc > pcc ) cc = pcc;
    while( --cc >= 0 ) *to++ = *from++;
    pcc = ++cc;
    ptyip = from;
    nfrontp = to;
}

static void
CopyFromSocket()
{
    register int cc;
    register char *from;
    register char *to;
    
    if( ncc <= 0 ) return;
    to = pfrontp;
    from = netip;
    cc = &ptyobuf[(sizeof ptyobuf) - 2] - to; /* ?? */
    if( cc > ncc ) cc = ncc;
    while( --cc >= 0 ) *to++ = *from++;
    ncc = ++cc;
    netip = from;
    pfrontp = to;
}

static void
PTYFlush()
{
	int n;

	if ((n = pfrontp - pbackp) > 0)
		n = write(PTY, pbackp, n);
	if (n < 0)
		return;
	pbackp += n;
	if (pbackp == pfrontp)
		pbackp = pfrontp = ptyobuf;
}

static void
SocketFlush()
{
	int n;

	if ((n = nfrontp - nbackp) > 0)
		n = SPPWrite(STDSTREAM, nbackp, n, /*setEM=*/ 1);
	if (n < 0) {
		if (sppResult == -EWOULDBLOCK)
			return;
		/* should blow this guy away... */
		return;
	}
	nbackp += n;
	if (nbackp == nfrontp)
		nbackp = nfrontp = netobuf;
}

#define BREAK(n) break

int
PTYCopy()
{
    int doclose = 1;
    int netbit = (1 << STDSTREAM);
    int ptybit = (1 << PTY);
    static int on = 1;
    static int off = 0;
    int p = PTY;
    int f = STDSTREAM;

    ioctl(f, FIONBIO, &on);
    ioctl(p, FIONBIO, &on);

        for(;;) {
	    int ibits = 0, obits = 0;

            if (ncc < 0 && pcc < 0) BREAK(0);

	    /*
	     * Never look for input if there's still
	     * stuff in the corresponding output buffer
	     */
	    if( nfrontp - nbackp ) {
		obits |= netbit;
	    } else {
	        if( pcc < 0 ) BREAK(1);
		ibits |= ptybit;
	    }

	    if( pfrontp - pbackp ) {
		obits |= ptybit;
	    } else {
	        if( ncc < 0 ) BREAK(2);
		ibits |= netbit;
	    }

	    select(NUM_DESCRIPTORS, &ibits, &obits, 0, 0);
	    if (ibits == 0 && obits == 0) {
		sleep(5);
		continue;
	    }

	    /*
	     * Something to read from the network...
	     */
	    if (ibits & netbit) {
	        netip = netibuf;
		ncc = SPPRead(f, netibuf, (sizeof netibuf), /*readToEM=*/0 );
                if( sppResult == -ECONNRESET )
		    { doclose = 0; ncc = -ECONNRESET; }
		if (ncc > 0) CopyFromSocket();
	    }

	    /*
	     * Something to read from the pty...
	     */
	    if (ibits & ptybit) {
	        ptyip = ptyibuf;
		pcc = read(p, ptyibuf, (sizeof ptyibuf));
		if (pcc < 0 && errno == EWOULDBLOCK) {
		    pcc = 0;
		}
		if( pcc > 0 ) CopyFromPTY();
	    }

            /*
	     * Something to write on the net
	     */
	    if( obits & netbit ) {
		SocketFlush();
		if( pcc > 0 ) CopyFromPTY();
	    }


            /*
	     * Something to write on the pty
	     */
	    if( obits & ptybit ) {
		PTYFlush();
		if (ncc > 0) CopyFromSocket();
	    }
	}

	ioctl(f, FIONBIO, &off);
	ioctl(p, FIONBIO, &off);
        return(doclose);
}

#undef BREAK

/*
 * Find and open an unused pty
 */

#define NPTY (NPTYGROUPS * LINESPERPTYGROUP)

static
mkname(buf, kind, n)
    char *buf;
    char kind; /* p or t */
    unsigned n;
{
    register unsigned group = (n / LINESPERPTYGROUP);
    register unsigned line = (n % LINESPERPTYGROUP);
#   if NPTYGROUPS != 1
        group = group % NPTYGROUPS;
#   endif
    sprintf(buf,"/dev/%cty%c%c",
        kind, PTYGROUPNAMES [group], PTYLINENAMES [line] );
}

static unsigned steps[] = { 11, 13, 19, 23 };
#define NSTEPS ((sizeof steps)/sizeof(unsigned))

int
GetPTY(ptyname, ttyname)
    char *ptyname;
    char *ttyname;
{
    int p; /* pty descriptor */
    unsigned n; /* pty number */
    unsigned step;
    int i; /* loop index */
    
    n = getpid() & 0xff;  /* choose random starting point */
    step = steps[ n % NSTEPS ];  /* choose random prime stepsize */

    for( i = 0; i < NPTY; i++ ) {
        mkname(ptyname, 'p', n);
	if( /* (access(ptyname, (R_OK|W_OK)) == 0) && ... not needed */
	    /* ?? because PTY's are exclusive-open even for root ??  */
	        ((p = open(ptyname,2)) >= 0) ) goto found;
	n += step;
    }
    return( -1 );
found:
    mkname(ttyname, 't', n);
    return( p );
}
