/*
 * SPP functions for Unix Bridge stubs
 *
 * Define BSD42 for Maryland XNS implementation.
 * Define BSD43 for Berkeley XNS implementation.
 *
 * Edited by AJD on Sun Aug 11 14:37:10 EDT 1985
 * Edited by AJD on Fri Nov 29 13:32:42 PST 1985
 *
 */

#include <sys/types.h>

#include <errno.h>
#include <netinet/in.h>
#include <signal.h>
#include <stdio.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/uio.h>

#include "NetworkStream.h"


int sppResult = 0;
extern int errno;


#ifdef BSD42

/*
 * Initialize socket options.
 * These are the default options anyway, but they probably shouldn't be.
 * Note SIGPIPE is ignored; writes just return error.
 */
static int
InitializeStream(s)
    int s;
{
    static int on = 1;
    int savResult;
    
    signal( SIGPIPE, SIG_IGN );
    sppResult = SPPR_OK;
    return( s );
}

/*
 * Connect to given workstation.
 * Return descriptor for stream.
 */

int
SPPConnect(ap)
struct xn_addr *ap;
{
    static struct sockaddr_xn wsAddr;
    int s;

    wsAddr.sxn_family = AF_XNS;
    wsAddr.sxn_addr = *ap;

    if( ((s = socket(AF_XNS, SOCK_SEQPACKET,  IDPPROTO_SPP )) < 0)
	    || (connect(s, (struct sockaddr*)(&wsAddr), (sizeof wsAddr)) < 0) )
	return( sppResult = -errno );

    return( InitializeStream(s) );
}

/*
 * Accept a connection request.
 * Return descriptor for stream.
 */

int
SPPAccept(listener, sap)
    int listener; /* listener socket descriptor */
    struct sockaddr_xn *sap; /* optional ptr to peer address */
{
    static struct sockaddr_xn you;
    int s; /* descriptor for new stream */
    int addrLen = sizeof you;

    if( sap == NULL ) sap = &you;
    if( (s = accept( listener , sap, &addrLen)) < 0 )
        return( sppResult = -errno );
    return( InitializeStream(s) );
}

/*
 * Create a listener, bind to given port
 */
int
SPPCreateListener(port)
    short port;
{
    static struct sockaddr_xn myAddr, clearAddr;
    int s = -1;

    myAddr = clearAddr;
    myAddr.sxn_family = AF_XNS;
    myAddr.sxn_addr.xn_socket = htons(port);
    s = socket(AF_XNS,SOCK_SEQPACKET,IDPPROTO_SPP);
    if( s < 0 ) goto bad;
    if( bind( s, &myAddr, (sizeof myAddr) ) < 0 ) goto bad;
    if( listen( s, 5 ) < 0 ) goto bad;
    sppResult = SPPR_OK;
    return( s );

bad:
    sppResult = -errno;
    if( s >= 0 ) close(s);
    return( sppResult );
}

/*
 * Close SPP connection.
 * If initiate, wait for ENDREPLY but don't read through any
 * buffered packets looking for it.
 */
int
SPPClose(s, initiate)
    int s; /* descriptor for stream */
    int initiate; /* true iff should initiate close from this end */
{
    char sst;

    if( initiate ) {
	static struct timeval timeVal = {
	    /* tv_sec */ 30L,
	    /* tv_usec */ 0L
	};
	long ibits = (1 << s);

        sst = SPPSST_END;
	setsockopt(s, SOL_PROTO, SPPOPT_DATASTREAMTYPE, &sst, (sizeof sst));
	write(s, ((char *)0), 0);
	(void)select(32, &ibits, 0, 0, &timeVal); /* wait for ENDREPLY */
    }

    sst = SPPSST_ENDREPLY;
    setsockopt(s, SOL_PROTO, SPPOPT_DATASTREAMTYPE, &sst, (sizeof sst));
    write(s, ((char *)0), 0);
    sleep(2);
    
    shutdown(s, 2);
    return(0);
}

/*
 * Read from spp socket.
 *
 * If readToEM is TRUE, advance to next EndOfMessage, discarding
 *   data if necessary.
 *
 * If readToEM is FALSE, take care not to discard data.  This means
 *   for a long logical record, the actual number of bytes returned
 *   may be as much as (XNS_MAXDATA-1) bytes less than nWanted.
 *
 * In either case, return the number of bytes read, set sppResult
 *   to SPPR_SHORT if returned data doesn't reach to next EM in stream.
 * 
 * Works correctly on a nonblocking descriptor if readToEM is FALSE.
 */
static char sppReadOfloBuf[XNS_MAXDATA];

int
SPPRead(from, buf, nWanted, readToEM)
    int from; /* socket descriptor */
    char *buf; /* buffer */
    int nWanted; /* number of bytes to read */
    int readToEM;
{
    static struct sppidphdr readHdr;
    static struct iovec ioVec[3] = {
        { (caddr_t)(&readHdr), (sizeof readHdr) },
	{ (caddr_t)0, 0 },
	{ (caddr_t)(sppReadOfloBuf), 0 } };
    int nRead = 0;
    int nReq, nRcv;

    if( (nWanted < 0)
            || ((nWanted < XNS_MAXDATA) && (!readToEM)) )
        return( sppResult = -EINVAL );

    sppResult = SPPR_OK;

    for(;;) {
        nReq = nWanted - nRead;
	if( (nReq < XNS_MAXDATA) && (!readToEM) ) {
	    sppResult = SPPR_SHORT;
	    return( nRead );
	}
	if( nReq > XNS_MAXDATA ) nReq = XNS_MAXDATA;
	ioVec[1].iov_base = (caddr_t)(buf + nRead);
	ioVec[1].iov_len = nReq;
	ioVec[2].iov_len = XNS_MAXDATA - nReq;
	nRcv = readv(from, ioVec, 3);
	if( nRcv < 0 ) {
	    sppResult = -errno;
	    return( (errno == EWOULDBLOCK) ? nRead : sppResult );
	}
	if( (nRcv -= sizeof(readHdr)) < 0 ) {
	    sppResult = -EIO;
	    return( sppResult );
	}
	if( nRcv > nReq ) {
	    sppResult = SPPR_SHORT;
	    nRead += nReq;
	} else {
	    nRead += nRcv;
	}
	if( readHdr.si_s.sh_streamtype != SPPSST_NORMAL ) {
	    if( readHdr.si_s.sh_streamtype == SPPSST_END ) {
	        SPPClose( from, /*initiate=*/0 );
		sppResult = -ECONNRESET;
		return( nRead );
	    } else {
	        SPPClose( from, /*initiate=*/1 );
	        sppResult = -EIO;
	        return( sppResult );
	    }
        }
	if( readHdr.si_s.sh_eom )
	    return( nRead );
    }
    /*NOTREACHED*/
}

/*
 * Write to spp socket.
 * Works correctly on non-blocking descriptor, may perform short write.
 */
int
SPPWrite( to, buf, nBytes, setEM )
    int to;
    char *buf;
    int nBytes;
    int setEM;
{
    static struct spphdr writeHdr;
    static struct iovec ioVec[1] = {
	{ (caddr_t)0, 0 } };
    int nWritten = 0;
    int cnt;
    static char on = 1;
    static char off = 0;

    do {
        cnt = nBytes - nWritten;
	if( cnt > XNS_MAXDATA ) {
	    setsockopt(to, SOL_PROTO, SPPOPT_EOMBIT, &off, (sizeof off));
	    cnt = XNS_MAXDATA;
	} else {
	    setsockopt(to, SOL_PROTO, SPPOPT_EOMBIT,
	            (setEM ? &on : &off), (sizeof on));
	}
	ioVec[0].iov_base = (caddr_t)(buf + nWritten);
	ioVec[0].iov_len = cnt;
	cnt = writev(to, ioVec, 1);
	if( cnt < 0 ) {
	    sppResult = -errno;
	    return( (errno == EWOULDBLOCK) ? nWritten : sppResult );
	}
	nWritten += cnt;
    } while( nWritten < nBytes );
    sppResult = SPPR_OK;
    return( nWritten );
}

#endif BSD42

#ifdef BSD43

/*
 * Initialize socket options.
 * These are the default options anyway, but they probably shouldn't be.
 * Note SIGPIPE is ignored; writes just return error.
 */
static int
InitializeStream(s)
    int s;
{
    static int on = 1;
    int savResult;
    
    signal( SIGPIPE, SIG_IGN );
    if( (setsockopt(s, NSPROTO_SPP,
            	SO_HEADERS_ON_INPUT, &on, (sizeof on)) < 0)
	    || (setsockopt(s, NSPROTO_SPP,
	    	SO_HEADERS_ON_OUTPUT, &on, (sizeof on)) < 0) ) {
        savResult = -errno;
	(void)SPPClose( s );
	return( sppResult = savResult );
    }
    sppResult = SPPR_OK;
    return( s );
}

/*
 * Connect to given workstation.
 * Return descriptor for stream.
 */

int
SPPConnect(ap)
struct ns_addr *ap;
{
    struct sockaddr_ns wsAddr;
    int s;

    wsAddr.sns_family = AF_NS;
/* THE GOULD C COMPILER BREAKS ON THIS! */
    wsAddr.sns_addr = *ap;

    if( ((s = socket(AF_NS, SOCK_SEQPACKET,  0 )) < 0)
            /* ?? why 0 not NSPROTO_SPP ?? - ajd */
	    || (connect(s, (struct sockaddr*)(&wsAddr), (sizeof wsAddr)) < 0) )
	return( sppResult = -errno );

    return( InitializeStream(s) );
}

/*
 * Accept a connection request.
 * Return descriptor for stream.
 */

int
SPPAccept(listener, sap)
    int listener; /* listener socket descriptor */
    struct sockaddr_ns *sap; /* optional ptr to peer address */
{
    static struct sockaddr_ns you;
    int s; /* descriptor for new stream */
    int addrLen = sizeof you;

    if( sap == NULL ) sap = &you;
    if( (s = accept( listener , sap, &addrLen)) < 0 )
        return( sppResult = -errno );
    return( InitializeStream(s) );
}

/*
 * Create a listener, bind to given port
 */
int
SPPCreateListener(port)
    short port;
{
    static struct sockaddr_ns myAddr, clearAddr;
    int s = -1;

    myAddr = clearAddr;
    myAddr.sns_family = AF_NS;
    myAddr.sns_port = htons(port);
    s = socket(AF_NS,SOCK_SEQPACKET,0);
    if( s < 0 ) goto bad;
    if( bind( s, &myAddr, (sizeof myAddr) ) < 0 ) goto bad;
    if( listen( s, 5 ) < 0 ) goto bad;
    sppResult = SPPR_OK;
    return( s );

bad:
    sppResult = -errno;
    if( s >= 0 ) close(s);
    return( sppResult );
}

/*
 * Close SPP connection.
 * If initiate, wait for ENDREPLY but don't read through any
 * buffered packets looking for it.
 */
int
SPPClose(s, initiate)
    int s; /* descriptor for stream */
    int initiate; /* true iff should initiate close from this end */
{
    static struct sphdr closeHdr;


    if( initiate ) {
	static struct timeval timeVal = {
	    /* tv_sec */ 30L,
	    /* tv_usec */ 0L
	};
	long ibits = (1 << s);

	closeHdr.sp_cc = 0;
	closeHdr.sp_dt = SPPSST_END;
	(void)write( s, &closeHdr, (sizeof closeHdr) );
	(void)select(32, &ibits, 0, 0, &timeVal); /* wait for ENDREPLY */
    }

    closeHdr.sp_cc = 0;
    closeHdr.sp_dt = SPPSST_ENDREPLY;
    (void)write( s, &closeHdr, (sizeof closeHdr) );
    sleep(2);
    
    shutdown(s, 2);
    return(0);
}


/*
 * Read from spp socket.
 *
 * If readToEM is TRUE, advance to next EndOfMessage, discarding
 *   data if necessary.
 *
 * If readToEM is FALSE, take care not to discard data.  This means
 *   for a long logical record, the actual number of bytes returned
 *   may be as much as (XNS_MAXDATA-1) bytes less than nWanted.
 *
 * In either case, return the number of bytes read, set sppResult
 *   to SPPR_SHORT if returned data doesn't reach to next EM in stream.
 * 
 * Works correctly on a nonblocking descriptor if readToEM is FALSE.
 */
static char sppReadOfloBuf[XNS_MAXDATA];

int
SPPRead(from, buf, nWanted, readToEM)
    int from; /* socket descriptor */
    char *buf; /* buffer */
    int nWanted; /* number of bytes to read */
    int readToEM;
{
    static struct sphdr readHdr;
    static struct iovec ioVec[3] = {
        { (caddr_t)(&readHdr), (sizeof readHdr) },
	{ (caddr_t)0, 0 },
	{ (caddr_t)(sppReadOfloBuf), 0 } };
    int nRead = 0;
    int nReq, nRcv;

    if( (nWanted < 0)
            || ((nWanted < XNS_MAXDATA) && (!readToEM)) )
        return( sppResult = -EINVAL );

    sppResult = SPPR_OK;

    for(;;) {
        nReq = nWanted - nRead;
	if( (nReq < XNS_MAXDATA) && (!readToEM) ) {
	    sppResult = SPPR_SHORT;
	    return( nRead );
	}
	if( nReq > XNS_MAXDATA ) nReq = XNS_MAXDATA;
	ioVec[1].iov_base = (caddr_t)(buf + nRead);
	ioVec[1].iov_len = nReq;
	ioVec[2].iov_len = XNS_MAXDATA - nReq;
	nRcv = readv(from, ioVec, 3);
	if( nRcv < 0 ) {
	    sppResult = -errno;
	    return( (errno == EWOULDBLOCK) ? nRead : sppResult );
	}
	if( (nRcv -= sizeof(struct sphdr)) < 0 ) {
	    sppResult = -EIO;
	    return( sppResult );
	}
	if( nRcv > nReq ) {
	    sppResult = SPPR_SHORT;
	    nRead += nReq;
	} else {
	    nRead += nRcv;
	}
	if( readHdr.sp_dt != SPPSST_NORMAL ) {
	    if( readHdr.sp_dt == SPPSST_END ) {
	        SPPClose( from, /*initiate=*/0 );
		sppResult = -ECONNRESET;
		return( nRead );
	    } else {
	        SPPClose( from, /*initiate=*/1 );
	        sppResult = -EIO;
	        return( sppResult );
	    }
        }
	if( readHdr.sp_cc & SP_EM )
	    return( nRead );
    }
    /*NOTREACHED*/
}

/*
 * Write to spp socket.
 * Works correctly on non-blocking descriptor, may perform short write.
 */
int
SPPWrite( to, buf, nBytes, setEM )
    int to;
    char *buf;
    int nBytes;
    int setEM;
{
    static struct sphdr writeHdr;
    static struct iovec ioVec[2] = {
        { (caddr_t)(&writeHdr), (sizeof writeHdr) },
	{ (caddr_t)0, 0 } };
    int nWritten = 0;
    int cnt;

    do {
        cnt = nBytes - nWritten;
	if( cnt > XNS_MAXDATA ) {
   	    writeHdr.sp_cc = 0;
	    cnt = XNS_MAXDATA;
	} else {
	    writeHdr.sp_cc = ( setEM ? SP_EM : 0 );
	}
	ioVec[1].iov_base = (caddr_t)(buf + nWritten);
	ioVec[1].iov_len = cnt;
	cnt = writev(to, ioVec, 2);
	if( cnt < 0 ) {
	    sppResult = -errno;
	    return( (errno == EWOULDBLOCK) ? nWritten : sppResult );
	}
	if( (cnt -= (sizeof writeHdr)) < 0 ) {
	    sppResult = -EIO;
	    return( sppResult );
	}
	nWritten += cnt;
    } while( nWritten < nBytes );
    sppResult = SPPR_OK;
    return( nWritten );
}

#endif BSD43



static char sppIOBuf[4096];

/*
 * receive a 1-character command and argument from stream.
 * return 0 on end of file or error.
 */
int
RcvCmd(from, argBuf, bufLen)
    int from; /* socket descriptor */
    char *argBuf; /* buffer */
   int bufLen; /* number of bytes to read */
{
    int nRead;
    if( bufLen >= (sizeof sppIOBuf) )
        bufLen = (sizeof sppIOBuf) - 1;
    while( (nRead = SPPRead(from,sppIOBuf,bufLen,/*readToEM=*/1)) == 0 ) ;
    if( sppResult < 0 ) return( 0 );
    sppIOBuf[nRead] = 0;
    if( (sppResult != SPPR_SHORT)
            && (sppIOBuf[nRead-1] == '\r') )
        sppIOBuf[nRead-1] = 0;

    strcpy( argBuf, sppIOBuf+1 );
    return( sppIOBuf[0] & 0xff );
}
    
/*
 * Send a one-character command and argument on stream.
 */
int
SendCmd(to, cmd, arg)
    int to;
    char cmd;
    char *arg;
{
    int len;
    if( (len = strlen(arg)+1) >= (sizeof sppIOBuf) )
        return( sppResult = -EINVAL );
    sppIOBuf[0] = cmd;
    strcpy( sppIOBuf+1, arg );

    if( sppIOBuf[len-1] != '\r' ) {
        if( len >= ((sizeof sppIOBuf)-1) )
	    return( sppResult = -EINVAL );
        sppIOBuf[len++] = '\r';
    }

    return( SPPWrite(to, sppIOBuf, len, /*setEM=*/ 1) );
}



/*
 * Send a one-character command and argument on stream,
 * using sprintf to format the argument.
 */
int
SendCmdf(to, cmd, fmt, arg1, arg2, arg3)
    int to; /* spp socket descriptor */
    char cmd;
    char *fmt;
    int arg1, arg2, arg3;
{
    register int len;

    sppIOBuf[0] = cmd;
    if( fmt == NULL ) {
        sppIOBuf[1] = '\r';
	len = 2;
    } else {
        sprintf((sppIOBuf+1), fmt, arg1, arg2, arg3);
        len = strlen(sppIOBuf);

        if( sppIOBuf[len-1] != '\r' ) sppIOBuf[len++] = '\r';

        if( len > (sizeof sppIOBuf) ) /* too late to recover */ abort(1);
    }
    return( SPPWrite(to, sppIOBuf, len, /*setEM=*/ 1) );
}


/*
 * Copy from a file to the network.
 * Send maxBytes or until eof, whichever comes first.
 */
int
SendFile(to, from, maxBytes, xlate) 
    int to; /* spp socket descriptor */
    int from; /* local file descriptor */
    int maxBytes; /* max number of bytes to send */
    int xlate; /* cr/lf (and unimplemented tab) translation */
{
    int nSent;
    int nReq;
    int nr;
    int nw;

    for (nSent = 0; nSent < maxBytes; nSent += nr) {
	if( (nReq = maxBytes-nSent) > (sizeof sppIOBuf) )
	    nReq = (sizeof sppIOBuf);
        if( (nr = read(from, sppIOBuf, nReq)) < 0 )
	    return( sppResult = -errno );
	if( nr == 0 ) /* eof */ break;
	if( xlate & XLATE_TEXT ) {
	  register char *p = sppIOBuf;
	  register int cnt = nr;
	    while( (--cnt) >= 0 )
	        if( *p++ == '\n' ) p[-1] = '\r';
	}
	nw = SPPWrite(to, sppIOBuf, nr, /*setEM=*/ 0);
	if( nw < 0 ) return( nw );
	if( nw != nr ) return( sppResult = -EIO );
    }
    nw = SPPWrite(to, sppIOBuf, 0, /*setEM=*/ 1);
    return( (nw < 0) ? nw : nSent );
}


/*
 * Copy into a file from the network.
 */
int
RcvFile(from, to, xlate, maxLineLength)
    int from; /* spp socket descriptor */
    int to; /* local file descriptor -- may be pipe */
    int xlate; /* cr/lf and tab translation */
    int maxLineLength;
{
    return( (xlate & XLATE_TEXT)
	? RcvTextFile(from, to, xlate, maxLineLength)
        : RcvRawFile(from, to) );
}

/*
 * Copy raw data file from the network.
 */
int
RcvRawFile(from, to)
    int from; /* spp socket descriptor */
    int to; /* local file descriptor -- may be pipe */
{
    int nRead = 0;
    int eof = 0;
    int icc;

    while (!eof) {
        icc = SPPRead(from, sppIOBuf, (sizeof sppIOBuf), /*readToEM=*/ 0);
	if( sppResult < 0 ) return( sppResult );
	eof = (sppResult != SPPR_SHORT);
	nRead += icc;
	if( icc > 0 )
	    if( write(to, sppIOBuf, icc) < 0 ) return( sppResult = -errno );
    }
    return( nRead );
}

/*
 * Copy text file from the network.
 */
int
RcvTextFile(from, to, xlate, maxLineLength)
    int from; /* spp socket descriptor */
    int to; /* local file descriptor -- may be pipe */
    int xlate; /* cr/lf and tab translation */
    int maxLineLength;
{
    char obuf[4096];
    int nRead = 0;
    int eof = 0;
    int icc = 0;
    int occ = 0;
    register char *rip;
    register char *rop = obuf;
    char *opLim = obuf + (sizeof obuf);
    register char c;
    int col = 0;
    int hardColLim;
    int softColLim;
#   define MIN_RAGGEDNESS 8
    int raggedness;

    hardColLim =
        ( (maxLineLength > MIN_RAGGEDNESS) ? maxLineLength : 0x7fffffff );
    raggedness = hardColLim / 6;
    if( raggedness < MIN_RAGGEDNESS)
        raggedness = MIN_RAGGEDNESS;
    softColLim = hardColLim - raggedness;

    for(;;) {

        /* c := next character */

        if( --icc < 0 ) {
	    if( eof ) break;
	    rip = sppIOBuf + TABSIZE;
	    icc = SPPRead( from, rip, (sizeof sppIOBuf)-TABSIZE,
	            /*readToEM=*/ 0 );
	    if( sppResult < 0 ) return( sppResult );
	    eof = (sppResult != SPPR_SHORT);
	    nRead += icc;
	    continue;
	}
	c = *rip++;

	/* discard c if flushing blanks */

	if( col < 0 ) /* flushing blanks */ {
	    if( (c == ' ') || (c == '\t') ) continue;
	    col = 0;
	}

	/* put c in output buffer */

	if( col < softColLim ) {
	    switch( c ) {
	        case '\r':
		    c = '\n';
		case '\n':
		    *rop++ = c;
	       	    col = 0;
		    break;
		case '\t': {
		    int delta = TABSIZE - (col % TABSIZE);
		    if( xlate & XLATE_HTSP ) {
		        do {
			    *--rip = ' ';
			    icc++;
			} while( --delta > 0 );
		    } else {
		        *rop++ = '\t';
			col += delta;
		    }
		    break; }
		default:
		    *rop++ = c;
		    col++;
		    break;
	    }
	} else switch(c) {
	    case ' ':
	    case '\t':
	        *rop++ = '\n';
		col = -1; /* indicate flushing blanks */
		break;
	    case '\r':
	    case '\n':
	        *rop++ = '\n';
	        col = 0;
		break;
	    default:
	        *rop++ = c;
		if( (++col) >= hardColLim ) { *rop++ = '\n'; col = 0; }
		break;
	}
	
	/* Check output buffer length; write if necessary */
	if( rop >= opLim ) {
	    if( (occ = write(to,obuf,rop-obuf,0)) < 0 )
		return( sppResult = -errno );
	    rop = obuf;
	}

    }

    /* flush output buffer */

    if( rop > obuf ) {
        if( (occ = write(to,obuf,rop-obuf,0)) < 0 )
	    return( sppResult = -errno );
    }

    /* successful return */

    sppResult = SPPR_OK;
    return( nRead );
}
