/*
 * MsgHeap.c
 *
 * 
 * Edited on Sun Jul 21 23:38:01 EDT 1985
 *
 */

#include <sys/types.h>

#include <errno.h>
#include <stdio.h>
#include <sys/file.h>
#include <sys/stat.h>

#include "MsgHeap.h"


#ifndef TRUE
# define TRUE 1
# define FALSE 0
#endif

#define NMSGHEAP 2

#define VALID_HANDLE(h) (((h)>=0) && ((h)<NMSGHEAP))

#define GET_PTR_FROM_HANDLE(p,h,access) \
    if( !VALID_HANDLE(h) ) return(-1); \
    p = msgHeapInfo+(h); if(((p)->flags&(access))==0) return(-1)

#define DEFAULT_HEAP "mail"
#define INDEX_EXT ".index"
#define DATA_EXT ".data"

#define LongWrite write
#define LongRead read


/*
 * Structure of index file on disk
 *
 * File is <IndexHeader> <IndexItem> <IndexItem> <IndexItem> ...
 * Number of items is implicit in length of file.
 */

typedef struct {
    int lastMsgID;
} IndexHeader;


typedef struct {
    long msgID;
    long msgPos;
    long msgLength;
    long refCnt;
} IndexItem;

static IndexItem clearIndexItem; /* = all zeroes */

/*
 * In-core index structure
 */

typedef struct {
    unsigned short flags;	/* see below */
    unsigned short state;	/* see below */
    int indexFildes;		/* Index file file descriptor */
    FILE *heapStream;		/* Stream for actual data */
    IndexHeader indexHeader;	/* In-core copy */
    IndexItem *indexItem;	/* --> array of IndexItems */
    int nItems;			/* number of items in index */
    int maxItems;		/* amount of space allocated */
    int currentItemNo;		/* read/write ptr: which item */
    long currentPos;		/* read/write ptr: position in heapStream */
} MsgHeapInfo;

static MsgHeapInfo msgHeapInfo[NMSGHEAP];
static MsgHeapInfo clearMsgHeapInfo; /* = all zeroes */

/* flags bits */
#define HF_RD   	1	/* Open for ReaDing  */
#define HF_WR		2	/* Open for WRiting */
#define HF_DIRTY 	4	/* is DIRTY */

#define HF_OPEN		(HF_RD|HF_WR)	/* OPEN for anything */

/* state codes */
#define HS_IDLE		0
#define HS_READING	1
#define HS_WRITING	2



static int
FGetLength(d)
    int d;
{
    struct stat b;
    fstat(d,&b);
    return( b.st_size );
}



static void
CleanUpHeapInfo(p)
    register MsgHeapInfo *p;
{
    if( p->indexFildes >= 0 ) close( p->indexFildes );
    if( p->heapStream != NULL ) fclose( p->heapStream );
    if( p->indexItem != NULL ) free( p->indexItem );

    *p = clearMsgHeapInfo;
    p->indexFildes = -1;
}


/*
 * Search index for
 *   least message id >= msgID
 *   with reference count >= refCnt (so refCnt = 0 ignores
 *       reference counts, and refCnt = 1 searches for undeleted
 *       entries).
 * Return position in index, or -1 on failure.
 */
static int
Search( p, msgID, refCnt )
    register MsgHeapInfo *p;
    int msgID;
    int refCnt;
{
    register IndexItem *ip;
    int left, right;
    register int mid;

    ip = p->indexItem;

    left = 0;
    right = p->nItems - 1;
    if( ip[right].msgID < msgID ) return(-1);

    while( left < right ) {
        mid = (left + right) / 2;
	if( ip[mid].msgID >= msgID )
	    right = mid;
	else
	    left = mid + 1;
    }

    while( (right < p->nItems) && (ip[right].refCnt < refCnt) )
        right++;

    return( (right < p->nItems) ? right : -1 );
}


/*
 * Open the named heap/index pair
 *   and read the index into memory.
 * Space for at most nNewItems items will be allocated.
 * If nNewItems <= 0 the heap is read-only; otherwise
 *   an exclusive advisory lock is applied.
 * Returns small integer handle on success, -1 on failure.
 */

int
AcquireMsgHeap(name, nNewItems)
    char *name;
    unsigned nNewItems;
{
    char indexName[40];
    char heapName[40];
    int handle;
    register MsgHeapInfo *p;
    int writable;
    int heapFildes;
    int cc;
    int length;

    if( name == NULL ) name = DEFAULT_HEAP;
    strncpy( indexName, name, (sizeof indexName)-1 );
    strncat( indexName, INDEX_EXT, (sizeof indexName)-strlen(indexName)-1 );
    strncpy( heapName, name, (sizeof heapName)-1 );
    strncat( heapName, DATA_EXT, (sizeof heapName)-strlen(heapName)-1 );

    writable = (nNewItems > 0);

    for( handle = 0
        ; (handle < NMSGHEAP) && (msgHeapInfo[handle].flags & HF_OPEN)
	; handle++ ) continue ;
    if( handle >= NMSGHEAP ) return(-1);
    p = &(msgHeapInfo[handle]);

    p->flags |= HF_RD;
    if( writable ) {
        p->flags |= HF_WR;
	if( (p->indexFildes = open(indexName,(O_CREAT|O_RDWR),0666)) < 0 )
	    goto bad;
	if( flock(p->indexFildes,(LOCK_EX|LOCK_NB)) < 0 )
	    goto bad;
	if( (p->heapStream = fopen(heapName,"a+")) == NULL )
	    goto bad;
    } else {
        if( (p->indexFildes = open(indexName,O_RDONLY,0666)) < 0 )
	    goto bad;
	if( (p->heapStream = fopen(heapName,"r")) == NULL )
	    goto bad;
    }

    cc = read( p->indexFildes,
        &(p->indexHeader), (sizeof p->indexHeader) );
    if( (cc == 0) && writable )
        cc = write( p->indexFildes,
	    &(p->indexHeader), (sizeof p->indexHeader) );
    if( cc != (sizeof p->indexHeader) ) goto bad;

    length = FGetLength(p->indexFildes) - sizeof(IndexHeader);
    if( length < 0 ) goto bad;
    if( (length % sizeof(IndexItem)) != 0 ) {
        if( writable ) goto bad;
	else length -= (length % sizeof(IndexItem));
    }
    p->nItems = (length/sizeof(IndexItem));
    p->maxItems = p->nItems + nNewItems;
    p->indexItem = (IndexItem *)malloc( (p->maxItems)*sizeof(IndexItem) );
    if( p->indexItem == NULL ) goto bad;

    cc = LongRead( p->indexFildes, p->indexItem, length );
    if( cc != length ) goto bad;

    return( handle );

bad:

    CleanUpHeapInfo(p);
    return(-1);
}


int
GetMsgHeapSize(handle)
    int handle;
{
    register MsgHeapInfo *p;
    
    GET_PTR_FROM_HANDLE(p,handle,HF_OPEN);
    return( p->nItems );
}

/*
 * Write the message heap to disk.
 * Use fflush and fsync to be sure it's really out there.
 */  
int
WriteMsgHeap(handle)
    int handle;
{
    register MsgHeapInfo *p;
    int answer;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR);

    if( p->state != HS_IDLE ) return(-1);

    if( (fflush(p->heapStream) < 0)
            || (fsync(fileno(p->heapStream)) < 0) )
	return(-2);

    lseek( p->indexFildes, 0L, L_SET );
    answer = LongWrite( p->indexFildes,
    	    &(p->indexHeader), (sizeof p->indexHeader) );
    if( answer != (sizeof p->indexHeader) ) return(-3);
    answer = LongWrite( p->indexFildes,
            p->indexItem, p->nItems * sizeof(IndexItem) );
    if( answer != p->nItems * sizeof(IndexItem) ) return(-3);
    if( fsync(p->indexFildes) < 0 ) return(-3);
    
    answer = ftruncate( p->indexFildes,
            (sizeof p->indexHeader) + p->nItems * sizeof(IndexItem) );
    if( answer < 0 ) return(-4);

    p->flags &= (~HF_DIRTY);
    return(0);
}


int
ReleaseMsgHeap(handle)
    int handle;
{
    register MsgHeapInfo *p;

    GET_PTR_FROM_HANDLE(p,handle,HF_OPEN);
    CleanUpHeapInfo(p);
    return(0);
}


int
CheckMsgHeap(handle)
    int handle;
{
    register MsgHeapInfo *p;
    GET_PTR_FROM_HANDLE(p,handle,HF_OPEN);
    return(0);
}


int
CheckWritableMsgHeap(handle)
    int handle;
{
    register MsgHeapInfo *p;
    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    return(0);
}


int
CheckDirtyMsgHeap(handle)
    int handle;
{
    register MsgHeapInfo *p;
    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    return( (p->flags & HF_DIRTY) ? 0 : -1 );
}


int
StartMsg(handle)
    int handle;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    if( (p->state != HS_IDLE)
            || (p->nItems >= p->maxItems) ) return(-1);

    p->state = HS_WRITING;
    p->currentItemNo = p->nItems;
    p->flags |= HF_DIRTY;
    fseek( p->heapStream, 0L, L_XTND );
    p->currentPos = ftell( p->heapStream );

    ip = &(p->indexItem[p->currentItemNo]);
    *ip = clearIndexItem;
    ip->msgID = ++(p->indexHeader.lastMsgID);
    ip->msgPos = p->currentPos;
    ip->msgLength = 0;
    ip->refCnt = 0;

    return( ip->msgID );
}



int
WriteMsg(handle, buf, nBytes)
    int handle;
    char *buf;
    int nBytes;
{
    register MsgHeapInfo *p;
    int n;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    if( p->state != HS_WRITING ) return(-1);
    if( nBytes == 0 ) return(0);
    if( (n = fwrite( buf, 1, nBytes, p->heapStream )) <= 0 ) return(-1);
    p->currentPos += n;
    return(n);
}


int
AdjustMsgID(handle, newMsgID)
    int handle;
    int newMsgID;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    if( p->state != HS_WRITING ) return(-1);
    if( p->indexHeader.lastMsgID > newMsgID ) return(-2);
    ip = &(p->indexItem[p->currentItemNo]);
    ip->msgID = p->indexHeader.lastMsgID = newMsgID;
    return( newMsgID );
}


int
EndMsg(handle)
    int handle;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    if( p->state != HS_WRITING ) return(-1);
    if( p->currentItemNo != p->nItems ) abort(); /* DEBUG */
    fflush( p->heapStream );
    ip = &(p->indexItem[p->currentItemNo]);
    ip->msgLength = p->currentPos - ip->msgPos;
    ip->refCnt = 1;
    p->nItems = p->nItems + 1;
    p->state = HS_IDLE;
    return(0);
}



int
OpenMsg(handle, msgID)
    int handle;
    int msgID;
{
    register MsgHeapInfo *p;
    int itemNo;
    register IndexItem *ip;

    GET_PTR_FROM_HANDLE(p,handle,HF_RD);
    if( p->state != HS_IDLE ) return(-1);

    if( (itemNo = Search( p, msgID, 1 )) < 0 ) return(-1);
    ip = &(p->indexItem[itemNo]);

    p->state = HS_READING;
    p->currentItemNo = itemNo;

    p->currentPos = ip->msgPos;
    fseek( p->heapStream, p->currentPos, L_SET );

    return( ip->msgID );
}


int
GetMsgLength(handle)
    int handle;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;

    GET_PTR_FROM_HANDLE(p,handle,HF_RD);
    if( p->state != HS_READING ) return(-1);
    ip = &(p->indexItem[p->currentItemNo]);
    return( ip->msgLength );
}


int
ReadMsg(handle, buf, nBytes)
    int handle;
    char *buf;
    int nBytes;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;
    int n;

    GET_PTR_FROM_HANDLE(p,handle,HF_RD);
    if( p->state != HS_READING ) return(-1);
    ip = &(p->indexItem[p->currentItemNo]);
    n = (ip->msgPos + ip->msgLength) - p->currentPos;
    if( n <= 0 ) return(0);
    if( n > nBytes ) n = nBytes;
    if( (n = fread( buf, 1, n, p->heapStream )) <= 0 ) return(-1);
    p->currentPos += n;
    return(n);
}


int
CloseMsg(handle)
    int handle;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;

    GET_PTR_FROM_HANDLE(p,handle,HF_RD);
    if( p->state != HS_READING ) return(-1);
    p->state = HS_IDLE;
    return(0);
}



int
AdjustMsgRefCount(handle, msgID, delta)
    int handle;
    int msgID;
    int delta;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;
    register int itemNo;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR); /* ?? */
    if( (itemNo = Search( p, msgID, 0 )) < 0 ) return(-1);
    ip = &(p->indexItem[itemNo]);
    if( ip-> msgID != msgID ) return(-1);

    if( (ip->refCnt += delta) < 0 ) ip->refCnt = 0;
    p->flags |= HF_DIRTY; /* ?? */
    return( ip->refCnt );
}


int
ClearMsgRefCounts(handle)
    int handle;
{
    register MsgHeapInfo *p;
    register IndexItem *ip;
    register int n;

    GET_PTR_FROM_HANDLE(p,handle,HF_WR); /* ?? */
    for(  (n = 0),(ip = p->indexItem);  n < p->nItems;  n++,ip++  )
        ip->refCnt = 0;
    p->flags |= HF_DIRTY; /* ?? */

    return(0);
}


static int
IsPrefix(pattern, string)
    register char *pattern;
    register char *string;
{
    while( *pattern )
        if( *pattern++ != *string++ ) return(FALSE);
    return(TRUE);
}

static int
IsNullLine(s)
    register char *s;
{
    return( ((s[0] == '\n') && (s[1] == 0)) || (s[0] == 0) );
}


static int
CompareIndexItems(p,q)
    IndexItem *p;
    IndexItem *q;
{
    return( p->msgID - q->msgID );
}


int
RebuildIndex(handle)
    int handle;
{
    char lineBuf[1024];
    int eof;
    unsigned long startOfMessage;
    unsigned long startOfNextMessage;
    long newMsgID;
    int sawNullLine;
    int mustSort = 0;
    register MsgHeapInfo *p;
    register IndexItem *ip;
    
    GET_PTR_FROM_HANDLE(p,handle,HF_WR);
    if( p->state != HS_IDLE ) return(-2);

    p->nItems = 0;
    p->flags |= HF_DIRTY;
    ip = p->indexItem;

    fseek( p->heapStream, 0L, L_SET );

    do {
        startOfNextMessage = ftell(p->heapStream);
        eof = (fgets(lineBuf, (sizeof lineBuf), p->heapStream) == NULL);
    } while( (!eof) && (!IsPrefix("From ", lineBuf)) );

    while( !eof ) {
        newMsgID = 0;
	startOfMessage = startOfNextMessage;
	while( (!eof) && (!IsNullLine(lineBuf)) ) {
	    (void)sscanf(lineBuf, "X-MessageID:%d", &newMsgID);
	    eof = (fgets(lineBuf, (sizeof lineBuf), p->heapStream) == NULL);
	}
	while( !eof ) {
	    if( IsNullLine(lineBuf) ) {
	        sawNullLine = TRUE;
	    } else {
	        if( sawNullLine && IsPrefix("From ", lineBuf) ) break;
		sawNullLine = FALSE;
	    }
	    startOfNextMessage = ftell(p->heapStream);
	    eof = (fgets(lineBuf, (sizeof lineBuf), p->heapStream) == NULL);
	}
	if( p->nItems >= p->maxItems ) return(-3);
	if( newMsgID > p->indexHeader.lastMsgID )
	    p->indexHeader.lastMsgID = newMsgID;
	else
	    mustSort = 1;
	ip->msgID = newMsgID;
	ip->msgPos = startOfMessage;
	ip->msgLength = startOfNextMessage - startOfMessage;
	ip->refCnt = 1;
	ip++; p->nItems++;
    }

    if( mustSort )
        qsort( /*base=*/p->indexItem, /*nElem=*/ p->nItems,
                /*width=*/ sizeof(IndexItem), /*compar=*/ CompareIndexItems );
    return( 0 );
}