/*
 * Mail garbage collection and index rebuilding
 *
 * Last edited on Thu Dec 12 18:06:49 PST 1985
 */

#include <stdio.h>
#include <sysexits.h>

#include "CommandOps.h" 
#include "MsgHeap.h"

#define ENV_MAILDIR "RRMAIL_MAILDIR"

#define MAIL_HEAP_NAME "mail"
#define MAIL_HEAP_DATA_NAME "mail.data"
#define MAIL_HEAP_INDEX_NAME "mail.index"

#define TEMP_HEAP_NAME "tempmail"
#define TEMP_HEAP_DATA_NAME "tempmail.data"
#define TEMP_HEAP_INDEX_NAME "tempmail.index"

#define PLENTY_BIG 3000 /* extra heap space allocated if rebuilding index */

static int
UpdateRefCounts( f, h )
    FILE *f;
    int h;
{
    int msgID;
    int answer;
    char lineBuf[1024];
    
    while( (answer = fscanf(f,"%*c%d%[^\n]%*c",&msgID,lineBuf)) == 2 ) {
	answer = AdjustMsgRefCount(h, msgID, 1);
	if( answer < 0 )
	    LogMsg("Message %d (%s) nonexistent\n", msgID, lineBuf);
    }
    if( answer == EOF ) return( 0 );
    if( answer > 0 ) return( -answer );
    return(-99);
}

static int
CopyMsgBody(hFrom, hTo, msgID)
    int hFrom;
    int hTo;
    int msgID;
{
    char buf[2048];
    int icc, occ;
    icc = ReadMsg(hFrom,buf,(sizeof buf));
    if( strncmp(buf, "From ", 5) != 0 ) return(-1);
    while( icc > 0 ) {
        if( (occ = WriteMsg(hTo, buf, icc)) != icc ) return(-1);
        icc = ReadMsg(hFrom,buf,(sizeof buf));
    }
    return( icc );
}


LogMsg(fmt,x,y,z)
    char *fmt;
    int x,y,z;
{
    extern FILE *logFile;
    fprintf(logFile, fmt, x, y, z);
}


static char *logFileName = NULL;
static FILE *logFile = stderr;
static int replaceFlag; /* replace old heap with copy */
static int indexFlag; /* rebuild index first */
struct { int nNames; char **names; } tocNames;



main(argc,argv)
    int argc;
    char **argv;
{
    int h = -1;
    int hNew = -1;
    int answer;
    int tocFileNo;
    FILE *tocFile = NULL;
    int newHeapSize;
    int msgID;
    int nMsgs;

	/* hack to do the right stuff with no arguments */
        if( argc <= 1 ) {
	  char *mailDirName;
	  char *homeName;
	  int returnCode;
	  
	    if( ((homeName = GetEnv("HOME")) == NULL)
		    || (chdir(homeName) < 0)
		    || ((mailDirName = GetEnv(ENV_MAILDIR)) == NULL)
		    || (chdir(mailDirName) < 0) ) {
		fprintf(stderr,"Can't chdir to mail directory\n");
		exit(EX_UNAVAILABLE);
	    }
	    if( (returnCode = system(DEFAULT_COMMAND)) != EX_OK )
	        fprintf(stderr,"... errors\n");
	    exit(returnCode);
	}
	
        if( ParseArgs(argc, argv,
		"[-r(eplace] [-i(ndex rebuild] [-l logFileName] tocFile ...",
		"-rB",	&replaceFlag,
		"-iB",	&indexFlag,
		"-ls",	&logFileName,
		"*",	&tocNames,
		0 ) < 0 )
	    exit(EX_USAGE);

	if( logFileName != NULL )
	    logFile = fopen( logFileName, "a");

	LogMsg("Acquiring %s\n", MAIL_HEAP_NAME);
	h = AcquireMsgHeap( MAIL_HEAP_NAME,
		/*nNewItems=*/ (indexFlag ? PLENTY_BIG : 1) );
	if( h < 0 ) {
	    LogMsg("Error %d acquiring %s\n", h, MAIL_HEAP_NAME);
	    goto bad;
	}

	if( indexFlag ) {
	    LogMsg("Rebuilding index\n");
	    if( (answer = RebuildIndex(h)) < 0 ) {
	        LogMsg("Error %d rebuilding index\n", answer);
		goto bad;
	    }
	    if( (answer = WriteMsgHeap(h)) < 0 ) {
	        LogMsg("Error %d writing %s\n", answer, MAIL_HEAP_INDEX_NAME);
		goto bad;
	    }
	}
	
	if( tocNames.nNames == 0 ) {
	    LogMsg("No messages sets specified -- ref counts unchanged\n");
	    goto done;
	}

	LogMsg("Rebuilding ref counts\n");
	if( (answer = ClearMsgRefCounts(h)) < 0 ) {
	    LogMsg("Error %d clearing ref counts\n", answer);
	    goto bad;
	}
	for( tocFileNo = 0; tocFileNo < tocNames.nNames; tocFileNo++ ) {
	    char * tocFileName = tocNames.names[tocFileNo];
	    LogMsg("Examining %s\n", tocFileName);
	    tocFile = fopen(tocFileName, "r");
	    if( tocFile == NULL ) {
	        LogMsg("Cannot open %s\n", tocFileName);
		goto bad;
	    }
	    if( (answer = UpdateRefCounts( tocFile, h )) < 0 ) {
	        LogMsg("Error %d processing %s\n", answer, tocFileName);
		goto bad;
	    }
	    (void)fclose(tocFile);
	}
	
	newHeapSize = GetMsgHeapSize(h);

	LogMsg("Creating %s with %d entries\n", TEMP_HEAP_NAME, newHeapSize);
	hNew = AcquireMsgHeap( TEMP_HEAP_NAME, /*nNewItems=*/newHeapSize);
	if( hNew < 0 ) {
	    LogMsg("Error %d creating %s\n", hNew, TEMP_HEAP_NAME);
	    goto bad;
	}

	LogMsg("Copying referenced messages\n");

	nMsgs = 0;
	msgID = 0;
	while( (msgID = OpenMsg(h, msgID+1)) > 0 ) {
	    if( (answer = StartMsg(hNew)) < 0 ) {
	        LogMsg("Error %d writing message %d\n", answer, msgID);
		goto bad;
	    }
	    (void)AdjustMsgID(hNew, msgID);
	    if( (answer = CopyMsgBody(h, hNew, msgID)) < 0 ) {
	        LogMsg("Error %d copying message %d\n", answer, msgID);
		goto bad;
	    }
	    if( (answer = EndMsg(hNew)) < 0 ) {
	        LogMsg("Error %d ending message %d\n", answer, msgID);
		goto bad;
	    }
	    if( (answer = CloseMsg(h)) < 0 ) {
	        LogMsg("Error %d closing message %d\n", answer, msgID);
		goto bad;
	    }
	    nMsgs += 1;
	}

	LogMsg("Copied %d messages\n", nMsgs);

	if( (answer = WriteMsgHeap(hNew)) < 0 ) {
	    LogMsg("Error %d writing %s\n", answer, TEMP_HEAP_NAME);
	    goto bad;
	}
	(void)ReleaseMsgHeap(hNew);
	hNew = -1;

    good: /* Hooray! */
        if( replaceFlag ) {
            LogMsg("Renaming files\n");
	    answer = rename(TEMP_HEAP_DATA_NAME, MAIL_HEAP_DATA_NAME);
	    if( answer < 0 ) {
	        LogMsg("Error %d renaming %s to %s\n",
		    answer, TEMP_HEAP_DATA_NAME, MAIL_HEAP_DATA_NAME);
		goto bad;
	    }
	    answer = rename(TEMP_HEAP_INDEX_NAME, MAIL_HEAP_INDEX_NAME);
	    if( answer < 0 ) {
	        LogMsg("Error %d renaming %s to %s\n",
		    answer, TEMP_HEAP_INDEX_NAME, MAIL_HEAP_INDEX_NAME);
		goto bad;
	    }
	}

    done: 
        (void)ReleaseMsgHeap(h);
	LogMsg("Done\n");
	exit(EX_OK);

    bad:
        LogMsg("Giving up\n");
        if( h >= 0 ) (void)ReleaseMsgHeap(h);
	if( hNew >= 0 ) (void)ReleaseMsgHeap(hNew);
	if( tocFile != NULL ) fclose( tocFile );
        exit(EX_UNAVAILABLE);

}
