/*
 * CommandOps.c
 *
 * A simple command argument parser.
 * Utilities for dealing with .xxxxrc files.
 *
 * Created by AJD on Sun Jul  7 19:19:19 EDT 1985
 * Last edited by AJD on Mon Dec  2 23:59:36 PST 1985
 */

#include <stdio.h>
#include <setjmp.h>
#include <ctype.h>

#include "CommandOps.h"

#define TRUE 1
#define FALSE 0

#define LINE_SIZE 2048

static int argc;
static char **argv;
static char *usageMsg;
static char **argFmt;

static int argNo;
static int fmtNo;

static jmp_buf env;


static void
Usage()
{
    if( usageMsg != NULL )
        fprintf(stderr, "Usage: %s %s\n", argv[0], usageMsg);
    else
        fprintf(stderr, "Usage: %s ... ??? ... \n", argv[0]);
    longjmp(env, -1);
}


static  void
BadFlag(s)
    char *s;
{
    fprintf(stderr, "%s: bad flag %s\n", argv[0], s);
    Usage();
}


static void
MissingArg(f)
    char f;
{
    if( f != 0 )
        fprintf(stderr, "%s: -%c: missing operand\n", argv[0], f);
    else
        fprintf(stderr, "%s: missing argument\n", argv[0]);
    Usage();
}


static void
UnexpectedFlagArg(s)
    char *s;
{
    fprintf(stderr, "%s: unexpected flag argument: %s\n", argv[0], s);
    Usage();
}


static void
TooManyArgs()
{
    fprintf(stderr, "%s: too many arguments\n", argv[0]);
    Usage();
}


static int
ConvertNumber(s)
    char *s;
{
    register char *p = s;
    int sign = 1;
    int n = 0;

    while( *p == ' ' ) p++;
    if( *p == '-' )
        { sign = -1;  p++;  while( *p == ' ' ) p++; }
    while( (*p >= '0') && (*p <= '9') )
        { n = (n * 10) + (*p - '0');  p++; }
    while( *p == ' ' ) p++;
    if( *p != 0 ) {
        fprintf(stderr, "%s: bad number %s\n", argv[0], s);
	Usage();
    }
    return( sign * n );
}


struct argDescriptor {
    int a_argc;
    char **a_argv;
};

static void
DoFmt()
{
    register char *fp = argFmt[fmtNo];
    char theFlag = 0;
    char theFmtChar;

    if( *fp == '-' ) { fp++; theFlag = *fp++; }
    fmtNo += 1;
    while( (theFmtChar = *fp) != 0 ) {
        switch(theFmtChar) {
	    case 'B': /* Boolean flag */
	        *((int *)(argFmt[fmtNo])) = TRUE;
		break;
	    case 'b': /* inverted bOOLEAN flag */
	        *((int *)(argFmt[fmtNo])) = FALSE;
		break;
	    case 'e': /* Execute a function */ {
	        typedef int (*FuncPtr)();
		(void)(( *((FuncPtr)(argFmt[fmtNo])) )());
		break; }
	    case 's': /* string */
	        if( argNo >= argc ) MissingArg( theFlag );
		if( argv[argNo][0] == '-' ) UnexpectedFlagArg( argv[argNo] );
	        *((char **)(argFmt[fmtNo])) = argv[argNo];
		argNo += 1;
		break;
	    case 'd': /* decimal numeric argument */
	        if( argNo >= argc ) MissingArg( theFlag );
		if( argv[argNo][0] == '-' ) UnexpectedFlagArg( argv[argNo] );
	        *((int *)(argFmt[fmtNo])) = ConvertNumber(argv[argNo]);
		argNo += 1;
		break;
	    case '?': /* 0 or 1 non-flag */
	    case '*': /* sequence of 0 or more non-flag arguments */
	    case '+': /* sequence of 1 or more non-flag arguments */ {
	        register struct argDescriptor *p;
		int startingArgNo;

		p = (struct argDescriptor *)(argFmt[fmtNo]);
		startingArgNo = argNo;
		while( (argNo < argc) && (argv[argNo][0] != '-') )
		    argNo += 1;
		p->a_argv = argv + startingArgNo;
		p->a_argc = argNo - startingArgNo;
		if( (p->a_argc == 0) && (theFmtChar == '+') )
		    MissingArg(0);
		if( (p->a_argc > 1) && (theFmtChar == '?') )
		    TooManyArgs();
		break; }
	    case '$': /* all 0 or more remaining arguments */
	    case '#': /* all 1 or more remaining arguments */ {
	        register struct argDescriptor *p;

		p = (struct argDescriptor *)(argFmt[fmtNo]);
		p->a_argv = argv + argNo;
		p->a_argc = argc - argNo;
		if( (p->a_argc == 0) && (theFmtChar == '#') )
		    MissingArg(0);
		argNo = argc;
		break; }
	    default:
	        fprintf( stderr, "Internal ERROR %c in argFmt\n", *fp );
		abort(0);
	}
	fp++; fmtNo++;
    }
}


int
ParseArgsV( aArgc, aArgv, aUsageMsg, aArgFmt )
    int aArgc;
    char **aArgv;
    char *aUsageMsg;
    char **aArgFmt;
{
    int result;
    register char *fp;
    register char *ap;

    argc = aArgc;
    argv = aArgv;
    usageMsg = aUsageMsg;
    argFmt = aArgFmt;

    if( (result = setjmp(env)) == 0 ) {
    
	argNo = 1;
    
	/* process flag arguments */
	    while( (argNo < argc) && (argv[argNo][0] == '-') ) {
		ap = argv[argNo];
		if( strlen(ap) != 2 /* strlen("-c") */ ) BadFlag( ap );
		fmtNo = 0;
		for(;;) {
		    fp = argFmt[fmtNo];
		    if( (fp == NULL) || (fp[0] != '-') ) BadFlag( ap );
		    if( fp[1] == ap[1] ) break;
		    fmtNo += (strlen(fp) - 1);
		}
		argNo = argNo + 1; /* skip the flag itself */
		DoFmt();
	    }
    
	/* process other arguments */
	    fmtNo = 0;
	    for(;;) {
		fp = argFmt[fmtNo];
		if( (fp == NULL) || (fp[0] != '-') ) break;
		fmtNo += (strlen(fp) - 1);
	    }
	    if( fp != NULL ) DoFmt();
	    if( argNo < argc ) TooManyArgs();

    }

    return( result );
}


int
ParseArgs( aArgc, aArgv, aUsageMsg, aArgFmt0 /*, aArgFmt1, ..., 0 */ )
    int aArgc;
    char **aArgv;
    char *aUsageMsg;
    char *aArgFmt0;
{
    return( ParseArgsV( aArgc, aArgv, aUsageMsg, &aArgFmt0 ) );
}



/*
 * GetToken
 *
 * Entry: *pTokTpr points to null-terminated char buffer
 * Exit: *pTokPtr updated beyond next (whitespace-delimited) token,
 *   returned value is pointer to beginning of token.
 */
char *
GetToken(pTokPtr)
    char **pTokPtr;
{
    register char *p;
    register char c;
    char *answer;

    for( p = *pTokPtr; ((c = *p) != 0) && isspace(c); p++ )
        ;
    answer = p;
    for( ; ((c = *p) != 0) && (!isspace(c)); p++ )
        ;
    if( c != 0 ) *p++ = 0;
    *pTokPtr = p;
    return(answer);
}

/*
 * Primitives for dealing with environment
 */

extern char *getenv();
extern char *malloc();

extern char **environ;
int envEntryLim = 0;
#define ENVSIZEINCREMENT 20


char *
GetEnv(name)
    char *name;
{
    return( getenv(name) );
}


SetEnv(name, value)
    char *name;
    char *value;
{
    register char **envp;
    int nameLen;
    char *newEntryPtr;
    int entryLen;

    nameLen = strlen(name);
    entryLen = nameLen + 1 /* '=' */ + strlen(value) + 1 /* '\0' */ ;
    newEntryPtr = malloc(entryLen);
    sprintf(newEntryPtr, "%s=%s", name, value);

    for( envp = environ
        ; (*envp != NULL) && (strncmp(newEntryPtr, *envp, nameLen+1) != 0)
	; envp++ ) ;

    if( *envp == NULL ) {
        int newEntryIndex = (envp - environ);
        if( newEntryIndex >= envEntryLim ) {
	    int newEntryLim;
	    char **newEnviron;
	    register char **oenvp;

	    newEntryLim = newEntryIndex + ENVSIZEINCREMENT;
	    newEnviron = (char **)malloc( (newEntryLim+1)*sizeof(char *) );
	    oenvp=environ; envp = newEnviron;
	    while( *oenvp != NULL ) *envp++ = *oenvp++;
	    *envp = NULL;
	    if( envEntryLim > 0 ) free(environ);
	    environ = newEnviron;
	    envEntryLim = newEntryLim;
	}
	envp[1] = NULL;
    }

    *envp = newEntryPtr;
}


SetEnvI(name, ivalue)
    char *name;
    int ivalue;
{
    char buf[16];

    sprintf(buf, "%d", ivalue);
    SetEnv(name, buf);
}
