#ifndef _CMACCT_
#define _CMACCT_

#if !defined(lint) && !defined(LC) && !defined(__SABER__)
static char rcsid_cmacct[] = "$Id: cmacct.h,v 1.12 1992/01/16 05:36:26 dm Exp $";
#endif

/*****************************************************************************
* 	  (c) Copyright  1991 Thinking Machines Corporation, Inc.,	     *
*		of Cambridge, Mass.   All rights reserved.		     *
*									     *
*  This notice is intended as a precaution against inadvertent publication   *
*  and does not constitute an admission or acknowledgement that publication  *
*  has occurred or constitute a waiver of confidentiality.		     *
*									     *
*  Connection Machine software is the proprietary and confidential property  *
*  of Thinking Machines Corporation.					     *
*****************************************************************************/

#include "cmioctl.h"

#ifndef KERNEL
#include <sys/param.h>		/* for MAXHOSTNAMELEN */
#include <sys/types.h>
#ifndef SOCK_STREAM		/* vax headers aren't protected against */
#include <sys/socket.h>		/* multiple inclusion */
#endif /* SOCK_STREAM */
#include <sys/time.h>
#include <sys/resource.h>
#include <netdb.h>
#endif

/*
 * Here is a how to keep references to the accounting records in a CM_PROC
 * structure straight:
 *
 *			    .id.{pid, gid, sid}
 *		      ->acct.*
 * 	cmp->cmp_kacct->vfifo_data.*
 *
 *      ^    ^          ^
 *      |    |          +- CM_ACCT (acct) 
 *      |    |             OR a CMVFIFO_DATA (vfifo_data)
 *	|    |
 *	|    +------------ CM_KACCT *
 *      |
 *      +----------------- a CM_PROC *
 *
 *	cm accounting record id (one per process):
 *
 *		process-id
 *		user-id
 *		group-id
 *		process start time
 *		hostname
 *
 *	The kernel fills in all this data, save for the front-end address
 *	(the accounting log has that information).  The ts-daemon fills in
 *	all the data.  The CMFS fileserver fills in only the process-id,
 *	user-id, group-id, cm-pid, and front-end address, since that is
 *	all it has access to.
 *
 *	Why host_name?
 *	
 *		A front-end may have several addresses, but it has
 *		only one ``official host name'', obtainable as
 *		gethostent()->h_name.  MAXHOSTNAMELEN is defined in
 *		<param.h>
 *	
 *	Why time?
 *	
 *		The time is obtained by:
 *	
 *			struct timeval tp;
 *			gettimeofday(&tp, NULL);
 *			cmacct_id->time = tp.tv_sec;
 *	
 *		Since a daemon which is started out of /etc/rc.local
 *		will always have the same pid, uid, and gid, different
 *		incarnations of the daemon are distinguised by the
 *		time they are started.  
 *	
 *		Any process which uses the CM will have an accounting
 *		record generated that contains its *start* time (the
 *		time the process was created, not the time the process
 *		attached to a CM).  
 *	
 *		The accounting reports of the ts-daemon, fileserver,
 *		or socket-server will all contain the time that the
 *		account-id was generated.  That, plus the other
 *		information is enough to put the information together
 *		with the CM attachment record.
 *	
 *		Programs which use the filesystem, but which do not
 *		use the CM will only have filesystem accounting
 *		records, so there is no need to group them together.
 *		Programs which use the filesystem and the
 *		socket-server will get grouped together because
 *		CMACCT_get_accounting_id always returns the same data
 *		when called multiple times from within the same
 *		process. 
 *		
 */
typedef struct {
    short 	pid;
    short	spare0;
    uid_t 	ruid;		/* REAL uid */
    gid_t 	rgid;		/* REAL gid */
    uid_t	euid;		/* Effective uid */
    gid_t	egid;		/* Effective gid */
    pid_t	sid;		/* session-id */
    short	spare1;
    long	time;		/* start time of process (for kernel */
				/* records, date that the accounting */
				/* record was generated (for user-level */
				/* records). */
    long	spare2;
    long	spare3;
    char	host_name[MAXHOSTNAMELEN]; /* ``official host-name, i.e., */
					   /* gethostent()->h_name */
} CM_ACCT_ID;

/*
 *	kernel accounting structure (one for each attach to an
 *	interface).
 *
 *	kernel accounting structures are kept separate from the CM_PROC
 *	structure because, in order to get written out, a kernel
 *	accounting structure must get queued for the accounting daemon to
 *	read it.  Meanwhile the process may have gone and attached again.
 *		
 */
typedef struct {
    CM_ACCT_ID 	id;
    char	command[CM_MAX_CMD_LENGTH];
    int 	flags;
#   define TIMESHARED 01

    struct timeval attach_time;		/* Time of attach */
    struct timeval detach_time;		/* Time of detach */
    /* 
     * The next field indicates the total amount of
     * time that this process had exclusive access 
     * to the CM (either timeshared or exclusive) 
     */
    struct timeval pcur_duration;
    struct timeval last_excl_time;	/* Timestamp: last time this was active process */

    long	suspends;		/* # of times this process was suspended */
    long	spare0;
    long	spare1;

    int		uccs;			/* UCCS used */
    int 	interface;		/* FE interface number */
    long	ucc_busy_ticks;		/* # of ticks we found the UCC not idle */
    long 	ififo_busy_ticks;	/* # of ticks we found the IFIFO had been used */
    long	read_selects;		
    long	write_selects;
    long	exception_selects;
    struct rusage rusage;		/* UNIX resource usage for duration of record */
} CM_ACCT;

typedef struct {
    long 	vfifo_writes;
    long	vfifo_bytes_written;
    long	vfifo_reads;
    long	vfifo_bytes_read;
} CM_VFIFO_DATA; 

typedef struct cm_kacct {
    struct cm_kacct *next;
    struct cm_kacct *prev;
    CM_ACCT 	acct;
    CM_VFIFO_DATA vfifo_data;
    int reason;			/* a flag indicating WHY the accounting */
				/* record was created */
#   define CMKR_CONNECT_TO_INTERFACE 	0
#   define CMKR_IT_UCC_CHANGE		1
#   define CMKR_DISCONNECT_PROCESS	2
#   define CMKR_DELETE_PROCESS		3
#   define CMKR_REPEAT			1000
    /*
     * This is the rusage figure when the accounting record is created, it
     * is not reported as part of the accounting system, it is only used
     * to calculate the connected-time rusage of a process.
     */
    struct rusage rusage_begin;
} CM_KACCT;


/*
 * the ``give me the accounting data'' ioctl structure.
 */
typedef struct {
    int length;
    union {
	CM_ACCT *acct;
	CM_KACCT *kacct;
    } acctu;
} CM_ACCT_IOCTL;

/*
 * CMFS accounting entry.
 * This structure is filled in by the CMFS server for each
 * connection it manages (in general for each process using the
 * server).  It may span several CM_ACCT entries.
 */
typedef struct {
    CM_ACCT_ID		id;			/* CM Accounting ID for this record */
    char		server[MAXHOSTNAMELEN];	/* Server hostname */
    char		sv_hw_type;		/* Server hardware type */
    char		sv_sw_type;		/* Server software type */
    short		spare1;
    struct timeval	sv_utime;		/* Server user time used on our behalf */
    struct timeval	sv_stime;		/* Server sys time used on our behalf */
    unsigned long	p_writes;		/* # I/O bus writes */
    unsigned long	p_kb_written;		/* KB written to server over CMIO bus */
    unsigned long	p_reads;		/* # I/O bus reads */
    unsigned long	p_kb_read;		/* KB read from server over CMIO bus */
    unsigned long	s_writes;		/* # Serial (Ethernet) writes */
    unsigned long	s_kb_written;		/* KB written to server (Ethernet) */
    unsigned long	s_reads;		/* # Serial (Ethernet) reads */	
    unsigned long	s_kb_read;		/* KB read from server (Ethernet) */
    unsigned long	opens;			/* # file opens */
    unsigned long	creates;		/* # file creates */
    unsigned long	other_ops;		/* # other ops */
    long		spare2[16];
} CMFS_ACCT;

/* sv_hw_type values */
#define S_DV	1
#define S_PDA	2
#define S_VMEIO	3
#define S_FE	4
#define S_HIPPI	5

/* sv_sw_type values */
#define S_FSSERVER 1
#define S_SOCKSERVER 2

/*
 * Timesharing accounting structure.
 * This structure is filled in by the timesharing daemon for each
 * process it services, and is passed to the CM accounting daemon
 * for logging.
 */
typedef struct {
    CM_ACCT_ID		id;
    char		command[CM_MAX_CMD_LENGTH];
    unsigned long	memory_integral;
    unsigned long	max_size;	/* Maximum size (in 1kbit pages of proc	*/
    unsigned long	t_elapsed;	/* Elapsed time using CM (seconds) */
    unsigned long	t_cm;		/* Time scheduled on CM (seconds) */
    unsigned long	mem_requests;	/* Requests for more memory */
    unsigned long	mem_frees;	/* Frees of memory */
    unsigned long	io_requests;	/* I/O lockdown requests */
    struct timeval	sv_utime;	/* TS daemon user time used on our behalf */
    struct timeval	sv_stime;	/* TS daemon sys time used on our behalf */
    unsigned short	last_pri;	/* Last TS priority of this process */
    unsigned short	max_pri;	/* Highest priority of this process */
    long		spare[15];
} CMTS_ACCT;

/*
 * Data is stored by the CM accounting daemon to disk in this form.
 * This structure is also used to pass a CMFS_ACCT from the CMFS server
 * via the network to the CM accounting daemon, and also from the TS daemon
 * to the accounting daemon. 
 */
struct cm_acct_file_entry {
    short acf_record_type;
    short acf_record_length;
    union  {
	CM_ACCT_ID cm_id;	/* All records have ID at beginning.  This is useful for sort/merge */
	CM_ACCT cm_acct;
	CMFS_ACCT cmfs_acct;
	CMTS_ACCT cmts_acct;
	char dummy[512 - (2*sizeof (short))];
    } acf_u;
};

#define acf_cm acf_u.cm_acct
#define acf_cmfs acf_u.cmfs_acct
#define acf_ts acf_u.cmts_acct
#define acf_id acf_u.cm_id

#define ACF_OLD_TYPE_CM	1
#define ACF_OLD_TYPE_CMFS	2
#define ACF_OLD_TYPE_TS	3
#define ACF_OLD_TYPE_LIMIT	4
#define ACF_OLD_RECORD_LENGTH 512

#define ACF_TYPE_CM 	5
#define ACF_TYPE_CMFS	6
#define ACF_TYPE_TS	7

/*
 * Note that CMTS records are padded to CMFS length in the accounting file.
 */
#define ACF_VALID_LENGTH(type, length) \
	((type == ACF_TYPE_CM && length == sizeof(CM_ACCT)) \
	 || (type == ACF_TYPE_CMFS && length == sizeof(CMFS_ACCT)) \
	 || (type == ACF_TYPE_TS && length == sizeof(CMFS_ACCT)) \
	 || ((type > 0 && type < ACF_OLD_TYPE_LIMIT) \
	     && length == ACF_OLD_RECORD_LENGTH))

#define CM_ACCT_DAEMON_UNIX_PORT	"/dev/cmaccount"
#define CM_ACCT_DAEMON_INET_PORT	202
#define CM_ACCOUNTING_SERVICE		"cm-accounting"
#define CM_ACCOUNTING_FILE		"/usr/spool/cm/cmacct"

#endif _CMACCT_

