/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Vdirectory.h
 *
 * Include file defining the standard context directory entry formats
 *   and message types.
 * Conventions:  The first field of every descriptor must be "unsigned
 *   descriptortype".  The second field should be "char name[MAX_NAME_LENGTH]"
 *   unless there is some good reason for something else to be there.
 * Note: If a descriptor struct definitions is changed, then the byte-swapping
 * macros for this struct may also have to be changed.
 */

#ifndef VDIRECTORY
#define VDIRECTORY

#include "Vteams.h"

/* Message types */
typedef struct
  {
#ifndef LITTLE_ENDIAN
    SystemCode		requestcode;
    short unsigned	nameindex;	/* Start of name within segment */
    InstanceId		fileid;
    short unsigned	dataindex;	/* Start of data within segment */
#else LITTLE_ENDIAN
    short unsigned	nameindex;	/* Start of name within segment */
    SystemCode		requestcode;
    short unsigned	dataindex;	/* Start of data within segment */
    InstanceId		fileid;
#endif LITTLE_ENDIAN
    unsigned		filler[3];
    ContextId		namecontextid;	/* Context used to interpret name */
    char		*segmentptr;
    unsigned		segmentlen;
  } DescriptorRequest;
    

typedef struct
  {
#ifndef LITTLE_ENDIAN
    SystemCode		replycode;
    short unsigned	filler1;
    short unsigned	filler2;
    short unsigned	dataindex;	/* Start of data within segment */    
#else LITTLE_ENDIAN
    short unsigned	filler1;
    SystemCode		replycode;
    short unsigned	dataindex;	/* Start of data within segment */
    short unsigned	filler2;
#endif LITTLE_ENDIAN
    unsigned		filler3[4];
    char		*segmentptr;
    unsigned		segmentlen;
  } DescriptorReply;

/* Descriptor type flags */
#define EMPTY_DESCRIPTOR		0
#define FILE_DESCRIPTOR			1
#define DEVICE_DESCRIPTOR		2
#define TEAM_DESCRIPTOR			4
#define UNIXFILE_DESCRIPTOR		5
#define VGT_DESCRIPTOR			6
#define HOST_DESCRIPTOR			8
#define QUEUE_DESCRIPTOR		10
#define INTERNET_DESCRIPTOR		11
#define CONTEXT_DESCRIPTOR		13

/* Max name length for most things */
#define MAX_NAME_LENGTH			32


/* Empty descriptor (unused directory slot) */
typedef struct
  {
    unsigned		descriptortype; /* This field must be the first
					 *   one in each descriptor struct */
  } EmptyDescriptor;
 
#define ByteSwapEmptyDescriptor(edesc) do { \
    ByteSwapLongInPlace(&((edesc)->descriptortype), 1*sizeof(long)); \
  } while (0)


/* File on a V storage server */
typedef struct
  {
    unsigned		descriptortype;
    char                name[MAX_NAME_LENGTH];
    unsigned            perms;
    unsigned            timestamp;
    unsigned            size;
  } FileDescriptor;
 
#define ByteSwapFileDescriptor(fdesc) do { \
    ByteSwapLongInPlace(&((fdesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((fdesc)->perms), 3*sizeof(long)); \
  } while (0)



/* File on the UNIX server */
#define UnixUserNameLen		8
typedef struct
  {
    unsigned		descriptortype;
    char		name[MAX_NAME_LENGTH]; /* name in directory */
    /* The following fields are direct copies of the entries in
       "struct stat" in the UNIX include file <stat.h>. */
    short		st_dev;			/* (major,minor) device #s */
    unsigned short 	st_ino;			/* inode number */
    unsigned short	st_mode;		/* protection bits for file */
    short		st_nlink;		/* # of directory entries 
    						   for this object */
    short		st_uid;		        /* unix user number of owner */
    short		st_gid;			/* group "owner" of file */
    short		st_rdev;		/* raw device # */
    short		alignment2;		/* needed for BS */
    long		st_size;		/* number of bytes in file */

    /* NOTE: the following times are from Jan 1, 1970 */
    long 		st_atime;		/* time of last	access */
    long		st_mtime;		/* last modification */
    long		st_ctime;		/* creation date */

    char		ownername[UnixUserNameLen];	/* userid expanded */
    char		groupname[UnixUserNameLen];	/* groupid expanded */
  } UnixFileDescriptor;
 
#define ByteSwapUnixDescriptor(udesc) do { \
    ByteSwapLongInPlace(&((udesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapShortInPlace(&((udesc)->st_dev), 8*sizeof(short)); \
    ByteSwapLongInPlace(&((udesc)->st_size), 4*sizeof(long)); \
  } while (0)


/* Kernel device server */
typedef struct
  {
    unsigned		descriptortype;
    char		name[MAX_NAME_LENGTH]; /* Name of this entry */
    unsigned short	unit;	/* Unit number passed to driver */
  } DeviceDescriptor;
 
#define ByteSwapDeviceDescriptor(ddesc) do { \
    ByteSwapLongInPlace(&((ddesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapShortInPlace(&((ddesc)->unit), 1*sizeof(short)); \
  } while (0)



/* Team server */
typedef struct
  {
    unsigned	descriptortype;
    char fileName[MAX_NAME_LENGTH];
				/* File name team was loaded from. */
    ProcessId ownerPid;		/* Pid of process owning the loaded team. */
    ProcessId rootPid;		/* Pid of root process of loaded team. */
    ProcessId fileserver;	/* Pid of file instance from which to load 
				   team. */
    InstanceId fileid;		/* Fileid of file instance from which to load
				   team. */
    RootMessage rtMsg;		/* Root message for postmortem debugger. */
    short runPriority;		/* Team priority. */
  } TeamDescriptor;
 
#define ByteSwapTeamDescriptor(tdesc) do { \
    ByteSwapLongInPlace(&((tdesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((tdesc)->ownerPid), 3*sizeof(long)); \
    ByteSwapShortInPlace(&((tdesc)->fileid), 1*sizeof(short)); \
    ByteSwapRootMessage(&((tdesc)->rtMsg)); \
    ByteSwapShortInPlace(&((tdesc)->runPriority), 1*sizeof(short)); \
  } while (0)


/*
 * Service Server Definitions
 */
#define MaxObjectTypeLength 32	/* Maximum length of type string. */
#define MaxObjectNameLength 32	/* Maximum length of name string. */
#define MaxObjectRestartLength 80 
				/* Maximum length of name string. */

/* Host type */
#define HOST_TYPE "host"	/* Type for hosts. */

typedef struct
  {
    unsigned descriptortype;
    char serverType[MaxObjectTypeLength];
    char userName[MaxObjectNameLength];
    char hostName[MaxObjectNameLength];
    ProcessId regPid;
    int	availability;
  } HostDescriptor;

#define ByteSwapHostDescriptor(hdesc) do { \
    ByteSwapLongInPlace(&((hdesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((hdesc)->regPid), 2*sizeof(long)); \
  } while (0)


/* Virtual Graphics Terminal server */
#define VgtNameLength 64
#define VgtBannerLength 128

typedef struct
  {
    unsigned	descriptortype;
    char	name[VgtNameLength];
    char	banner[VgtBannerLength];
    int		typeVGT;
    int		mode;
    short	vgt, master;
    short	lines, columns;
    ProcessId 	ownerPid;
    ProcessId	breakPid;
    InstanceId	fileid;
  } VgtDescriptor;
 
#define ByteSwapVgtDescriptor(vdesc) do { \
    ByteSwapLongInPlace(&((vdesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((vdesc)->typeVGT), 2*sizeof(long)); \
    ByteSwapShortInPlace(&((vdesc)->vgt), 4*sizeof(short)); \
    ByteSwapLongInPlace(&((vdesc)->ownerPid), 2*sizeof(long)); \
    ByteSwapShortInPlace(&((vdesc)->fileid), 1*sizeof(short)); \
  } while (0)


/* Internet (including TCP, IP, PUP, IKC Gateway) Descriptor */
typedef struct
  {
    unsigned 	descriptortype;
    char	name[MAX_NAME_LENGTH];
    int		typeInternet;
    unsigned	addr1;
    unsigned	addr2;
    unsigned    addr3;
  } InternetDescriptor;
 
#define ByteSwapInternetDescriptor(idesc) do { \
    ByteSwapLongInPlace(&((idesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((idesc)->typeInternet), 4*sizeof(long)); \
  } while (0)


/* Print server queue descriptor */
#define QUserLen	10
#define QHostLen	20
#define	QFileNameLen	30
#define QSpoolFileLen	25

typedef struct
  {
    unsigned	descriptortype;
    unsigned	rank;
    char	user[QUserLen];
    char	host[QHostLen];
    char	filename[QFileNameLen];
    char	spoolfile[QSpoolFileLen];
  } QueueDescriptor;

#define ByteSwapQueueDescriptor(desc) do { \
    ByteSwapLongInPlace(&((desc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((desc)->rank), 1*sizeof(long)); \
  } while (0)



/* Context with no special attributes */
typedef struct
  {
    unsigned		descriptortype;
    char                name[MAX_NAME_LENGTH];
    ContextPair		ctxpair;	/* (0, 0) if none unassigned */
  } ContextDescriptor;
 
#define ByteSwapContextDescriptor(fdesc) do { \
    ByteSwapLongInPlace(&((fdesc)->descriptortype), 1*sizeof(long)); \
    ByteSwapLongInPlace(&((fdesc)->ctxpair), 2*sizeof(long)); \
  } while (0)


/* Arbitrary descriptor */
typedef union
  {
    EmptyDescriptor		e;
    FileDescriptor		f;
    DeviceDescriptor		d;
    TeamDescriptor		t;
    UnixFileDescriptor		u;
    VgtDescriptor		v;
    HostDescriptor		h;
    QueueDescriptor		q;
    InternetDescriptor		i;
    ContextDescriptor		c;
  } ArbitraryDescriptor;

#endif VDIRECTORY
