/*
 * table.c
 *
 * Reverse Address Resolution Protocol server
 * Copyright (c) 1984 Stanford University.
 *
 * Routines for reading and searching the table of
 * (Ethernet address, IP address) pairs.
 *
 * HISTORY:
 *
 * 28 May 1984	Sally Ahnger	Stanford (Gregorio)
 *	- Created
 *
 * Ross Finlayson, October 1984.
 * Moved this code to a separate file, and added some error checking.
 *
 */

#include "rarpd.h"

/* Size of line input buffer from database file. */
#define MAX_LINE 200


BringInTable(pn)
    PerNet *pn;
    /* read file until eof, filling each element of the table with a line.
     * each line of the file looks like this:
     *     <host name, no spaces> Ethernet: x.y.z IP: a.b.c.d 
     * or  <host name, no spaces> Ethernet: x IP: a.b.c.d
     * (for a 3Mb Ethernet)
     *     (x, y, z must be hexadecimal - a, b, c, d decimal.)
     */
 {
    FILE *fileid;		/*data base file*/
    char *fname;		/*name of file to open from getfilename*/
    char buff[MAX_LINE];	/*buffer to read a line*/
    char *eof;		/*fgets returns NULL for eof*/
    int lineCount, parseResult;

    char *GetFileName();	/*forward declaration for function*/

    /* open the file, named by function GetFileName() */
    fname = GetFileName(pn);
    if (debug)
	printf("Reading \"%s\"\n", fname);
    fileid = fopen( fname, "r");
    if (fileid==NULL)
      {
	printf("rarpd: couldn't open \"%s\"\n", fname);
	perror("");
	exit(1);
      }

    /* Skim quickly through the file, counting the number of lines. */
    for (lineCount = 1;
	 fgets(buff, sizeof(buff), fileid) != NULL;
	 ++lineCount)
	;
    rewind(fileid);
    
    /* Free any space taken up by an existing table. */
    if (pn->Table != NULL)
	free(pn->Table);
    
    /* Allocate space for the new table. */
	if ((pn->Table = (TabEnt *)malloc(sizeof(TabEnt)*lineCount)) == 0)
	  {
	    perror("rarpd: malloc error in BringInTable()");
	    exit(1);
	  }
      
    /* Read in the new table. */
    pn->Size = 0;			/*start with element zero*/
    eof = fgets (buff, sizeof(buff), fileid);/*read first line*/

    while (eof != NULL)
      {
	parseResult = ParseLine(buff, &(pn->Table[pn->Size]), pn->enetType);
	if (parseResult < 0)
	    fprintf(stderr, "rarpd: bad line in RARP database: %s\n", buff);
	else if (parseResult > 0)
	    ++(pn->Size);
	
	/*read another line of the file, incl the CR, followed by null*/
	eof = fgets (buff, sizeof(buff), fileid);
      }
    fclose( fileid);
    pn->When_Checked = time(0);
    pn->When_Read = pn->When_Checked;
  }


int ParseLine(line, tabEnt, enetType)
    char line[];
    TabEnt *tabEnt;
    EnetType enetType;
    /* Parses the input line "line", entering the IP and Ethernet addresses
     * found into "tabent".  This routine returns:
     *     1 if the line was successfully parsed,
     *     0 if the line is empty or begins with a comment character (; or #),
     *    -1 if a syntax error was found in the line.
     */
  {
    int count;		/*number of fields scanned in a line*/
    uchar IPstr[20];	/*larger than IP address x.x.x.x */
    uchar Estr[20];	/*larger than ethernet address x.x.x */

    if ( (line[0]==';') || (line[0]=='#') )
        return(0); /* comment line. */

    count = sscanf(line, "%*s%*s%s%*s%s", Estr, IPstr);
    if (count <= 0)
        return (0); /* empty line */
    else if (count != 2)
	return (-1);

    if ( ConvHard( tabEnt->Hard_Addr, Estr, enetType) < 0 )
	return (-1);
    if (debug)
	PrintEtherAddr(tabEnt->Hard_Addr, enetType);

    if ( ConvSoft( tabEnt->Soft_Addr, IPstr) < 0 )
	return (-1);
    if (debug)
      {
	PrintIPAddr(tabEnt->Soft_Addr);
	printf("\n");
      }
	
    return (1);
  }


int ConvSoft(dest,src)
    IPAddr dest;
    uchar *src;
/* ConvSoft scans the IP address string to pick off the decimal digits. */
 {
    int i;
    unsigned int num[4];

    if (sscanf( src, "%d%*c%d%*c%d%*c%d",
		&num[0], &num[1], &num[2], &num[3]) != 4)
	return (-1);
    for (i=0; i< 4; ++i)
	dest[i] = num[i];	/* correct bits truncated automagically */
    return (0);
  }


int ConvHard(dest, src, enetType)
    EtherAddr dest;
    uchar *src;
    EnetType enetType;
/* ConvHard scans the incoming string and picks off the hex numbers and then
   splits each number into two bytes for the EtherAddr type format. */
 {
    if (enetType == enet10)
      {
	/* 10Mb Ethernet address. */
	int i;
	unsigned int num[3];
	    
	if (sscanf( src, "%x%*c%x%*c%x", &num[0], &num[1], &num[2]) != 3)
	    return (-1);
	for (i=0; i<3; i++)
	  {
	    dest[i*2] = (num[i]>>8) & 0xFF;
	    dest[(i*2)+1] = num[i] & 0xFF;
	  }
	return (0);
      }
    else
      {
	/* 3Mb Ethernet address. */
	int num;

	if (sscanf( src, "%x", &num) != 1)
	    return (-1);
	dest[0] = num & 0xFF;
	return (0);
      }
  }


char rarpFileName[] = "/etc/rarpdb.XXXXXXXXXXXXXXXX";


static char *baseOf(fileName)
    register char *fileName;
  {
    char *base = fileName;
    
    while (*fileName != '\0')
        if (*fileName++ == '/')
	    base = fileName;
    return (base);
  }

char *GetFileName(pn)
    PerNet *pn;
    /* Returns the name of the RARP database file for the net "pn".  This
     * is (by convention) "/etc/rarpdb." concatenated with the 'basename' of
     * the device filename (e.g. "eneta").
     */
  {
    strncpy(&rarpFileName[12], baseOf(pn->FileName), 15);
    return (rarpFileName);
  }


LookUp(hard, soft, pn)
    /* Find the software address corresponding to the given hardware address.
       pn points to the pernet table containing the table to search. */
    register EtherAddr hard;
    register IPAddr *soft;
    register PerNet *pn;
  {
    int j;
    int i = 0;
    int found = 0;
    EnetType enetType = pn->enetType;

    if (enetType == enet10)
      {
	register uchar *ha;
	
	while ((i < pn->Size) && (!found))
	  {
	    ha = pn->Table[i++].Hard_Addr;
	    found = 1;
	    for (j=0; j < enet10AddrSize; j++)
	      {
		if (ha[j] != hard[j])
		  {
		    found = 0;
		    break;
		  }
	      }
	  }
      }
    else /* enet3Mb */
      {
	register uchar ha;
	
        while (i < pn->Size)
	  {
	    ha = pn->Table[i++].Hard_Addr[0];
	    if (ha == hard[0])
	      {
		found = 1;
		break;
	      }
	  }
      }

    --i;   /* since it got bumped once too often */

    if (found)
    	bcopy( pn->Table[i].Soft_Addr, soft, sizeof(IPAddr));
    return(found);
  }
