/*
 * rarpd.c
 *
 * Reverse Address Resolution Protocol server
 * Copyright (c) 1984 Stanford University.
 *
 * HISTORY:
 *
 * 28 May 1984	Sally Ahnger	Stanford (Gregorio)
 *	- Created
 * Ross Finlayson, October 1984.
 * Broke up the code into several files, added some error checking, and
 * added support for 3Mbps Ethernets.
 *
 ****************************************************************
 *
 * execution: 1 arg => debug 
 *
 * This server uses files (/etc/rarpdb.*) to create a table of mappings 
 * between Ethernet (hardware) addresses and 32-bit IP protocol (logical)
 * addresses.
 * It can be expanded to include other kinds of hardware and protocol
 * addresses, if needed.
 *
 * It provides a service for client machines (usually diskless workstations)
 * to return a logical address when given a hardware address.
 *
 ****************************************************************
 * Possible future extensions:
 * 1)	Fix Our_Ether_Addr to reflect multiple networks.  Right now
 *	gethostid is used, which returns the IP address on the first
 *	network, not necessarily the 10 Mbit one.  (in OpenEthers)
 * 2)	Change LookUp to use a faster search than sequential.
 * 3)	Support other kinds of 'hardware' or 'protocol' addresses.
 */

#include "rarpd.h"

/*interval to wait before checking to see if disk file has been changed*/
#define CHECK_INTERVAL (60 * 10)	/* 10 minutes */

/*
 * Map from fid's to PerNet structures
 */
PerNet *FidMap[NOFILE];

long ethermask;		/* used in select */

int debug;


main(argc, argv)
    int argc; /*number of parameters*/
    char **argv; /*array of parameters*/
    /* Main mostly does debug and error-checking things.  It calls OpenEthers
       to establish the networks, and then calls MainLoop to do the serving. */
  {
    register int i;

    if (NOFILE >= 32)
	fprintf(stderr, "%s: WARNING: NOFILE >= 32\n", argv[0]);

    /* if there are more args than the name of the program, it must be the 
       debug flag. */
    if (argc > 1)	
	debug = 1;
    
    /* disable interrupts*/
    if (!debug)
      {
	signal(SIGHUP, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
      }
    
    for (i = 0; i < NOFILE; i++)
	FidMap[i] = NULL; 
    
    ethermask = 0;
    
    OpenEthers();  /* changes ethermask, builds FidMap table, reads file */
    

    if (debug)
      {
	for (i = 0; i < NOFILE; i++)
	    if (FidMap[i] != NULL)
	      {
		PrintEtherAddr(FidMap[i]->Our_Eth_Addr, FidMap[i]->enetType);
		printf("\n");
	      }
      }
    
    MainLoop();
  }


MainLoop()
/* MainLoop contains an infinite loop that reads packets and checks to see if
   disk data file has been changed since last it was read. */
  {
    int nfound, readfds;
    register int i;
    long now;
    struct timeval tv;

    tv.tv_sec = 60;		/*wait 1 minute for a packet*/
    tv.tv_usec = 0;
    
    while (1)
      {
	/* save ethermask because "select" changes the bits to indicate
	   which of the bits that were on in readfds correspond to 
	   files that have packets waiting on their net */
	readfds = ethermask;	
	nfound = select(NOFILE, &readfds, 0, 0, &tv);	
	if (debug)
	  {
	    switch (nfound)
	      {
		case -1:
		    printf("error in select!\n"); break;
		case 0:
		    printf("timed out from select.\n"); break;
		default:
		    printf("select - packets available on %d files.\n",
			   nfound);
	      }
	  }
    
	for (i = 0; readfds; i++)	/*until all bits turned off*/
	  {
	    if (readfds & (1<<i))
		/* something to read from this ether */
		EtherRead(i);
	    readfds &= ~(1<<i);	/*turn this bit off*/
	  }
	if (debug)
	    printf("past EtherRead loop\n");
    
	/* Check the disk file for changes, if it hasn't been checked
	   in the last 10 minutes */
	now = time(0);
	for (i=0; i < NOFILE; i++)
	    if (FidMap[i] != NULL)
	      if (FidMap[i]->When_Checked < (now - CHECK_INTERVAL))
		{
		  FidMap[i]->When_Checked = now;
		  if (NeedDiskFile(FidMap[i]))
		      BringInTable(FidMap[i]);
		}
	if (debug)
	    printf("past time check loop\n");
	
      }   /* end while (1) */
  }


NeedDiskFile(pn)
    /* look on disk to see if the file GetFileName(fid) has been changed since 
       FidMap[fid]->When_Checked.  If not, return true. */
    PerNet *pn;
  {
	struct stat statbuf;
	char *fname;
	char *GetFileName();

	fname = GetFileName(pn);
	if (stat(fname, &statbuf) != 0 )
	  {
		perror(fname);
		return(0);
  	  }
	if (statbuf.st_mtime > pn->When_Read)
		return(1);
	else
		return(0);
  }

OpenEthers()
    /* OpenEthers goes through all nets on this machine and opens a file for
       each Ethernet interface.  It builds a pernet table for each file
       opened.  It calls BringInTable to build a table of address pairs for each
       net.  It sets up a packet filter for each net for RARP packets. */
 {
    struct pupnettab *pnt;
    int fid;
    union DevParams devp;
    union PakFilter filter;
    PerNet *pn;
    int n;
    long OurIPAddr;
    EnetType enetType;

    if (setpupnettab() == 0)
      {
	fprintf(stderr, "rarpd: cannot open %s\n", PUPNETTAB);
	exit(1);
      }
    
    while (pnt = getpupnettab())
      {
	if (debug)
	    printf("Trying to open %s\n", pnt->pnt_devname);
	fid = enopen(pnt->pnt_devname, &devp);
	if (fid < 0)
	    continue;
	if (devp.en.dp.end_dev_type == ENDT_10MB)
	    enetType = enet10;
	else if ( (devp.en.dp.end_dev_type == ENDT_3MB) ||
		  (devp.en.dp.end_dev_type == ENDT_BS3MB) )
	    enetType = enet3;
	else
	  {
	    close(fid);
	    if (debug)
		printf("%s - bad device type\n", pnt->pnt_devname);
	    continue;
	  }
	if (debug)
	    printf("Opened %s\n", pnt->pnt_devname);

	if ((pn = (PerNet *)malloc(sizeof(PerNet))) == 0)
	  {
	    perror("rarpd: malloc error in OpenEthers()");
	    exit(1);
	  }

	/* fill in pernet table entries: */
	bcopy(devp.en.dp.end_addr, pn->Our_Eth_Addr,
	      (enetType == enet10) ? enet10AddrSize : enet3AddrSize);
	pn->enetType = enetType;
	/* gethostid really gets the main id and won't work if > 1 net: */
	OurIPAddr = gethostid();	
	*(long *)pn->Our_IP_Addr = OurIPAddr;
	if (debug)
	  {
	    printf("returned from gethostid: %x\n", OurIPAddr);
	    PrintIPAddr(pn->Our_IP_Addr);
	  }
	pn->Fid = fid;		/* BringInTable uses this */
	strcpy(pn->FileName, pnt->pnt_devname);
	pn->DevP= devp;
	pn->Table = NULL;	/* for check in BringInTable */

	BringInTable(pn);	/* fills in Table, Size, When_Checked,Read */

	FidMap[fid] = pn;
	ethermask |= (1<<fid);
	
	efinit(&devp, &filter, 63, makeshort(ETHERTYPE_RARPTYPE));
	ensetfilt(fid, &filter);
    }
    endpupnettab();
  }

