/*
 * CPARSE tree expression handling
 */

#include "defs.h"
#include "y.tab.h"

/*
 * Tmkname - allocate a T_NAME node
 */
public Tnode Tmkname(name, son)
Name name;
Tnode son;
{
	register Tnode t;

	Ttrace(1, printf("Tmkname %s %s\n", Nstring(name), xTop(son)));
	t = Talloc(T_NAME);
	t->nm.name = name;
	t->T_type = TUNDEF;
	t->nm.dimptr = TNIL;
	t->T_son = son;
	return(t);
}

/*
 * Tmkvardef - make a T_VARDEF node
 */
public Tnode Tmkvardef(att, initlist)
Tnode att, initlist;
{
	register Tnode t, tt;

	t = Talloc(T_VARDEF);
	t->T_son = Talloc(T_TYPEDEC);
	if (att->T_op != T_CLASS) {
		Ttrace(1, printf("Tmkvardef add class\n"));
		tt = Talloc(T_CLASS);
		tt->cl.class = SNULL;
		tt->T_sib = att;
	} else {
		Ttrace(1, printf("Tmkvardef\n"));
		tt = att;
	}
	t->T_son->T_son = tt;
	t->T_son->T_sib = initlist;
	return(t);
}

/*
 * Tmkumul - make a unary * node
 */
public Tnode Tmkumul(node)
Tnode node;
{
	register Tnode t;

	Ttrace(1, printf("Tmkumul %s\n", xTop(node)));
	t = Talloc(T_UMUL);
	t->T_son = node;
	return(t);
}

/*
 * Tmkarray - make an array node
 *
 * dimension size is in sibling pointer
 */
public Tnode Tmkarray(node, size)
Tnode node, size;
{
	register Tnode t;

	Ttrace(1, printf("Tmkarray %s %s\n", xTop(node), xTop(size)));
	t = Talloc(T_UARRAY);
	if (size==TNIL)
		size = Talloc(T_NULL);	/* make sure there is a placeholder */
	t->sz.dim = size;
	t->T_son = node;
	return(t);
}

/*
 * Tmkbinop - binary op expr node
 */
public Tnode Tmkbinop(left, op, right)
Tnode left, right;
Integer op;
{
	register Tnode t;
	register unsigned short act;

	/*
	 * left and right have been checked for existence,
	 * but here they should be checked for compatibility.
	 */
	TTtrace(2, printf("Tmkbinop %s\n",Opstr(op)));
	t = Talloc(T_BINOP);
	t->ic.value = op;
	left->T_sib = right;
	t->T_son = left;

	act = Opaction(op, left, right);

	if (act & LVAL) {
		if (!Oplval(left)) {
			uerror("illegal LHS of assignment");
		}
	}

	if (act & PUN) {
		Opchkpun(t);
	}

	if (act & TYPL) {
		t->T_type = left->T_type;
	}
	if (act & TYPR) {
		t->T_type = right->T_type;
	}

	if (act & TYMATCH) {
		Optymatch(t);
	}
	if (act & PTMATCH) {
		Opptmatch(t);
	}

	if (act & OTHER) {
	   switch (op) {
	   case LS:
	   case RS:
	   case ASG LS:
	   case ASG RS:
		/* cast to int if needed */
		break;
	   case ASSIGN:
		/* get here if it's a struct assignment */
		break;
	   default:
		cerror("OTHER binop code = %d", op);
	   }
	}

	return(t);
}

/*
 * Tmkunop - unary op expr node
 */
public Tnode Tmkunop(op, term)
Integer op;
Tnode term;
{
	register Tnode t;
	register unsigned short act;

	TTtrace(2, printf("Tmkunop %d\n",op));
	t = Talloc(T_UNOP);
	t->ic.value = op;
	t->T_son = term;

	act = Opaction(op, term, TNIL);

	if (act & TYPL) {
		t->T_type = term->T_type;
	}

	if (act & OTHER) {
		switch (op) {
		case UNARY MUL:
			if (!ISPTR(term->T_type))
				uerror("illegal indirection");
			t->T_type = DECREF(term->T_type);
			break;
		case UNARY AND:
			if (ISFTN(term->T_type) || ISARY(term->T_type)) {
				werror("& before function or array ignored");
				dispose(t);
				return(term);
			}
			/*
			 * all of these cases may in fact be the same.
			 * I'll decide later.
			 */
			if (term->T_op==T_SYM) {
				t->T_type = INCREF(term->T_type);
			} else if (term->T_op==T_UNOP && term->ic.value==UNARY MUL) {
				t->T_type = INCREF(term->T_type);
			} else if (term->T_op==T_ARYREF) {
				t->T_type = INCREF(term->T_type);
			} else if (term->T_op==T_CMEXPR) {
				/* type is that of the last expr in the list */
				t->T_type = INCREF(term->T_type);
			} else if (term->T_op==T_COND) {
				/* just grab the type from the conditional */
				t->T_type = INCREF(term->T_type);
			} else {
				uerror("unacceptable operand of &");
			}
			break;
		}
	}
	return(t);
}

/*
 * Tmkvaref - variable reference node
 */
public Tnode Tmkvaref(name)
Name name;
{
	register Tnode t;
	register Symbol s;

	Strace(2, printf("Tmkvaref %s\n",Nstring(name)));
	s = Symlookup(name, 0, Curlevel);
	if (s==SNIL) {
		if (Globallevel) {
			uerror("undeclared initializer name %s", Nstring(name));
			s = Syminsert(name, SEXTERN, 0, true);
			s->s_type = TINT;
		} else {
			uerror("%s undefined", Nstring(name));
			/*
			 * cannot be a param, because they are stashed in the
			 * symtab right away.
			 */
			s = Syminsert(name, SAUTO, Curlevel, true);
			s->s_type = TINT;
		}
	}

	t = Talloc(T_SYM);
	t->sym.symptr = s;
	t->T_type = s->s_type;
	TTtrace(2, printf("Tmkvaref %s ",Nstring(s->s_name)));
	TTtrace(2, Stprint(t->T_type, TNIL));
	TTtrace(2, printf("\n"));
	return(t);
}

/*
 * Tmkstunref - struct/union reference node
 */
public Tnode Tmkstunref(term, op, member)
register Tnode term;
Token op;
Name member;
{
	register Tnode t;
	register Symbol s, tagptr;

	TTtrace(2, printf("Tmkstunref %s\n",Nstring(member)));

	t = Talloc(T_STREF);
	t->ic.value = op;
	/* allocate a T_SYM for the member */
	term->T_sib = Talloc(T_SYM);
	t->T_son = term;
	t->T_type = TINT;	/* in case of error */

	if ((op==DOT && ISPTR(term->T_type))
	    || (op==STREF && !ISPTR(term->T_type))
	    || (BTYPE(term->T_type) != TSTRTY && BTYPE(term->T_type) != TUNIONTY)
	    || ((tagptr = Structsym(term))==SNIL)) {
		uerror("illegal struct/union ref");
	} else {
		while (tagptr->s_class==STYPEDEF) {
			TTtrace(2, printf(">skipping %s typedef\n",Nstring(tagptr->s_name)));
			tagptr = tagptr->s_typtr;
		}
		for (s=tagptr->s_typtr; s != SNIL; s = s->s_xlink) {
			TTtrace(3, Symprint(s));
			if (s->s_name==member) {
				term->T_sib->sym.symptr = s;
				term->T_sib->T_type = s->s_type;
				t->T_type = s->s_type;
				return(t);
			}
		}
		uerror("illegal struct/union member ref");
	}

	return(t);
}

/*
 * Tmkaryref - array reference node
 */
public Tnode Tmkaryref(term, expr)
Tnode term, expr;
{
	register Tnode t;
	register Stype btype, type;

	TTtrace(2, printf("Tmkaryref %s[%s]\n", xTop(term), xTop(expr)));
	t = Talloc(T_ARYREF);
	term->T_sib = expr;
	t->T_son = term;
	type = term->T_type;
	if (!ISARY(type) && !ISPTR(type))
		uerror("illegal array reference");
	if (!(Opmod(expr->T_type) & MINT))
		uerror("illegal array index");
	t->T_type = DECREF(type);
	return(t);
}

/*
 * Tmklabel - label a node
 */
public Tnode Tmklabel(name, expr)
Name name;
Tnode expr;
{
	register Tnode t, tt;
	register Symbol s;

	/*
	 * null name == "default" statement
	 * null name + expr == "case" statement
	 * null expr == "name:" statement
	 */
	t = Talloc(T_LABEL);
	/* T_son will be a statement */

	if (name==NNIL) {
		Ttrace(2, printf("Tmklabel case %s\n",xTop(expr)));
		t->label.ptr = expr;
		if (SwitchDepth==0)
			uerror("case/default stmt outside switch");
	} else {
		Ttrace(2, printf("Tmklabel %s\n",Nstring(name)));
		s = Symlookup(name, 0, Curlevel);
		if (s==SNIL) {
			s = Syminsert(name, SULABEL, Curlevel, true);
		}
		tt = Talloc(T_SYM);
		tt->sym.symptr = s;
		Symdef(tt, SLABEL, TNULL, IN_OTHER);
		t->label.ptr = tt;
	}

	return(t);
}

/*
 * Tmkopinc - prefix inc op node
 */
public Tnode Tmkincop(op, term)
Integer op;
Tnode term;
{
	register Tnode t;
	register Stype btype, type;

	TTtrace(2, printf("Tmkincop %d %s\n",op,xTop(term)));
	t = Talloc(T_INCPRE);
	t->ic.value = op;
	t->T_son = term;
	type = term->T_type;
	if (!ISPTR(type) && ((btype=BTYPE(type))==TSTRTY || btype==TUNIONTY
				|| btype==TENUMTY || btype==TMOETY))
		uerror("illegal operand of %s", Opstr(op));
	t->T_type = term->T_type;
	return(t);
}

/*
 * Tmkincop - suffix inc op node
 */
public Tnode Tmkopinc(op, term)
Integer op;
Tnode term;
{
	register Tnode t;
	register Stype btype, type;

	TTtrace(2, printf("Tmkopinc %d %s\n",op,xTop(term)));
	t = Talloc(T_INCPOST);
	t->ic.value = op;
	t->T_son = term;
	type = term->T_type;
	if (!ISPTR(type) && ((btype=BTYPE(type))==TSTRTY || btype==TUNIONTY
				|| btype==TENUMTY || btype==TMOETY))
		uerror("illegal operand of %s", Opstr(op));
	t->T_type = term->T_type;
	return(t);
}

/*
 * Tmksizeof - sizeof expr node
 */
public Tnode Tmksizeof(term)
Tnode term;
{
	register Tnode t;

	/*
	 * need to make sure we can take the size of the item
	 */
	TTtrace(2, printf("Tmksizeof %s\n",xTop(term)));
	t = Talloc(T_SIZEOF);
	t->T_son = term;
	t->T_type = TUSHORT;
	return(t);
}

/*
 * Tmkgoto - generate a goto
 */
public Tnode Tmkgoto(name)
Name name;
{
	register Tnode t;
	register Symbol s;

	Ttrace(2, printf("Tmkgoto %s\n",Nstring(name)));
	t = Talloc(T_GOTO);
	s = Symlookup(name, 0, Curlevel);
	if (s==SNIL) {
		s = Syminsert(name, SULABEL, Curlevel, true);
		s->s_type = TNULL;
	}

	/* the symbol will be defined later if it isn't alreay */
	t->sym.symptr = s;
	return(t);
}

/*
 * Tmkfunsym - make a function reference node
 */
public Tnode Tmkfunsym(name)
Name name;
{
	register Tnode t;
	register Symbol s;

	TTtrace(2, printf("Tmkfunsym %s\n",Nstring(name)));
	t = Talloc(T_SYM);
	s = Symlookup(name, 0, 0);
	if (s==SNIL) {
		Strace(1, printf("Tmkfunsym undef ftn %s -> SEXTERN\n",Nstring(name)));
		s = Syminsert(name, SNULL, 0, true);
		t->sym.symptr = s;
		Symdef(t, SEXTERN, TFTN|TINT, IN_OTHER);
	} else
		t->sym.symptr = s;

	if (!ISFTN(s->s_type))
		uerror("illegal function symbol");
	t->T_type = DECREF(s->s_type);
	return(t);
}

/*
 * Tmkcond - conditional expr node
 */
public Tnode Tmkcond(cond, texpr, fexpr)
Tnode cond, texpr, fexpr;
{
	register Tnode t;

	TTtrace(2, printf("Tmkcond %s ? %s : %s\n",xTop(cond),xTop(texpr),xTop(fexpr)));
	t = Talloc(T_COND);
	texpr->T_sib = fexpr;
	cond->T_sib = texpr;
	t->T_son = cond;
	/* check for texpr and fexpr compatibility */
	t->T_type = texpr->T_type;
	return(t);
}

/*
 * Tmkcmlist - make a list of exprs
 */
public Tnode Tmkcmlist(expr1, expr2)
Tnode expr1, expr2;
{
	register Tnode t;

	TTtrace(2, printf("Tmkcmlist %s , %s\n",xTop(expr1),xTop(expr2)));
	t = Talloc(T_CMEXPR);
	expr1->T_sib = expr2;
	t->T_son = expr1;
	t->T_type = expr2->T_type;	/* type is right-most expr type */
	return(t);
}

/*
 * Tmkcast - make a typecast
 */
public Tnode Tmkcast(castype, term)
Tnode castype, term;
{
	register Tnode t;

	TTtrace(2, printf("Tmkcast %s %s\n",xTop(castype),xTop(term)));
	t = Talloc(T_CAST);
	term->T_sib = castype;
	t->T_son = term;
	t->T_type = castype->T_type;
	return(t);
}

/*
 * Tmkcastype - evaluate a typecast
 */
public Tnode Tmkcastype(tnode, nmnode)
Tnode tnode, nmnode;
{
	register Tnode t, tynode;

	Ttrace(2, printf("Tmkcastype\n"));
	t = Talloc(T_TYPEDEC);
	tynode = TYmerge(nmnode, TYresolve(tnode));
	t->T_type = tynode->T_type;
	t->ty.symptr = tnode->ty.symptr;
	TTtrace(2, printf("Tmkcastype type=("));
	TTtrace(2, Stprint(t->T_type, TNIL));
	TTtrace(2, printf(") sym="));
	TTtrace(2, Symprint(t->ty.symptr));
	t->T_son = tnode;
	t->T_sib = nmnode;
	return(t);
}
