/*
 * CPARSE symbol table management
 *
 * Alan Hastings 2/17/85
 */

#include "defs.h"

#define SYMHASHSIZE	2047

#define Symhash(ptr)		(((unsigned) ptr) >> 2) % SYMHASHSIZE

private Symbol Symhtab[SYMHASHSIZE];

private Symbol Symlist = SNIL;

/*
 * insert a symbol
 */
public Symbol Syminsert(name, class, level, hashflg)
Name name;
Sclass class;
Integer level;
Boolean hashflg;
{
	register Hashval hval;
	register Symbol s;

	s = new(Symbol);
	s->s_name = name;
	s->s_class = class;
	s->s_level = level;
	s->s_type = 0;
	s->s_typtr = SNIL;
	s->s_line = Lineno;
	s->s_file = Filename;

	if (hashflg) {
		if (name==NNIL)
			cerror("attempted to hash null name");
		hval = Symhash(name);
		s->s_next = Symhtab[hval];
		Symhtab[hval] = s;
		s->s_xlink = Symlist;
		Symlist = s;
		Strace(2, printf("Syminsert hash "));
	} else {
		Strace(2, printf("Syminsert "));
		s->s_xlink = SNIL;
	}

	Strace(2, Symprint(s));

	return(s);
}

/*
 * Symlookup - lookup symbol by name and flags
 *
 * assumes LIFO hash queues for scoping
 */
public Symbol Symlookup(name, flag, level)
Name name;
Sflag flag;
Integer level;
{
	register Hashval hval;
	register Symbol s;

	hval = Symhash(name);
	s = Symhtab[hval];

	while (s != SNIL
		&& (s->s_level > level
		    || s->s_name != name
		    || ((s->s_flag&(S_STAG|S_MOS)) != flag))) {
			s = s->s_next;
	}

	if (s==SNIL)
		Strace(2, printf("Symlookup %s failed\n", Nstring(name)));
	else {
		Strace(2, printf("Symlookup "));
		Strace(2, Symprint(s));
	}
	return(s);
}

/*
 * Symclear - clear a scoping level out of the symbol table
 *
 * Symlist points to a FIFO linked list of visible symbols.
 */
public void Symclear(level)
Integer level;
{
	register Symbol s, xs, ps;
	register Hashval hval;

	Strace(1, printf("Symclear %d\n", level));
	for (xs=s=Symlist; s != SNIL; ) {
	    if (s->s_level >= level) {
		/* remove from hash list if it has a name */
		if (s->s_name) {
	    		Strace(1, printf("removing "));
	    		Strace(1, Symprint(s));
			hval = Symhash(s->s_name);
			ps = Symhtab[hval];
			if (ps==s) {
				Symhtab[hval] = s->s_next;
				s->s_next = SNIL;
			} else {
				register Boolean symfnd = false;
				while (ps && !symfnd) {
					if (ps->s_next==s) {
						ps->s_next = s->s_next;
						s->s_next = SNIL;
						symfnd = true;
					} else
						ps = ps->s_next;
				}
				if (!symfnd)
					cerror("hash entry missing");
			}
		}
		if (s==Symlist) {
			Symlist = s->s_xlink;
			s->s_xlink = SNIL;
			s = Symlist;
		} else {
			xs->s_xlink = s->s_xlink;
			s->s_xlink = SNIL;
			s = xs->s_xlink;
		}
	    } else {
		/* follow along with xs */
		xs = s;
		s = s->s_xlink;
	    }
	}
}

#ifdef	DEBUG
/*
 * Symprint - print a symbol table entry for debugging
 */
public Symprint(s)
register Symbol s;
{
	String name, type, class, flag;

	if (s != SNIL) {
		name = s->s_name ? s->s_name->n_string : "(null)";
		if ( ( BTYPE(s->s_type)==TSTRTY
		    || BTYPE(s->s_type)==TUNIONTY
		    || BTYPE(s->s_type)==TENUMTY )
		    && ! ( s->s_class==SSTNAME
			|| s->s_class==SUNAME
			|| s->s_class==SENAME )
		    && ! ( s->s_typtr==SNIL )) {
			printf("%s=(%s,",name,xSymclass(s->s_class));
			Stprint(s->s_type, s->s_szptr);
			printf(" %s,%d",Nstring(s->s_typtr->s_name),s->s_level);
		 } else {
			printf("%s=(%s,", name, xSymclass(s->s_class));
			Stprint(s->s_type, s->s_szptr);
			printf(",%d", s->s_level);
		}
		xSymflagpr(s->s_flag);
		printf(")\n");
	} else
		printf("NIL pointer in Symprint\n");
}

public Stprint(type, sznode)
Stype type;
Tnode sznode;
{
	register Stype t;

	for (t=type; ;t=DECREF(t)) {
		if (ISPTR(t)) {
			printf("PTR ");
		} else if (ISFTN(t)) {
			printf("FTN ");
		} else if (ISARY(t)) {
			printf("ARY ");
		} else if (ISFLD(t)) {
			printf("FLD ");
		} else {
			printf("%s", xtypes[t]);
			return;
		}
	}
}

public xSymflagpr(flag)
register Sflag flag;
{
	if (flag&S_STAG) printf(",TAG");
	if (flag&S_FARG) printf(",FARG");
	if (flag&S_MOS) printf(",MOS");
	if (flag&S_NONUNIQ) printf(",NONUNIQ");
}
 
/*
 * Symdump - dump the symbol table LIFO
 */
public Symdump(flag)
Boolean flag;
{
	register Name voidname = NNIL;

	if (!flag)
		voidname = Nlookup("void");
	printf("%d links in overflow\n", _Symdump(Symlist, voidname, 0));
}

/*
 * _Symdump - recursive symbol table dump
 *
 * follows stptr lists
 */
private _Symdump(start, last, level)
Symbol start;
Name last;
Integer level;
{
	register Symbol s;
	register Integer linkct = 0;
	register Integer i;

	for (s=start; s; s=s->s_xlink) {
		if (last != NNIL && s->s_name==last)
			break;
		printf("-> ");
		for (i=level; i > 0; i--)
			printf("\t");
		Symprint(s);
		if ((s->s_class==SSTNAME || s->s_class==SUNAME)
		    && s->s_typtr != SNIL)
			(void) _Symdump(s->s_typtr, NNIL, level+1);
		if (s->s_next != SNIL)
			linkct++;
	}
	return(linkct);
}
#endif	DEBUG

/*
 * Sfixclass - adjust class
 */
public Sclass Sfixclass(class, type, location)
register Sclass class;
register Stype type;
register Slocation location;
{
	register sclass = class;

	if (class==SNULL) {
		if (location==IN_STRUCT)
			class = SMOS;
		else if (location==IN_UNION)
			class = SMOU;
		else if (location==IN_ENUM)
			class = SMOE;
		else if (Globallevel)
			class = SEXTDEF;
		else if (Paramlevel)
			class = SPARAM;
		else
			class = SAUTO;
	}

	if (ISFTN(type)) {
		switch (class) {
		default:
			uerror("function has illegal storage class");
		case SAUTO:
			class = SEXTERN;
		case SEXTERN:
		case SEXTDEF:
		case SFORTRAN:
		case STYPEDEF:
		case SSTATIC:
		case SUFORTRAN:
		case SUSTATIC:
			;
		}
	}

	Strace(2, printf("Sfixclass %s -> %s\n", xSymclass(sclass), xSymclass(class)));

	if ((class & SFIELD) && location != IN_STRUCT)
		uerror("field outside structure");
	else
		return(class);

	switch (class) {
	case SMOS:
		if (location != IN_STRUCT)
			uerror("illegal class (struct)");
		return(class);
	case SMOU:
		if (location != IN_UNION)
			uerror("illegal class (union)");
		return(class);
	case SMOE:
		if (location != IN_ENUM)
			uerror("illegal class (enum)");
		return(class);
	case SREGISTER:
		if (Globallevel)
			uerror("illegal register declaration");
		/*
		 * normally count registers in use and adjust this
		 * class to SPARAM or SAUTO if all spare registers are in use.
		 */
		return(class);
	case SLABEL:
	case SULABEL:
		if (!Funclevel)
			uerror("label outside function");
		return(class);
	case SAUTO:
		if (!Funclevel)
			uerror("auto outside function");
		return(class);
	case SPARAM:
		if (!Paramlevel)
			uerror("param decl misplaced");
		return(class);
	case SUFORTRAN:
	case SFORTRAN:
		if (!ISFTN(type))
			uerror("fortran decl must apply to function");
		else {
			type = DECREF(type);
			if (ISFTN(type) || ISARY(type) || ISPTR(type))
				uerror("fortran function has illegal type");
		}
	case SSTNAME:
	case SUNAME:
	case SENAME:
	case SEXTERN:
	case SSTATIC:
	case SEXTDEF:
	case STYPEDEF:
	case SUSTATIC:
		return(class);

	default:
		cerror("illegal class: %d", class);
	}
}

/*
 * Symdec - declare a symbol
 */
public Tnode Symdec(att, initlist)
register Tnode att;
Tnode initlist;
{
	register Tnode t;

	assert(att->T_op==T_CLASS);
	/*
	 * snarf class and resolve type
	 */
	Strace(2, printf("Symdec %s %s\n", xSymclass(att->cl.class), xTop(initlist)));
	if (att->cl.class==SNULL) {
		if (Funclevel)
			att->cl.class = SAUTO;
		else if (!Globallevel)
			cerror("Symdec called in wrong context");
		else 	/* Global level */
			att->cl.class = SEXTERN;
	}

	Syminst(att->cl.class, att->T_sib, initlist, IN_OTHER);

	/* make a T_VARDEF instance */
	return(Tmkvardef(att, initlist));
}

/*
 * Syminst - install a symbol
 */
public void Syminst(class, typenode, nodes, loc)
Sclass class;
Tnode typenode;
Tnode nodes;
Slocation loc;
{
	register Tnode t, namenode;
	register Symbol s;
	register Stype ttype;
	Stype type;
	Sclass sclass;

	if (nodes != TNIL) {
	   type = TYresolve(typenode);

	   for (t=nodes; t != TNIL; t = t->T_sib) {
		Strace(2, printf("Syminst op=%s\n", xTop(t)));
		namenode = TYmerge(t, type);
		ttype = namenode->T_type;

		/* for functions, fix the class up */
		if (ISFTN(ttype)) {
			if (class==SNULL)
				sclass = SEXTERN;
			else if (class==SSTATIC)
				sclass = SUSTATIC;
			else if (class==SFORTRAN)
				sclass = SUFORTRAN;
			else
				sclass = class;
		} else
			sclass = class;

		s = Symlookup(namenode->nm.name, 0, Curlevel);
		if (s==SNIL) {
			s = Syminsert(namenode->nm.name, sclass, Curlevel, true);
			s->s_szptr = namenode->nm.dimptr;
		}

		Tmksym(namenode, s);
		Symdef(namenode, sclass, ttype, loc);

		/*
		 * connect this symbol to its type symbol.
		 * In the case of multiple types (e.g. unsigned long int)
		 * this only connects the first one.
		 */
		assert(typenode->sym.symptr!=SNIL);
		namenode->sym.symptr->s_typtr = typenode->ty.symptr;
	   }
	}
}

/*
 * Symdef - define a symbol in the symbol table
 */
public void Symdef(node, class, type, location)
register Tnode node;
register Sclass class;
register Stype type;
Slocation location;
{
	register Symbol s;
	Stype stype;		/* type from symbol table */
	Integer slevel;		/* level from symbol table */
	Sclass sclass;		/* class from symbol table */
	Boolean Ssizecmp();

	if (node==TNIL)
		return;

	if (node->T_op != T_SYM)
		cerror("Symdef called with no name");
	s = node->sym.symptr;

	Strace(1, printf("Symdef(%s, ",xSymclass(class)));
	Strace(1, Stprint(type, node->nm.dimptr));
	Strace(1, printf(", %s)\n",xloc[((unsigned)location)]));
	Strace(1, printf("Symdef previous "));
	Strace(1, Symprint(s));

	type = TYfix(type, class, location);
	class = Sfixclass(class, type);

	stype = s->s_type;
	slevel = s->s_level;
	sclass = s->s_class;

	if (s->s_type==TVOID && s->s_class==SNULL)
		goto enter;

	if (s->s_type==TUNDEF || (s->s_flag&S_FARG)) {
		Strace(3, printf(">TUNDEF|FARG level=%d %s\n", Curlevel, xSymclass(class)));
		if (Paramlevel && !(s->s_flag&S_FARG)) {
			switch (class) {
			default:
				if (!(class&SFIELD))
					uerror("declared argument %s is missing",
						Nstring(s->s_name));
			case SMOS:
			case SSTNAME:
			case SMOU:
			case SUNAME:
			case SMOE:
			case SENAME:
			case STYPEDEF:
				;
			}
		} else if (!Paramlevel && (s->s_flag&S_FARG)) {
			/*
			 * new declaration hides a formal parameter:
			 * grab a new symtab entry for it.
			 */
			s = Syminsert(s->s_name, SNULL, Curlevel, true);
		}
		goto enter;
	}

	if (type != stype) {
		Strace(2, printf(">mismatch on type 0%o != 0%o\n",type,stype));
		goto mismatch;
	}

	/*
	 * check dimension declaration
	 */
	if (Ssizecmp(s->s_szptr, node->nm.dimptr))
		goto mismatch;
	s->s_szptr = node->nm.dimptr;

	if (class&SFIELD) {
		/* reallocate field */
		Strace(2, printf(">field class\n"));
		return;	/* always succeed for now */
	} else
		Strace(2, printf(">mismatch switch %s sclass=%s slevel=%d\n",xSymclass(class),xSymclass(sclass),slevel));
		switch (class) {
		case SEXTERN:
			switch (sclass) {
			case SSTATIC:
			case SUSTATIC:
				if (slevel==0)
					return;	/* static at level 0 is extern */
				break;
			case SEXTDEF:
			case SEXTERN:
			case SFORTRAN:
			case SUFORTRAN:
				return;	/* another extern reference */
			}
			break;
		case SSTATIC:
			if (sclass==SUSTATIC || (sclass==SEXTERN && Globallevel)) {
				s->s_class = SSTATIC;
				return;
			}
			break;
		case SUSTATIC:
			if (sclass==SSTATIC || sclass==SUSTATIC)
				return;
			break;
		case SLABEL:
			if (sclass==SULABEL) {
				s->s_class = SLABEL;
				return;
			}
			break;
		case STYPEDEF:
			if (sclass==class)
				return;
			break;
		case SUFORTRAN:
			if (sclass==SUFORTRAN || sclass==SFORTRAN)
				return;
			break;
		case SFORTRAN:
			if (sclass==SUFORTRAN) {
				s->s_class = SFORTRAN;
				return;
			}
			break;
		case SMOU:
		case SMOS:
			if (sclass==class) {
				/* allocate structure/union member */
				return;
			}
			break;
		case SMOE:
			if (sclass==class) {
				/* assign enum value */
			}
			break;
		case SEXTDEF:
			if (sclass==SEXTERN) {
				s->s_class = SEXTDEF;
				return;
			}
			break;
		case SSTNAME:
		case SUNAME:
		case SENAME:
			if (sclass != class)
				break;
			/* check for previous mention */
			return;	/* always succeed for now */
		case SULABEL:
			if (sclass==SLABEL || sclass==SULABEL)
				return;	/* another forward reference to label */
		case SPARAM:
		case SAUTO:
		case SREGISTER:
			;	/* mismatch ... */
		}

   mismatch:

	Strace(2, printf(">mismatch, %s\n", xSymclass(class)));

	if (class==SMOU || class==SMOS || (class&SFIELD)) {
		cerror("Symdef asked to re-define struct mem or field");
	}

	if (Curlevel > slevel && class != SEXTERN && class != SFORTRAN
	    && class != SUFORTRAN && !(class==SLABEL && slevel >= 2)) {
		Strace(1, printf(">hide old symbol %s\n", Nstring(s->s_name)));
		s = Syminsert(s->s_name, SNULL, Curlevel, true);
		node->sym.symptr = s;
		goto enter;
	}

	uerror("redeclaration of %s", Nstring(s->s_name));
	return;

    enter:

	Strace(2, printf(">enter: %s %s %s line=%d\n",Nstring(s->s_name),xSymtype(type),xSymclass(class),Lineno));
	if (type==TUNDEF)
		uerror("void type for %s", Nstring(s->s_name));

	s->s_type = type;
	s->s_class = class;
	if (ISFTN(type))
		s->s_level = 0;	/* make function symbols global */

	Strace(1, printf("Symdef new      "));
	Strace(1, Symprint(s));

}

/*
 * Ssizecmp - compare and resolve dimension nodes
 */
private Boolean Ssizecmp(symnode, decnode)
Tnode symnode, decnode;
{
	if (symnode==TNIL && decnode==TNIL) {
		return(false);
	} else if ((symnode==TNIL && decnode!=TNIL) || (decnode==TNIL && symnode!=TNIL)) {
		cerror("dim nodes of different length");
		return(true);
	} else {
		if (symnode->sz.dim->T_op != T_NULL) {
			if ((decnode->sz.dim->T_op != T_NULL)
			     && (Iconval(symnode->sz.dim) != Iconval(decnode->sz.dim)))
				return(true);
		}
		return(Ssizecmp(symnode->T_sib, decnode->T_sib));
	}
}
