/*
 * xVprint.c - decode and print xV kernel packet
 *
 * Bill Nowicki September 1982
 *
 * Copyright (c) 1982 Stanford University
 *
 * This source code can be copied ONLY if all changes and improvements
 * are promptly sent to the Author.
 *
 * Whenever an xV -> V cataclysm occurs, this file (xvprint.c) should be
 * copied to vprint.c and all occurrences of "xV" changed to "V" (and this
 * comment removed).  Then xvprint.c can track any changes in the xV protocol.
 */

# include <Venviron.h>

# define ENET3MEG 1

# include <Vikc.h>

	/*
	 * The !$^&%#@ include files are incompatible!
	 */
# undef EOF
# undef FILE
# undef getc
# undef putc
# undef getchar
# undef putchar
# define fopen foo
# define freopen bar

# include <stdio.h>

extern int PrintData;

static char *VType(kind)
    unsigned kind;
  {
    static char temp[64];
    
    switch (kind&0xFFFF)
      {
	case remoteSend:		return( "Send");
	case remoteReply:		return( "Reply");
	case remoteForward:		return( "Forward");
	case remoteReceiveSpecific:	return( "Receive Specific");
	case breathOfLife:		return( "Breath Of Life");
	case remoteGetPid:		return( "GetPid");
	case remoteGetPidReply:		return( "GetPid Reply");
	case remoteMoveFromReq:		return( "Move From Request");
	case remoteMoveFromReply:	return( "Move From Reply");
	case remoteMoveToReq:		return( "Move To Request");
	case remoteMoveToReply:		return( "Move To Reply");
	case nAck:			return( "nAck");
	case lhnCollisionCheck:		return( "LHN Collision Check");
	case lhnCollisionReply:		return( "LHN Collision Reply");
      }
    sprintf( temp, "Unknown Packet Type (0x%x)", kind&0xFFFF);
    return ( temp );
  }

static char *Pid(p)
    unsigned p;
  {
    static char temp[64];
    
    sprintf(temp, "%04x-%04x", p >> 16, p & 0xffff);
    return(temp);
  }

static char *LogicalPid(p)
    unsigned p;
  {
    static char temp[128];
  	/*
	 * returns a readable version of a Logical pid
	 */

    int host = (p >> 16) & 0xFFFF;
    int smallPid = p & 0xFFFF;
    
    if (p==0) return("0");
    
    if (host)
      sprintf( temp, "%04x-", host);
    else
      strcpy( temp, "" );

    switch (smallPid)
      {
	case ACTIVE_PROCESS:	strcat(temp,"Active Process"); break;
	case EXCEPTION_SERVER:	strcat(temp,"Exception Server"); break;
	case DEVICE_SERVER:	strcat(temp,"Device Server"); break;
	case INTERNET_SERVER:	strcat(temp,"Internet Server"); break;
	case STORAGE_SERVER:	strcat(temp,"Storage Server"); break;
	case UNIX_SERVER:	strcat(temp,"Unix Server"); break;
	case TIME_SERVER:	strcat(temp,"Time Server"); break;
	case PRINT_SERVER:	strcat(temp,"Print Server"); break;
	case VGT_SERVER:	strcat(temp,"Vgt Server"); break;
	case NAME_SERVER:	strcat(temp,"Name Server"); break;
	case CONTEXT_SERVER:	strcat(temp,"Context Server"); break;
	case TEAM_SERVER:	strcat(temp,"Team Server"); break;
	case PIPE_SERVER:	strcat(temp,"Pipe Server"); break;
	case KERNEL_PROCESS:	strcat(temp,"Kernel Process"); break;
	case SERVICE_SERVER:	strcat(temp,"Service Server"); break;
	case VSTORAGE_SERVER:	strcat(temp,"V Storage Server"); break;
	case VMS_SERVER:	strcat(temp,"VMS Server"); break;
	case LINK_SERVER:	strcat(temp,"Link Server"); break;
	case EXEC_SERVER:	strcat(temp,"Exec Server"); break;
	case XVSTORAGE_SERVER:	strcat(temp,"XV Storage Server"); break;
	case AUTH_SERVER:	strcat(temp,"Authentication Server"); break;
	case SERVICE_SERVER_WATCHDOG:
				strcat(temp,"Service Server Watchdog"); break;
	default:		sprintf( temp, "%s%04x", temp, smallPid  );
      }
    return(temp);
  }


xVPrint( buf, out, count)
  short *buf;
  FILE *out;
  int count;
  {
    register kPacketWithSegment *p = (kPacketWithSegment *)buf;
    char *VType();
    register int i;
    int endian = p->packetType & IKC_LITTLE_ENDIAN;

    if ( DifferentIKCByteOrder(p) )
      {
	SwapIKPacket(p);
      }
    p->packetType &= ~IKC_LITTLE_ENDIAN;/* The bits are just a nuisance now */

    fprintf( out, "xV    %s (%s), sequence no %d\n",
	VType( p->packetType ),
	endian == IKC_LITTLE_ENDIAN ? "little-endian" :
	endian == 0 ? "big-endian" : "middle-endian?",
	p->sequenceNo );

    fprintf( out, "xV source pid %s", Pid( p->srcPid ) );
    fprintf( out, ", destination pid %s", Pid( p->dstPid ) );
    fprintf( out, ", forwarder pid %s\n",
	(p->packetType == remoteGetPid || p->packetType == remoteGetPidReply)
	? LogicalPid( p->forwarder ) : Pid( p->forwarder ) );

    fprintf( out, "xV user no %d, length %d, local addr 0x%x, remote addr 0x%x\n",
	p->userNumber,
	p->length,
	p->localaddress,
	p->remoteaddress );

    if ((p->packetType >= remoteSend) && (p->packetType <= remoteForward))
	fprintf( out, "xV msg %x%04x %x %x %x %x %x %x %x (%s)\n",
            p->msg.sysCode,
	    p->msg.fill & 0xFFFF,
	    p->msg.unspecified[0], 
	    p->msg.unspecified[1],
	    p->msg.unspecified[2],
	    p->msg.unspecified[3],
            p->msg.delivery,
	    p->msg.segmentPtr,
	    p->msg.segmentSize,
	    ErrorString( p->msg.sysCode ) );
    else if (p->packetType == nAck)
	fprintf( out, "xV error %x (%s)\n",
            p->msg.sysCode,
	    ErrorString( p->msg.sysCode ) );
    else if (p->packetType == remoteMoveFromReq ||
    	     p->packetType == remoteMoveToReq)
	fprintf( out, "xV Send sequence no %d\n",
	    p->msg.unspecified[0] );

    if ( count < 64 ) 
	fprintf(out,"xV packet too small, only %d bytes!\n", count);
    count -= 64;
    if ( count > 0 ) 
      {
	fprintf( out, "xV data (%d bytes) ", count );
        for(i=0;i<count;i++) 
	  {
            PrettyPutchar( p->data[i] & 0xFF, out );
            if( (i%16) == 15 )
	      {
		if( !PrintData )
		  {
		    fprintf( out, "..." );
		    break;
		  }
		fprintf( out,"\n");
	      }
	  }
	 if( (i%16) != 0 ) fprintf( out,"\n");
      }
  }
