/*
 * xnsprint.c - decode Xerox Network Systems Internet Datagrams
 *
 * Bill Nowicki October 1983
 *
 * Copyright (c) 1983 Stanford University
 *
 * This source code can be copied ONLY if all changes and improvements
 * are promptly sent to the Author.
 */

# include <stdio.h>
# include "xns.h"

char *PrintAddress(addr)
    unsigned short *addr;
  {
  	/*
	 * prints a 10Mb Ethernet address, returns string
	 */
     static char buf[20];
     
     if (addr[0]==0xFFFF && addr[1]==0xFFFF && addr[2]==0xFFFF)
     	return("Broadcast");
     if (addr[0]==0 && addr[1]==0 && addr[2]==0)
     	return("Unknown");

     sprintf(buf,"%x.%x.%x", addr[0], addr[1], addr[2] );
     return(buf);
  }

char *PrintPupAddress(addr)
    unsigned char addr;
  {
  	/*
	 * prints a 3Mb Ethernet address, returns string
	 */
     static char buf[20];
     
     if (addr==0)
     	return("Unknown");

     sprintf(buf,"0%o", addr);
     return(buf);
  }


char *PrintXNSnet(addr)
    long addr;
  {
  	/*
	 * prints an XNS Network number
	 */
     static char buf[20];
     
     if (addr==0)
     	return("Unknown");
     if (addr==-1)
     	return("Broadcast");

     sprintf(buf,"0%o", addr);
     return(buf);
  }


char *XNStype(type)
    char type;
  {
  	/*
	 * Returns a string representation of the XNS packet type
	 */
    static char buf[128];

    switch (type)
      {
	case 0: 		return("Unknown");
        case RoutingType:	return("Routing");
	case EchoType:		return("Echo");
	case ErrorType:		return("Error");
	case PIType: 		return("Packet Exchange");
	case SPPType: 		return("Sequenced Packet");
	case PupLookupType: 	return("PUP Lookup");
	case -1: return("Broadcast");
      }
    sprintf(buf, "0%o", type & 0xFF );
    return(buf);
  }


char *XNSsocket(socket)
    short socket;
  {
  	/*
	 * Returns a string representation of the XNS socket
	 */
    static char buf[128];

    switch (socket)
      {
	case 0: 		return("Unknown");
        case RoutingSocket:	return("Routing");
	case EchoSocket:	return("Echo");
	case RoutingErrorSocket:return("Routing Error");
	case PupLookupSocket: 	return("PUP Lookup");
	case 040:		return("SGI Graphics");
	case 041:		return("SGI Serial");
	case 042:		return("SGI Load");
	case 043:		return("SGI Status");
	case 044:		return("SGI Login");
	case 050:		return("SGI Name");
	case 051:		return("SGI Boot");
	case 052:		return("SGI File");
	case 053:		return("SGI VRing");
	case 054:		return("SGI Key");
	case 055:		return("SGI Exec");
	case -1: 		return("Broadcast");
      }
    sprintf(buf, "0%o", socket & 0xFFFF );
    return(buf);
  }


XNSPrint( buf, out, count )
  struct XNSpacket *buf;
  FILE *out;
  {
    /*
     * Decode an XNS packet, and print information on the
     * given file about it.
     */
    fprintf(out,"XNS length=%d, type=%s\n", buf->length, 
    	XNStype(buf->type) );

    fprintf(out,"Destination Network %s, Host %s, Socket %s\n",
	PrintXNSnet(buf->destNet),
  	PrintAddress(buf->destHost.addr),
	XNSsocket(buf->destSocket)
      );

    fprintf(out,"Source Network %s, Host %s, Socket %s\n",
	PrintXNSnet(buf->sourceNet),
  	PrintAddress(buf->sourceHost.addr),
	XNSsocket(buf->sourceSocket)
      );
    
    	/*
	 * compare packet and protocol byte count
	 */
    if (count < buf->length)
      fprintf(out,"Physical length is: %d bytes, but header says %d\n",
          count, buf->length);
    
    count = buf->length - (sizeof(struct XNSpacket) - 2);

    switch (buf->type)
     {
        case RoutingType: PrintRouting(buf->data,out,count);		break;
	case EchoType: PrintEcho(buf->data,out,count);			break;
	case ErrorType:	PrintErrorXNS(buf->data,out,count);		break;
	case PIType: PrintPacketExchange(buf->data,out,count);		break;
	case SPPType: PrintSequencedPacket(buf->data,out,count);	break;
        case PupLookupType:  PrintPupLookup(buf->data,out,count);	break;

        default:
          if (count>0)
            {
		extern int PrintData;
		int i;
		
		count /= 2;

	        fprintf( out, "Data (hex): ");
		for (i=0; i<count && (i<32 || PrintData); i++)
	  	    fprintf( out, "%x", buf->data[i]);
		if (i<count) fprintf(out,"...");
        	fprintf( out, "\n");
	   }
      }
  }

static PrintRouting(buf,out,count)
    unsigned short *buf;
    FILE *out;
  {
  	/*
	 * Printout routing Packets
	 */
   switch (*buf++)
      {
        case RoutingRequest: 	fprintf(out,"Request\n");	break;
        case RoutingResponse: 	fprintf(out,"Response\n");	break;
      }

    count -= 2;

    while (count>0)
     {
        fprintf( out, "Network: %s", PrintXNSnet( *(long *)(buf) ) );
	buf += 2;
       	fprintf( out, " Delay: %d\n", *buf++ );
	count -= 6;
     }
   }


static PrintEcho(buf,out,count)
    unsigned short *buf;
    FILE *out;
  {
  	/*
	 * Printout XNS Echo Packets
	 */

   switch (*buf++)
      {
        case EchoRequest: 	fprintf(out,"Request\n");	break;
        case EchoResponse: 	fprintf(out,"Response\n");	break;
      }

    count -= 2;

    if (count>0)
      {
	extern int PrintData;
	int i;
	
        fprintf( out, "Data (hex): ");
	for (i=0; i<count && (i<32 || PrintData); i += 2)
  	    fprintf( out, "%x", *buf++);
	if (i<count) fprintf(out,"...");
       	fprintf( out, "\n");
       }
   }


static PrintErrorXNS(buf,out,count)
    unsigned short *buf;
    FILE *out;
  {
  	/*
	 * Printout XNS Error Packets
	 */

    switch (*buf)
      {
        case 0:  	fprintf(out,"Destination Error");		break;
        case 1:  	fprintf(out,"Checksum Error");			break;
        case 2:  	fprintf(out,"No such socket");			break;
        case 3:  	fprintf(out,"No Resources");			break;
        case 01000:  	fprintf(out,"Before Destination Error");	break;
        case 01001:  	fprintf(out,"Checksum Before Destination");	break;
        case 01002:  	fprintf(out,"Destination Unreachable");		break;
        case 01003:  	fprintf(out,"Too many hops");			break;
        case 01004:  	fprintf(out,"Packet too large");		break;
        default: 	fprintf(out,"Error number 0%o,");
      }
    
    buf++;
    fprintf(out," Parameter 0%o\n", *buf++);
    count -= 4;

    if (count>0)
      {
	extern int PrintData;
	int i;
		
        fprintf( out, "Data (hex): ");
	for (i=0; i<count && (i<32 || PrintData); i += 2)
  	    fprintf( out, "%x", *buf++);
	if (i<count) fprintf(out,"...");
       	fprintf( out, "\n");
     }
   }


static PrintPacketExchange(buf,out,count)
    unsigned short *buf;
    FILE *out;
  {
  	/*
	 * Printout XNS Packet Exchange Packets
	 */
    long id;
    
    id = (*buf++) << 16;
    id |= *buf++;
    fprintf(out,"ID=0x%x Client=0x%x", id, *buf++);
    count -= 6;
    if (count>0)
      {
	extern int PrintData;
	int i;
		

        fprintf( out, "Data (hex): ");
	for (i=0; i<count && (i<32 || PrintData); i += 2)
  	    fprintf( out, "%x", *buf++);
	if (i<count) fprintf(out,"...");
       	fprintf( out, "\n");
      }
   }

static PrintSequencedPacket(buf,out,count)
    unsigned short *buf;
    FILE *out;
  {
  	/*
	 * Printout XNS Sequenced Packets
	 */
    
    fprintf(out,"DataStream type=%d. ", *buf & 0xFF);
    if (*buf & 0x8000) fprintf(out,"System ");
    if (*buf & 0x4000) fprintf(out,"Send ACK ");
    if (*buf & 0x2000) fprintf(out,"Attention ");
    if (*buf & 0x1000) fprintf(out,"EOM ");
    fprintf(out,"\n");
    buf++; count -= 2;
    
    fprintf(out,"Source ID=0x%x ", *buf++); count -= 2;
    fprintf(out,"Destination ID=0x%x\n", *buf++); count -= 2;
    fprintf(out,"Sequence number=0x%x ", *buf++); count -= 2;
    fprintf(out,"ACK=0x%x ", *buf++); count -= 2;
    fprintf(out,"Allocation=0x%x\n", *buf++); count -= 2;

    if (count>0)
      {
	extern int PrintData;
	int i;
		
        fprintf( out, "Data (hex): ");
	for (i=0; i<count && (i<32 || PrintData); i++)
  	    fprintf( out, "%x", *buf++);
	if (i<count) fprintf(out,"...");
       	fprintf( out, "\n");
      }
   }




PrintPupLookup(buf,out,count)
    unsigned short *buf;
    FILE *out;
  {
  	/*
	 * Printout the XIP Pup Lookup Packets
	 */
    long id;
    char *p;
    
    id = (*buf++) << 16;
    id |= *buf++;
    fprintf(out,"ID=0x%x ", id);
    switch (*buf)
      {
        case XipRequest: 
		fprintf(out,"XIP Request\n");
		buf++;
		fprintf(out,"XNS Address is %s\n",
			PrintAddress(buf) );
		break;

        case XipResponse: 
		fprintf(out,"XIP Response\n");
		buf++;
		fprintf(out,"PUP Address is %s\n",
			PrintPupAddress(*buf >> 8) );
		break;

        case XipError: 
		fprintf(out,"XIP Error: ");
		buf++;
		p = (char *)buf;
		while (count-->0)
		  {
		    PrettyPutchar( *p++, out);
		  }
		fprintf(out,"\n");
		break;
      }
  }


PupToNSPrint( buf, out, count)
    struct HtPacket *buf;
    FILE *out;
  {
    fprintf(out,"Pup to XNS Translation ");
    HtPrint(buf,out,count);
  }


NSToPupPrint( buf, out, count)
    struct HtPacket *buf;
    FILE *out;
  {
    fprintf(out,"XNS to PUP Translation ");
    HtPrint(buf,out,count);
  }


HtPrint( buf, out, count)
    struct HtPacket *buf;
    FILE *out;
  {
  	/*
	 * Prints out the Xerox Hardware Translation packets
	 */
    switch (buf->operation)
      {
        case HtRequest: fprintf(out, "Request\n"); break;
        case HtReply: fprintf(out, "Reply\n"); break;
        default: fprintf(out, "Operation %d", buf->operation); break;
      }
    fprintf(out,"Desired XNS Address=%s, Desired PUP Address=%s\n",
      PrintAddress(buf->desiredXNS.addr), 
      PrintPupAddress(buf->desiredPUP));
    fprintf(out,"Sender's XNS Address=%s, Sender's PUP Address=%s\n",
      PrintAddress(buf->sendersXNS.addr), 
      PrintPupAddress(buf->sendersPUP));
  }


SgDiagPrint(buf, out, count)
    short *buf;
    FILE *out;
  {
  	/*
	 * Print the special SGI Diagnostic kludge
	 */
    register char *p;

    fprintf(out,"SGI Diagnostic ");

    p = (char *)buf;
    while (count-->0)
      {
	PrettyPutchar( *p++, out);
      }
    fprintf(out,"\n");
  }


SgGamePrint(buf, out, count)
    short *buf;
    FILE *out;
  {
  	/*
	 * Print the special SGI Game kludge
	 */
    register char *p;

    fprintf(out,"SGI Game ");

    p = (char *)buf;
    while (count-->0)
      {
	PrettyPutchar( *p++, out);
      }
    fprintf(out,"\n");
  }

SgBouncePrint(buf, out, count)
    short *buf;
    FILE *out;
  {
  	/*
	 * Print the special SGI "bounce" (?) kludge
	 */
    register char *p;

    fprintf(out,"SGI Bounce ID=%d ", *buf++ & 0xFFFF);

    p = (char *)buf;
    count -= 2;
    while (count-->0)
      {
        if (*p==0) break;
	PrettyPutchar( *p++, out);
      }
    fprintf(out,"\n");
  }
