/*
 * Netwatch - a network monitoring program.
 *
 * Bill Nowicki September 1983
 *
 * Copyright (c) 1983 Stanford University, all rights reserved.
 *
 * This is the 10Mbit version, which can be linked to
 * any of the available drivers, such as ExceLan or 3Com.
 * This module has stuff which is dependent on the 10Mbit net,
 * but independent on the controller board used.
 */

#define ENET10MEG

# include <Venviron.h>
# include <Vikc.h>
# include <Vio.h>
# include "ip.h"
# include "xns.h"
# include "dec.h"
# include "chaos.h"
# include <pup/puplib.h>
# include <pup/pupconstants.h>
# include <pup/puppacket.h>

# ifndef PUP
# define PUP ETHERNET_TYPE_PUP
# endif PUP

#define BUFFERS 128	/* number of Network buffers */
#define SIZE 1600	/* size in bytes of each buffer */
int MaxBuffers = BUFFERS;
int BufferSize = SIZE;

#define FILTER_WORDS (0x10000 / sizeof(unsigned long))
#define Word(addr)(addr>>5)
#define Bit(addr)(1 << (addr & 0x1f))
#define InFilter(filter,addr)(filter[Word(addr)] & Bit(addr))
#define FLAG( state, flag, meaning ) \
	printf("  %s %c %s\n", (state) ? " ON" : "OFF", flag, meaning);

char *PrintAddress();
char userName[64];
char cwd[128];
short myAddress[3];
/*
 * Within each buffer, things are organized as follows:
 *
 * Shorts 0, 1		Time in ms. from start
 * Short 2		Status word
 * Short 3		Length in bytes
 * Shorts 4-6		Ethernet Source
 * Shorts 7-9		Ethernet Destination
 * Short 10		Ethernet packet type
 * Short 11-end		Rest of packet
 */
char Buffers[BUFFERS][SIZE];
unsigned long Filter1[FILTER_WORDS];
unsigned long Filter2[FILTER_WORDS];
int MultiFilter1 = 0;
int MultiFilter2 = 0;

int FirstBuffer = 0;		/* index of firrst buffer */
int LastBuffer = 0;		/* index of next usable buffer */

int TickFlag = 1;		/* print exclamation on each packet */
int PupFlag = 0;		/* listen to PUPs */
int IPFlag = 0;			/* listen to IPs */
int VFlag = 1;			/* listen to V kernel packets */
int XNSFlag = 0;		/* listen to XNS packets */
int CHAOSFlag = 0;		/* listen to ChaosNet packets */
int ARPFlag = 0;		/* listen to ARP and RARP packets */
int OtherFlag = 0;		/* listen to other random types */
int PrintData = 0;		/* do not print data in packets */
int StartTime = 0;		/* K_ticks when we started */
int HitFlag;			/* a key was hit */
int Annoying = 0;		/* Always print option list */

char *Buffer(i)
  {
    return(Buffers[i]);
  }

int FilterOK( dest, source )
unsigned short source[], dest[];
  {
    register unsigned short s = source[2];
    register unsigned short d = dest[2];

    if (dest[0] & 0x0100)
	if ( ( InFilter(Filter1,s) && MultiFilter2 ) ||
	     ( InFilter(Filter2,s) && MultiFilter1 ) )
	    return( 1 );
	else
	    return( 0 );

    if ( ( InFilter(Filter1,s) && InFilter(Filter2,d) ) ||
         ( InFilter(Filter1,d) && InFilter(Filter2,s) ) )
        return( 1 );
    return( 0 );
  }

ReadPacket()
  {
    /*
     * Read a packet into the next buffer.
     * returns true if we one matched the filter.
     */
     
    register short *buf;
    register short *packet;
    short *EtherPeek();

    while (!EtherReady())
        if (KeyHit()) return(0);

    buf = (short *)Buffers[LastBuffer];
    *(int *)buf = K_ticks() - StartTime;
    buf += 2;

    /* Try to peek at the packet without reading it; 
     *  returns NULL if not possible.
     */
    packet = EtherPeek();
    if (packet == NULL)
      {
        EtherRead( buf );
	packet = buf + 2;	/* skip over interface status */
      }

    /*
     * check that the sending host is in our software filter
     */
    if ( !FilterOK(packet, packet+3) ) goto Quit;
    packet += 6;

    /*
     * now we have read the packet, throw away Ethernet types
     * that we do not want.
     */
    switch (*packet & 0xFFFF)
      {
        case NSToPup:
        case PupToNS:
        case PUP:
	    if (!PupFlag) goto Quit;
	    break;

	case ARP_TYPE:
	case RARP:
	    if (!ARPFlag) goto Quit;
	    break;

        case IP:
	    if (!IPFlag) goto Quit;
	    break;

	case SG_DIAG:
	case SG_GAMES:
	case SG_BOUNCE:
        case XNS:
	    if (!XNSFlag) goto Quit;
	    break;

	case V_KERNEL_PACKET:
	case XV_KERNEL_PACKET:
	    if (!VFlag)	goto Quit;
	    break;

	case CHAOS:
	    if (!CHAOSFlag) goto Quit;
	    break;

	default:
	    if (!OtherFlag) goto Quit;
	    break;
      }

    /* Read the rest of the packet if we haven't done so already */
    if (packet != buf + 8) EtherRead(buf);

    LastBuffer = (LastBuffer + 1) % MaxBuffers;
    if( LastBuffer == FirstBuffer )
	FirstBuffer = (FirstBuffer + 1) % MaxBuffers;

    return(1);

Quit:
    EtherDrop();
    return(0);
  }



PrintBuffer( buf, out )
 register short *buf;
  {
    /*
     * Decode and print the buffer indicated on the
     * given file.
     */
    short rxStatus;
    short *source, *dest;
    int i;
    int count;  /* Number of bytes in interesting portion of packet; */
		/*   used later on as a word-count.		     */
    
    fprintf( out, "\n%d.%02ds ",
    	(*(int *)buf)/1000, ( (*(int *)buf)/10)%100 );
    buf += 2;
    rxStatus = *buf++;
    count = *buf++;
    dest = buf;
    buf += 3; count -= 6;
    source = buf;
    buf += 3; count -= 6;

    fprintf( out, "[%s ", PrintAddress(source) );
    fprintf( out, "--> %s]\n", PrintAddress(dest) );

    if ( EtherBadStatus(rxStatus) )
    	fprintf( out, "Ethernet status: %s", PrintEtherStatus(rxStatus) );

    switch (*buf & 0xFFFF)
      {
        case PUP:	PupPrint( ++buf, out, count);		break;
	case ARP_TYPE:
	case RARP:	ARPPrint( ++buf, out, count);		break;
        case IP:	IPPrint(  ++buf, out, count);		break;
	case SG_DIAG:	SgDiagPrint(++buf, out, count);		break;
	case SG_GAMES:	SgGamePrint(++buf, out, count);		break;
	case SG_BOUNCE: SgBouncePrint(++buf, out, count);	break;
        case XNS:	XNSPrint( ++buf, out, count);		break;
	case CHAOS:	CHAOSPrint(++buf, out, count);		break;
        case NSToPup:	NSToPupPrint( ++buf, out, count);	break;
        case PupToNS:	PupToNSPrint( ++buf, out, count);	break;
	case V_KERNEL_PACKET:
			VPrint(  ++buf, out, count);		break;
	case XV_KERNEL_PACKET:
			xVPrint(  ++buf, out, count);		break;
	default:	
	    if (*buf == DECLOOP) fprintf(out,"DEC LoopBack Packet");
	    else fprintf( out, "Ether Type %04x Packet", *buf&0xFFFF );
	    buf++;

	    if( count == 0 ) fprintf( out, "-- No Ether Data!\n" );
	    else
	      {
		fprintf( out, "(%d bytes)", count );
	        count = (count + 1) / 2;  /* Now it's a word count */
		for( i=0; i<count; ++i )
		  {
		    fprintf( out, " %04x", *buf++ & 0xFFFF );
		    if( i % 16 == 15 )
		      {
			if( !PrintData )
			  {
			    fprintf( out, "..." );
			    break;
			  }
			fprintf( out, "\n" );
		      }
		  }
		if( i % 16 != 0 ) fprintf( out, "\n" );
	      }
      }
  }

PrintSpecifications()
  {
    printf("\nFilter specification\n");
    printf("      a Add a primary host\n");
    printf("      s Make a host secondary (and not primary)\n");
    printf("      i Ignore all packets to/from a host\n");
    printf("      1 Make multicast primary\n");
    printf("      2 Make multicast secondary\n");
    printf("      0 Ignore multicast\n");
    printf("      o Only hear packets between primary hosts\n");
    printf("      e Hear packets between primary hosts and Everybody\n");
    printf("      c Clear primary list\n");
    printf("      n Listen to Nobody\n");
    printf("      * Promiscuous (listen to all packets)\n");
    printf("      p Print out filters\n");
    printf("return or space - Go back to main menu\n");
  }

PrintFilter( filter )
unsigned long filter[];
  {
    register int except; /* 1 = print entries NOT in filter */
    register unsigned long bit, word, entry;
    register unsigned printed = 0;

    bit = 0;
    for (word = 0; word < 10; word++) if (filter[word] == 0xffffffff) bit++;
    if (bit > 3)
      {
	except = 1;
	printf("all hosts");
      }
    else except = 0;
    
    for (word = 0; word < FILTER_WORDS; word++)
      {
	entry = filter[word];
	/* We can skip this word if there is nothing interesting in it. */
	if ( (except) ? (entry != 0xffffffff) : (entry != 0) )
	    for (bit = 0; bit < 32; bit++)
		if (except)
		  {
		    if ( !(entry & Bit(bit)) )
		      {
			if (!printed++)
			    printf(" except ");
		        printf("%x ", (word<<5) + bit);
		      }
		  }
		else
		    if ( entry & Bit(bit) )
		      {
		        printf("%x ", (word << 5) + bit);
			printed++;
		      }
      }
    if (!except && !printed)
        printf("none");
  }

HostSpecify()
  {
	/*
	 * specify hosts to watch
	 */
    register short host;
    int inputHost;
    register char c;

    while (1)
      {
	if (Annoying)
	  {
	    PrintSpecifications();
	    printf("Enter suboption: ");
	  }
	else
	    printf("Host suboption (? for list): ");
	switch( c = getchar() )
	  {
	    case 'a':
	    case 's':
	        printf("\nAdd host (HEX):");
		Echo(); Cooked();
		scanf( "%x", &inputHost );
		NoEcho(); Raw();
		if (c == 'a')
		    Filter1[Word(inputHost)] |= Bit(inputHost);
		else
		    Filter1[Word(inputHost)] &= ~Bit(inputHost);
        	Filter2[Word(inputHost)] |= Bit(inputHost);
		break;		
		
	    case '1':
	    case '2':
	        printf("\nAdding multicast\n");
		if (c == '1')
		    MultiFilter1 = 1;
		else
		    MultiFilter1 = 0;
		MultiFilter2 = 1;
		break;

  	    case '*':
	    case 'e':
	        if ( c == '*' ) printf("\nWill hear all packets\n");
		else printf("\nWill hear between primaries and any host\n");
    		for (host=0; host < FILTER_WORDS; host++)
		  {
        	    if ( c == '*') Filter1[host] = 0xffffffff;
		    Filter2[host] = 0xffffffff;
		  }
		if ( c == '*' )
		    MultiFilter1 = 1;
		MultiFilter2 = 1;
		break;
		
	    case 'n':
	    case 'c':
	        if ( c == 'n' )
		  {
		    printf("\nClearing all lists\n");
		    MultiFilter2 = 0;
		  }
		else printf("\nClearing the primary list\n");
    		for (host=0; host < FILTER_WORDS ; host++)
		  {
        	    Filter1[host] = 0;
		    if ( c == 'n') Filter2[host] = 0;
		  }
		MultiFilter1 = 0;
		break;
		
	    case 'o':
    		for (host=0; host < FILTER_WORDS ; host++)
		    Filter2[host] = Filter1[host];
		MultiFilter2 = MultiFilter1;
		break;
		
	    case 'i':
	        printf("\nIgnore host (HEX):");
	        fflush(stdin);
		Echo(); Cooked();
		scanf( "%x", &inputHost );
		NoEcho(); Raw();
        	Filter1[Word(inputHost)] &= ~Bit(inputHost);
        	Filter2[Word(inputHost)] &= ~Bit(inputHost);
		break;		
		
	    case '0':
	    	printf("\nIgnore multicast\n");
		MultiFilter1 = 0;
		MultiFilter2 = 0;
		break;

	    case 'p':
		printf("\nHear packets between ");
		if ( MultiFilter1 )
		    printf("multicast + ");
		PrintFilter( Filter1 );
		printf("\nand ");
		if ( MultiFilter2 )
		    printf("multicast + ");
		PrintFilter( Filter2 );
		printf("\n");
		break;
	    
	    case '?':
		printf("\n");
		PrintSpecifications();
		break;

	    case ' ':
	    case '\n':
	    case '\r':
	        printf( "\nBack to main menu\n");
	    	return;

    
	    default:
	        printf("\nUnknown option!\n");
		break;
	  }
      }
    
  }

PrintOptions()
  {
	 printf("\n%s Netwatch [ local address %s ]\n",
		EtherKind(), PrintAddress(myAddress) );
	FLAG(ARPFlag,	'a', "ARP packets");
	FLAG(CHAOSFlag,	'k', "Chaos packets");
	FLAG(IPFlag,	'i', "IP packets");
	FLAG(PupFlag,	'p', "Pup packets");
	FLAG(VFlag,	'v', "V kernel packets");
	FLAG(XNSFlag,	'x', "XNS packets");
	FLAG(OtherFlag,	'o', "Other packets");
	FLAG(PrintData,	'd', "Print all packet data");
	FLAG(TickFlag,	'!', "Report ! when packets are received");
	FLAG(Annoying,	'm', "Print annoying option menus all the time");
	printf("      t Type incoming packets (^S to pause)\n");
        printf("      r Record incoming packets to buffer\n");
	printf("      b Display buffer (^S to pause)\n");
	printf("      s Display buffer, skipping initial packets\n");
	printf("      h Modify host filter\n");
	printf("      l Login to network\n");
	printf("      c Change directory\n");
	printf("      w Write buffer to a file\n");
	printf("      q Quit\n");
  }

InitFilter()
  {
    register unsigned long host;
    for (host=0; host < FILTER_WORDS; host++)
      {
        Filter1[host] = 0;
	Filter2[host] = 0xffffffff;
      }
    MultiFilter1 = 0;
    MultiFilter2 = 1;
  }

#define TOGGLE(flag,string) \
flag = !flag; \
printf("\n%s %s packets\n", ((flag) ? "Print" : "Ignore"), string);

main()
  {
    int	j;

    NoEcho(); Raw();			/* no input buffering */
    
    strcpy(userName, "unknown");
    cwd[0] = 0;

    EnetPowerup();		/* initialize V simulator */

    EtherInit(myAddress);
    
    InitFilter();

    putchar('\f');
    PrintOptions();

    while (1)
      {
 	if (Annoying)
	  {
	    PrintOptions();
	    printf("Enter option: ");
	  }
	else printf("\nOption (? for list): ");
	switch (getchar())
	 {
	  case 'a': TOGGLE(ARPFlag, "ARP"); break;
	  case 'k': TOGGLE(CHAOSFlag, "Chaos"); break;
	  case 'i': TOGGLE(IPFlag, "IP"); break;
	  case 'p': TOGGLE(PupFlag, "PUP"); break;
	  case 'v': TOGGLE(VFlag, "V kernel"); break;
	  case 'x': TOGGLE(XNSFlag, "XNS"); break;
	  case 'o': TOGGLE(OtherFlag, "other"); break;
	    
	  case 'd': PrintData = !PrintData; break;
	  case '!': TickFlag = !TickFlag; break;
	  case 'm': Annoying = !Annoying; printf("\n"); break;

	  case 't': netwatch(1); break;
	  case 'r': netwatch(0); break;

	  case 'b': showbuffer(0); break;
	  case 's': showbuffer(1); break;

	  case 'h': printf("\n"); HostSpecify(); break;

	  case 'l': login(); break;
	  case 'c': cd(); break;
	  case 'w': WriteFile(); break;

	  case 'q': printf("\nQuit the program\n"); return;
	  case '?': PrintOptions(); break;

	  case ' ': case '\n': case '\r': break;

	  default: printf( "\nIllegal option!\n"); break;
	 }
      }
  }

cd()
  {
    char dir[128];

    Cooked(); Echo();
    printf("\nDirectory (user: %s cwd: %s): ", userName, cwd[0]?cwd:"<none>");
    gets(dir);
    if (dir[0])
	if (dir[0] != '[')
	    printf("Directories must start with a '['.\n");
	else
	    strcpy(cwd, dir);
    NoEcho(); Raw();
  }

ARPPrint( buf, out, count )
  short *buf;
  FILE *out;
  {
    /*
     * Decode an Address Resolution Protocol packet, 
     * as described in RFC 826
     */
    int hln, pln;
    short type;

    /* semi-kludge: to distinguish ARP from reverse ARP (RARP), look at the
     * previous location for the ethertype.
     */
    type = *(buf-1) & 0xFFFF;

    fprintf( out, (type == RARP) ? "Reverse ARP" : "ARP" );

    fprintf(out,"   hardware: ");
    switch (*buf)
      {
        case 1:	fprintf(out,"Ethernet  ");	break;
	default: fprintf(out,"Type %d  ", *buf & 0xFFFF);
      }
     buf++;
    
    fprintf(out,"software: ");
    switch (*buf)
      {
        case CHAOS:	fprintf(out,"Chaos  ");		break;
        case IP:	fprintf(out,"IP  ");	break;
	default: fprintf(out,"Type 0x%04x ", *buf & 0xFFFF);
      }
     buf++;
    
     hln = (*buf >> 8) & 0xFF;
     pln = *buf & 0xFF;
     buf++;
     fprintf(out,"operation: ");

     if (type == ARP_TYPE)
        switch (*buf)
          {
            case 1: fprintf(out,"Request "); break;
	    case 2: fprintf(out,"Reply "); break;
	    default: fprintf(out,"%d ", *buf & 0xFFFF);
          }
     else /* RARP */
        switch (*buf)
          {
            case 3: fprintf(out,"Request Reverse "); break;
	    case 4: fprintf(out,"Reply Reverse"); break;
	    default: fprintf(out,"%d ", *buf & 0xFFFF);
          }
     buf++;
 
     fprintf(out,"\nsource hardware address: %s ",PrintAddress(buf));
     buf += (hln/2);

     fprintf(out,", software address: ");
     if (pln==4) IPaddrPrint( GetInt(buf), out );
     else PrintBytes(out,buf,pln);
     buf += (pln/2);

     fprintf(out,"\ntarget hardware address: %s", PrintAddress(buf));
     buf += (hln/2);

     fprintf(out,", software address: ");
     if (pln==4) IPaddrPrint( GetInt(buf), out );
     else PrintBytes(out,buf,pln);
     fprintf(out,"\n");
  }


PrettyPutchar (ch, fp)
register unsigned char ch;
FILE *fp;
  {
    fprintf (fp, ((ch>=0x20) && (ch<=0x7E)) ? "%c" : "`%02x", ch);
  }
