/*
 * V  - Copyright (c) 1982 by David Cheriton, Tim Mann
 * All rights reserved.
 *
 * Team file loading function
 * Hacked up for use in boot loader.  TPM 11/11/83
 */

#include <Vioprotocol.h>
#include <Vprocess.h>
#include <b.out.h>

extern File *Open();

#define TRUE 1
#define FALSE 0
#define NUL '\0'
#define BUFFERSIZE 1024

SystemCode LoadFile( filename, address, header, zerobss )
    char *filename, *address;
    struct bhdr *header;
    int zerobss;
  /*
   * Load a program file into memory.
   *
   * The file must be in b.out format. The b.out header is checked for a 
   *   valid "magic number."
   * The file is loaded beginning at address, and the bss segment is zeroed
   *   if zerobss is true.
   * Returns OK on success, possibly useful system error codes on failure.
   *   This is incompatible with program-loading routines which run on top of
   *   a real kernel - they return the teampid, or NULL if errors occur, and
   *   use a separate parameter to return error codes.
   * Program layout:
   *  |text|data|uninitialized data(bss)|stack space|
   */

  {
    File *teamfile;
    int  rsize, bytes, blocksize;

    register char *bssptr, *readend, *loadptr, *p;
    SystemCode ercode = OK;

#ifdef DEBUG
    puts("Open File\n");
#endif
    teamfile = Open(filename, FREAD | FBLOCK_MODE, &ercode);
    if (ercode != OK)
	return (ercode);
    if (teamfile == NULL) /* Don't see how this could happen, but play safe */
	return (NO_MEMORY);  /* Maybe */

#ifdef DEBUG
    puts("Read header\n");
#endif

    bytes = 0;
    blocksize = teamfile->blocksize;

    /* Read b.out header.  Must read max(sizeof(struct bhdr)), blocksize); */
    /*   we need the former here, and the routines below assume we've read */
    /*   at least a whole block.					   */
    rsize = Read(teamfile, address, 
	(blocksize > sizeof(struct bhdr) ? blocksize : sizeof(struct bhdr)) );

    if( rsize < sizeof(struct bhdr) )
      {
#ifdef VAX
#ifdef DEBUG
	printf("Read failed on header; rsize=%d, lastexception=0x%x\n",
		rsize, teamfile->lastexception);
#endif DEBUG
#endif
	Close(teamfile);
	return (teamfile->lastexception);
      }

    /* Copy b.out header to header buffer */
    *header = *(struct bhdr *) address;
#ifdef VAX
#ifdef DEBUG
    printf("Magic #\tText\tData\tBss\tSymbols\tEntry\n");
    printf("0%o\t0x%x\t0x%x\t0x%x\t0x%x\t0x%x\n",
		header->fmagic,header->tsize,header->dsize,
		header->bsize ,header->ssize,header->entry );
#endif
#endif
    /* Check the header has a legal magic number */
    if (N_BADMAG(*header))
      {
	Close(teamfile);
	return (BAD_STATE);
      }

#ifdef VAX
    /* If OMAGIC or NMAGIC, we have to ignore the first 32 bytes; if ZMAGIC,*/
    /*    the first 1K.  If the first interesting data is on a block        */
    /*    boundary, we don't do anything here except set the block number.  */
      {
	unsigned blocknumber, blockoffset;

	blocknumber = N_TXTOFF(*header) / blocksize;
	blockoffset = N_TXTOFF(*header) % blocksize;
        if ( blockoffset != 0 )
	  {
	    if ( blocknumber != teamfile->block )
	      {
		teamfile->block = blocknumber;
		rsize = Read(teamfile, address, blocksize);
		if ( rsize < header->tsize + header->dsize ) 
		  /* Good enough - could add ssize, rtsize, rdsize too. */
		  /* Can't expect blocksize, since may be a small file. */
		  {
		    Close(teamfile);
		    return(teamfile->lastexception);
		  }
	      }
	    p = address + blockoffset;
	    for (loadptr = address;
		 loadptr < address + blocksize - blockoffset; )
		*loadptr++ = *p++;
	    teamfile->block++;
	  }
	else
	  {
	    teamfile->block = blocknumber;
	    loadptr = address;
	  }
      }
#else VAX
#ifdef MC68000
    /* Currently we only use fmagic = FMAGIC, so text always starts at 32, */
    /*   and we assume blocksize >= 32.  If this ever changes, the	   */
    /*   following will have to be revised.  We could just use the VAX     */
    /*   code above, but size is critical in the Sun versions of Vload.	   */
    p = address + sizeof(struct bhdr);
    for (loadptr = address; 
	 loadptr < address + blocksize - sizeof(struct bhdr); )
	*loadptr++ = *p++;
    teamfile->block++;
#else  MC68000
--- bad buildfile; no machine-type specified
#endif MC68000
#endif VAX

    /* Calculate the required sizes and pointers */
    readend = address + header->tsize + header->dsize;

#ifdef DEBUG
    puts("Read rest of file.\n");
#endif DEBUG
#ifdef SLOWLOAD
    while( loadptr < readend )
      {
	if ( ( bytes = (int) (readend - loadptr) ) > blocksize )
	    bytes = blocksize;

	if( (rsize = Read(teamfile, loadptr, bytes)) < bytes )
	  {
#ifdef VAX
#ifdef DEBUG
	    printf(
	"Slow read failed; loadptr=0x%x, rsize=%d (<%d), lastexception=0x%x\n",
		   (unsigned)loadptr, rsize, bytes, teamfile->lastexception);
#endif DEBUG
#endif
	    Close(teamfile);
	    return (teamfile->lastexception);
	  }
#ifdef VAX
#ifdef DEBUG
	printf("  Returned from Read, loadptr (before) = 0x%x, bytes = %d.\n",
		(int)loadptr, rsize);
#endif DEBUG
#endif
	loadptr += rsize;
	teamfile->block++; /* move to next file block */
      }
#else SLOWLOAD
    /* Fast team load in one giant Read */
     {
	bytes = readend - loadptr;
	if( (rsize = Read(teamfile, loadptr, bytes)) < bytes )
	  {
#ifdef DEBUG
	    printf("Read failed; rsize= %d (< %d), lastexception=0x%x\n",
		   rsize, bytes, teamfile->lastexception);
#endif DEBUG
	    Close(teamfile);
	    return (teamfile->lastexception);
	  }

	loadptr = readend;
      }
#endif SLOWLOAD
#ifdef DEBUG
    printf("Close file.\n");
#endif DEBUG
    Close(teamfile);

    /* If fmagic is NMAGIC (0410 on VAX or 68000), we must move the data   */
    /*   section up to the next SEGSIZE boundary (1K on VAX, 32K on 68000) */
    /*   and jigger the start of bss appropriately.			   */
    /* We assume that "address" is a multiple of SEGSIZE.		   */
    /* In practice, we don't use NMAGIC at present on the 68000, so we	   */
    /*   #ifdef this out as another sacrifice to the great god Space.	   */
#ifndef MC68000
    if ( header->fmagic == NMAGIC )
      {
	unsigned offset = (- header->tsize) & (SEGSIZE-1); /* Poor man's % */
	if ( offset )
	  {
	    p = readend + offset; /* and loadptr == readend */
	    while ( loadptr > address + header->tsize )
	      *--p = *--loadptr;
	  }
	bssptr = readend + offset;
      }
    else
#endif  MC68000
	bssptr = readend;

    /* Now zero the bss area. */
    if (zerobss)
      {
#ifdef DEBUG
	puts("Zero bss segment\n");
#endif DEBUG
	loadptr = bssptr + header->bsize;
	while ( bssptr < loadptr ) *bssptr++ = 0;
      }
    return (OK);
  }
