/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1984			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: draw1.c
 *
 * This file (and draw2.c) contains the low-level routines that interface with
 * the graphics libraries.
 */

/*
 * the ifdefs in this file look horrible, because a diff -D merge was done.
 */

#include <Vioprotocol.h>
#include <Vgts.h>
#include <rasterops.h>
#include "sdf.h"

char CanDisableScreen = 1;

/* Include the appropriate header file, depending on the framebuffer type. */
#ifdef SUN100FB

#include <Vfont.h>
#include "gl_sun100.h"

#else
#ifdef  SUN120FB

#include "gl_sun120.h"

#else
#ifdef  QVSS

#include "gl_mvax.h"

#else

Hey, somebody blew it!

#endif  QVSS
#endif  SUN120FB
#endif  SUN100FB

#define FontHeight 16

short ScreenLimitX, ScreenLimitY;	/* framebuffer limits */


short GridStip[] = {
			0x8000, 0x0000, 0x0000, 0x0000,
			0x0000, 0x0000, 0x0000, 0x0000,
			0x0000, 0x0000, 0x0000, 0x0000,
			0x0000, 0x0000, 0x0000, 0x0000 };

short Background[] = {
			0xEEEE, 0xDDDD, 0x7777, 0xBBBB,
			0xEEEE, 0xDDDD, 0x7777, 0xBBBB,
			0xEEEE, 0xDDDD, 0x7777, 0xBBBB,
			0xEEEE, 0xDDDD, 0x7777, 0xBBBB };


/*********************************************************************
 * InitDraw:
 ********************************************************************/

/*
 * Initialization routine for screen drawing routines.
 */

#include <Venviron.h>
#include <Vquerykernel.h>

#ifdef SUN100FB

int GXBase;
short RowOrder = 0;	/* rasters are stored by columns */

InitDraw()
  {
    /* Assume that we have a Sun-1 framebuffer.  Check whether this machine
     * is a Sun-1, or a Sun-2 upgrade.  In the former case we also check the
     * configuration register (if any) to determine whether the screen is
     * portrait mode or landscape mode.
     */
    MachineConfigurationReply mreply;    
    short config;
    
    QueryKernel(0, MACHINE_CONFIG, &mreply);
    if (mreply.machine == MACH_SMI_SUN2)
      { /* we have a Sun-2 upgrade (i.e. with a Sun-1 framebuffer). */
	GXBase = SUN2upgFbBase;
        ScreenLimitX = 1024;
	ScreenLimitY = 800;
       }
    else
      {
	GXBase = GXDefaultBase;
	config = mreply.confreg;
	if ( ( (struct ConfReg*)&config)->FBType )
	  {
	    ScreenLimitX = 1024;
	    ScreenLimitY = 800;
	  }
	else
	  {
	    ScreenLimitX = 800;
	    ScreenLimitY = 1024;
	  }
      }
    ScreenEnable();

    DefineFont(PAD_FONT_NAME, ""); /* needed in case of clipping */
    DefineFont(MENU_FONT_NAME, "");
  }

#endif  SUN100FB

#ifdef  SUN120FB

int GXBase;
short RowOrder = 1;	/* rasters are stored by rows */

InitDraw()
  { SystemCode error;

    MachineConfigurationReply mreply;    

    /* This might be a Sun-2 or Sun-3 */
    QueryKernel(0, MACHINE_CONFIG, &mreply);
    if (mreply.machine == MACH_SMI_SUN2)
        GXBase = SUN2FbBase;
    else
      {
	GXBase = SUN3FbBase;
	Sun120Fb.start = (BitUnit *) SUN3FbBase;
	Sun2FrameBuffer.start = (u_short*) SUN3FbBase;
      }
    ScreenLimitX = 1152;
    ScreenLimitY = 900;
    ScreenEnable();

    DefineFont(MENU_FONT_NAME, "");
  }

#endif  SUN120FB

#ifdef  QVSS

short RowOrder = 1;	/* rasters are stored by rows */
extern VRaster *WholeScreen;
InitDraw()
  {
    UseQvss(WholeScreen);
    ScreenLimitX = WholeScreen[-1].bBox.h;
    ScreenLimitY = WholeScreen[-1].bBox.v;
    DefineFont(MENU_FONT_NAME, "");
  }

#endif  QVSS


ScreenEnable()
  {
    if (CanDisableScreen) gl_EnableScreen();
  }

ScreenDisable()
  {
    if (CanDisableScreen) gl_DisableScreen();
  }


/*********************************************************************
 * ClearRegion:
 ********************************************************************/

/*
 * Clears specified region of the screen.  
 */

ClearRegion(x, y, w, h)
  short x, y, w, h;
  {
#ifdef MC68000
    struct fb_raster window;

    window.x = x;
    window.y = y;
    window.height = h;
    window.width = w;

    gl_WhitenRaster(&window);
#else
    VRaster temp;
    RestrictRaster(&temp+1, WholeScreen, y, x, h, w);
    RasterSet(&temp+1);
#endif
  }


/*********************************************************************
 * FillRegion:
 ********************************************************************/

/*
 * Blackens a portion of the screen.
 */

FillRegion(x, y, h, w)
  short x, y, h, w;
  {
#ifdef MC68000
    struct fb_raster dst;
    if (x<0 || x>ScreenLimitX || y<0 || y>ScreenLimitY || w<=0 || h<=0)
	return;
    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;

    gl_BlackenRaster(&dst);
#else
    VRaster temp;
    if (x<0 || x>ScreenLimitX || y<0 || y>ScreenLimitY || w<=0 || h<=0)
	return;
    RestrictRaster(&temp+1, WholeScreen, y, x, h, w);
    RasterClear(&temp+1);
#endif
  }


/*********************************************************************
 * InvertRegion:
 ********************************************************************/

/*
 * Inverts specified region.
 */

InvertRegion(x, y, h, w)
    short x, y, h, w;
  {
#ifdef MC68000
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.height = h;
    dst.width = w;

    gl_InvertRaster(&dst);
#else
    VRaster temp;
    RestrictRaster(&temp+1, WholeScreen, y, x, h, w);
    RasterInvert(&temp+1);
#endif
  }


/*********************************************************************
 * DrawOutline:
 ********************************************************************/

/*
 * DrawOutline() "ors" in a black outline with absolute corner (x, y) and
 * width and height w and h, respectively.  Thickness is the thickness (in
 * pixels) of the outline.  Select is four bits in the order left, right,
 * top, and bottom for bits 0, 1, 2, and 3 respectively.  Only these parts
 * of the outline are printed.
 */

DrawOutline(x, y, w, h, thickness, select)
  short x, y, w, h, thickness, select;
{
    if (select & LeftEdge)
        FillRegion(x, y, h, thickness);
    if (select & RightEdge)
        FillRegion(x+w-thickness, y, h, thickness);
    if (select & TopEdge)
        FillRegion(x, y, thickness, w);
    if (select & BottomEdge)
        FillRegion(x, y+h-thickness, thickness, w);
}


FlashOutline(xmin, ymin, xmax, ymax, select)
  {
	/* 
	 * Invert the indicated edges
	 */
#ifndef MC68000
    Rectangle dst;
    register Rectangle *r = &dst;
#else MC68000
    struct fb_raster dst;
    register struct fb_raster *r = &dst;
#endif MC68000
    short clipXmin, clipYmin, clipXmax, clipYmax;

    ymax -= 2;		/* Since lines go inside region */

    clipXmin = max(xmin, 2);
    clipYmin = max(ymin, 2);
    clipXmax = min(xmax, ScreenLimitX-2);
    clipYmax = min(ymax, ScreenLimitY-2);
    
    if (clipXmin >= ScreenLimitX -2 ) return;
    if (clipXmax < 2 ) return;
    if (clipYmin >= ScreenLimitY -2 ) return;
    if (clipYmax < 2 ) return;

#ifndef MC68000
    r->size.h = 2;
    r->size.v = clipYmax - clipYmin - 1;
    r->start.v = clipYmin + 2;
    if ( (select & LeftEdge) && xmin>2 && r->size.v>2)
      {
        r->start.h = xmin;
	gl_InvertRaster(r);
      }
    if ( (select & RightEdge) && xmax<ScreenLimitX-2 && r->size.v>2)
      {
        r->start.h = xmax-1;
	gl_InvertRaster(r);
      }
    r->size.v = 2;
    r->size.h = clipXmax - clipXmin + 1;
    r->start.h = clipXmin;
    if ( (select & TopEdge) && ymin>2 && r->size.h>2)
      {
        r->start.v = ymin;
	gl_InvertRaster(r);
      }
    if ( (select & BottomEdge) && ymax<ScreenLimitY-2 && r->size.h>2)
      {
        r->start.v = ymax+1;
	gl_InvertRaster(r);
      }
#else MC68000
    r->width = 2;
    r->height = clipYmax - clipYmin - 1;
    r->y = clipYmin + 2;
    if ( (select & LeftEdge) && xmin>2 && r->height>2)
      {
        r->x = xmin;
	gl_InvertRaster(r);
      }
    if ( (select & RightEdge) && xmax<ScreenLimitX-2 && r->height>2)
      {
        r->x = xmax-1;
	gl_InvertRaster(r);
      }
    r->height = 2;
    r->width = clipXmax - clipXmin + 1;
    r->x = clipXmin;
    if ( (select & TopEdge) && ymin>2 && r->width>2)
      {
        r->y = ymin;
	gl_InvertRaster(r);
      }
    if ( (select & BottomEdge) && ymax<ScreenLimitY-2 && r->width>2)
      {
        r->y = ymax+1;
	gl_InvertRaster(r);
      }
#endif MC68000
}



/*********************************************************************
 * ClearRectangle:
 ********************************************************************/

/*
 * Clears designated rectangle with or without a grid redrawn.
 */

ClearRectangle(x, y, h, w, withGrid)
    short x, y, h, w;
    BOOLEAN withGrid;
  {
    ClearRegion(x, y, w, h);
    if (withGrid)
	PaintRectangle( x, y, w, h, GridStip);
  }


/*********************************************************************
 * DrawRectangle:
 *
 * Draw a rectangle with the indicated stipple pattern.
 ********************************************************************/

DrawRectangle(x, y, w, h, stip_ptr)
  short x, y, w, h;
  short *stip_ptr;
{
#ifdef MC68000
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;
#else
    Rectangle dst;
    dst.start.h = x;
    dst.start.v = y;
    dst.size.h = w;
    dst.size.v = h;
#endif

    gl_PutPattern(&dst, stip_ptr);
}

/* "PaintRectangle" is the same as "DrawRectangle", except that the pattern is
 * "or"d into the framebuffer.
 */
PaintRectangle(x, y, w, h, stip_ptr)
  short x, y, w, h;
  short *stip_ptr;
{
#ifdef MC68000
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;
#else
    Rectangle dst;
    dst.start.h = x;
    dst.start.v = y;
    dst.size.h = w;
    dst.size.v = h;
#endif

    gl_PaintPattern(&dst, stip_ptr);
}


DrawBackground(x, y, w, h)
  short x, y, w, h;
{
#ifdef MC68000
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;
#else
    Rectangle dst;
    dst.start.h = x;
    dst.start.v = y;
    dst.size.h = w;
    dst.size.v = h;
#endif

    gl_PutPattern(&dst, Background);
}


/*******************************************************************/

#define CursorSize 32		/* number of shorts in the cursor */
#define CursorWidth 32		/* width in bits */
#define CursorHeight 16		/* height in bits */

BitWord PadCursor[32] = {
    0x0,0x0,
    0x0,0x0,
    0xfffc,0xffff,
    0xf01c,0x9fff,
    0xe01c,0x9fff,
    0xcf9c,0x9fff,
    0xcf9c,0x9fff,
    0xcf9c,0x83c1,
    0xe01c,0x81c0,
    0x701c,0x9cce,
    0x7f9c,0x9cce,
    0x7f9c,0x9cce,
    0x7f9c,0x8cc6,
    0x7f9c,0x80c0,
    0xff9c,0x91c8,
    0xfffc,0xffff,
};

BitWord ExecCursor[32] = {
    0x0,0x0,
    0x0,0x0,
    0xfffc,0xffff,
    0xfe0c,0xffff,
    0xfe0c,0xffff,
    0xffcc,0xffff,
    0xffcc,0xffff,
    0x73cc,0xc386,
    0x230c,0x8102,
    0x70c,0x9933,
    0x8fcc,0xf903,
    0x8fcc,0xf903,
    0x7cc,0x99f3,
    0x220c,0x8102,
    0x720c,0xc306,
    0xfffc,0xffff,
};

BitWord ZoomCursor[32] = {
    0xffff,0xffff,
    0xffc1,0xffff,
    0xffe1,0xffff,
    0xfff1,0xffff,
    0xffe9,0xffff,
    0xffdd,0xffff,
    0xffbf,0xffff,
    0xff01,0xffff,
    0xfe81,0xffff,
    0x71cf,0xc44c,
    0x24e7,0x8009,
    0x24f3,0x9999,
    0x24f9,0x9999,
    0x2481,0x9999,
    0x7181,0x999c,
    0xffff,0xffff,
};

BitWord ViewCursor[32] = {
    0x0,0x0,
    0x0,0x0,
    0xfffc,0xffff,
    0xe7e4,0xffff,
    0x27e4,0xffff,
    0x27e4,0xffff,
    0xe7e4,0xffff,
    0x33cc,0x9cc3,
    0x33cc,0x9c81,
    0x399c,0x9c99,
    0x399c,0x9c81,
    0x3c3c,0x9481,
    0x3c3c,0x80f9,
    0x3e7c,0x8881,
    0x3e7c,0x9c83,
    0xfffc,0xffff,
};

BitWord NormalCursor[32] = {
    0xffff,0xffff,
    0xff01,0xffff,
    0xff81,0xffff,
    0xffc1,0xffff,
    0xffc1,0xffff,
    0xff81,0xffff,
    0xff19,0xffff,
    0xfe3d,0xffff,
    0xfc7f,0xffff,
    0xf8ff,0xffff,
    0xf1ff,0xffff,
    0xe3ff,0xffff,
    0xc7ff,0xffff,
    0xefff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
};

BitWord AuxNormalCursor[32] = {
    0x7ff,0x0,
    0x301,0x0,
    0x181,0x0,
    0xc1,0x0,
    0xc1,0x0,
    0x181,0x0,
    0x319,0x0,
    0x63d,0x0,
    0xc67,0x0,
    0x18c3,0x0,
    0x3181,0x0,
    0x6300,0x0,
    0xc600,0x0,
    0x6c00,0x0,
    0x3800,0x0,
    0x1000,0x0,
};

BitWord AuxZoomCursor[32] = {
    0x7f,0x0,
    0x7f,0x0,
    0x7f,0x0,
    0x3f,0x0,
    0x7f,0x0,
    0xff,0x0,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
};

BitWord WhiteCursor[32] = {
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
    0xffff,0xffff,
};


static enum CursorStateType
  {
    Disabled,			/* don't touch it */
    Enabled,			/* move it normally when mouse moves */
    Changed,			/* rewrite it ASAP */
    Selecting,			/* XOR popup menu selections */
    WasSelecting,		/* in menu, but interrupted */
    WasChanged			/* changed, but interrupted */
  }
 CursorState = Disabled;	

static short CursorSave[CursorSize];		/* buffer saved from screen */
static BOOLEAN CursorOn = FALSE;		/* cursor is On */
short	 MouseX, MouseY;

#ifndef MC68000

static VRaster 
    _CursorRaster =
	{{0,0},{0,0},(BitUnit*)NormalCursor, 0,2, {CursorHeight,CursorWidth}},
    _AuxCursorRaster =
	{{0,0},{0,0},(BitUnit*)AuxNormalCursor,0,2,{CursorHeight,CursorWidth}},
    _SaveRaster =
	{{0,0},{0,0},(BitUnit*)CursorSave,0,2,{CursorHeight,CursorWidth}};
#define CursorRaster (&_CursorRaster+1)
#define AuxCursorRaster (&_AuxCursorRaster+1)
#define SaveRaster (&_SaveRaster)[1]

static Rectangle DrawRaster =  { {0, 0}, {CursorHeight, CursorWidth} };
static Rectangle flash;

#else MC68000

static struct mem_raster 
    CursorRaster =    { CursorHeight, CursorWidth, (short *) NormalCursor },
    AuxCursorRaster = { CursorHeight, CursorWidth, (short *) AuxNormalCursor },
    SaveRaster =      { CursorHeight, CursorWidth, (short *) CursorSave };

static struct fb_raster DrawRaster =  { CursorHeight, CursorWidth, 0, 0 };
static struct fb_raster flash;

#endif MC68000

/*
 * ChangeCursor:
 *
 * Changes to the new cursor.
 */

ChangeCursor(newCursorPtr, newAuxPtr)
#ifndef MC68000
  BitUnit *newCursorPtr, *newAuxPtr;
  {
   CursorRaster[-1].start = newCursorPtr;
   AuxCursorRaster[-1].start = newAuxPtr;
#else MC68000
  short *newCursorPtr, *newAuxPtr;
  {
   CursorRaster.start = newCursorPtr;
   AuxCursorRaster.start = newAuxPtr;
#endif MC68000
   CursorState = Changed;
   Cursor();
  }

SetViewCursor()
{
    ChangeCursor(ViewCursor, WhiteCursor);
}

SetZoomCursor()
{
    ChangeCursor(ZoomCursor, AuxZoomCursor);
}

SetPadCursor()
{
    ChangeCursor(PadCursor, WhiteCursor);
}

SetExecCursor()
{
    ChangeCursor(ExecCursor, WhiteCursor);
}

SetMainCursor()
{
    ChangeCursor(NormalCursor, AuxNormalCursor);
}

static mxmin,mxmax,mymin,mymax;		/* bounding box of pop-up menu */
static lastStart;			/* last entry we flashed */
static HaveToRedraw;			/* pop-up interference flag */

MenuCursor(xmin,xmax,ymin,ymax)
  {
	/*
	 * Change to menu mode
	 */
    mxmin = xmin;
    mxmax = xmax;
    mymin = ymin;
    mymax = ymax;
    lastStart = -1;
    HaveToRedraw = 0;
    CursorOn = 0;
    CursorState = Selecting;
    DoSelect();
  }


/* Globals used by the following three routines: */

int	NewSaveSpaceNeeded = 0;
int	CurrentSaveSpaceSize = 0;

#ifndef MC68000
struct Rectangle MenuSpace;
VRaster SaveSpace;
#else MC68000
struct fb_raster MenuSpace;
struct mem_raster SaveSpace;
#endif MC68000

SaveBeforeMenu(x, y, w, h)
    unsigned short x, y, w, h;
    /* Saves away the indicated portion of the screen, so that a menu can be
     * popped up in its place.
     */
  {
#ifndef MC68000
    MenuSpace.start.h = x;
    MenuSpace.start.v = y;
    MenuSpace.size.v = SaveSpace.bBox.v = h;
    MenuSpace.size.h = SaveSpace.bBox.h = w;
    SaveSpace.stride = ((w + 15)>>BitWordLog)<<BitAlignLog;

    NewSaveSpaceNeeded = h * SaveSpace.stride;
    if ( CurrentSaveSpaceSize < NewSaveSpaceNeeded )
      {
	if ( CurrentSaveSpaceSize > 0 )
	    free( SaveSpace.start );
	if (( SaveSpace.start = (BitUnit *) malloc( NewSaveSpaceNeeded ))
	    == NULL )
	    CurrentSaveSpaceSize = 0;
	else
	    CurrentSaveSpaceSize = NewSaveSpaceNeeded;
      }
    if ( CurrentSaveSpaceSize > 0 )
	gl_GetRaster( &SaveSpace+1, &MenuSpace );
#else MC68000
    MenuSpace.x = x;
    MenuSpace.y = y;
    MenuSpace.height = SaveSpace.height = h;
    MenuSpace.width = SaveSpace.width = w;

    NewSaveSpaceNeeded = h * ((w + 7)>>3);
    if ( CurrentSaveSpaceSize < NewSaveSpaceNeeded )
      {
	if ( CurrentSaveSpaceSize > 0 )
	    free( SaveSpace.start );
	if (( SaveSpace.start = (short *)malloc( NewSaveSpaceNeeded )) == NULL)
	    CurrentSaveSpaceSize = 0;
	else
	    CurrentSaveSpaceSize = NewSaveSpaceNeeded;
      }
    if ( CurrentSaveSpaceSize > 0 )
	gl_GetRaster( &SaveSpace, &MenuSpace );
#endif MC68000
  }

RestoreAfterMenu()
    /* Replaces the portion of the screen saved away by "SaveBeforeMenu". */
  {
#ifndef MC68000
    if ( CurrentSaveSpaceSize > 0 )
	gl_PutRaster( &MenuSpace, &SaveSpace + 1 );
#else MC68000
    if ( CurrentSaveSpaceSize > 0 )
	gl_PutRaster( &MenuSpace, &SaveSpace );
#endif MC68000
  }

FreeMenuSaveSpace()
  {
  }


DisableCursor()
  {
  	/*
	 * called after we do a pop-up menu, to indicate that we
	 * are out of selecting mode.
	 *
	 * Returns 0 if everything went OK, true if we had to
	 * redraw everything behind the popup menu.
	 */
    int prev = HaveToRedraw;

    CursorState = Disabled;
    if (HaveToRedraw)
      {
	EraseCursor(mxmin, mxmax, mymin, mymax); /* clean up state */
	RestoreAfterMenu();
        RedrawScreen(mxmin,mxmax,mymin,mymax,NULL);
      }
    HaveToRedraw = 0;
    return(prev);
  }

/*
 * EraseCursor:
 * Erase the cursor image from the screen so that it doesn't
 * conflict with other drawing going on.
 *
 * The bounding box of the area being worked on is passed,
 * so we can just leave the cursor on if it is outside.
 *
 * Returns true if the area might conflict with a popup menu.
 */

EraseCursor(xmin, xmax, ymin, ymax)
  int xmin, xmax, ymin, ymax;
{
    switch (CursorState)
      {
        case Selecting:
	case WasSelecting:
	    if ( (xmin < mxmax && xmax > mxmin) 
	      && (ymin < mymax && ymax > mymin) )
	    	{
			/*
			 * Darn! we want to draw part of the screen
			 * that might overlap the pop-up menu.
			 * Set a flag and return an error.
			 */
		  HaveToRedraw = 1;
		  return(1);
		}
	    CursorState = WasSelecting;
	    break;

	case Changed:
	    CursorState = WasChanged;
	    break;

        default:
            CursorState = Disabled;
      }
    if (! CursorOn)
        return(0);
#ifndef MC68000
    if ( xmin>DrawRaster.start.h+CursorWidth || xmax<DrawRaster.start.h ||
    	 ymin>DrawRaster.start.v+CursorHeight || ymax<DrawRaster.start.v )
#else MC68000
    if ( xmin>DrawRaster.x+CursorWidth || xmax<DrawRaster.x ||
    	 ymin>DrawRaster.y+CursorHeight || ymax<DrawRaster.y )
#endif MC68000
      return(0);
    CursorOn = FALSE;
    gl_PutRaster( &DrawRaster, &SaveRaster ); 
    return(0);
}


/*******************************************************************/


/*
 * ConstrainCursorXY:
 * Constrain the given X,Y coordinates to be valid cursor coordinates.
 */

ConstrainCursorXY(xPtr, yPtr)

  short *xPtr, *yPtr;

{

    if (*xPtr < 0) *xPtr = 0;
    else if (*xPtr > ScreenLimitX-CursorWidth) *xPtr = ScreenLimitX-CursorWidth;

    if (*yPtr < 0) *yPtr = 0;
    else if (*yPtr > ScreenLimitY-CursorHeight) *yPtr=ScreenLimitY-CursorHeight;

}


/*******************************************************************/


/*
 * Cursor:
 * Redraw the cursor image on the screen.  This routine is used in
 * conjunction with EraseCursor() to prevent the cursor image from
 * interfering with other drawing that is to be done.
 * The routine also prevents the cursor from going off the screen.
 *
 * If the position has not changed since the last write,
 * it is not re-written.
 */

Cursor()
{

    if (CursorOn &&  ( CursorState==Enabled || CursorState==Disabled)
#ifndef MC68000
    	 && DrawRaster.start.h == MouseX 
	 && DrawRaster.start.v == MouseY)
#else MC68000
    	 && DrawRaster.x == MouseX 
	 && DrawRaster.y == MouseY)
#endif MC68000
      {
        CursorState = Enabled;
    	return;
      }

    if (CursorState==WasSelecting)
      {
        CursorState = Selecting;
        return;
      }
    if (CursorOn)
	gl_PutRaster( &DrawRaster, &SaveRaster ); 
#ifndef MC68000
    DrawRaster.start.h = MouseX;
    if (DrawRaster.start.h < 0)  DrawRaster.start.h = 0;
    if (DrawRaster.start.h > ScreenLimitX - CursorWidth)
    	 DrawRaster.start.h = ScreenLimitX - CursorWidth;
    DrawRaster.start.v = MouseY;
    if (DrawRaster.start.v < 0)  DrawRaster.start.v = 0;
    if (DrawRaster.start.v > ScreenLimitY - CursorHeight)
    	 DrawRaster.start.v = ScreenLimitY - CursorHeight;
    if (DrawRaster.start.h != MouseX || DrawRaster.start.v != MouseY)
    	SetMouseTo(DrawRaster.start.h, DrawRaster.start.v);
#else MC68000
    DrawRaster.x = MouseX;
    if (DrawRaster.x < 0)  DrawRaster.x = 0;
    if (DrawRaster.x > ScreenLimitX - CursorWidth)
    	 DrawRaster.x = ScreenLimitX - CursorWidth;
    DrawRaster.y = MouseY;
    if (DrawRaster.y < 0)  DrawRaster.y = 0;
    if (DrawRaster.y > ScreenLimitY - CursorHeight)
    	 DrawRaster.y = ScreenLimitY - CursorHeight;
    if (DrawRaster.x != MouseX || DrawRaster.y != MouseY)
    	SetMouseTo(DrawRaster.x,DrawRaster.y);
#endif MC68000

    gl_GetRaster( &SaveRaster, &DrawRaster );

/*
 * Draw the cursor background part first
 */
#ifndef MC68000
    gl_InvPaintRaster( &DrawRaster, AuxCursorRaster );
#else MC68000
    gl_InvPaintRaster( &DrawRaster, &AuxCursorRaster );
#endif MC68000

/*
 * Now draw the cursor body
 */
#ifndef MC68000
    gl_PaintRaster( &DrawRaster, CursorRaster );
#else MC68000
    gl_PaintRaster( &DrawRaster, &CursorRaster );
#endif MC68000
    CursorOn = TRUE;
    CursorState = Enabled;
}

static DoSelect()
{	
	/*
	 * select an item out of a menu.
	 * If we have not moved, do nothing.
	 * Otherwise, save the cursor area, 
	 *  then invert the menu item that we would select,
	 *  and finally Xor the cursor image.
	 */
#ifndef MC68000
    if (CursorOn
    	 && DrawRaster.start.h == MouseX 
	 && DrawRaster.start.v == MouseY)
	   return;

    if (CursorOn)
        gl_PutRaster( &DrawRaster, &SaveRaster ); 

    flash.size.v = FontHeight-1;
    flash.size.h  = mxmax - mxmin - 2;
    flash.start.h = mxmin+1;
    if (MouseX >= mxmin && MouseX < mxmax &&
        MouseY >= mymin && MouseY < mymax )
	  {
	    /*
	     * flash a part of the pop-up menu
	     */
	     short start = mymin;

	     while (start+FontHeight < MouseY) start += FontHeight;
	     if (lastStart!=start)
	       {
	         if (lastStart>=0)
	           {
	             flash.start.v = lastStart+1;
	             gl_InvertRaster(&flash);
	           }
	         flash.start.v = start+1;
	         gl_InvertRaster(&flash);
	         lastStart = start;
	       }
	  }
	else if (lastStart>=0)
	       {
	         flash.start.v = lastStart+1;
	         gl_InvertRaster(&flash);
		 lastStart = -1;
	       }
    DrawRaster.start.h = MouseX;
    if (DrawRaster.start.h < 0)  DrawRaster.start.h = 0;
    if (DrawRaster.start.h > ScreenLimitX - CursorWidth)
    	 DrawRaster.start.h = ScreenLimitX - CursorWidth;
    DrawRaster.start.v = MouseY;
    if (DrawRaster.start.v < 0)  DrawRaster.start.v = 0;
    if (DrawRaster.start.v > ScreenLimitY - CursorHeight)
    	 DrawRaster.start.v = ScreenLimitY - CursorHeight;
    if (DrawRaster.start.h != MouseX || DrawRaster.start.v != MouseY)
    	SetMouseTo(DrawRaster.start.h,DrawRaster.start.v);
    gl_GetRaster( &SaveRaster, &DrawRaster );
    if (lastStart>=0 && DrawRaster.start.v < mymax - FontHeight/2)
    	/*
	 * The Cursor Xor's instead of OR's when you are selecting
	 * something, except for the last half line
	 */
	{gl_XorRaster( &DrawRaster, CursorRaster );}
    else
      {
/*
 * Draw the cursor background part first
 */
	gl_InvPaintRaster( &DrawRaster, AuxCursorRaster );
/*
 * Now draw the cursor body
 */
	gl_PaintRaster( &DrawRaster, CursorRaster );
      }
    CursorOn = TRUE;
#else MC68000
    if (CursorOn
    	 && DrawRaster.x == MouseX 
	 && DrawRaster.y == MouseY)
	   return;

    if (CursorOn)
        gl_PutRaster( &DrawRaster, &SaveRaster ); 

    flash.height = FontHeight-1;
    flash.width  = mxmax - mxmin - 2;
    flash.x = mxmin+1;
    if (MouseX >= mxmin && MouseX < mxmax &&
        MouseY >= mymin && MouseY < mymax )
	  {
	    /*
	     * flash a part of the pop-up menu
	     */
	     short start = mymin;

	     while (start+FontHeight < MouseY) start += FontHeight;
	     if (lastStart!=start)
	       {
	         if (lastStart>=0)
	           {
	             flash.y = lastStart+1;
	             gl_InvertRaster(&flash);
	           }
	         flash.y = start+1;
	         gl_InvertRaster(&flash);
	         lastStart = start;
	       }
	  }
	else if (lastStart>=0)
	       {
	         flash.y = lastStart+1;
	         gl_InvertRaster(&flash);
		 lastStart = -1;
	       }
    DrawRaster.x = MouseX;
    if (DrawRaster.x < 0)  DrawRaster.x = 0;
    if (DrawRaster.x > ScreenLimitX - CursorWidth)
    	 DrawRaster.x = ScreenLimitX - CursorWidth;
    DrawRaster.y = MouseY;
    if (DrawRaster.y < 0)  DrawRaster.y = 0;
    if (DrawRaster.y > ScreenLimitY - CursorHeight)
    	 DrawRaster.y = ScreenLimitY - CursorHeight;
    if (DrawRaster.x != MouseX || DrawRaster.y != MouseY)
    	SetMouseTo(DrawRaster.x,DrawRaster.y);
    gl_GetRaster( &SaveRaster, &DrawRaster );
    if (lastStart>=0 && DrawRaster.y < mymax - FontHeight/2)
    	/*
	 * The Cursor Xor's instead of OR's when you are selecting
	 * something, except for the last half line
	 */
	{gl_XorRaster( &DrawRaster, &CursorRaster );}
    else
      {
/*
 * Draw the cursor background part first
 */
	gl_InvPaintRaster( &DrawRaster, &AuxCursorRaster );
/*
 * Now draw the cursor body
 */
	gl_PaintRaster( &DrawRaster, &CursorRaster );
      }
    CursorOn = TRUE;
#endif MC68000
}


/*
 * CursorUpdate:
 * Updates the cursor image position if a cursor image is allowed on
 * the screen.
 */

CursorUpdate()
  {
    switch (CursorState)
      {
        case Changed:
        case Enabled:   Cursor();	break;
	case Selecting:	DoSelect();	break;
      }
  }
