/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: tty.c
 *
 * Gus Fernandez 5/18/85 - Fixed dprintf to pads.
 */

#include "Vgts.h"
#include "sdf.h"
#include "vgt.h"
#include "pad.h"
#include <chars.h>


short TtyVGT;	/* Index of the "command" tty VGT.  This
		   VGT is used by the vgts for communications
		   with the user. */

extern short TtySDF;		/* SDF used for internal communication */
extern short Debug;		/* print extra debugging information */
extern TtyPadType *DebugPad;	/* pad to which debug info should go */
extern short Unique;		/* symbol identifier */

#define MaxInputStringLength 75
#define PauseTime 15			/* Number of clock clicks to delay
		   			   when inverting text line. */

static struct TtyPadRec *TtyPad;


/*******************************************************************/


/*
 * InitTtyVGT:
 * Creates the vgts' tty window.  This window is used by the
 * Vgts to communicate with the user.
 * It is positioned near the bottom, since only the last few lines
 * are visible.  The user can move the top edge up to see more
 * if he so desires.
 */

#define VGTS_VERSION "VGTS a.2"

InitTtyVGT(xmin,ymin,xmax,ymax)
  {
    TtySDF = CreateSDF();
    TtyVGT = NewTextVGT( TtySDF, VGTS_VERSION, 20, 64 );
    CreateView( TtyVGT, xmin, ymin, xmax, ymax, 2, -2, 0, 0);
    TtyPad = TtyPadList + TtyVGT;
    PadPosition(TtyPad, 24, 0);
  }

/*******************************************************************/


/*
 * TtyGetPromptedString:
 * Clears the vgts ttywindow, puts up the prompt, and
 * accepts a string of characters terminated by a newline.  The string
 * is left in the local character array, so the caller must copy it out if
 * he wishes to keep it.  It also flashes the prompt a couple of times.
 */

char *TtyGetPromptedString(prompt, echo)
 char *prompt;
  BOOLEAN echo;
{
    short i;
    char *TtyGetString(), *response;

    MakePadTop(TtyVGT);

    while (*prompt)
        PadPutChar(*(prompt++), TtyPad);	/* dump the prompt */
    PadPutChar(' ', TtyPad);
    PadClearToEOL(TtyPad);
    
    RedrawVGT( TtyVGT, 0, 3000, 0, 3000 );
    for (i = 0; i < 2; i++)
      {
        PadBlinkLine(TtyVGT);
        Pause(PauseTime);
      } 
    
    response = TtyGetString(echo);

    MakePadBottom(TtyVGT);

    return(response);
}


/*******************************************************************/


/*
 * TtyGetNumber:
 * Displays the prompt in the vgts ttywindow
 * and returns the number which is typed in, zero if the number was too
 * big to fit in a short, if it was negative, or if it was garbage.
 */

short TtyGetNumber(prompt)
  char *prompt;
{
    char *TtyGetPromptedString();
    int i, atoi();
    
    i = atoi( TtyGetPromptedString(prompt, TRUE) );
    return ( (short)(((unsigned)i > (unsigned short)~0 || i <= 0) ? 0 : i) );
}


/*******************************************************************/


/*
 * TtyBlinkError:
 * Puts up the error string in the top line of the vgts ttywindow,
 * and blinks it a couple of times.
 */

TtyBlinkError(errorString)
char *errorString;
{
    TtyBlinkHelper(errorString, 6);
}


/*******************************************************************/


/*
 * TtyMessage:
 * Displays the designated string in the vgts ttywindow.
 */

TtyMessage(string)
char *string;
{
    TtyBlinkHelper(string, 2);
}


/*******************************************************************/


/*
 * TtyBlinkHelper:
 * Primary support routine for displaying text in the vgts ttywindow.
 * Displays the indicated text string and flashes it a few times.
 */
static TtyBlinkHelper(errorString, blinkNumber)
  short blinkNumber;
  char *errorString;
{
    short lineSave, columnSave, blinkCount;

    PadClearToEOL(TtyPad);
    while (*errorString)
        PadPutChar(*errorString++, TtyPad);
    
    for (blinkCount = 0; blinkCount < blinkNumber; blinkCount++)
      {
        PadBlinkLine(TtyVGT);
        Pause(PauseTime);
      }
    PadReturn(TtyPad);
    PadNewLine(TtyPad);    
    RedrawVGT( TtyVGT, 0, 3000, 0, 3000 );
}

/* The following two routines added by TRD to fix yale */

/* TtyPutString: dumps a string to the tty.  It does not bother with
 * a final newline or any blinking as in TtyMessage.
 */

TtyPutString(string)
  char *string;
{
  while (*string)
      PadPutChar(*string++, TtyPad);
  RedrawVGT(TtyVGT, 0, 3000, 0, 3000);
}


/* TtyGetString: just gets a string from the user, echoing it in the
 * tty window.  It is like TtyGetPromptedString, except that it does not put out a
 * prompt first.
 */

char *TtyGetString(echo)
  BOOLEAN echo;
{
    static char locString[MaxInputStringLength + 1];
    char GetInputChar();
    char c;
    short locStringPtr = 0;
    short i;

    while (1)
     {
      switch (c = GetInputChar())
       {
	case LF:
	case CR:
	    locString[locStringPtr] = 0;
	    PadReturn(TtyPad);
	    PadNewLine(TtyPad);
            RedrawVGT( TtyVGT, 0, 3000, 0, 3000 );
	    return(locString);

	case BS:
	case DEL:
	    if (locStringPtr)
	      {
		locStringPtr--;
		if (echo)
		  {
		    PadPutChar( BS, TtyPad);
		    PadPutChar( ' ', TtyPad);
  		    PadPutChar( BS, TtyPad);
                    PadRedrawLine( TtyVGT );
		  }
	      }
	    break;

	default:
	    locString[locStringPtr++] = c;
	    if (locStringPtr == MaxInputStringLength)
	      {
		locString[locStringPtr] = 0;
		return(locString);
	      }
	    if (echo)
	      {
		PadPutChar(c, TtyPad);
                PadRedrawLine(TtyVGT);
	      }
	    break;
	}
    }

}

TtyTrace()
  {
  	/*
	 * Debugging routine
	 */
    dprintf( "Tracing TTy vgt\n");
    SdfTrace(VGTtable[TtyVGT].topSymbol, 0 );
  }


/**************************  Debug output system  ***************************/
/* Vgts writes debug info to one of its own pads, unless pad -1 is specified,
 * in which case it writes to the console.
 */

dputchar(c)
  char c;
  {
    if (DebugPad == NULL)
      {
	putchar(c);
	if (c == '\n')  putchar('\r');
      }
    else
      {
	PadPutChar(c, DebugPad);
      }
  }

dprintf(format, a, b, c, d, e, f, g, h, i, j)
  char *format;
  int a, b, c, d, e, f, g, h, i, j;
  {
    char buffer[256], buf2[256],*p,*q;

    sprintf(buffer, format, a, b, c, d, e, f, g, h, i, j);
    if (DebugPad == NULL)
      {
	printf("%s", buffer);
      }
    else
      {
	p = &buffer[0];
	q = &buf2[0];
	while (*p) {
	    if (*p=='\n')
		*q++='\r';
	    *q++ = *p++;
	}
	*q++='\0';
	PadPutString(buf2, DebugPad);
      }
  }
