/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: pad.c
 *
 * Bill Nowicki September 1982
 *
 * Terminal emulator and Pad maintenance routines.
 *
 * April 1983, from a suggestion by Per Bothner, we actually allocate
 * an extra column on each line for a "continuation character".
 *
 * Craig Dunwoody July 1983
 *	Added PadGetCursor and PadGetSize routines for extracting information
 *	about pad parameters.
 */


#include <Vgts.h>
#include <Vgtp.h>
#include "sdf.h"
#include "vgt.h"
#include "pad.h"
#include <chars.h>
#include <stdio.h>

extern short Debug;			/* print extra debugging information */

TtyPadType TtyPadList[MaxVGT];		/* Special TTY window stuff */

#define LeftMargin 8			/* pixels left of first char */
#define BottomMargin 4			/* pixels below bottom line */
#define ContinuationChar 031		/* "continued on next line" */


/*******************************************************************/


short PadInit(sdf, topSymbol, name, lines, columns)
    char *name;
  {
	/*
	 * Initialize a pad, and return the pad number
	 */
    int line;
    register struct TtyPadRec *pad;
    register char *linebuf;
    DisplayRecord *Find();
    short vgt;
    
    vgt = CreateVGT(sdf, TTY, 0, name);
    if (vgt<0 || (linebuf=(char *)malloc(lines*(columns+2)))==NULL)
	return(-1);
    
    pad = TtyPadList + vgt;
    pad->strings = linebuf;
    pad->width = columns-1;
    pad->length = pad->pageLength = lines-1;
    pad->regionTop = 0;
    pad->regionBot = lines-1;
    if (Debug & DebugVgtsPads) dprintf("creating pad %d\n", vgt);
    DefineSymbol(sdf, topSymbol, name);
    for (line=pad->length; line>=0; line--, linebuf += pad->width+3 )
      AddItem(sdf, 0, LeftMargin, (columns+1)*FontWidth+LeftMargin,
            line*FontHeight+BottomMargin,
        (line+1)*FontHeight+BottomMargin-1, 0, SDF_InternalText, linebuf );
    EndSymbol(sdf, topSymbol, 0);
    pad->cursorFlag = 0;
    pad->standout = 0;
    pad->state = Normal;
    pad->vgt = vgt;
    pad->yMinChanged = MaxCoord;
    pad->yMaxChanged = -MaxCoord;
    DisplayItem(sdf, topSymbol, vgt);
    pad->sdfPtr = VGTtable[vgt].topSymbol;
    PadClear(pad);
    return(vgt);
 }


/* "NewTextVGT" is an intermediate function that is used for
 * source-code compatibility between the old VGTS and the GKS-compatible
 * version.
 */
short NewTextVGT(sdf, name, lines, columns)
    char *name;
  {
    extern short Unique;

    return(PadInit(sdf, Unique++, name, lines, columns));
  }

/* "DeleteTextVGT" is also an intermediate function used for compatibility. */
DeleteTextVGT(vgtindex, andViews)
    short vgtindex, andViews;
  {
    PadDelete(vgtindex);
    DeleteVGT(vgtindex, andViews);
  }

extern short TtySDF;

PadDelete( vgt )
  {
  	/*
	 * Free up the string space allocated to the given pad.
	 */
    register struct TtyPadRec *pad = TtyPadList + vgt;
    
    if ( vgt < 0 || vgt >= MaxVGT ) return;
    if ( VGTtable[ vgt ].type == UNUSED ) return;
    if ( pad->strings ) free( pad->strings );
    pad->strings = NULL;
    if ( pad->sdfPtr )
      {
	DeleteHash( TtySDF, pad->sdfPtr->item );
/*
	if ( pad->sdfPtr->leftChild )
	    free( pad->sdfPtr->leftChild );
*/
	FreeSdfDef( 0, pad->sdfPtr, FALSE );
	pad->sdfPtr = NULL;
      }
  }


/*
 * PadClear:
 * Clears the entire specified tty pad, and leaves
 * the cursor at the top left corner.
 */

PadClear(pad)
 register struct TtyPadRec *pad;
  {
    int i;

    PadHome(pad);
    for (i=0; i<=pad->length; i++, PadCursorDown(pad) )
        PadClearToEOL(pad);
    PadHome(pad);
  }


/*
 * PadClearToEOS:
 * Clears the pad to the End of the screen.
 * Leaves cursor where it is.
 */

PadClearToEOS(pad)
    register struct TtyPadRec *pad;
  {
    int i, col, line;

    col = pad->curColumn;
    line = pad->curLine;
    PadClearToEOL(pad);
    PadReturn(pad);
    for (i=line+1; i<=pad->length; i++ )
      {
        PadCursorDown(pad);
        PadClearToEOL(pad);
      }
    PadPosition(pad, line, col);
  }


/*
 * PadHome:
 * Move the cursor to the upper left corner.
 */

PadHome(pad)
    register struct TtyPadRec *pad;
  {
    QuickCursorOff(pad);
    pad->curLine = 0;
    pad->curColumn = 0;
    pad->newlinesLeft = pad->pageLength;
    pad->curItem = pad->sdfPtr->rightSib;
    if ((Debug & DebugVgtsPads) && pad->curItem->type != SDF_InternalText)
        dprintf( "Bad type in pad Home, %d ", pad->curItem->type);
    if (pad->curItem==NULL) printf("NULL CurItem  in PadHome\n");
    pad->cursorPos = (char *)pad->curItem->leftChild;
    if (pad->yMaxChanged < pad->curItem->ymax)
    	pad->yMaxChanged = pad->curItem->ymax;
  }

/*
 * PadPosition:
 * position the cursor on the indicated line and column
 */

PadPosition(pad, line, column)
   register struct TtyPadRec *pad;
   short line, column;
  {
    register short l = line;
    
    QuickCursorOff(pad);
    if (l > pad->length) l = pad->length;
    if (l < 0 ) l = 0;
    if (pad->curLine>l) 
      {
        pad->curLine = 0;
        pad->curItem = pad->sdfPtr->rightSib;
      }
    while (pad->curLine<l)
      {
         pad->curItem = pad->curItem->rightSib;
	 pad->curLine++;
      }
    if (column > pad->width) column = pad->width+1;
    if (column < 0) column = 0;
    pad->curColumn = column;
    pad->cursorPos = (char *)(pad->curItem->leftChild) + column;
  }

/*******************************************************************/


/*
 * PadPutString:
 * Output a string to a tty pad.
 */

PadPutString(s, pad)
  register char *s;
  register struct TtyPadRec *pad;

  {
    register char c;

    while (c = *s++) PadPutChar(c, pad);
  }


/*******************************************************************/


/*
 * PadPutChar:
 * Output a character to a tty pad.
 */

PadPutChar(ch, pad)
    register  ch;
    register struct TtyPadRec *pad;
  {
     ch &= 0177;
     if (pad->state==Normal && ch >= ' ' && ch < 0177)
       {
    	    if (pad->curColumn > pad->width)
		PadContinueLine(pad);
	      	/*
		 * In-line for speed -- this is the usual case
		 */
	    if (pad->cursorFlag) *pad->cursorPos ^= 0200;
    	    pad->cursorFlag = 0;
    	    *(pad->cursorPos)++ = ch | pad->standout;
    	    if (pad->yMaxChanged < pad->curItem->ymax)
    		pad->yMaxChanged = pad->curItem->ymax;
    	    if (pad->yMinChanged > pad->curItem->ymin)
    		pad->yMinChanged = pad->curItem->ymin;
   	    pad->curColumn++;
	    return;
	}

    switch (pad->state)
     {
      case Normal:
        switch (ch)
         {
          case BS:
	    PadCursorBackward(pad); 
    	    break;

          case CR:
	    PadReturn(pad);
	    break;

          case BELL:
	    InvertVGT(pad->vgt);
	    putchar(BELL);
	    Flush(stdout);
	    Delay(0,10);
	    InvertVGT(pad->vgt);
	    break;

          case TAB:
	    PadTab(pad);
	    break;

          case FF:
	    PadClear(pad);
	    break;
	    
          case LF:
    	    PadNewLine(pad);
    	    break;

          case ESC:
    	    pad->state = Escape;
    	    break;
	    
	  case NULL:
          case DEL:
	  case SO:	/* select alternate character set */
	  case SI:	/* (currently ignored) */
	    break;

          default:
    	    if (pad->curColumn > pad->width)
    	        PadContinueLine(pad);
	    PadDrawChar(pad, ch);
	    PadCursorForward(pad);
	    break;
	}
	break;
	
	case Escape:
	    pad->argCount = 0;
	    pad->args[0] = 0;
	    pad->args[1] = 0;
	    pad->args[2] = 0;
	    pad->args[3] = 0;
	    pad->args[4] = 0;
	    pad->state = Normal;
	    PadEscape( ch, pad, NonAnsi );
	    break;
	    
	  case GettingArgs:
	    if ('0' <= ch && ch <= '9' )
	      {
	         pad->args[pad->argCount] *= 10;
		 pad->args[pad->argCount] += ch - '0';
	      }
	    else switch (ch)
	      {
	        case ';':	pad->argCount++;	break;
		case '?':				break;
		default:	pad->state = Normal;
				PadEscape(ch, pad, Ansi);
	      }
	    break;
	    
	case GettingCharSet:
		/*
		 * for now ignore the  VT-100 character set descriptor
		 */
	    pad->state = Normal;
	    break;

	default:
	    return(0);
      }
}

PadContinueLine(pad)
register struct TtyPadRec *pad;
  {
    register int oldstandout;

    oldstandout = pad->standout;
    pad->standout = 0;
    PadDrawChar(pad, ContinuationChar);
    PadReturn(pad);
    PadNewLine(pad);
    pad->standout = oldstandout;
  }


/*******************************************************************/


/*
 * PadDrawChar:
 * Helper routine which
 * draws the specified character at the indicated line and column.
 * and records it in the SDF for posterity.
 */

static PadDrawChar(pad,c)
    register struct TtyPadRec *pad;
    char c;
  {

    QuickCursorOff(pad);
    *(pad->cursorPos) = (c&0177) | pad->standout;
    if (pad->yMaxChanged < pad->curItem->ymax)
    	pad->yMaxChanged = pad->curItem->ymax;
    if (pad->yMinChanged > pad->curItem->ymin)
    	pad->yMinChanged = pad->curItem->ymin;
  }


/*******************************************************************/


/*
 * PadTab:
 * Tabulate to the next multiple of eight columns
 */

PadTab(pad)
  register struct TtyPadRec *pad;
{
   do 
     {
       PadCursorForward(pad);
     } while (pad->curColumn & 7 && pad->curColumn < pad->width);
}

/*
 * PadClearToEOL:
 * Routine which clears to the end of the current line.
 * This routine leaves CurLine and CurColumn unchanged.
 */

PadClearToEOL(pad)
  register struct TtyPadRec *pad;
{
  register char *p;
  register int i;
  
   QuickCursorOff(pad);
   if ((Debug & DebugVgtsPads) && pad->curItem->type != SDF_InternalText)
      dprintf( "Bad type in pad Clear to EOL, %d ", pad->curItem->type);
    if (pad->yMaxChanged < pad->curItem->ymax)
    	pad->yMaxChanged = pad->curItem->ymax;
    if (pad->yMinChanged > pad->curItem->ymin)
    	pad->yMinChanged = pad->curItem->ymin;
   p = pad->cursorPos;
   for (i=pad->curColumn; i <= pad->width+1; i++)
     *p++ = ' ';
   *p = 0;
}

/*
 * PadInsertChar:
 * Insert a character, by moving all of them over to the right one.
 * This routine leaves CurLine and CurColumn unchanged.
 */

PadInsertChar(pad)
    register struct TtyPadRec *pad;
{
  register char *from, *to;
  
   QuickCursorOff(pad);
   to = (char *)pad->curItem->leftChild;
   to += pad->width;
   from = to - 1;
   for (; to>pad->cursorPos;)
     *to-- = *from--;
   *to = ' ';
}

/*
 * PadDeleteChar:
 * Delete a character, by moving all of them over to the left one.
 * This routine leaves CurLine and CurColumn unchanged.
 */

PadDeleteChar(pad)
  register struct TtyPadRec *pad;
{
  register char *from, *to;
  int i;
  
   QuickCursorOff(pad);
   to = pad->cursorPos;
   from = to + 1;
   for (i=pad->curColumn; i<pad->width; i++)
     *to++ = *from++;
   *to = ' ';
}

/*
 * PadInsertLine:
 * Insert a line, by moving all of the lines below this one
 * down, and putting the last one in the current position
 * and clearing it.
 * This routine leaves CurLine is unchanged; column set to 0;
 */

PadInsertLine(pad)
    register struct TtyPadRec *pad;
{
  register DisplayRecord *sdf;
  register char *temp, *last;
  int i;
  
   QuickCursorOff(pad);
   sdf = pad->curItem;
   if (pad->yMaxChanged < sdf->ymax) 
       pad->yMaxChanged = sdf->ymax;
   last = (char *)sdf->leftChild;
   for (i=pad->curLine; i<pad->length; i++)
     {
       sdf = sdf->rightSib;
       temp = (char *)sdf->leftChild;
       sdf->leftChild = (DisplayRecord *)last;
       last = temp;
     }
    if (pad->yMinChanged > sdf->ymin)
    	pad->yMinChanged = sdf->ymin;
   pad->curItem->leftChild = (DisplayRecord *)last;
   for (i=0; i<=pad->width+1; i++)
     *last++ = ' ';
   PadReturn(pad);
}

/*
 * PadDeleteLine:
 * Delete a line, by moving all of the following lines
 * up to their previous item, and erasing the last line
 * This routine leaves CurLine and CurColumn unchanged.
 */

PadDeleteLine(pad)
    register struct TtyPadRec *pad;
{
  register char *temp;
  register DisplayRecord *sdf;
  int i;
  
   QuickCursorOff(pad);
   sdf = pad->curItem;
   if (pad->yMaxChanged < sdf->ymax) 
       pad->yMaxChanged = sdf->ymax;
   temp = (char *)sdf->leftChild;
   for (i=pad->curLine; i<pad->length; i++)
     {
       sdf->leftChild = sdf->rightSib->leftChild;
       sdf = sdf->rightSib;
     }
   sdf->leftChild = (DisplayRecord *)temp;
    if (pad->yMinChanged > sdf->ymin)
    	pad->yMinChanged = sdf->ymin;
   for (i=0; i<=pad->width; i++)
     *temp++ = ' ';
   PadReturn(pad);
}


/*
 * PadIndex:
 * Scroll the indicated region up the screen.  This is simulated
 * with line inserts and deletes.
 * This routine leaves CurLine and CurColumn unchanged.
 */

PadIndex(pad, top, bot)
    register struct TtyPadRec *pad;
    short top, bot;
{
  short col, line;

   col = pad->curColumn;
   line = pad->curLine;
   PadPosition(pad,top,0);
   PadDeleteLine(pad);
   PadPosition(pad,bot,0);
   PadInsertLine(pad);
   PadPosition(pad,line,col);
}

/*
 * PadReverseIndex:
 * move the indicated region down on the screen, by deleting and
 * inserting a line.
 * This routine leaves CurLine and CurColumn unchanged.
 */

PadReverseIndex(pad, top, bot)
    register struct TtyPadRec *pad;
    short top, bot;
{
  short col, line;

   col = pad->curColumn;
   line = pad->curLine;
   PadPosition(pad,bot,0);
   PadDeleteLine(pad);
   PadPosition(pad,top,0);
   PadInsertLine(pad);
   PadPosition(pad,line,col);
}


/*
 * PadBlinkLine:
 * invert a line by Xoring the 0200 bit on each character
 */

PadBlinkLine(vgt)
    short vgt;
{
  register struct TtyPadRec *pad = TtyPadList + vgt;
  register char *p;
  register int i;
  
  if ((Debug & DebugVgtsPads) && pad->curItem->type != SDF_InternalText)
      dprintf( "Bad type in pad blink line, %d ", pad->curItem->type);
  p = (char *)(pad->curItem->leftChild);
  for (i=0; i<pad->width; i++)
    *p++ ^= 0200;
   PadRedrawLine(vgt);
}

PadRedrawLine(vgt)
  {
    register struct TtyPadRec *pad = TtyPadList + vgt;    

    RedrawVGT( vgt, LeftMargin, 3000, 
       pad->curItem->ymin, pad->curItem->ymax );
  }



/*******************************************************************/


/*
 * PadNewLine:
 * Helper routine which advances to the same column in the next line,
 * scrolling if necessary.
 */

PadNewLine(pad)
  register struct TtyPadRec *pad;
{
  register DisplayRecord *sdf;
  int saveCol;

   if (pad->curLine == pad->regionBot && pad->regionBot < pad->length)
     {
       PadIndex(pad, pad->regionTop, pad->regionBot);
       pad->newlinesLeft--;
       return;
     }

   QuickCursorOff(pad);
   if (++pad->curLine > pad->length)
    {
      int i;

      pad->curLine--;
      saveCol = pad->curColumn;
      sdf = pad->sdfPtr->rightSib;
      pad->cursorPos = (char *)sdf->leftChild;
      if (pad->yMaxChanged < sdf->ymax)
    	  pad->yMaxChanged = sdf->ymax;
      for (i=0; i<pad->length; i++)
        {
	  if ((Debug & DebugVgtsPads) && sdf->type != SDF_InternalText)
	   {
	     dprintf( "Bad type Scrolling\n");
	     SdfTrace(pad->sdfPtr, 0 );
	   }
	  sdf->leftChild = sdf->rightSib->leftChild;
	  sdf = sdf->rightSib;
	}
      if (pad->yMinChanged > sdf->ymin)
    	  pad->yMinChanged = sdf->ymin;
      pad->curItem->leftChild = (DisplayRecord *)pad->cursorPos;
      PadReturn(pad);
      PadClearToEOL(pad);
      pad->curColumn = saveCol;
    }
  else 
    {
      if (pad->curItem==NULL) printf("NULL CurItem  in PadNewLine\n");
      if ((Debug & DebugVgtsPads) && 
	  pad->curItem->rightSib->type != SDF_InternalText)
          dprintf( "Bad type in pad NewLine, %d ", pad->curItem->type);
      pad->curItem = pad->curItem->rightSib;
    }
  pad->cursorPos = (char *)(pad->curItem->leftChild) + pad->curColumn;
  pad->newlinesLeft--;
}

/*******************************************************************/


/*
 * PadReturn:
 * Helper routine which returns to the beginning of the line.
 */

PadReturn(pad)
  register struct TtyPadRec *pad;
{
   QuickCursorOff(pad);
   if (pad->curItem==NULL) printf("NULL CurItem  in PadReturn\n");
   pad->cursorPos = (char *)(pad->curItem->leftChild);
   if (pad->cursorPos==NULL) printf("NULL CursorPos in PadReturn\n");
   pad->curColumn = 0;
}



/*******************************************************************/


/*
 * PadCursorForward:
 * Helper routine which moves the cursor forward.
 */

PadCursorForward(pad)
  register struct TtyPadRec *pad;
{
   QuickCursorOff(pad);
   if (pad->curColumn > pad->width) return;
   pad->cursorPos++;
   pad->curColumn++;
}



/*
 * PadCursorUp:
 * Helper routine which moves the cursor up.
 */

PadCursorUp(pad)
    register struct TtyPadRec *pad;
{
   QuickCursorOff(pad);
   if (pad->curLine<=0) return;
   PadPosition( pad, pad->curLine-1, pad->curColumn);
}


/* 
 * PadCursorDown:
 * Helper routine which moves the cursor down.
 */

PadCursorDown(pad)
    register struct TtyPadRec *pad;
{
    QuickCursorOff(pad);
    if (pad->curLine >= pad->length) return;
    PadPosition( pad, pad->curLine+1, pad->curColumn);
}


PadCursorBackward(pad)
   register struct TtyPadRec *pad;
  {
    QuickCursorOff(pad);
    if (pad->curColumn > 0)
      {
        pad->cursorPos--;
	pad->curColumn--;
      }
  }


PadCursorOff(vgt)
  {
   register struct TtyPadRec *pad = TtyPadList + vgt;

    QuickCursorOff(pad);
  }


static QuickCursorOff(pad)
   register struct TtyPadRec *pad;
  {
  	/*
	 * turns the cursor off if it is on.
	 */

    if (pad->cursorPos==NULL || pad->curItem==NULL) return;
    if (pad->cursorFlag) *pad->cursorPos ^= 0200;
    pad->cursorFlag = 0;
    if (pad->yMaxChanged < pad->curItem->ymax)
    	pad->yMaxChanged = pad->curItem->ymax;
    if (pad->yMinChanged > pad->curItem->ymin)
    	pad->yMinChanged = pad->curItem->ymin;
  }


PadCursorOn(vgt)
  {
   register struct TtyPadRec *pad = TtyPadList + vgt;
  	/*
	 * turns the cursor on if it is off.
	 */
    if (pad->cursorPos==NULL || pad->curItem==NULL) return;
    if (pad->cursorFlag==0) 
      {
        *pad->cursorPos ^= 0200;
        if (pad->yMaxChanged < pad->curItem->ymax)
        	pad->yMaxChanged = pad->curItem->ymax;
        if (pad->yMinChanged > pad->curItem->ymin)
        	pad->yMinChanged = pad->curItem->ymin;
      }
    pad->cursorFlag = 1;
  }


PadCursorBlink(vgt)
  {
  	/*
	 * turns the cursor on if it is off, off if on.
	 */
   register struct TtyPadRec *pad = TtyPadList + vgt;

    if (pad->cursorFlag==0) pad->cursorFlag=1;
    else pad->cursorFlag=0;;
    *pad->cursorPos ^= 0200;
    RedrawVGT( vgt, LeftMargin,
	(pad->width+2)*FontWidth + LeftMargin,
    	pad->curItem->ymin, pad->curItem->ymax );
  }


PadRedraw(vgt)
  {
  	/*
	 * redraw those parts of the pad that have changed
	 */
   register struct TtyPadRec *pad = TtyPadList + vgt;
   
   if (pad->yMinChanged < pad->yMaxChanged)
       RedrawVGT( vgt, LeftMargin,
       (pad->width+2)*FontWidth + LeftMargin,
    	pad->yMinChanged, pad->yMaxChanged );
    pad->yMinChanged = MaxCoord;
    pad->yMaxChanged = -MaxCoord;
  }


/*
 * PadOp:
 * perform the indicated operation a number of times.
 * That is, assume the arg[0] is an operation count.
 */
PadOp( op, pad )
    int (*op)();
    register struct TtyPadRec *pad;
  {
    int i = pad->args[0];
    
    if (i==0) i=1;
    while (i--)
        (*op)(pad);
  }


/*
 * PadEscape:
 * Handle an ANSI standard Escape sequence
 */

PadEscape( ch, pad, kind )
     char ch;
     enum EscapeKind kind;
     register struct TtyPadRec *pad;
  {

    BOOLEAN resetNewlinesLeft = TRUE;

     switch (ch)
       {
	 case 'f':
	 case 'H':	if (kind==Ansi)
	 		    PadPosition(pad, 
		 	      pad->args[0]-1, pad->args[1]-1 );	break;
         case 'A':	PadOp( PadCursorUp, pad);		break;
	 case 'B':	PadOp( PadCursorDown, pad);		break;
	 case 'C':	PadOp( PadCursorForward, pad);		break;
	 case 'D':	if (kind==Ansi) 
			    PadOp( PadCursorBackward, pad);		
	 		else
			    PadIndex(pad, pad->regionTop, pad->regionBot );
		    						break;
	 case 'E':	PadOp( PadNewLine, pad);		break;
	 case 'J':	if (pad->args[0]==2) PadHome(pad);
	 		PadClearToEOS(pad);			break;
	 case 'K':	PadClearToEOL(pad);			break;
	 case 'L':	PadOp( PadInsertLine, pad);		break;
	 case 'M':	if (kind==Ansi) 
	 		    PadOp( PadDeleteLine, pad);
	 		else
	  		    PadReverseIndex(pad, pad->regionTop,
			    			 pad->regionBot );
								break;
	 case 'P':	PadOp( PadDeleteChar, pad);		break;
	 case '@':	PadOp( PadInsertChar, pad);		break;
	 case 'm':	if (pad->args[pad->argCount])
	 			pad->standout = 0200;	
			  else	pad->standout = 0;
			  resetNewlinesLeft = FALSE;		break;
	 case 'r':	if (pad->args[0] == 0)
	 		    pad->args[0] = 1;
			if (pad->args[1] == 0)
			    pad->args[1] = pad->length+1;
			if (pad->args[0] >= 1 &&
	 		    pad->args[1] <= pad->length+1 &&
			    pad->args[0] < pad->args[1])
			  {
			    pad->regionTop = pad->args[0]-1;
	 		    pad->regionBot = pad->args[1]-1;
			    pad->pageLength =
			    	pad->regionBot - pad->regionTop;
			  }
			break;
	 case '[':	pad->state = GettingArgs;
	 		resetNewlinesLeft = FALSE;		break;
	 case ')':	pad->state = GettingCharSet;		break;
	 case '(':	pad->state = GettingCharSet;		break;
	 case '<':	/* ANSI mode (default) */		break;
	 
	 case CR:
	 case LF:
         case BS:	/* special Sail graphics */
         case DEL:
         case FF:
         case TAB:
	 case BELL:
	 case SO:
	 case SI:
	 case NULL:
	 case ESC:
    	    if (pad->curColumn > pad->width)
    	        PadContinueLine(pad);
    	    PadDrawChar(pad, ch);
	    PadCursorForward(pad);
	    break;
       }

    if (resetNewlinesLeft) pad->newlinesLeft = pad->pageLength;

  }


PadGetCursor( pad, line, col )
  register struct TtyPadRec *pad;
  short *line, *col;
  {
	/*
	 * Return the character coordinates of the current cursor
	 * position.
	 */

    *line = pad->curLine+1;
    *col = pad->curColumn+1;
  }


PadGetSize( pad, length,  width )
  register struct TtyPadRec *pad;
  short *length, *width;
  {
	/*
	 * Return the length and width of the given pad.
	 *
	 */

    *length = pad->length+1;
    *width = pad->width+1;

  }

