/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* mux.c - Virtual Graphics Terminal Server Master Multiplexor
 *
 *
 * Bill Nowicki February 1983
 *
 * April 1983 (WIN)
 *	- Integrated mouse changes by Kenneth Brooks
 * 
 * Marvin Theimer, 7/83
 *	Added SET_BREAK_PROCESS case to the message handling.
 *	Changed SetBanner into a msg-sent request.
 *
 * Craig Dunwoody July 1983
 * 	Added initialization of lineEditBuf and call to InitEditor in
 *	InitVgts, added calls to CreateBuffer and DestroyBuffer in
 *	HandleCreate, HandleRelease, HandleModify, and SetPadMode.
 *
 * The process structure of the vgts consists of 4 processes: 
 *	the vgts process,
 * 	a mouse device helper process,
 *	a keyboard helper process.
 *	a timer helper process
 * The helper processes block on input from resp. the mouse
 * and the keyboard kernel calls and then send the input to the
 * vgts process.  The vgts process receives messages from both the
 * helper processes and from applications processes who desire 
 * virtual graphics terminal services.
 *
 */

#include <Venviron.h>
#include <Vioprotocol.h>
#include <Vtermagent.h>
#include <Vteams.h>
#include <Vgroupids.h>
#include <Vgtp.h>
#include "sdf.h"
#include "vgt.h"
#include "pad.h"

short Debug = 0;	/* bit vector of flags to print out debug info */
TtyPadType *DebugPad = NULL; /* place to put debug info: NULL means console */

short Unique = 37;	/* my favorite random number */

BOOLEAN GraphicsInputSeen = FALSE; /* True if we've ever seen graphics input */

extern short MouseX, MouseY, MouseButtons;

struct Event *FreeEvents;
ProcessId VgtsPid;	/* Well-known process id for the vgts process. */

/*
 * Number of clock ticks before declaring screen to be idle
 */
#define TicksPerMinute 240
static IdleLimit = 10*TicksPerMinute;
static IdleTime = 0;
static LastTimeBannerUpdate = 0;

#define DateAndTimeLength 24	/* size of VGTS banner */

/*
 * Process stack sizes.
 */
#define HelperSize 1024


/*******************************************************************/

/*
 * InitHelperProcesses:
 * Starts up keyboard and mouse helper processes for Vgts.
 */

#define KeyboardProcessPriority 5
#define MouseProcessPriority 6
#define TimerProcessPriority 10

InitHelperProcesses()
  {
    Message msg;
    ProcessId pid;
    int MouseProcess(), KeyboardProcess(), TimerProcess();

    pid = Create(KeyboardProcessPriority, KeyboardProcess, HelperSize );
    Ready(pid, 0);
    pid = Create(MouseProcessPriority, MouseProcess, HelperSize);
    Ready(pid, 0);
    pid = Create(TimerProcessPriority, TimerProcess, HelperSize);
    Ready(pid, 0);
  }


/*******************************************************************/

short InputVGT = -1;	/* the current input VGT */
short TtyVGT;		/* VGT for view manager interaction */


InitVgts(xmin,ymin,xmax,ymax)
  {
/*
 * InitVgts:
 * Initializes the Vgts multiplexor.
 * Called with the coordinates of the first VGTS view.
 * This process must eventually be the one who calls GetMessage()
 */
    int i;
    register Term1 *term1 = Term1Table;
    register struct Event *e = EventTable;
    SystemCode JoinGroup();
    SystemCode r;
    
    VgtsPid = GetPid(VGT_SERVER,LOCAL_PID);
    if (VgtsPid && ValidPid(VgtsPid))
      {
        printf("You can only run one copy of the VGTS\r\n");
	exit();
      }

    VgtsPid = GetPid(ACTIVE_PROCESS,LOCAL_PID);
    SetPid(VGT_SERVER,VgtsPid,LOCAL_PID);
    ChangeTeamPriority(VgtsPid, REAL_TIME4);
				/* Put the vgts in the least important
				   realtime priority class. */
    r = JoinGroup(LTERMINAL_SERVER_GROUP, VgtsPid);
    if (r != OK)
      {
        fprintf(stderr,
		"VGTS: Couldn't join local terminal server group: %s\n",
		ErrorString(r));
	fflush(stderr);
      }

    InitHelperProcesses();

    for (i = 0; i < MaxVGT; i++, term1++)
      {
	term1->owner = 0;
	term1->reader = 0;
	term1->requestFlag = FALSE;
	term1->eventQ = NULL;
	term1->eventTail = NULL;
	term1->mode = CR_Input+LF_Output+Echo+LineBuffer
		       +PageOutput+PageOutputEnable;
	term1->writerPid = 0;
	term1->termPid = 0;
	term1->master = -1;
	term1->isexec = FALSE;
	term1->lineEditBuf = NULL;
	term1->bytesWritten = 0;
	term1->outputFlag = 0;
      }
 
    for (i = 0; i < MaxEvents; i++, e++)
      {
      		/*
		 * put all of the event descriptors onto the
		 * free event list
		 */
        e->next = FreeEvents;
	FreeEvents = e;
      }

    InitViewPackage();
    InitTtyVGT(xmin,ymin,xmax,ymax);
    Term1Table[TtyVGT].requestFlag = TRUE;
    Term1Table[TtyVGT].owner = VgtsPid;

    ModifyPad(stdin, 0);  /* allows this vgts to be tested under other
			     terminal agents. */
  }


/*******************************************************************/


GetMessage(mouseFlag, keyboardFlag, msg)
  int mouseFlag;	/* true if we return on mouse input */
  int keyboardFlag;	/* true if we retun on keyboard input */
  register IoRequest *msg;	/* message that we return */
   {
	/*
	 * handle a request by dispatching to the correct handler
	 */
    ProcessId pid;
    int index;
    char c;

    while (1)
      {
	pid = Receive(msg);
	if (Debug & DebugVgtsMessages)
	    if ((msg->requestcode != (SystemCode)Timer) &&
	        (msg->requestcode != (SystemCode)Keyboard) &&
	        (msg->requestcode != (SystemCode)GraphicsHelperMsg))
	        dprintf("mux receives message - requestcode = 0x%x\n", 
		        msg->requestcode);
	index = msg->fileid;
	if (msg->requestcode != CREATE_INSTANCE &&
	    msg->requestcode != SET_DEBUG_MODE)
	  {
	  	/*
		 * Do all the file id Validity checking here.
		 * after this point, we can index the table
		 * with impunity.
		 */
	    if (( index < 0 ) || ( index >= MaxVGT ))
	      {
		msg->requestcode = INVALID_FILE_ID;
		Reply( msg, pid);
	  	return;
	      }
	  }
	switch (msg->requestcode)
	  {
	    case QUERY_INSTANCE:
		if ( index != DirectoryInstance )
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleQuery(msg, pid);
	        Reply(msg, pid);
	        break;

	    case CREATE_INSTANCE:
		HandleCreate(msg, pid);
	        Reply(msg, pid);
	        break;

	    case SET_BREAK_PROCESS:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleSetBreak(msg);
		Reply(msg, pid);
		break;

	    case SET_INSTANCE_OWNER:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleSetOwner(msg);
		Reply(msg, pid);
		break;

	    case SetBannerRequest:
		if ( VGTtable[ index ].type == UNUSED )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleBanner(msg, pid);
		Reply(msg, pid);
		break;

	    case RELEASE_INSTANCE:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleRelease(msg);
	        Reply(msg, pid);
	        break;

	    case WRITE_INSTANCE:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
	        HandleWrite(msg, pid);
	        break;

	    case WRITESHORT_INSTANCE:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
	        HandleShortWrite(msg, pid);
	        break;

	    case READ_INSTANCE:
		if (index == DirectoryInstance)
		  HandleDirectoryRead(msg, pid);
		else if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
	        else HandleRead(Term1Table+index,pid);
		break;

	    case QueryPadRequest:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleQueryFile(msg, pid);
	        Reply(msg, pid);
	        break;

	    case ModifyPadRequest:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleModify(msg);
	        Reply(msg, pid);
	        break;

	    case EventRequest:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleEventRequest(Term1Table+index, msg, pid);
		break;

	    case LineEditRequest:	 
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleShoveInput(Term1Table+index, msg, pid);
		break;

	    case RedrawRequest:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		if (Term1Table[index].mode & NoCursor)  PadCursorOff(index);
		  else PadCursorOn(index);
		if (!mouseFlag) PadRedraw(index);
		msg->requestcode = OK;
		Reply( msg, pid);
		break;

	    case SwitchInput:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		msg->requestcode = OK;
		Reply( msg, pid );
		HandleSwitchInput( index, pid );
		break;

	    case GetRawIO:
		if ( ClientTable[ index ].owner == 0 )
		  {
		    msg->requestcode = INVALID_FILE_ID;
		    Reply( msg, pid);
		    return;
		  }
		HandleGetRawIO(msg);
		Reply(msg, pid);
		break; 

	    case SET_DEBUG_MODE:
		HandleDebug(msg);
		Reply(msg, pid);
		break;

	    case Keyboard:
		if (IdleTime > IdleLimit) ScreenEnable(); /* Wake up screen */
		IdleTime = 0;
		msg->requestcode = OK;
		Reply(msg, pid);
		if (keyboardFlag) return(pid);
		c = (char) msg->shortbuffer[0];
		if (!GraphicsInputSeen)
		  {
		    BOOLEAN ConvertKeyToGraphics();

		    if (ConvertKeyToGraphics(msg))
		      {
			if (mouseFlag) return(pid);
			else HandleGraphics(msg);
		      }
		  }
		else UserCook(Term1Table+InputVGT,c);
		break;

	    case GraphicsHelperMsg:
		if (IdleTime > IdleLimit) ScreenEnable(); /* Wake up screen */
		IdleTime = 0;
		msg->requestcode = OK;
		Reply(msg, pid);
		MouseX = ((struct EventMsg *)msg)->x;
		MouseY = ((struct EventMsg *)msg)->y;
        	MouseButtons = ((struct EventMsg *)msg)->buttons;
		if (mouseFlag) return(pid);
	        HandleGraphics(msg);
		break;
		
	    case Timer:
		HandleTimer(mouseFlag);
		msg->requestcode = OK;
		Reply(msg, pid);
		break;

	    case Die: /* a request to commit suicide */
		msg->requestcode = OK;
		Reply(msg, pid);
		DestroyProcess(0);
		break;

	    default:
		if (Debug & DebugVgtsMessages)
			dprintf("Bad request: 0x%x, pid 0x%x\n", 
				msg->requestcode, pid);
		msg->requestcode = ILLEGAL_REQUEST;
		Reply(msg, pid);
		break;
	  }
      }
  }


HandleTimer(mouseFlag)
  {
  	/*
	 * handle clean-up operations for the pad driven
	 * by the timer, and flash the cursor in the input pad
	 */
    register Term1 *term1 = Term1Table;
    register int index;
    int seconds, clicks;

    for (index=0; index<MaxVGT; index++, term1++)
      {
	register Term2 *term2 = term2Ptr(term1);
	static blinkCount = 0;

        if (term1->owner && ValidPid(term1->owner)==0)
	  {
	    IoReply m;
	  	/*
		 * we just encountered a vgt whose owner has died. RIP.
		 */
  	    if (index==InputVGT) MakeNotCurrent(index);
            if (Debug & DebugVgtsEvents) 
		    dprintf("Owner died in vgt %d\n", index );
	    DeleteTextVGT(index, TRUE);
	    FlushEvents(term1);
	    m.replycode = END_OF_FILE;
	    m.bytecount = 0;
	    Reply(&m, term1->reader);
	    m.replycode = INVALID_FILE_ID;
	    Reply(&m, term2->writerPid);
	    term1->owner = 0;
	    term2->isexec = FALSE;
	    if (term1->interp)
		free(term1->interp);
	    term1->interp = NULL;
	    if ( term1->lineEditBuf )
		free( term1->lineEditBuf );
	    term1->lineEditBuf = NULL;
	  }

	if (index==InputVGT && !mouseFlag && 
        	term1->requestFlag &&
		!(term1->mode & NoCursor) &&
		blinkCount++ == 2)
          {
	    PadCursorBlink(index);
	    blinkCount = 0;
          }
	else if (term1->mode & NoCursor)
	    PadCursorOff(index);
	else 
	    PadCursorOn(index);

	if (!mouseFlag)
	    PadRedraw(index);
      }

     /* Turn off the screen if no user events (keyboard or mouse) have
      * occurred in the last several minutes.
      */
    if (IdleTime++ > IdleLimit)
	  ScreenDisable();
    seconds = GetTime( &clicks);
    if ( (seconds % 10) == 0 && seconds != LastTimeBannerUpdate)
      {
        char dateAndTime[32];

	strcpy( dateAndTime, ctime( &seconds ));
	dateAndTime[DateAndTimeLength] = 0;
        SetBanner( TtyVGT, dateAndTime);
	LastTimeBannerUpdate = seconds;
      }

    if (!GraphicsInputSeen) AgeKbdEscSequence(); /* handle timeout of	*/
						 /* escape sequences    */	
						 /* for mouse emulator  */

  }
