/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * mouse.c - Mouse helper process 
 *
 * Bill Nowicki November 1982
 */

#ifdef MC68000
# define MouseGain 2	/* MC68000 factor to multiply the mouse motion */
#else
# define MouseGain 4	/* MVAX factor to multiply the mouse motion */
#endif MC68000

# include <Venviron.h>
# include <Vgts.h>
# include <Vmouse.h>
# include <Vioprotocol.h>
# include "sdf.h"
# include "vgt.h"

extern Process_id VgtsPid; 	/* master multiplexor */

static File *MouseFAD;	/* mouse globals */
short MouseX, MouseY, MouseButtons;
extern short Debug;
extern BOOLEAN GraphicsInputSeen;

/*
 * MouseProcess:
 * It blocks on input from the kernel mouse device, updates the
 * cursor image on the screen, and sends a mouse click message
 * to the vgts process if a mouse button has been clicked.
 */

#ifndef MC68000
/* To make microVAX conform to brain-damaged Sun button numbering */
static char MouseTable[8] = {
    0,				RightButton,
    MiddleButton,		RightButton+MiddleButton,
    LeftButton,			RightButton+LeftButton,
    LeftButton+MiddleButton,	RightButton+LeftButton+MiddleButton};
#define ConvertButtons(x) MouseTable[x&7]
#else
#define ConvertButtons(x) (x)
#endif MC68000

MouseProcess()
  {
    struct EventMsg msg;
    static MouseBlock buffer;
    int size;
    BOOLEAN firstTime = TRUE;	/* mega-kludge:  first Read from mouse may */
				/* not block if mouse not plugged in, so   */
				/* don't believe mouse is there till after */
				/* second Read returns			   */

    if (!InitMouse())
        return;
    /*
     * Initially put the mouse in a reasonable place
     */
    SetMouseTo(100,100);

    while (1)
      {
	msg.requestcode = (SystemCode)GraphicsHelperMsg;
	msg.fileid = 0;
	msg.x = (short) buffer.xcoordinate*MouseGain;
	msg.y = (short) buffer.ycoordinate*MouseGain;
	msg.buttons = ConvertButtons(buffer.buttons); 
	Send( &msg, VgtsPid);

        size = Read(MouseFAD, &buffer, sizeof(MouseBlock));
		/* Block on input from the kernel mouse device. */
        if (size <= 0)
	  {
	    if (MouseFAD->lastexception==END_OF_FILE)
	        while (1) Receive(&msg);  /* Hang if mouse has been released */
	    printf("Error %s on Mouse\n", ErrorString(MouseFAD->lastexception));
	    return;
	  }
	else if (firstTime) firstTime = FALSE;
	else GraphicsInputSeen = TRUE;
      }
  }


/*
 * InitMouse:
 * Initializes the mouse device.
 */

static int InitMouse()
  {
    SystemCode error;
    Message msg;
    File *Open();

    MouseFAD = Open("[device]mouse", FCREATE, &error);

    /*
     * Hang ourselves if we don't have a mouse.
     */
    if (error==NOT_FOUND)
      {
	MouseX = MouseY = 800;
    	while (1) Receive(msg);
      }

    if (error || (MouseFAD == NULL))
      {
	printf("Error %s in Mouse Open\n", ErrorString(error));
	return (0);
      }
    return (1);
  }


/*
 * ReadMouseInput:
 *
 */

ReadMouseInput()
  {
    struct EventMsg msg;

    GetMessage( 1, 0, &msg );
  }

/*
 * GetMouseInput: waits until some button is pressed, then returns as
 * soon as all buttons are released.  This is in contrast with
 * ReadMouseInput above, which returns on an button change.
 * Returns with the button set.
 */

GetMouseInput()
  {
    short buttons = 0;

    WaitForDownButton();
    while (MouseButtons)
      {
	buttons |= MouseButtons;
        CursorUpdate();
	ReadMouseInput();
      }
    return(buttons);
  }

WaitForDownButton()
  {
	/*
	 * Wait until a button goes down, tracking the mouse all the time
	 */
    while (MouseButtons==0)
      {
        CursorUpdate();
	ReadMouseInput();
      }
  }

/*******************************************************************/


SetMouseTo(x,y)
    short x,y;
  {
  	/*
	 * defines the mouse to be at a given location.
	 * This is because the mouse is a relative device,
	 * but we do not want it going too far off the screen.
	 */
    MouseBlock buffer;
    short size;

    buffer.xcoordinate = x/MouseGain;
    buffer.ycoordinate = y/MouseGain;
    buffer.buttons = MouseButtons;
    size = Write(MouseFAD, &buffer, sizeof(MouseBlock));
    if (size != sizeof(MouseBlock))
      {
	printf("Error %s on Mouse Write\n", ErrorString(MouseFAD->lastexception));
	return;
      }
  }
